﻿using System;
using System.IO;
using System.Runtime.InteropServices;
using NaGet.Tasks;
using NaGet.InteropServices;

namespace NaGet.SubCommands.SubTask
{
	public class VirusScanSubTask : NaGetSubTask
	{
		private string targetFilePath = null;
		
		private string targetUrl = null;
		
		private DownloadScannerService scanner = null;
		
		public VirusScanSubTask(DownloadScannerService scanner)
			: this(scanner, null, null)
		{
		}
		
		public VirusScanSubTask(DownloadScannerService scanner, string targetFilePath, string targetUrl)
		{
			this.scanner = scanner;
			this.targetFilePath = targetFilePath;
			this.targetUrl = targetUrl;
		}
		
		public string TargetFilePath {
			get { return targetFilePath; }
			set { targetFilePath = value; }
		}
		
		public string TargetUrl {
			get { return targetUrl; }
			set { targetUrl = value; }
		}
		
		private void runScanFile()
		{
			DownloadScannerResult result;
			try {
				RaiseTaskSetEvent(TaskEventType.PING, string.Empty, -1);
				result = scanner.Scan(targetFilePath, targetUrl);
			} catch (COMException ex) {
				result = DownloadScannerResult.ScannerNotFound;
				RaiseTaskSetEvent(TaskEventType.WARNING, string.Format("{0} (E{1})", ex.Message, ex.ErrorCode), 0);
			} catch (FileNotFoundException) {
				result = DownloadScannerResult.ErrorNotFound;
			} catch (Exception) {
				result = DownloadScannerResult.ScannerNotFound;
			}
			
			Exception e = null;
			switch (result) {
				case DownloadScannerResult.OK:
					RaiseTaskSetEvent(TaskEventType.PING, string.Empty, 100);
					break;
				case DownloadScannerResult.ScannerNotFound:
					RaiseTaskSetEvent(TaskEventType.INFO, "ダウンロードしたファイルへのウイルススキャンはされなかったか、スキャン結果は無視されました", 0);
					break;
				case DownloadScannerResult.InfectedAndCleaned:
					e = new ApplicationException("インストーラーファイルからウイルス感染が検出されたため、削除されました。");
					break;
				case DownloadScannerResult.InfectedButNotCleaned:
					e = new ApplicationException("インストーラーファイルからウイルス感染が検出されました。");
					break;
				case DownloadScannerResult.ErrorNotFound:
					e = new FileNotFoundException(string.Empty);
					break;
			}
			
			if (e != null) {
				throw e;
			}
		}
		
		public override void Run()
		{
			NotifyStarted();
			RaiseTaskSetEvent(TaskEventType.STARTED, string.Format("ウイルススキャン: {0}", targetFilePath), 0);
			
			try {
				if (! NaGet.Env.EnableScanInstallerFile) {
					RaiseTaskSetEvent(TaskEventType.INFO, "ウイルススキャンを行わない設定のため、ダウンロードしたファイルはウイルススキャンされませんでした", 0);
				} else if (!scanner.HasScanner) {
					RaiseTaskSetEvent(TaskEventType.INFO, "ダウンロードしたファイルはウイルススキャンされませんでした（ウイルススキャンソフトが検出できませんでした）", 0);
				} else if (! File.Exists(targetFilePath)) {
					RaiseTaskSetEvent(TaskEventType.ERROR, "ダウンロードしたファイルは存在しませんでした（ウイルススキャンソフトが自動的に即座に削除した可能性があります）", 0);
				} else {
					runScanFile();
				}
			} finally {
				RaiseTaskSetEvent(TaskEventType.COMPLETED, string.Format("ウイルススキャン: {0}", targetFilePath), 0);
				NotifyCompleted();
			}
		}
	}
}
