﻿using System;
using System.Drawing;
using System.Windows.Forms;
using System.Text;
using System.Text.RegularExpressions;
using System.IO;
using NaGet.Packages.Install;
using NaGet.Packages;

namespace AppliStation.PackageInfo
{
	/// <summary>
	/// Description of PackageInfoForm.
	/// </summary>
	public partial class PackageInfoForm : Form
	{
		private Package pkg;
		
		public PackageInfoForm()
		{
			InitializeComponent();
		}
		
		/// <summary>
		/// 現在選択中(=表示中)のパッケージ
		/// </summary>
		public Package SelectedObject {
			get { return pkg; }
			set {
				pkg = value;
				updateSelectedObject();
			}
		}
		
		private void updateSelectedObject()
		{
			InstalledPackage instPkg = pkg as InstalledPackage;
			bool isInstalledPackage = instPkg != null;
			
			iconLabel.Image = null;
			iconLabel.Size = Size.Empty;
			if (isInstalledPackage) {
				Icon icon = AppliStation.Util.GUIUtils.GetIconForPackage(instPkg);
				if (icon != null) {
					iconLabel.Image = icon.ToBitmap();
					iconLabel.Size = icon.Size;
				}
			}
			nameLabel.Text = string.Format("{0} ({1})", pkg.Name, pkg.Version);
			nameLabel.Font = new Font(SystemFonts.CaptionFont.FontFamily, SystemFonts.CaptionFont.Size*1.25f);
			
			summaryTextBox.Text = pkg.Summary;
			
			if (isInstalledPackage) {
				UninstallInformation uninstInfo = instPkg.UninstallInfo;
				
				string instPath = null;
				if ((pkg.Type == InstallerType.ARCHIVE)
				    ||(pkg.Type == InstallerType.ITSELF)) {
					instPath = Path.Combine(NaGet.Env.ArchiveProgramFiles, pkg.Name);
				} else if (Directory.Exists(uninstInfo.InstallLocation)) {
					instPath = uninstInfo.InstallLocation;
				}
				
				statusLinkLabel.Links.Clear();
				if (string.IsNullOrEmpty(instPath)) {
					statusLinkLabel.Text = "情報が提供されていません";
					statusLinkLabel.Enabled = false;
				} else {
					statusLinkLabel.Text = instPath;
					statusLinkLabel.Enabled = true;
					statusLinkLabel.Links.Add(0, instPath.Length, instPath);
				}
			}
			urlLinkLabel.Links.Clear();
			if (pkg.Url != null) {
				urlLinkLabel.Text = pkg.Url.Href;
				urlLinkLabel.Links.Add(0, pkg.Url.Href.Length, pkg.Url.Href);
				urlLinkLabel.Enabled = true;
				urlLabel.Enabled = true;
			} else {
				urlLinkLabel.Text = string.Empty;
				urlLinkLabel.Enabled = false;
				urlLabel.Enabled = false;
			}
			tagsTextBox.Text = pkg.Tags;
			setLinkLabelFromLicense(licenseLinkLabel, pkg);
			typeValueLabel.Text = PackageDescripter.GetInstallerTypeLabel(pkg.Type);
			if (! string.IsNullOrEmpty(pkg.PackageListName)) {
				packageListNameValueLabel.Text = pkg.PackageListName;
				packageListNameValueLabel.Enabled = true;
			} else {
				packageListNameValueLabel.Enabled = false;
			}
			
			tabControl1.TabPages.Clear();
			if (isInstalledPackage && string.IsNullOrEmpty(pkg.Version)) {
				tabControl1.Visible = false;
			} else {
				tabControl1.Visible = true;
				foreach (Installer inst in pkg.Installer) {
					TabPage page = new TabPage(string.Format("インストーラー その{0}", tabControl1.TabCount+1));
					page.Dock = DockStyle.Fill;
					
					InstallerInfoForm instForm = new InstallerInfoForm();
					instForm.SelectedObject = inst;
					instForm.Dock = DockStyle.Fill;
					page.Controls.Add(instForm);
					
					tabControl1.TabPages.Add(page);
				}
			}
			
			this.Size = new Size(this.Size.Width, (tabControl1.TabCount>0)? 380 : 250);
		}
		
		/// <summary>
		/// インストーラーの種類についてラベルにテキストを設定する
		/// </summary>
		/// <param name="typeValueLabel">設定先のラベル</param>
		/// <param name="instType">インストーラーの種類</param>
		private static void setLinkLabelFromLicense(LinkLabel linklabel, Package pkg)
		{
			linklabel.Enabled = true;
			
			string text;
			LinkLabel.Link[] links;
			PackageDescripter.GetLicenseValueLabel(pkg, out text, out links);
			
			linklabel.Text = text;
			linklabel.Links.Clear();
			foreach (LinkLabel.Link link in links) {
				linklabel.Links.Add(link);
			}
		}
		
		private void linkLabelLinkClicked(object sender, LinkLabelLinkClickedEventArgs e)
		{
			string linkData = e.Link.LinkData.ToString();
			
			if (Directory.Exists(linkData)) {
				e.Link.Visited = true;
				System.Diagnostics.Process.Start(e.Link.LinkData.ToString());
			} else if (System.Text.RegularExpressions.Regex.IsMatch(linkData, @"^(https?|file)://")) {
				e.Link.Visited = true;
				try {
					System.Diagnostics.Process.Start(e.Link.LinkData.ToString());
				} catch (System.ComponentModel.Win32Exception) {
					MessageBox.Show(string.Format("{0}を開くのに失敗しました。", e.Link.LinkData), "ブラウザ起動エラー", MessageBoxButtons.OK, MessageBoxIcon.Error);
				}
			}
		}
		
		void CloseButtonClick(object sender, EventArgs e)
		{
			if (! this.Modal) {
				this.DialogResult = DialogResult.OK;
				this.Close();
			}
		}
		
		void PackageInfoFormFormClosed(object sender, FormClosedEventArgs e)
		{
			this.Dispose(true);
		}
	}
}
