﻿using System;
using System.Reflection;
using System.Xml.Serialization;

namespace NaGet.Packages
{
	/// <summary>
	/// PlatformのOSの種類をあらわす
	/// </summary>
	public enum PlatformOSType : ushort
	{
		WIN95 = 40,
		WIN98 = 41,
		WINME = 42,
		
		WINNT4 = 140,
		WIN2K = 150,
		WINXP = 151,
		WIN2003 = 152,
		VISTA = 160,
		WIN7 = 161,
	}
	
	public class Platform
	{
		/// <summary>
		/// ソフトの動作するアーキテクチャ
		/// </summary>
		[XmlAttribute]
		public ProcessorArchitecture Arch = ProcessorArchitecture.X86;
		
		/// <summary>
		/// 動作するOSの種類の配列
		/// </summary>
		[XmlIgnore]
		public PlatformOSType[] OsType;
		
		/// <summary>
		/// OsTypeの文字列表現
		/// </summary>
		[XmlAttribute]
		public string Os {
			get {
				if (OsType == null) return null;
				string[] strs = new string[OsType.Length];
				for (int i = 0; i < OsType.Length; i++) {
					strs[i] = OsType[i].ToString("G");
				}
				return string.Join(",", strs);
			}
			set {
				string[] strs = (value ?? string.Empty).Split(',');
				System.Collections.Generic.List<PlatformOSType> list = new System.Collections.Generic.List<PlatformOSType>();
				for (int i = 0; i < strs.Length; i++) {
					try {
						list.Add((PlatformOSType) Enum.Parse(typeof(PlatformOSType), strs[i], true));
					} catch (ArgumentException) {
					}
				}
				list.Sort();
				OsType = list.ToArray();
			}
		}
		
		/// <summary>
		/// 現在のプラットホームで動くか否か?
		/// </summary>
		/// <returns>動く場合はtrue</returns>
		public bool IsRunnable()
		{
			return IsRunnableArch() && IsRunnableOS();
		}
		
		/// <summary>
		/// 現在のマシンのアーキテクチャで動くか否か?
		/// </summary>
		/// <returns>動く場合はtrue</returns>
		public bool IsRunnableArch()
		{
			return Arch == GetArch() ||
				Arch == ProcessorArchitecture.None ||
				Arch == ProcessorArchitecture.MSIL;
		}
		
		/// <summary>
		/// 現在のマシンのアーキテクチャで動かないが、Wow64で動くか否か?
		/// 64ビット環境でない場合は常にfalse
		/// </summary>
		/// <returns></returns>
		public bool IsRunnableArchOnWow64()
		{
			if (IntPtr.Size == 8) {
				return Arch == ProcessorArchitecture.X86;
			} else {
				return false;
			}
		}
		
		/// <summary>
		/// 現在のマシンのアーキテクチャを得る
		/// </summary>
		/// <returns>現在のマシンのアーキテクチャ</returns>
		public static ProcessorArchitecture GetArch()
		{
			Module[] moduleArray = Assembly.GetExecutingAssembly().GetModules();
			Module md = moduleArray[0];

			PortableExecutableKinds pekinds;
			ImageFileMachine ifm;
			md.GetPEKind(out pekinds, out ifm);
			
			switch (ifm) {
				case ImageFileMachine.AMD64:
					return ProcessorArchitecture.Amd64;
				case ImageFileMachine.I386:
					return (IntPtr.Size == 4)? ProcessorArchitecture.X86 : ProcessorArchitecture.Amd64;
				case ImageFileMachine.IA64:
					return ProcessorArchitecture.IA64;
				default:
					return ProcessorArchitecture.None;
			}
		}
		
		/// <summary>
		/// 現在のマシンで動くかを返す
		/// </summary>
		/// <returns>現在のマシンで動くか</returns>
		public bool IsRunnableOS()
		{
			if (OsType == null || OsType.Length <= 0) {
				return true; // 記述なしはOS全部で動く扱い
			}
			
			PlatformOSType? thisOs = GetOSType();
			return thisOs != null && Array.BinarySearch(OsType, (PlatformOSType) thisOs) >= 0;
		}
		
		/// <summary>
		/// 現在のマシンのOSを得る
		/// </summary>
		/// <returns>現在のマシンのOS</returns>
		public static PlatformOSType? GetOSType()
		{
			OperatingSystem os = Environment.OSVersion;
			Version osVer = os.Version;
			
			switch (os.Platform) {
				case PlatformID.Win32Windows:
					if (osVer.Major == 4) {
						switch (osVer.Minor) {
							case 4:
								return PlatformOSType.WIN95;
							case 10:
								return PlatformOSType.WIN98;
							case 90:
								return PlatformOSType.WINME;
						}
					}
					break;
				case PlatformID.Win32NT:
					if (osVer.Major == 4) {
						return PlatformOSType.WINNT4;
					} else if (osVer.Major == 5) {
						switch (osVer.Minor) {
							case 0:
								return PlatformOSType.WIN2K;
							case 1:
								return PlatformOSType.WINXP;
							case 2:
								return PlatformOSType.WIN2003;
						}
					} else if (osVer.Major == 6) {
						switch (osVer.Minor) {
							case 0:
								return PlatformOSType.VISTA;
							case 1:
								return PlatformOSType.WIN7;
						}
					}
					break;
//				case PlatformID.WinCE:
//					return null;
//				case PlatformID.Unix:
//					return null;
			}
			
			return null;
		}
	}
}
