﻿// project created on 2007/09/08 at 20:20
using System;
using System.IO;
using System.Collections.Generic;
using NaGet.Packages;
using NaGet.Packages.Install;
using NaGet.SubCommands;
using NaGet.Tasks;

namespace AllGet
{
	sealed class TaskSetEventHandlers
	{
		public static void OnTaskSetEvent(object sender, TaskEventArgs e) { 
			switch (e.Type) {
//				case NaGetTaskSetEventType.COMPLETED_TASKSET
//					break;
				case TaskEventType.COMPLETED:
					Console.WriteLine("Done.");
					break;
				case TaskEventType.STARTED_SUBTASK:
					Console.Write("  " + e.TaskMessage);
					break;
				case TaskEventType.COMPLETED_SUBTASK:
					Console.WriteLine(" ... Done. [{0}%]", (int) e.ProgressPercent);
					break;
				case TaskEventType.INFO:
					Console.WriteLine("  " + e.TaskMessage);
					break;
				case TaskEventType.ERROR:
				case TaskEventType.WARNING:
					Console.WriteLine("  [Error]" + e.TaskMessage);
					break;
			}
		}
		
		public static void OnDownloadSubTaskEvent(object sender, TaskEventArgs e)
		{
			if (sender is NaGet.SubCommands.SubTask.DownloadSubTask) {
				if (e.Type == TaskEventType.STARTED) {
					Console.WriteLine();
				}
				
				int origPosX = Console.CursorLeft;
				for (int i = 0; i < Console.WindowWidth - 1; i++) {
					Console.Write(' ');
				}
				Console.CursorLeft = origPosX;
				
				switch (e.Type) {
					case TaskEventType.PING:
						{
							int origPos = Console.CursorLeft;
							Console.Write(e.TaskMessage);
							Console.CursorLeft = origPos;
						}
						break;
					case TaskEventType.WARNING:
					case TaskEventType.ERROR:
						Console.WriteLine("  [Error]" + e.TaskMessage);
						break;
					default:
						Console.WriteLine(e.TaskMessage);
						break;
				}
			}
		}
		
		public static NaGetTaskQueryResult OnTaskQueryEvent(object sender, NaGetTaskQueryArgs e)
		{
			char result = '\u0000';
			if (e.SelectionFlag == (NaGetTaskQueryResult.CONTINUE | NaGetTaskQueryResult.RETRY | NaGetTaskQueryResult.CANCEL)) {
				result = AllGet.Util.Query(e.Message + " [y/R/n]?", "yrn", 'r');
			} else if (e.SelectionFlag == (NaGetTaskQueryResult.RETRY | NaGetTaskQueryResult.CANCEL)) {
				result = AllGet.Util.Query(e.Message + " [R/n]?", "rn", 'r');
			} else if (e.SelectionFlag == NaGetTaskQueryResult.CONTINUE) {
				result = AllGet.Util.Query(e.Message + " [Y]?", "y", 'y');
			} else {
				result = AllGet.Util.Query(e.Message + " [Y/n]?", "yn", 'y');
			}
			
			switch (result) {
				case 'y':
					return NaGetTaskQueryResult.CONTINUE;
				case 'n':
					return NaGetTaskQueryResult.CANCEL;
				case 'r':
					return NaGetTaskQueryResult.RETRY;
				default:
					return NaGetTaskQueryResult.CANCELED_AUTOMATICALLY;
			}
		}
	}
	
	
	class MainClass
	{
		private PackageListsManager pkgListMan;
		
		/// <summary>
		/// サイレントインストールをするか否か
		/// </summary>
		private bool silent = false;
		
		public MainClass()
		{
			pkgListMan = new PackageListsManager();
			pkgListMan.LoadPackageLists();
		}
		
		public void Update()
		{
			update(true);
		}
		
		public void LocalUpdate()
		{
			update(false);
		}
		
		public void update(bool downloadPackageListFlag)
		{
			NaGet.SubCommands.NaGetUpdate2 tasks = new NaGet.SubCommands.NaGetUpdate2(pkgListMan, downloadPackageListFlag);
			tasks.TaskEventRaised += TaskSetEventHandlers.OnTaskSetEvent;
			tasks.SubTaskEventRaised += TaskSetEventHandlers.OnDownloadSubTaskEvent;
			
			tasks.Run();
		}
		
		public void CheckUpgrade()
		{
			foreach (Package pkg in UpgradeFinder.GetUpgradePackages(pkgListMan)) {
				bool isSystem = false;
				Package instPkg = null;
				
				instPkg = pkgListMan.SystemInstalledPkgList.GetPackageForName(pkg.Name);
				if (instPkg != null) {
					isSystem = true;
				} else {
					instPkg = pkgListMan.InstalledPkgList.GetPackageForName(pkg.Name);
					if (instPkg != null) {
						isSystem = false;
					} else {
						System.Diagnostics.Debug.Fail("internal error.");
					}
				}
				
				Console.WriteLine("{0} ({1}){3} => ({2})", pkg.Name, instPkg.Version, pkg.Version, (isSystem)? "@sys" : "");
			}
		}
		
		public void Search(string keys)
		{
			foreach(Package package in pkgListMan.AvailablePkgList.Search(keys)) {
				Console.WriteLine("{0} ({1}) - {2}", package.Name, package.Version, package.Summary);
			}
			
			foreach(InstalledPackage package in pkgListMan.InstalledPkgList.Search(keys)) {
				Console.WriteLine("{0} ({1})[@install] - {2}", package.Name, package.Version, package.Summary);
			}
			
			foreach(InstalledPackage package in pkgListMan.SystemInstalledPkgList.Search(keys)) {
				Console.WriteLine("{0} ({1})[@sys] - {2}", package.Name, package.Version, package.Summary);
			}
		}
		
		public void Show(string packagename)
		{
			PackageList<Package> allPkgs = new PackageList<Package>();
			allPkgs.AddPackages(pkgListMan.GetAllPackages());
			
			foreach (Package pkg in allPkgs.GetPackagesForName(packagename)) {
				Console.WriteLine("Package: {0}", pkg.Name);
				Console.WriteLine("Version: {0}", pkg.Version);
				Console.WriteLine("Summary: {0}", pkg.Summary);
				if (pkg.Url != null) Console.WriteLine("Website: {0}", pkg.Url.Href);
				if (pkg.Tags != null) Console.WriteLine("Tag: {0}", pkg.Tags.ToLower());
				Console.WriteLine("Type: {0}", pkg.Type);
				if (pkg.License != null) Console.WriteLine("License: {0}", pkg.License);
				if (pkg is InstalledPackage) Console.WriteLine("State: Installed");
				Console.WriteLine("Repository: {0}", pkg.PackageListName);
				
				if (pkg.Description != null) {
					Console.WriteLine("Description:");
					Console.WriteLine(pkg.Description);
				}
				Console.WriteLine();
			}
		}
		
		
		public void Download(string[] packagenames)
		{
			Installation[] installations = null;
			{
				List<Package> downloadList = new List<Package>();
			
				foreach(string packagename in packagenames) {
					Package foundPackage = pkgListMan.AvailablePkgList.GetPackageForName(packagename);
					if (foundPackage == null) {
						Console.WriteLine("E: Couldn't find package "+packagename);
						Environment.Exit(1);
					}
					if (! downloadList.Contains(foundPackage)) {
						downloadList.Add(foundPackage);
					}
				}
				
				installations = new Installation[downloadList.Count];
				for (int i = 0; i < installations.Length; i++) {
					installations[i] = new Installation(downloadList[i]);
				}
			}
			
			Console.WriteLine("The following packages will be downloaded:");
			Console.WriteLine("  {0}", Installation.ToString(installations));
			if (AllGet.Util.Confirm("Do you want to continue [Y/n]?", true) == false) {
				Console.WriteLine("Abort.");
				Environment.Exit(0);
			}
			
			NaGet.SubCommands.NaGetDownloadToCache2 tasks = new NaGet.SubCommands.NaGetDownloadToCache2(pkgListMan, installations);
			tasks.TaskEventRaised += TaskSetEventHandlers.OnTaskSetEvent;
			tasks.SubTaskEventRaised += TaskSetEventHandlers.OnDownloadSubTaskEvent;
			tasks.TaskQueryRaised += TaskSetEventHandlers.OnTaskQueryEvent;
			
			tasks.Run();
		}
		
		public void Install(string[] packagenames)
		{
			if (! NaGet.Utils.IsAdministrators()) {
				Console.WriteLine("W: you are not administrators!");
			}
			
			Installation[] installations = null;
			{
				List<Package> downloadList = new List<Package>();
			
				foreach(string packagename in packagenames) {
					Package foundPackage = pkgListMan.AvailablePkgList.GetPackageForName(packagename);
					if (foundPackage == null) {
						Console.WriteLine("E: Couldn't find package "+packagename);
						Environment.Exit(1);
					}
					if (! downloadList.Contains(foundPackage)) {
						downloadList.Add(foundPackage);
					}
				}
				
				installations = new Installation[downloadList.Count];
				for (int i = 0; i < installations.Length; i++) {
					installations[i] = new Installation(downloadList[i]);
				}
			}
			
			Installation[] depInstallations;
			{
				Installation[] resolved;
				DependeciesResolver.ResolveInstallations(installations, pkgListMan, out resolved, out depInstallations);
				installations = resolved;
			}
			
			Console.WriteLine("The following packages will be downloaded:");
			Console.WriteLine("  {0}", Installation.ToString(installations));
			if (AllGet.Util.Confirm("Do you want to continue [Y/n]?", true) == false) {
				Console.WriteLine("Abort.");
				Environment.Exit(0);
			}
			
			NaGet.SubCommands.NaGetInstall2 tasks = new NaGet.SubCommands.NaGetInstall2(pkgListMan, installations);
			tasks.TaskEventRaised += TaskSetEventHandlers.OnTaskSetEvent;
			tasks.SubTaskEventRaised += TaskSetEventHandlers.OnDownloadSubTaskEvent;
			tasks.TaskQueryRaised += TaskSetEventHandlers.OnTaskQueryEvent;
			
			tasks.Run();
		}
		
		public void Remove(string[] packagenames)
		{
			if (! NaGet.Utils.IsAdministrators()) {
				Console.WriteLine("W: you are not administrators!");
			}
			
			this.LocalUpdate();
			
			PackageList<InstalledPackage> installedPkgList = new PackageList<InstalledPackage>();
			installedPkgList.AddPackages(pkgListMan.GetAllInstalledPackages());
			
			InstalledPackage[] insPkgs = new InstalledPackage[packagenames.Length];
			for (int i = 0; i < packagenames.Length; i++) {
				insPkgs[i] = installedPkgList.GetPackageForName(packagenames[i]);
				
				if (insPkgs[i] == null) {
					Console.WriteLine("E: could not found package " + packagenames[i]);
					Environment.Exit(1);
				}
			}
			
//			if (AllGet.Util.Confirm("Do you want to continue [Y/n]?", true) == false) {
//				Console.WriteLine("Abort.");
//				Environment.Exit(0);
//			}
			
			NaGet.SubCommands.NaGetUninstall2 tasks = new NaGet.SubCommands.NaGetUninstall2(pkgListMan, insPkgs);
			tasks.TaskEventRaised += TaskSetEventHandlers.OnTaskSetEvent;
			
			tasks.Run();
		}
		
		public void CleanCache(string[] packages)
		{
			if (! Directory.Exists(NaGet.Env.ArchiveFolderPath)) {
				return;	
			}
			
			int i = 0;
			if (packages.Length == 0) {
				foreach (string folder in Directory.GetDirectories(NaGet.Env.ArchiveFolderPath)) {
					Directory.Delete(folder, true);
					i ++;
				}
			} else {
				foreach (string package in packages) {
					foreach (string folder in Directory.GetDirectories(NaGet.Env.ArchiveFolderPath, package+"*", SearchOption.TopDirectoryOnly)) {
						Directory.Delete(folder, true);
						i ++;
					}
				}
			}
			if (i > 0) {
				Console.WriteLine("... Done.");
			}
		}
		
		public void Help()
		{
			string executeFileName = System.AppDomain.CurrentDomain.FriendlyName;
			Console.Write("Usage:");
			
			Console.WriteLine("\t{0} update|localupdate", executeFileName);
			Console.WriteLine("\t{0} search|show pkg1 [pkg2 ...]", executeFileName);
			Console.WriteLine("\t{0} cleancache [pkg ...]", executeFileName);
			Console.WriteLine("\t{0} download pkg1 [pkg2 ...]", executeFileName);
			Console.WriteLine("\t{0} [--quiet] install|uninstall pkg1 [pkg2 ...]", executeFileName);
			Console.WriteLine();
			Console.WriteLine("{0} is a simple command line interface for downloading and "+
			                  "installing packages. The most frequently used commands are update "+
			                  "and install.", executeFileName);
			Console.WriteLine();
			Console.WriteLine("Commands:");
			Console.WriteLine("   update - Retrieve new lists of packages");
			Console.WriteLine("   localupdate - Update installed-soft-list only");
			Console.WriteLine("   checkupgrade - Show upgraded-soft list");
			Console.WriteLine("   search - Search the package list for not a regex pattern");
			Console.WriteLine("   show - Show package detail");
			Console.WriteLine("   cleancache - Clear cached archived file(s)");
			Console.WriteLine("   download - Download only - do NOT install or unpack archives");
			Console.WriteLine("   install - Install new packages");
			Console.WriteLine("   remove - Uninstall packages");
			Console.WriteLine();
		}
		
		public void FooBar()
		{
			foreach (InstalledPackage pkg in pkgListMan.SystemInstalledPkgList) {
				Console.WriteLine("{0} : {1}", pkg.Name, pkg.UninstallInfo.UninstallString);
			}
		}
		
		public void Hoge()
		{
			foreach (UninstallInformation uInfo in RegistriedUninstallers.Uninstallers) {
				if (! uInfo.IsOSPatch && ! uInfo.IsSystemComponent) {
					Console.WriteLine("{0}", uInfo.DisplayName);
				}
			}
		}
		
		public void Moo()
		{
			// TODO スーパー牛さんパワー化
			
			string executeFileName = System.AppDomain.CurrentDomain.FriendlyName;
			Console.WriteLine("誰か {0} をスーパー牛さんパワー化してくれ", executeFileName);
		}
		
		[STAThread]
		public static void Main(string[] args)
		{
			// アーカイブSYSTEM32をパスに足す
			NaGet.Utils.AddDirectoryToPath(NaGet.Env.ArchiveSystem32);
			
			MainClass mc = new MainClass();
			
			if (args.Length == 0) {
				mc.Help();
				
				Environment.Exit(1);
			} else {
				try {
					NaGet.ArgParser argParser = new NaGet.ArgParser();
					argParser["quiet"] = false;
					args = argParser.Parse(args);
					mc.silent = (bool) argParser["quiet"];
				} catch (ApplicationException e) {
					Console.WriteLine(e.Message);
					Environment.Exit(1);
				}
			}
			
			// 引数ごとに操作を決定
			switch (args[0])
			{
				case "update":
					if (args.Length != 1) {
						Console.WriteLine("E: The update command takes no arguments");
						Environment.Exit(1);
					}
					mc.Update();
					break;
				case "localupdate":
					if (args.Length != 1) {
						Console.WriteLine("E: The update command takes no arguments");
						Environment.Exit(1);
					}
					mc.LocalUpdate();
					break;
				case "search":
					if (args.Length <= 1) {
						Console.WriteLine("E: You must give exactly one pattern");
						Environment.Exit(1);
					}
					
					mc.Search(string.Join(" ", args, 1, args.Length - 1));
					break;
				case "show":
					if (args.Length <= 1) {
						Console.WriteLine("E: You must give exactly one pattern");
						Environment.Exit(1);
					}
					
					for (int i = 1; i < args.Length; i++) {
						mc.Show(args[i]);
					}
					break;
				case "download":
					if (args.Length <= 1) {
						Console.WriteLine("E: Invalid operation download");
						Environment.Exit(1);
					} else {
						string[] packages = new string[args.Length - 1];
						Array.Copy(args, 1, packages, 0, packages.Length);
						
						mc.Download(packages);
					}
					break;
				case "install":
					if (args.Length <= 1) {
						Console.WriteLine("E: Invalid operation install");
						Environment.Exit(1);
					} else {
						string[] packages = new string[args.Length - 1];
						Array.Copy(args, 1, packages, 0, packages.Length);
						
						mc.Install(packages);
					}
					break;
				case "checkupgrade":
					mc.CheckUpgrade();
					break;
				case "remove":
					if (args.Length <= 1) {
						Console.WriteLine("E: Invalid operation remove");
						Environment.Exit(1);
					} else {
						string[] packages = new string[args.Length - 1];
						Array.Copy(args, 1, packages, 0, packages.Length);
						
						mc.Remove(packages);
					}
					break;
				case "cleancache":
					{
						string[] packages = new string[args.Length - 1];
						Array.Copy(args, 1, packages, 0, packages.Length);
						
						mc.CleanCache(packages);
					}
					break;
				case "foobar":
					mc.FooBar();
					break;
				case "hoge":
					mc.Hoge();
					break;
				case "moo":
					mc.Moo();
					break;
				default:
					mc.Help();
					
					Environment.Exit(1);
					break;
			}
		}
	}
}
