﻿using System;
using System.Drawing;
using System.Windows.Forms;

namespace AppliStation.Util
{
	/// <summary>
	/// Description of OptionDialog.
	/// </summary>
	public partial class OptionDialog : Form
	{
		private object userInputValue = null;
		
		public OptionDialog()
		{
			//
			// The InitializeComponent() call is required for Windows Forms designer support.
			//
			InitializeComponent();
		}
		
		private void SomeButtonClick(object sender, EventArgs e)
		{
			this.Close();
		}
		
		private void HandleUserInput(object sender, EventArgs e)
		{
			TextBoxBase textBox = sender as TextBoxBase;
			ButtonBase button	= sender as ButtonBase;
			
			if (textBox != null) {
				this.userInputValue = textBox.Text;	
			}
			if (button != null) {
				this.userInputValue = button.Tag;
			}
		}
		
		public string MainInstructionText {
			set {
				mainInstLabel.Text = value ?? string.Empty;
				mainInstLabel.Visible = ! string.IsNullOrEmpty(value);
			}
			get {	return mainInstLabel.Text;	}
		}
		public string ContentText {
			set {
				contentLabel.Text = value ?? string.Empty;
				contentLabel.Visible = ! string.IsNullOrEmpty(value);
			}
			get {	return mainInstLabel.Text;	}
		}
		public Image Image {
			set {
				iconPictureBox.Image = value;
				if (value != null) {
					iconPictureBox.Size = new Size(32, 32);
				} else {
					iconPictureBox.Size = new Size(0, 0);
				}
			}
			get {	return iconPictureBox.Image;	}
		}
		public MessageBoxButtons Buttons {
			set {
				okButton.Enabled	 = okButton.Visible		= (value == MessageBoxButtons.OK) || (value == MessageBoxButtons.OKCancel);
				cancelButton.Enabled = cancelButton.Visible	= (value == MessageBoxButtons.OKCancel) || (value == MessageBoxButtons.YesNoCancel);
				yesButton.Enabled	 = yesButton.Visible	= (value == MessageBoxButtons.YesNo) || (value == MessageBoxButtons.YesNoCancel);
				noButton.Enabled	 = noButton.Visible		= (value == MessageBoxButtons.YesNo) || (value == MessageBoxButtons.YesNoCancel);
				
				if (okButton.Enabled) {
					this.AcceptButton = okButton;
				} else if (yesButton.Enabled) {
					this.AcceptButton = yesButton;
				}
			}
			get {
				bool ok = okButton.Enabled;
				bool cancel = cancelButton.Enabled;
				bool yes = yesButton.Enabled;
				bool no = noButton.Enabled;
				
				if (ok && !cancel && !yes && !no) {
					return MessageBoxButtons.OK;
				} else if (ok && cancel && !yes && !no) {
					return MessageBoxButtons.OKCancel;
				} else if (!ok && !cancel && yes && no) {
					return MessageBoxButtons.YesNo;
				} else if (!ok && cancel && yes && no) {
					return MessageBoxButtons.YesNoCancel;
				} else {
					return (MessageBoxButtons) 0xFFFFu;
				}
			}
		}
		public Control.ControlCollection Content {
			get {	return contentFlowLayoutPanel.Controls;	}
		}
		
		public object UserInputValue {
			get { return userInputValue; }
		}
		
		public void BuildRadioButtons(string[] options, int initialOption) {
			this.Content.Clear();
			
			for (int i = 0; i < options.Length; i++) {
				RadioButton optRadios = new RadioButton();
				optRadios.Text = options[i];
				optRadios.Tag = i;
				optRadios.Click += HandleUserInput;
				optRadios.AutoSize = true;
				
				if (i == initialOption) {
					optRadios.Checked = true;
					this.userInputValue = initialOption;
				} else {
					optRadios.Checked = false;
				}
				
				this.Content.Add(optRadios);
			}
		}
		
		public void BuildCommandLinkButtons(string[] options, int initialOption)
		{
			OperatingSystem os = Environment.OSVersion;
			
			this.Content.Clear();
			
			int defaultButtonWidth = this.Content.Owner.Width;
			for (int i = 0; i < options.Length; i++) {
				Button button = null;
				if (os.Platform == PlatformID.Win32NT && os.Version.Major >= 6) {
					button = new CommandLinkButton();
					string[] labelData = options[i].Split(new char[]{';'}, 2);
					button.Text = labelData[0];
					if (labelData.Length > 1 && !string.IsNullOrEmpty(labelData[1])) {
						((CommandLinkButton) button).Note = labelData[1];
						button.Size = new Size(defaultButtonWidth, 58);
					} else {
						button.Size = new Size(defaultButtonWidth, 41);
					}
					
					this.Content.Add(button);
				} else {
					button = new Button();
					string[] labelData = options[i].Split(new char[]{';'}, 2);
					button.Text = labelData[0];
					button.Font = new Font(button.Font.FontFamily, button.Font.Size * 1.25f, FontStyle.Underline);
					button.Size = new Size(defaultButtonWidth, button.Height);
					button.TextAlign = ContentAlignment.MiddleLeft;
					button.FlatStyle = FlatStyle.Flat;
					button.FlatAppearance.BorderSize = 0;
					button.FlatAppearance.MouseDownBackColor = SystemColors.ButtonShadow;
					button.FlatAppearance.MouseOverBackColor = button.BackColor;
					button.ForeColor = SystemColors.HotTrack;
					button.Cursor = Cursors.Hand;
					
					this.Content.Add(button);
					
					if (labelData.Length > 1 && !string.IsNullOrEmpty(labelData[1])) {
						Label label = new Label();
						label.Text = labelData[1];
						label.Margin = new Padding(10, 0, 0, 5);
						label.AutoSize = true;
						label.UseMnemonic = false;
						this.Content.Add(label);
					}
				}
				
				button.Tag = i;
				button.Click += HandleUserInput;
				button.DialogResult = DialogResult.OK;
				
				if (i == initialOption) {
					button.Focus();
					this.userInputValue = initialOption;
				}
			}
			this.Invalidate();
		}
		
		public static OptionDialog createMessageDialog(string message, string title, string mainInstruction, Icon icon)
		{
			return createOptionDialog(message, title, mainInstruction, icon, MessageBoxButtons.OK, null, -1);
		}
		
		public static OptionDialog createConfirmDialog(string message, string title, string mainInstruction, Icon icon, MessageBoxButtons buttons)
		{
			return createOptionDialog(message, title, mainInstruction, icon, buttons, null, -1);
		}
		
		public static OptionDialog createOptionDialog(string message, string title, string mainInstruction, Icon icon, MessageBoxButtons buttons, string[] options, int initialOption)
		{
			OptionDialog dialog = new OptionDialog();
			dialog.Text = title;
			dialog.ContentText = message;
			dialog.MainInstructionText = mainInstruction;
			dialog.Image = (icon != null)? icon.ToBitmap() : null;
			dialog.Buttons = buttons;
			
			if (options != null) {
				dialog.BuildRadioButtons(options, initialOption);
			}
			
			return dialog;
		}
		
		public static OptionDialog createCommandSelectionDialog(string message, string title, string mainInstruction, Icon icon, string[] options, int initialOption)
		{
			OptionDialog dialog = new OptionDialog();
			dialog.Text = title;
			dialog.ContentText = message;
			dialog.MainInstructionText = mainInstruction;
			dialog.Image = (icon != null)? icon.ToBitmap() : null;
			dialog.Buttons = MessageBoxButtons.OKCancel;
			
			if (options != null) {
				dialog.BuildCommandLinkButtons(options, initialOption);
				dialog.okButton.Visible = false;
			}
			
			return dialog;
		}
		
		void OptionDialogStyleChanged(object sender, EventArgs e)
		{
			mainInstLabel.Font = SystemFonts.CaptionFont;
			if (System.Windows.Forms.VisualStyles.VisualStyleInformation.IsEnabledByUser) {
				mainInstLabel.ForeColor = Color.FromArgb(0x003399);
				
				this.BackColor = SystemColors.Window;
				this.ForeColor = SystemColors.WindowText;
				separatorLabel.Visible = true;
			} else {
				mainInstLabel.ForeColor = Color.Empty;
				
				this.BackColor = Color.Empty;
				this.ForeColor = Color.Empty;
				separatorLabel.Visible = false;
			}
		}

		#region CommandLinkButton

		private class CommandLinkButton : Button
		{
			private string note = null;
			
			private bool isCommandLink = false;
			
			public CommandLinkButton()
			{
				this.FlatStyle = FlatStyle.System;
			}
			
			protected override CreateParams CreateParams {
				get {
					OperatingSystem os = Environment.OSVersion;
					
					CreateParams cParams = base.CreateParams;
					// Vista未満はなにもしない
					if (os.Platform == PlatformID.Win32NT && os.Version.Major >= 6) {
						cParams.Style |= 0x0000000E;	// BS_COMMANDLINK
						this.isCommandLink = true;
					} else {
						this.Padding = new Padding(5);
					}
					return cParams;
				}
			}
			
			public string Note {
				set {
					if (this.isCommandLink) {
						this.note = value;
						//SendMessage(hWnd, BCM_SETNOTE, NULL, (PCWSTR)value);
						NativeMethods.SendMessage(this.Handle, 0x00001609, IntPtr.Zero, System.Runtime.InteropServices.Marshal.StringToHGlobalAuto(value));
					}
				}
				get { return this.note; }
			}
			
		}
		
		#endregion
	}
}
