﻿using System;
using System.IO;
using System.Collections.Generic;
using System.Xml.Serialization;

namespace NaGet.Packages
{
	/// <summary>
	/// パッケージのリストの情報を示すクラス
	/// </summary>
	[XmlRoot("PackageList", Namespace="http://diffshare.tv/xmlns/2007/na-get/PackageList/")]
	public class PackageList<TPackage> where TPackage : Package
	{
		/// <summary>
		/// パッケージリスト自体の識別名
		/// </summary>
		public string Name;

		/// <summary>
		/// パッケージの配列のリスト
		/// </summary>
		private List<TPackage> packageArrayList;

		/// <summary>
		/// コンストラクタ
		/// </summary>
		public PackageList()
		{
			packageArrayList = new List<TPackage>();
		}

		/// <summary>
		/// 本リストが格納しているパッケージのリスト
		/// </summary>
		[XmlElement("Package", Namespace="http://diffshare.tv/xmlns/2007/na-get/PackageList/")]
		public TPackage[] Packages
		{
			get
			{
				return packageArrayList.ToArray();
			}
			set
			{
				SetPackages(value);
			}
		}
		
		#region パッケージの追加/削除とイテレーション
		
		/// <summary>
		/// パッケージの中身を書き換える。
		/// </summary>
		/// <param name="pkgs"></param>
		public void SetPackages(IEnumerable<TPackage> pkgs)
		{
			packageArrayList.Clear();
			if (pkgs != null) {
				packageArrayList.AddRange(pkgs);
			}
		}
		
		/// <summary>
		/// 本リストが格納しているパッケージのイテレータを返す
		/// </summary>
		/// <returns>パッケージのイテレータ</returns>
		public IEnumerator<TPackage> GetEnumerator()
		{
			return packageArrayList.GetEnumerator();
		}

		/// <summary>
		/// パッケージをリストに追加する
		/// </summary>
		/// <param name="package">追加するパッケージ</param>
		public void AddPackages(IEnumerable<TPackage> package)
		{
			packageArrayList.AddRange(package);
		}
		
		/// <summary>
		/// パッケージをリストにマージ(追加)する
		/// </summary>
		/// <param name="package">追加するパッケージ</param>
		public void AddPackages(PackageList<TPackage> pkgList)
		{
			packageArrayList.AddRange(pkgList.packageArrayList);
		}
		
		/// <summary>
		/// パッケージをリストに追加する
		/// </summary>
		/// <param name="package">追加するパッケージ</param>
		public void AddPackage(TPackage package)
		{
			packageArrayList.Add(package);
		}
		
		/// <summary>
		/// パッケージをリストから削除
		/// </summary>
		/// <param name="package">削除するパッケージ</param>
		public void RemovePackage(TPackage package)
		{
			packageArrayList.Remove(package);
		}
		
		#endregion
		
		#region パッケージ検索
		
		#region 検索用Predicate
		
		/// <summary>
		/// パッケージを検索をする際のPredicateを返す 
		/// </summary>
		/// <param name="key">検索キー</param>
		protected internal static Predicate<TPackage> GetPredicateForSearch(string key)
		{
			string lowerKey = key.ToLower();
			return delegate(TPackage package) {
				return package.Name.ToLower().IndexOf(lowerKey) >= 0 ||
					package.Summary.ToLower().IndexOf(lowerKey) >= 0 ||
					((package.Tags ?? string.Empty).ToLower().IndexOf(lowerKey) >= 0);
			};
		}
		
		/// <summary>
		/// パッケージ名で検索をする際のPredicateを返す 
		/// </summary>
		/// <param name="name">
		/// パッケージ名
		/// </param>
		/// <returns>
		/// Predicate
		/// </returns>
		protected internal static Predicate<TPackage> GetPredicateForPackageName(string name)
		{
			return delegate(TPackage package) {
				return package.Name == name;
			};
		}
		
		/// <summary>
		/// エントリで検索をする際のPredicateを返す 
		/// </summary>
		/// <param name="entry">
		/// エントリ
		/// </param>
		/// <returns>
		/// Predicate
		/// </returns>
		protected internal static Predicate<TPackage> GetPredicateForEntry(Entry entry)
		{
			return entry.Match;
		}
		
		#endregion
				
		/// <summary>
		/// パッケージを検索して、それにマッチしたパッケージをイテレータとして返す。
		/// </summary>
		/// <param name="predicate">検索条件のPredicate</param>
		/// <returns>マッチしたパッケージのイテレータ</returns>
		protected internal IEnumerable<TPackage> Search(Predicate<TPackage> predicate)
		{
			return packageArrayList.FindAll(predicate);
		}
		
		/// <summary>
		/// パッケージを検索して、それにマッチしたパッケージをイテレータとして返す。
		/// </summary>
		/// <param name="key">検索キー</param>
		/// <returns>マッチしたパッケージのイテレータ</returns>
		public IEnumerable<TPackage> Search(string key)
		{
			if (string.IsNullOrEmpty(key)) {
				return packageArrayList.AsReadOnly();
			} else {
				return Search(GetPredicateForSearch(key));
			}
		}
		
		/// <summary>
		/// 指定した名前に対応するパッケージを返す
		/// </summary>
		/// <param name="name">検索にかけるパッケージ名</param>
		public TPackage GetPackageForName(string name)
		{
			return packageArrayList.Find(GetPredicateForPackageName(name));
		}
		
		/// <summary>
		/// 指定した名前に対応するパッケージを返す
		/// </summary>
		/// <param name="name">検索にかけるパッケージ名</param>
		public IEnumerable<TPackage> GetPackagesForName(string name)
		{
			return packageArrayList.FindAll(GetPredicateForPackageName(name));
		}
		
		/// <summary>
		/// 名前・バージョンが対応するパッケージを返す
		/// </summary>
		/// <param name="name">検索にかけるパッケージ名</param>
		/// <param name="version">検索にかけるパッケージのバージョン</param>
		public TPackage GetPackageForPackage(string name, string version)
		{
			return GetPackageForEntry(new Entry(name, null, version));
		}
		
		/// <summary>
		/// パッケージ参照エントリが対応するパッケージを返す
		/// </summary>
		/// <param name="name">検索にかけるパッケージ参照エントリ</param>
		public TPackage GetPackageForEntry(Entry entry)
		{
			return packageArrayList.Find(GetPredicateForEntry(entry));
		}
		
		/// <summary>
		/// パッケージ参照エントリが対応するパッケージをすべて返す
		/// </summary>
		/// <param name="name">検索にかけるパッケージ参照エントリ</param>
		public IEnumerable<TPackage> GetPackagesForEntry(Entry entry)
		{
			return packageArrayList.FindAll(GetPredicateForEntry(entry));
		}
		
		#endregion
		
		/// <summary>
		/// パッケージリストの各PackageListNameを設定する
		/// </summary>
		/// <param name="pkgList">設定するパッケージが入っているパッケージリスト</param>
		public void FixPackageListName()
		{
			foreach (TPackage pkg in packageArrayList) {
				pkg.PackageListName = this.Name;
			}
		}
	}
}
