﻿using System;
using System.Runtime.InteropServices;
using System.IO;

namespace NaGet.InteropServices
{
	/// <summary>
	/// CommonArchiverDllの設定
	/// </summary>
	public struct CommonArchiverDllConfig
	{
		public CommonArchiverDllConfig(string dllName, string cmdName, string cmdLineFmt, int versionreq)
		{
			DllName = dllName;
			CmdName = cmdName;
			CmdLineFmt = cmdLineFmt;
			VersionMoreThan = versionreq;
		}
		
		/// <summary>
		/// DLLの名称。LoadLibraryに渡される
		/// </summary>
		public string DllName;
		/// <summary>
		/// コマンド名称。
		/// </summary>
		public string CmdName;
		/// <summary>
		/// コマンドライン形式"{0}"がアーカイブ名、"{1}"が展開先
		/// </summary>
		public string CmdLineFmt;
		/// <summary>
		/// 最低バージョン要求
		/// </summary>
		public int VersionMoreThan;
	}
	
	/// <summary>
	/// アーカイバDLLを使った書庫展開器
	/// </summary>
	public class CommonArchiverExtracter
	{
		
		#region DLL関数のdelegate
		
		/// <summary>
		/// DLL の版の取得
		/// </summary>
		private delegate ushort ArcGetVersion();
		
		/// <summary>
		/// DLL の実行状況の取得
		/// </summary>
		private delegate bool ArcGetRunning();
		
		/// <summary>
		/// 書庫のチェック
		/// </summary>
		private delegate bool ArcCheckArchive(string szFileName, ArchCheckFlag iMode);
		
		/// <summary>
		/// 書庫のチェックの精度
		/// </summary>
		private enum ArchCheckFlag {
			/// <summary>
			/// 簡易モード。先頭の数個のヘッダのみ確認
			/// </summary>
			RAPID = 0,
			/// <summary>
			/// 通常モード。すべてのヘッダを確認
			/// </summary>
			BASIC = 1,
			/// <summary>
			/// 厳密モード。CRCも含めすべて確認
			/// </summary>
			FULLCRC = 2,
			//RECOVERY = 4,
			//SFX = 8,
			//ALL = 16,
		}
		
		/// <summary>
		/// 格納ファイル数の取得
		/// </summary>
		private delegate int ArcGetFileCount(string szFileName);
		
		/// <summary>
		/// 書庫操作一般
		/// </summary>
		private delegate int ArcMain(IntPtr hWnd,
		                            [MarshalAs(UnmanagedType.LPStr)] string szCmdLine,
		                            [MarshalAs(UnmanagedType.LPStr)] System.Text.StringBuilder szOutput, uint dwSize);
		
		#endregion
		
		/// <summary>
		/// アーカイバDLLの設定
		/// </summary>
		public static readonly CommonArchiverDllConfig[] Configs = {
			new CommonArchiverDllConfig("7-ZIP32", "SevenZip", "x -y \"{0}\" \"-o{1}\"", 423),
			new CommonArchiverDllConfig("UNZIP32", "UnZip", "-x -o \"{0}\" \"{1}\" *", 541),
			new CommonArchiverDllConfig("UNLHA32", "Unlha", "x \"{0}\" \"{1}\" *", 240),
			new CommonArchiverDllConfig("CAB32", "Cab", "-x \"{0}\" \"{1}\" *", 98),
			new CommonArchiverDllConfig("TAR32", "Tar", "-x \"{0}\" -o \"{1}\"", 218),
			new CommonArchiverDllConfig("UNGCA32", "UnGCA", "e \"{0}\" \"{1}\"", 10), // いるかな?
			
			//new CommonArchiverDllConfig("UNRAR32", "Unrar", "x -y \"{0}\" \"{1}\" *", -1),
		};
		
		/// <summary>
		/// アーカイブを展開する。適切なDLLで自動的に展開する。
		/// </summary>
		/// <param name="arcFile">アーカイブのパス</param>
		/// <param name="targetDir">展開先ディレクトリ</param>
		/// <param name="output">アーカイバDLLの展開時の標準出力を格納する</param>
		/// <param name="hWnd">親フレーム</param>
		/// <returns>アーカイバDLLの展開時のエラーコード(0なら正常終了)</returns>
		public static int ExtractArchive(string arcFile, string targetDir, System.Text.StringBuilder output, IntPtr hWnd)
		{
			foreach (CommonArchiverDllConfig config in Configs) {
				try {
					return ExtractArchiveWith(arcFile, targetDir, config, output, hWnd);
				} catch (DllNotFoundException) {
				} catch (ApplicationException) {
				}
			}
			throw new DllNotFoundException("Not found dll matched for " + arcFile);
		}
		
		/// <summary>
		/// 指定したDLLの設定を使ってアーカイブを展開する
		/// </summary>
		/// <param name="arcFile">アーカイブのパス</param>
		/// <param name="targetDir">展開先ディレクトリ</param>
		/// <param name="config">使用するアーカイバDLLの設定</param>
		/// <param name="output">アーカイバDLLの展開時の標準出力を格納する</param>
		/// <param name="hWnd">親フレーム</param>
		/// <returns>アーカイバDLLの展開時のエラーコード(0なら正常終了)</returns>
		public static int ExtractArchiveWith(string arcFile, string targetDir, CommonArchiverDllConfig config, System.Text.StringBuilder output, IntPtr hWnd)
		{
			if (! File.Exists(arcFile) ) {
				throw new FileNotFoundException("File not found: ", arcFile);
			}
			if (! Directory.Exists(targetDir)) {
				throw new DirectoryNotFoundException("Directory not found: " + targetDir);
			}
			
			// 統合アーカイバDLLの解凍先ディレクトリは\で終わらないといけない(重要)
			if (! targetDir.EndsWith("\\")) {
				targetDir += "\\";
			}
			
			// DLLバージョンチェック
			using (DllAccess dll = new DllAccess(config.DllName)) {
				ArcGetVersion GetVersion = (ArcGetVersion) dll.GetFunction(config.CmdName+"GetVersion", typeof(ArcGetVersion));
				ushort version = GetVersion();
				if (version < config.VersionMoreThan) {
					throw new DllNotFoundException("Dll version is too old.");
				}
			}
			
			string cmdLine = string.Format(config.CmdLineFmt, arcFile, targetDir);
			return ExtractArchiveWith(config.DllName, config.CmdName, cmdLine, arcFile, output, hWnd);
		}
		
		protected static int ExtractArchiveWith(string dllName, string cmdName, string cmdLine, string arcFile, System.Text.StringBuilder output, IntPtr hWnd)
		{
			using (DllAccess dll = new DllAccess(dllName)) {
				ArcGetRunning GetRunning = (ArcGetRunning) dll.GetFunction(cmdName+"GetRunning", typeof(ArcGetRunning));
				if ( GetRunning() ) { // マルチスレッド対応でないのでビジー時には例外
					throw new SystemException(dllName + " is busy");
				}
				
				ArcCheckArchive CheckArchive = (ArcCheckArchive) dll.GetFunction(cmdName+"CheckArchive", typeof(ArcCheckArchive));
				if (! CheckArchive(arcFile, ArchCheckFlag.BASIC)) {
					throw new ApplicationException(dllName + " could not treat the archive file");
				}
				
				ArcMain Exec = (ArcMain) dll.GetFunction(cmdName, typeof(ArcMain));
				int retVal = Exec(hWnd, cmdLine, output, (uint)((output != null)? output.Capacity : 0) );
				
				return retVal;
			}
		}
	}
}
