﻿using System;
using NaGet.Packages.Install;
using NaGet.Packages;
using NaGet.Net;

namespace NaGet.SubCommands
{
	public class NaGetInstall : NaGetTaskSet
	{
		private bool done = false;
		
		private int currentTaskSetIndex = -1;
		
		private PackageListsManager pkgListMan;
		
		/// <summary>
		/// ダウンロードに使うダウンローダオブジェクト
		/// </summary>
		public Downloader Downloader {
			get {
				if (downloader == null) {
					downloader = new Downloader();
				}
				return downloader;
			}
		}
		
		private Downloader downloader;
		
		private bool packageInstallerDownloaded = false;
		
		public override bool Cancelable {
			get { return ! done; }
		}
		
		/// <summary>
		/// インストールするパッケージ
		/// </summary>
		public Installation[] Installations;
		
		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="pkgs">インストールするパッケージ</param>
		public NaGetInstall(PackageListsManager pkgListMan, Package[] pkgs)
			: this(pkgListMan, Installation.ConvertInstallations(pkgs))
		{
		}
		
		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="installations">インストール処理の配列</param>
		public NaGetInstall(PackageListsManager pkgMan, Installation[] installations)
		{
			pkgListMan = pkgMan;
			
			Installations = installations;
			initializeMainTaskSetNames();
		}
		
		private void initializeMainTaskSetNames()
		{
			System.Collections.Generic.List<string> taskSetNames = new System.Collections.Generic.List<string>();
			
			for (int i =0; i < Installations.Length; i++) {
				taskSetNames.Add(string.Format("取得: {0}", Installations[i].ToString()));
			}
			taskSetNames.Add("インストーラの検証");
			for (int i =0; i < Installations.Length; i++) {
				taskSetNames.Add(string.Format("インストール: {0}", Installations[i].ToString()));
			}
			taskSetNames.Add(string.Format("リスト更新: {0}", NaGet.Env.ArchiveInstalledPackageListFile));
			taskSetNames.Add(string.Format("リスト更新: {0}", NaGet.Env.SystemInstalledPackageListFile));
			
			TaskSetNames = taskSetNames.ToArray();
		}
				
		public override void Run()
		{
			currentTaskSetIndex = 0;
			RaiseTaskSetEvent(NaGetTaskSetEventType.STARTED, "インストール処理開始");
			
			{
				// ハッシュ非適合なインストーラの表
				System.Collections.Generic.List<Installation> invalidInstallers = null;
				
				while (invalidInstallers == null || invalidInstallers.Count > 0) {
					currentTaskSetIndex = 0;
					packageInstallerDownloaded = false;
					
					runDownloadInstallers();
					if (done) return; // もしrunDownloadInstallers()内でエラー終了していたなら終了
					
					packageInstallerDownloaded = true;
					
					// ハッシュの壊れているインストーラを取得
					invalidInstallers = runCheckHashForInstaller();
					
					// ハッシュが壊れているときの対策
					if (invalidInstallers.Count > 0) {
						string msg = string.Format("{0}個のパッケージでファイルが壊れている可能性があります\n強制的にインストールを続行しますか?",
						                           invalidInstallers.Count);
						NaGetTaskQueryResult result = RaiseTaskSetQueryEvent(msg, NaGetTaskQueryResult.CONTINUE
						                                                     | NaGetTaskQueryResult.RETRY
						                                                     | NaGetTaskQueryResult.CANCEL);
						
						switch (result) {
							case NaGetTaskQueryResult.CONTINUE:
								RaiseTaskSetEvent(NaGetTaskSetEventType.WARNING, "ハッシュの非整合を無視してインストールを継続");
								invalidInstallers.Clear(); // ハッシュ非適合パッケージを強制的に抹消
								break;
							case NaGetTaskQueryResult.RETRY:
								RaiseTaskSetEvent(NaGetTaskSetEventType.INFO, "ダウンロード処理を再試行");
								
								foreach (Installation invalidInst in invalidInstallers) {
									invalidInst.RemoveDownloadedFile();
								}
								
								break;
							//case NaGetTaskQueryResult.CANCEL:
							default:
								RaiseTaskSetEvent(NaGetTaskSetEventType.CANCELED, "パッケージのインストール処理がキャンセルされました");
								done = true;
								return;
						}
					}
					currentTaskSetIndex ++;
				}
			}
			
			foreach (Installation inst in Installations) {
				string installTaskMsg = inst.ToString();
				if (inst.Silent && (!inst.SupportsSilentOnly)) {
					installTaskMsg += " (サイレントインストール)";
				}
				
				RaiseTaskSetEvent(NaGetTaskSetEventType.STARTED_TASKSET, installTaskMsg);
				
				try {
					
					inst.ErrorDataReceived += this.ReceivedErrorData;
					inst.OutputDataReceived += this.ReceivedOutputData;
					int exitCode = inst.Install();
					if (exitCode != 0) {
						RaiseTaskSetEvent(NaGetTaskSetEventType.WARNING, "インストールが正常に終えていない可能性があります。インストーラの終了コード:"+exitCode);
					}
					
					pkgListMan.WriteInstallationLog(inst);
				} catch (Exception e) {
					RaiseTaskSetEvent(NaGetTaskSetEventType.ERROR, e.Message);
					done = true;
					return;
				}
				currentTaskSetIndex ++;
				
				RaiseTaskSetEvent(NaGetTaskSetEventType.COMPLETED_TASKSET, installTaskMsg);
				
				if (cancelCalled) {
					RaiseTaskSetEvent(NaGetTaskSetEventType.CANCELED, "パッケージのインストール処理がキャンセルされました");
					done = true;
					return;
				}
			}
			pkgListMan.SaveSystemInstalledLogList(); // ログのコミット
			
			runLocalUpdate();
			
			done = true;
			
			RaiseTaskSetEvent(NaGetTaskSetEventType.COMPLETED, "終了", 100);
		}
		
		
		/// <summary>
		/// 処理内容のダウンロード部分のサブルーチン
		/// </summary>
		private void runDownloadInstallers()
		{
			foreach (Installation inst in Installations) {
				if (! inst.IsInstallablePackage()) {
					string msg = string.Format("{0}はインストールすることができません", inst.ToString());
					
					RaiseTaskSetEvent(NaGetTaskSetEventType.ERROR, msg);
					done = true;
					return;
				}
				
				RaiseTaskSetEvent(NaGetTaskSetEventType.STARTED_TASKSET, inst.ToString());
				
				if (! inst.Downloaded) {
					try {
						inst.Download(Downloader);
					} catch (NaGetTaskCanceledException) {
						RaiseTaskSetEvent(NaGetTaskSetEventType.CANCELED, "インストーラのダウンロード処理がキャンセルされました");
						done = true;
						return;
					} catch (System.Net.WebException e) {
						RaiseTaskSetEvent(NaGetTaskSetEventType.WARNING, e.Message);
						if (System.Net.NetworkInformation.NetworkInterface.GetIsNetworkAvailable()) {
							RaiseTaskSetEvent(NaGetTaskSetEventType.ERROR, "ネットワークに接続されていません。");
						} else {
							RaiseTaskSetEvent(NaGetTaskSetEventType.ERROR, "ネットワークに接続できませんでした。ネットワークが切断されているか、ファイアウォールによって遮断された可能性があります。");
						}
						done = true;
						return;
					} catch (Exception e) {
						RaiseTaskSetEvent(NaGetTaskSetEventType.ERROR, e.Message);
						done = true;
						return;
					}
				}
				currentTaskSetIndex ++;
				
				if (inst.Downloaded) { // 正常終了
					RaiseTaskSetEvent(NaGetTaskSetEventType.COMPLETED_TASKSET, inst.ToString());
				} else { // インストールが完了せずに終わった=失敗=エラー
					RaiseTaskSetEvent(NaGetTaskSetEventType.ERROR, string.Format("{0}のインストーラを正常にダウンロードできませんでした", inst.ToString()));
				}
			}
		}
		
		/// <summary>
		/// ダウンロードしたパッケージが整合したか否かハッシュでチェック
		/// </summary>
		/// <returns>整合しなかったインストーラのリスト</returns>
		private System.Collections.Generic.List<Installation> runCheckHashForInstaller()
		{
			System.Collections.Generic.List<Installation> invalidInstallers = new System.Collections.Generic.List<Installation>();
			
			int i = 0;
			foreach (Installation inst in Installations) {
				float percent = (CurrentTaskSetIndex+((float)i / Installations.Length))*100f/TaskSetNames.Length;
				
				if (inst.GetRegisteredHashCount() > 0) {
					RaiseTaskSetEvent(NaGetTaskSetEventType.INFO, "検証: "+inst.ToString(), percent);
					
					if (inst.IsInstallablePackage() && inst.VerifyHashValues() == false) {
						invalidInstallers.Add(inst);
						RaiseTaskSetEvent(NaGetTaskSetEventType.WARNING, "検証: "+inst.ToString() + " 非整合", percent);
					} else {
						RaiseTaskSetEvent(NaGetTaskSetEventType.INFO, "検証: "+inst.ToString() + " OK", percent);
					}
				}
				i++;
			}
			
			return invalidInstallers;
		}
		
		private void runLocalUpdate()
		{
			// インストールトリストの更新
			RaiseTaskSetEvent(NaGetTaskSetEventType.STARTED_TASKSET, TaskSetNames[currentTaskSetIndex]);
			pkgListMan.DetectInstalledPkgs();
			pkgListMan.SaveInstalledPackageList();
			currentTaskSetIndex++;
			RaiseTaskSetEvent(NaGetTaskSetEventType.COMPLETED_TASKSET, TaskSetNames[currentTaskSetIndex-1]);
		
			// システムにインストールされているリストの更新
			RaiseTaskSetEvent(NaGetTaskSetEventType.STARTED_TASKSET, TaskSetNames[currentTaskSetIndex]);
			pkgListMan.DetectSystemInstalledPkgs();
			pkgListMan.SaveSystemInstalledPackageList();
			currentTaskSetIndex++;
			RaiseTaskSetEvent(NaGetTaskSetEventType.COMPLETED_TASKSET, TaskSetNames[currentTaskSetIndex-1]);
		}
		
		public override bool Done {
			get { return done; }
		}
		
		public override int CurrentTaskSetIndex {
			get { return currentTaskSetIndex; }
		}
		
		private bool cancelCalled = false;
		
		public override bool Cancel()
		{
			cancelCalled = true;
			if (! packageInstallerDownloaded) {
				return Downloader.Cancel();
			} else return true;
		}
	}
}
