﻿using System;
using System.IO;
using System.Xml.Serialization;
using System.Security.Cryptography;

namespace NaGet.Packages
{
	/// <summary>
	/// ハッシュ値の種類(計算法)を表す。
	/// </summary>
	public enum HashValueType
	{
		/// <summary>
		/// ファイルのサイズ(容量)をバイト単位で返す
		/// </summary>
		[XmlEnum(Name="size")]
		SIZE,
		[XmlEnum(Name="md5")]
		MD5SUM,
		[XmlEnum(Name="sha1")]
		SHA1SUM,
		[XmlEnum(Name="sha256")]
		SHA256SUM,
	}
	
	public class HashValue
	{
		/// <summary>
		/// ハッシュ値の種類(計算法)
		/// </summary>
		[XmlAttribute]
		public HashValueType Type;
		
		/// <summary>
		/// ハッシュ値そのものをあらわす
		/// </summary>
		[XmlText]
		public string Value;
		
		/// <summary>
		/// コンストラクタ
		/// </summary>
		public HashValue()
		{
		}
		
		/// <summary>
		/// 与えられたファイルのハッシュ値が同一であるか検証する
		/// </summary>
		/// <param name="path">ハッシュ計算を行う対象のファイル</param>
		/// <returns>ファイルのハッシュ値が妥当な場合true。</returns>
		public bool Validate(string path)
		{
			return string.Compare(this.Value, HashValueFor(path, this.Type), true) == 0;
		}
		
		/// <summary>
		/// ファイルのハッシュを計算する
		/// </summary>
		/// <param name="file">計算対象のファイル</param>
		/// <param name="type">ハッシュの種類(計算法)</param>
		/// <returns></returns>
		public static string HashValueFor(string file, HashValueType type)
		{
			using (FileStream fs = new FileStream(file, FileMode.Open)) {
				return HashValue.HashValueFor(fs, type);
			}
		}
		
		/// <summary>
		/// ストリーム入力からハッシュを計算する
		/// </summary>
		/// <param name="stream">ストリーム入力</param>
		/// <param name="type">ハッシュの種類(計算法)</param>
		/// <returns>ハッシュ値</returns>
		public static string HashValueFor(Stream stream, HashValueType type)
		{
			byte[] hash;
			
			switch (type) {
				case HashValueType.SIZE:
					return stream.Length.ToString();
				case HashValueType.MD5SUM:
					hash = MD5.Create().ComputeHash(stream);
					break;
				case HashValueType.SHA1SUM:
					hash = SHA1.Create().ComputeHash(stream);
					break;
				case HashValueType.SHA256SUM:
					hash = SHA256.Create().ComputeHash(stream);
					break;
				default:
					throw new NotSupportedException(string.Format("Hash type {0} does not supported", type));
			}
			
			return BitConverter.ToString(hash).Replace("-",string.Empty);
		}
	}
}
