﻿using System;
using System.Reflection;
using System.Runtime.InteropServices;

namespace NaGet.InteropServices
{
	/// <summary>
	/// DLLへの動的アクセスを行うクラス
	/// </summary>
	public class DllAccess : IDisposable
	{
		#region Win32 API
		
		[DllImport("kernel32.dll", CharSet=CharSet.Auto)]
		private extern static IntPtr LoadLibrary(string lpFileName);
		
		[DllImport("kernel32.dll")]
		private extern static bool FreeLibrary(IntPtr hModule);
		
		[DllImport("kernel32.dll", CharSet=CharSet.Ansi)]
		private extern static IntPtr GetProcAddress(IntPtr hModule, string lpProcName);
		
		#endregion
		
		private IntPtr hModule = IntPtr.Zero;
		
		/// <summary>
		/// コンストラクタ。DLLを読み込む
		/// </summary>
		/// <param name="dllName">
		/// DLLの名称。直接Win32APIのLoadLibraryにわたされる
		/// </param>
		public DllAccess(string dllName)
		{
			hModule = LoadLibrary(dllName);
			if (hModule == IntPtr.Zero) {
				Exception innerEx = null;
				try {
					innerEx = Marshal.GetExceptionForHR(Marshal.GetHRForLastWin32Error());
				} catch {
				}
				throw new DllNotFoundException("Failed to LoadLibrary " + dllName, innerEx);
			}
		}
		
		/// <summary>
		/// 関数をdelegateに変換して得る
		/// </summary>
		/// <param name="strProcName">
		/// 関数名
		/// </param>
		/// <param name="type">
		/// 関数の型。戻り値のdelegateのタイプ
		/// </param>
		/// <returns>
		/// delegateに変換された関数
		/// </returns>
		public Delegate GetFunction(string strProcName, Type type)
		{
			IntPtr pFunc = GetProcAddress(hModule, strProcName);
			if (pFunc == IntPtr.Zero) {
				int result = Marshal.GetHRForLastWin32Error();
				throw Marshal.GetExceptionForHR(result);
			}
			
			return Marshal.GetDelegateForFunctionPointer(pFunc, type);
		}
		
		/// <summary>
		/// DLLの読み込みを閉じる
		/// </summary>
		public void Dispose()
		{
			if (hModule != IntPtr.Zero) {
				FreeLibrary(hModule);
				hModule = IntPtr.Zero;
			}
		}
	}
}
