﻿using System;
using System.Drawing;
using System.Windows.Forms;
using NaGet.Packages;
using NaGet.Packages.Install;

namespace AppliStation
{
	/// <summary>
	/// Description of PackagesInstallConfirmForm.
	/// </summary>
	public partial class InstallationConfirmForm : Form
	{
		private Installation[] installations;
		
		private PackageListsManager pkgListMan;
		
		/// <summary>
		/// パッケージインストール処理の集合
		/// </summary>
		public Installation[] Installations {
			get { return installations; }
			set {
				installations = value;
				
				// 基本はサイレントインストールを行うこととする
				foreach (Installation inst in installations) {
					inst.Silent = inst.IsSupportsSilent;
				}
				
				updateInstsListView();
			}
		}
		
		public PackageListsManager PkgListsManager {
			get { return pkgListMan; }
			set {
				pkgListMan = value;
				
				updateInstsListView();
			}
		}
		
		public InstallationConfirmForm()
		{
			//
			// The InitializeComponent() call is required for Windows Forms designer support.
			//
			InitializeComponent();
			
			// 管理者権限で動いているならばrunasが必要にはならないので表示しない
			if (NaGet.Utils.IsAdministrators()) {
				runasCheckBox.Checked = false;
				runasCheckBox.Visible = false;
			}
		}
		
		/// <summary>
		/// インストールリストを更新したなどで、リストの表示を更新する
		/// </summary>
		private void updateInstsListView()
		{
			instsListView.Items.Clear();
			
			if (installations != null) {
				foreach (Installation inst in installations) {
					Package pkg = inst.InstalledPackage;
					
					string[] itemData = new string[instsListView.Columns.Count];
					itemData[nameColumnHeader.Index] = pkg.Name;
					
					Package curPkg = null;
					if (pkgListMan != null) {	
						curPkg = pkgListMan.InstalledPkgList.GetPackageForName(pkg.Name) ??
							pkgListMan.SystemInstalledPkgList.GetPackageForName(pkg.Name);
					}
					itemData[versionColumnHeader.Index]        = pkg.Version;
					itemData[currentVersionColumnHeader.Index] = (curPkg != null)? curPkg.Version : "-";
					// itemData[silentInstColumnHeader.Index] の設定は instViewUpdateSilentInstallView で
					
					ListViewItem item = new ListViewItem(itemData);
					item.Tag = inst;
					item.ToolTipText = pkg.Summary;
					item.Checked = true;
					instViewUpdateSilentInstallView(item);
					
					instsListView.Items.Add(item);
				}
			}
			
			InstsListViewItemChecked(instsListView, null);
			instsListView.Refresh();
		}
		
		/// <summary>
		/// アイテムのサイレントインストール部分の表示の更新を行う。
		/// </summary>
		/// <param name="item">対象のインストーラのリストアイテム</param>
		private void instViewUpdateSilentInstallView(ListViewItem item)
		{
			Installation inst = (Installation) item.Tag;
			item.SubItems[silentInstColumnHeader.Index].Text =
				(inst.SupportsSilentOnly)? "サイレントインストールのみサポート" :
				(inst.Silent)? "サイレントインストール" :
				(inst.IsSupportsSilent)? "手動でインストール" :
				"サイレントインストールできませんので、手動でインストールします";
		}
		
		#region instsListViewのオーナードドロー関連
		
		void InstsListViewDrawSubItem(object sender, DrawListViewSubItemEventArgs e)
		{
			if (e.Header == silentInstColumnHeader) {
				Installation inst = ((Installation) e.Item.Tag);
				
				//e.DrawBackground();
				e.Graphics.Clip.Intersect(e.Bounds);
				
				if (inst.Silent) {
					AppliStation.Util.GUIUtils.Graphics_DrawCenterImage(
						e.Graphics,
						instListViewSilentInstallImageList.Images[0],
						e.Bounds, null);
				} else if (inst.IsSupportsSilent) {
					AppliStation.Util.GUIUtils.Graphics_DrawCenterImage(
						e.Graphics,
						instListViewSilentInstallImageList.Images[0],
						e.Bounds,
						AppliStation.Util.GUIUtils.GetImageAttributeToGrayOut(0.5f));
				}
			} else {
				e.DrawDefault = true;
			}
		}
		
		void InstsListViewDrawColumnHeader(object sender, DrawListViewColumnHeaderEventArgs e)
		{
			if (e.Header == silentInstColumnHeader) {
				e.DrawBackground();
				e.Graphics.Clip.Intersect(e.Bounds);
				AppliStation.Util.GUIUtils.Graphics_DrawCenterImage(
					e.Graphics,
					instListViewSilentInstallImageList.Images[0],
					e.Bounds, null);
			} else {
				e.DrawDefault = true;
			}
		}
		
		#endregion
		
		void InstsListViewItemChecked(object sender, ItemCheckedEventArgs e)
		{
			System.Windows.Forms.ListView.CheckedListViewItemCollection checkeds = instsListView.CheckedItems;
			
			okButton.Enabled = checkeds != null && checkeds.Count > 0;
		}
		
		AppliStation.Util.ListViewItemSortComparer packageListViewSortComparer;

		void InstsListViewColumnClick(object sender, ColumnClickEventArgs e)
		{
			SortOrder order = SortOrder.None;
			
			if (packageListViewSortComparer == null) {
				order = SortOrder.Ascending;
				packageListViewSortComparer = new AppliStation.Util.ListViewItemSortComparer(e.Column, order);
				instsListView.ListViewItemSorter = packageListViewSortComparer;
			} else {
				if (packageListViewSortComparer.Column == e.Column) {
					order = (packageListViewSortComparer.Order == SortOrder.Ascending)? SortOrder.Descending : SortOrder.Ascending;
					
					packageListViewSortComparer.Order = order;
				} else {
					order = packageListViewSortComparer.Order;
					packageListViewSortComparer.Column = e.Column;
				}
				
				instsListView.Sort();
			}
			AppliStation.Util.NativeMethods.ColumnHeader_SetSortState(instsListView, e.Column, order);
			
			// ソート対象列の色付け
			try {
				// SendMessage(hWnd, LVM_SETSELECTEDCOLUMN, column, NULL);
				AppliStation.Util.NativeMethods.SendMessage(instsListView.Handle, 0x1000+140, (uint) e.Column, 0);
			} catch (Exception) {	
			}
		}
		
				
		void InstsListViewContextMenuStripOpening(object sender, System.ComponentModel.CancelEventArgs e)
		{
			System.Windows.Forms.ListView.CheckedIndexCollection  chkIdxes = instsListView.CheckedIndices;
			System.Windows.Forms.ListView.SelectedIndexCollection selIdxes = instsListView.SelectedIndices;
			
			silentInstallStripMenuItem.Visible = selIdxes.Count > 0;
			if (selIdxes.Count > 0) {
				Installation inst0th = ((Installation) instsListView.Items[selIdxes[0]].Tag);
				bool bChecked = inst0th.Silent;
				bool bEnabled = (inst0th.IsSupportsSilent && (!inst0th.SupportsSilentOnly));
				silentInstallStripMenuItem.Checked = bChecked;
				silentInstallStripMenuItem.Enabled = bEnabled;
				
				for (int i = 1; i < selIdxes.Count; i++) {
					Installation inst = ((Installation) instsListView.Items[selIdxes[i]].Tag);
					
					if ( (bChecked != inst.Silent) ||
					    (bEnabled != (inst.IsSupportsSilent && (!inst.SupportsSilentOnly))) ) {
						silentInstallStripMenuItem.CheckState = CheckState.Indeterminate;
						silentInstallStripMenuItem.Enabled = false;
						break;
					}
				}
			}
			instListToolStripSeparator.Visible = selIdxes.Count > 0;
		
			
			selectAllInstsStripMenuItem.Enabled = chkIdxes.Count < instsListView.Items.Count;
			unselectAllInstsStripMenuItem.Enabled = chkIdxes.Count > 0;
		}
		
		void SilentInstallStripMenuItemClick(object sender, EventArgs e)
		{
			bool silent = ! silentInstallStripMenuItem.Checked;
			foreach (ListViewItem item in instsListView.SelectedItems) {
				((Installation) item.Tag).Silent = silent;
				instViewUpdateSilentInstallView(item);
			}
		}
				
		void SelectAllInstsStripMenuItemClick(object sender, EventArgs e)
		{
			foreach (ListViewItem item in instsListView.Items) {
				item.Checked = true;
			}
			instsListView.Refresh();
		}
		
		void UnselectAllInstsStripMenuItemClick(object sender, EventArgs e)
		{
			foreach (ListViewItem item in instsListView.CheckedItems) {
				item.Checked = false;
			}
			instsListView.Refresh();
		}
		
		/// <summary>
		/// インストールするよう選択されたパッケージの配列
		/// </summary>
		public Installation[] CheckedInstallations {
			get {
				System.Collections.Generic.List<Installation> insts = new System.Collections.Generic.List<Installation>();
				foreach (ListViewItem item in instsListView.CheckedItems) {
					insts.Add((Installation) item.Tag);
				}
				return insts.ToArray();
			}
		}
		
		#region runas関連
		
		/// <summary>
		/// runasで実行するか否か
		/// </summary>
		public bool UseRunas {
			set {
				runasCheckBox.Checked = (! NaGet.Utils.IsAdministrators()) && value;
				
				updateUseRunas();
			}
			get {
				return runasCheckBox.Checked;
			}
		}
		
		/// <summary>
		/// 選択されたパッケージを調査して、Runasを使うべきかいなかを返す
		/// </summary>
		public bool GetShouldUseRunas()
		{
			if (NaGet.Utils.IsAdministrators()) {
				// 管理者権限で動いている場合は不要
				return false;	
			} else if (NaGet.Utils.IsUACEnabled()) {
				// UACが適用されている場合は標準では不要とする
				return false;	
			}
			
			// ひとつでもPCターゲットなインストーラがあれば必要とする
			foreach (Installation inst in CheckedInstallations) {
				if (inst.TargetPC) return true;
			}
			// それ以外は不要
			return false;
		}
		
		void RunasCheckBoxCheckedChanged(object sender, EventArgs e)
		{
			updateUseRunas();
		}
		
		private void updateUseRunas()
		{
			if (UseRunas) {
				System.ComponentModel.ComponentResourceManager resources = new System.ComponentModel.ComponentResourceManager(typeof(InstallationConfirmForm));
				okButton.Image = ((System.Drawing.Bitmap)(resources.GetObject("okButton.Image")));
			} else {
				okButton.Image = null;
			}
		}
		
		#endregion
		
	}
}
