﻿using System;
using System.IO;
using System.Drawing;
using System.Drawing.Imaging;
using NaGet.Packages;
using NaGet.Packages.Install;

namespace AppliStation.Util
{
	/// <summary>
	/// GUIに関する雑多な関数
	/// </summary>
	public sealed class GUIUtils
	{
		/// <summary>
		/// パッケージに対応するアイコンを返す
		/// </summary>
		/// <param name="pkg">パッケージ</param>
		/// <returns>対応するアイコン。検出できなかった場合はnull。</returns>
		public static Icon GetIconForPackage(InstalledPackage pkg)
		{
			string iconPath = pkg.UninstallInfo.IconPath;
			if (! string.IsNullOrEmpty(iconPath)) {
				if (iconPath.EndsWith(",0") || iconPath.EndsWith(",-0")) {
					iconPath = iconPath.Substring(0, iconPath.LastIndexOf(','));
				}
				if (File.Exists(iconPath)) {
					return Icon.ExtractAssociatedIcon(iconPath);
				}
			} else if (pkg.Type == InstallerType.ARCHIVE) {
				string progGrp = Path.Combine(NaGet.Env.ArchiveProgramGroup, pkg.Name);
				if (Directory.Exists(progGrp)) {
					string[] lnkFiles = Directory.GetFiles(progGrp, "*.lnk");
					
					if (lnkFiles.Length >= 1) {
						using (NaGet.InteropServices.ShellLink link = new NaGet.InteropServices.ShellLink(lnkFiles[0])) {
							if (File.Exists(link.GetPath(0))) {
								return Icon.ExtractAssociatedIcon(link.GetPath(0));
							}
						}
					}
				}
			}
			return null;
		}
		
		/// <summary>
		/// グレーアウトアイコンを作るための ImageAttributes を作って返す。
		/// </summary>
		/// <param name="alpha">透明度。(1.0が不透明、0.0が完全透明)</param>
		/// <returns>生成されたImageAttributes</returns>
		public static ImageAttributes GetImageAttributeToGrayOut(float alpha)
		{
			// RGB比率として、YIQカラーモデルの値を採用する
			const float r = 0.298912f;
			const float g = 0.586611f;
			const float b = 0.114478f;
			
			ColorMatrix cm = new ColorMatrix(new float[][]{
		         	new float[]{r, r, r,     0, 0},
		         	new float[]{g, g, g,     0, 0},
		         	new float[]{b, b, b,     0, 0},
		         	new float[]{0, 0, 0, alpha, 0},
		         	new float[]{0, 0, 0,     0, 1},
		         });
			ImageAttributes ia = new ImageAttributes();
			ia.SetColorMatrix(cm);
			
			return ia;
		}
		
		/// <summary>
		/// 画像を指定領域の真中に描く
		/// </summary>
		/// <param name="g">描画対象のグラフィックス</param>
		/// <param name="img">画像</param>
		/// <param name="b">指定領域</param>
		/// <param name="ia">ImageAttributes。nullでもかまわない</param>
		public static void Graphics_DrawCenterImage(Graphics g, Image img, Rectangle b, ImageAttributes ia)
		{
			int x = b.Left + (b.Width  - img.Width ) / 2;
			int y = b.Top  + (b.Height - img.Height) / 2;
			
			g.DrawImage(img,
			            new Rectangle(x, y, img.Width, img.Height),
			            0, 0, img.Width, img.Height,
			            GraphicsUnit.Pixel, ia);
		}
	}
}
