﻿using System;
using System.Drawing;
using System.Windows.Forms;
using System.Threading;
using NaGet.SubCommands;
using NaGet.Net;

namespace AppliStation.Util
{
	/// <summary>
	/// Description of ExecutionProgressViewer.
	/// </summary>
	public partial class ExecutionProgressViewer : Form
	{
		private NaGetTaskSet taskSet;
		
		private Thread tasksetRunningThread = null;
		
		/// <summary>
		/// 終了時に何を行うかのフラグ
		/// </summary>
		public enum ActionOnDoneFlags {
			None = 0,
			FlashWindow = 1,
			AutoCloseOnSuccess = 2,
		}
		
		/// <summary>
		/// 終了時に何を行うか
		/// </summary>
		public ActionOnDoneFlags ActionOnDone = ActionOnDoneFlags.FlashWindow;
				
		public ExecutionProgressViewer()
		{
			//
			// The InitializeComponent() call is required for Windows Forms designer support.
			//
			InitializeComponent();
			
			this.Icon = Icon.ExtractAssociatedIcon(Application.ExecutablePath);
		}
				
		private void onDownloadEvent(object sender, DownloadEventArgs a)
		{
			if (InvokeRequired) {
				Invoke(new EventHandler<DownloadEventArgs>(onDownloadEvent), sender, a);
				return;
			}
			
			
			if (a.TaskProgressPercent >= 0) {
				progressBarSub.Value = (int) a.TaskProgressPercent;
				progressBarSub.Style = ProgressBarStyle.Continuous;
			} else {
				progressBarSub.Style = ProgressBarStyle.Marquee;
			}
			
			switch (a.Type) {
				case DownloadEventType.INITED:
				case DownloadEventType.CONNECTED:
					progressSubLabel.Text = a.TaskMessage;
					
					progressBarSub.Visible = true;
					progressSubLabel.Visible = true;
					break;
				case DownloadEventType.DOWNLOADING:
					progressSubLabel.Text = a.TaskMessage;
					break;
				case DownloadEventType.COMPLETED:
					progressBarSub.Visible = false;
					progressSubLabel.Visible = false;
					break;
				case DownloadEventType.ERROR:
					progressBarSub.Visible = false;
					progressSubLabel.Visible = false;
					
					logBox.SelectionColor = System.Drawing.Color.Red;
					logBox.AppendText("   [エラー] " + a.TaskMessage + System.Environment.NewLine);
					logBox.SelectionColor = logBox.ForeColor;
					break;
			}
		}
		
		#region NaGetTaskSet関連
		
		private void onTaskSetRaised(object sender, NaGetTaskSetEventArgs e)
		{
			NaGetTaskSet taskSet = (NaGetTaskSet) sender;
			
			if (e.TaskProgressPercent >= 0) {
				progressBar.Value = (int) e.TaskProgressPercent;
				progressBar.Style = ProgressBarStyle.Continuous;
			} else {
				progressBar.Style = ProgressBarStyle.Marquee;
			}
			progressLabel.Text = e.TaskMessage ?? string.Empty;
			
			cancelButton.Enabled = taskSet.Running && taskSet.Cancelable;
			
			switch (e.Type) {
				case NaGetTaskSetEventType.COMPLETED:
					logBox.AppendText("完了." + System.Environment.NewLine);
					if (taskSet.Done) {
						okButton.Enabled = true;
						cancelButton.Enabled = false;
					}
					break;
				case NaGetTaskSetEventType.STARTED_TASKSET:
					subtitleLabel.Text = taskSet.TaskSetNames[taskSet.CurrentTaskSetIndex];
					logBox.AppendText("  " + e.TaskMessage + System.Environment.NewLine);
					break;
				case NaGetTaskSetEventType.COMPLETED_TASKSET:
					if (progressBarSub.Visible) progressBarSub.Hide();
					if (progressSubLabel.Visible) progressSubLabel.Hide();
					
					NativeMethods.ProgressBar_SetState(progressBar, 1); // VistaProgress色：ノーマル
					
					logBox.AppendText(string.Format(" ... 完了. [{0}%]", (int) e.TaskProgressPercent));
					logBox.AppendText(System.Environment.NewLine);
					break;
				case NaGetTaskSetEventType.INFO:
					logBox.AppendText("  " + e.TaskMessage + System.Environment.NewLine);
					break;
				case NaGetTaskSetEventType.ERROR:
					logBox.SelectionColor = System.Drawing.Color.Red;
					logBox.AppendText("  [エラー] " + e.TaskMessage + System.Environment.NewLine);
					logBox.SelectionColor = logBox.ForeColor;
					
					NativeMethods.ProgressBar_SetState(progressBar, 2); // VistaProgress色：エラー
					
					okButton.Enabled = true;
					cancelButton.Enabled = false;
					break;
				case NaGetTaskSetEventType.CANCELED:
					logBox.SelectionColor = System.Drawing.Color.Red;
					logBox.AppendText(e.TaskMessage + System.Environment.NewLine);
					logBox.SelectionColor = logBox.ForeColor;
					
					NativeMethods.ProgressBar_SetState(progressBar, 1); // VistaProgress色：中断
					
					okButton.Enabled = true;
					cancelButton.Enabled = false;
					break;
				case NaGetTaskSetEventType.WARNING:
					logBox.SelectionColor = System.Drawing.Color.Red;
					logBox.AppendText("  [エラー] " + e.TaskMessage + System.Environment.NewLine);
					logBox.SelectionColor = logBox.ForeColor;
					
					break;
			}
			
			if (taskSet.Done) {
				if ((ActionOnDone & ActionOnDoneFlags.FlashWindow) != 0) {
					NativeMethods.Form_FlashWindow(this,
				                               NativeMethods.FlashFlag.All | NativeMethods.FlashFlag.TimerNoFG,
				                               uint.MaxValue, 0);
				}
				if (okButton.Enabled && (ActionOnDone & ActionOnDoneFlags.AutoCloseOnSuccess) != 0) {
					this.DialogResult = DialogResult.OK;
					Close();
					Dispose();
				}
			}
		}
		
		private NaGetTaskQueryResult onTaskQueryRaised(object sender, NaGetTaskQueryArgs e)
		{
			MessageBoxButtons buttons = MessageBoxButtons.OKCancel;
			if (e.SelectionFlag == (NaGetTaskQueryResult.CONTINUE | NaGetTaskQueryResult.RETRY | NaGetTaskQueryResult.CANCEL)) {
				buttons = MessageBoxButtons.AbortRetryIgnore;
			} else if (e.SelectionFlag == (NaGetTaskQueryResult.RETRY | NaGetTaskQueryResult.CANCEL)) {
				buttons = MessageBoxButtons.RetryCancel;
			} else if (e.SelectionFlag == NaGetTaskQueryResult.CONTINUE) {
				buttons = MessageBoxButtons.OK;
			}
			
			DialogResult result = MessageBox.Show(e.Message, this.Text, buttons);
			
			switch (result) {
				case DialogResult.OK:
				case DialogResult.Ignore:
					return NaGetTaskQueryResult.CONTINUE;
				case DialogResult.Cancel:
				case DialogResult.Abort:
					return NaGetTaskQueryResult.CANCEL;
				case DialogResult.Retry:
					return NaGetTaskQueryResult.RETRY;
				default:
					return NaGetTaskQueryResult.CANCELED_AUTOMATICALLY;
			}
		}
		
		public void SetTaskSet(NaGetTaskSet taskSet)
		{
			this.taskSet = taskSet;
			
			taskSet.TaskSetRaised += delegate(object sender, NaGetTaskSetEventArgs e) {
				if (InvokeRequired) {
					Invoke(new EventHandler<NaGetTaskSetEventArgs>(onTaskSetRaised), taskSet, e);
				} else {
					onTaskSetRaised(taskSet, e);
				}
			};
			
			System.Reflection.PropertyInfo fDownloader = taskSet.GetType().GetProperty("Downloader");
			if (fDownloader != null && (fDownloader.PropertyType == typeof(Downloader))) {
				((Downloader) fDownloader.GetValue(taskSet, null)).DownloadEventRaised += onDownloadEvent;
			}
			
			taskSet.TaskQueryRaised += onTaskQueryRaised;
		}

		public void StartTaskSet()
		{
			tasksetRunningThread = new Thread(taskSet.Run);
			tasksetRunningThread.Start();
		}
		
		#endregion
		
		void OkButtonClick(object sender, EventArgs e)
		{
			if (taskSet == null || taskSet.Done) {
				this.Close();
				this.Dispose();
			}
		}
		
		void CancelButtonClick(object sender, EventArgs e)
		{
			if (InvokeRequired) {
				Invoke(new EventHandler(CancelButtonClickConcrete), sender, e);
			} else {
				CancelButtonClickConcrete(sender,e);
			}
		}
		
		void CancelButtonClickConcrete(object sender, EventArgs e)
		{
			if (taskSet != null && taskSet.Running && taskSet.Cancelable) {
				cancelButton.Enabled = false;
				
				NativeMethods.ProgressBar_SetState(progressBar, 3); // VistaProgress色：中断
				
				taskSet.Cancel();
			}
		}
		
		void ExecutionProgressViewerShown(object sender, EventArgs e)
		{
			this.BringToFront();
		}
	}
}
