using System;
using System.IO;
using System.Collections.Generic;
using NaGet.Packages;
using NaGet.Packages.Install;

namespace NaGet.Packages
{
	public class PackageListsManager
	{
		internal PackageList<Package> availablePkgList;
		internal PackageList<InstalledPackage> installedPkgList;
		internal PackageList<InstalledPackage> systemInstalledPkgList;
		
		public PackageList<Package> AvailablePkgList {
			get { return availablePkgList; }
		}
		
		public PackageList<InstalledPackage> InstalledPkgList {
			get { return installedPkgList; }
		}
		
		public PackageList<InstalledPackage> SystemInstalledPkgList {
			get { return systemInstalledPkgList; }
		}
		
		private List<InstallationLog> systemInstalledLogList;
		
		public PackageListsManager()
		{
			LoadPackageLists();
		}
		
		#region t@Cǂݏ
		
		/// <summary>
		/// t@CǂݍōXV
		/// </summary>
		public void LoadPackageLists()
		{
			try {
				availablePkgList = NaGet.Utils.GetDeserializedObject<PackageList<Package>>(NaGet.Env.PackageListFile);
			} catch (FileNotFoundException) {
				availablePkgList = new PackageList<Package>();
			}
			
			try {
				installedPkgList = NaGet.Utils.GetDeserializedObject<PackageList<InstalledPackage>>(NaGet.Env.ArchiveInstalledPackageListFile);
			} catch (FileNotFoundException) {
				installedPkgList = new PackageList<InstalledPackage>();
			}
			
			try {
				systemInstalledPkgList = NaGet.Utils.GetDeserializedObject<PackageList<InstalledPackage>>(NaGet.Env.SystemInstalledPackageListFile);
			} catch (FileNotFoundException) {
				systemInstalledPkgList = new PackageList<InstalledPackage>();
			}
			
			
			systemInstalledLogList = new List<InstallationLog>();
			try {
				systemInstalledLogList.AddRange(
					NaGet.Utils.GetDeserializedObject<InstallationLog[]>(NaGet.Env.SystemInstalledPackageLogFile)
				);
			} catch (Exception) {
				// do nothing
			}
		}
		
		public void SaveAvailablePackageList()
		{
			NaGet.Utils.PutSerializeObject(NaGet.Env.PackageListFile, availablePkgList);
		}
		public void SaveInstalledPackageList()
		{
			NaGet.Utils.PutSerializeObject(NaGet.Env.ArchiveInstalledPackageListFile, installedPkgList);
		}
		public void SaveSystemInstalledPackageList()
		{
			NaGet.Utils.PutSerializeObject(NaGet.Env.SystemInstalledPackageListFile, systemInstalledPkgList);
		}
		public void SaveSystemInstalledLogList()
		{
			NaGet.Utils.PutSerializeObject(NaGet.Env.SystemInstalledPackageLogFile, systemInstalledLogList);
		}
		
		public void CommitToFile()
		{
			SaveAvailablePackageList();
			SaveInstalledPackageList();
			SaveSystemInstalledPackageList();
			
			SaveSystemInstalledLogList();
		}
		
		#endregion
		
		#region pbP[WCXg[m֘A
		
		public void DetectInstalledPkgs()
		{
			PackageList<InstalledPackage> pkgList = new PackageList<InstalledPackage>();
			if (Directory.Exists(NaGet.Env.ArchiveProgramFiles)) {
				foreach (string path in NaGet.Utils.ExtendWildcardFile(NaGet.Env.ArchiveProgramFiles, Path.Combine("*", ".applistation.package.xml"))) {
					pkgList.AddPackage(NaGet.Utils.GetDeserializedObject<InstalledPackage>(path));
				}
				this.installedPkgList = pkgList;
			}
		}
		
		public void DetectSystemInstalledPkgs()
		{
			PackageList<InstalledPackage> installedPkgList = new PackageList<InstalledPackage>();
			foreach (InstalledPackage pkg in RegistriedUninstallers.DetectInstalledPackages(availablePkgList)) {
				InstallationLog log = this.GetLogFor(pkg);
				if (log != null) {
					InstalledPackage pkg2 = new InstalledPackage();
					NaGet.Utils.FieldCopy((Package) log.Package, ref pkg2);
					pkg2.UninstallInfo = pkg.UninstallInfo;
					
					installedPkgList.AddPackage(pkg2);
				} else {
					installedPkgList.AddPackage(pkg);
				}
			}
			this.systemInstalledPkgList = installedPkgList;
		}
		
		#endregion
		
		#region pbP[W֗\bh
		
		public IEnumerable<InstalledPackage> GetAllInstalledPackages()
		{
			foreach (InstalledPackage pkg in installedPkgList.Packages) {
				yield return pkg;
			}
			foreach (InstalledPackage pkg in systemInstalledPkgList.Packages) {
				yield return pkg;
			}
		}
		
		/// <summary>
		/// SẴpbP[WCe[^ŕԂ֗\bhB
		/// </summary>
		/// <returns>pbP[W̃Ce[^</returns>
		public IEnumerable<Package> GetAllPackages()
		{
			foreach (Package pkg in availablePkgList.Packages) {
				yield return pkg;
			}
			foreach (Package pkg in GetAllInstalledPackages()) {
				yield return pkg;
			}
		}
		
		#endregion
		
		#region CXg[O֘A
		
		private Predicate<InstallationLog> createPackageNamePredicator(Package pkg)
		{
			return delegate(InstallationLog log) {
				return log.Package.Name == pkg.Name;
			};
		}
		
		public InstallationLog GetLogFor(Package pkg)
		{
			return systemInstalledLogList.Find(createPackageNamePredicator(pkg));
		}
		
		public void WriteInstallationLog(Installation inst)
		{
			// d͍폜
			systemInstalledLogList.RemoveAll(createPackageNamePredicator(inst.InstalledPackage));
			
			InstallationLog newLog = new InstallationLog();
			newLog.Date = DateTime.Now;
			newLog.Package = inst.InstalledPackage;
			
			systemInstalledLogList.Add(newLog);
		}
		
		#endregion
	}
}
