package net.java.amateras.xlsbeans;

import java.awt.Point;
import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;

import jxl.Cell;
import jxl.Sheet;
import net.java.amateras.xlsbeans.annotation.Column;
import net.java.amateras.xlsbeans.annotation.MapColumns;
import net.java.amateras.xlsbeans.xml.AnnotationReader;

/**
 * 
 * @author Naoki Takezoe
 * @author Mitsuyoshi Hasegawa
 */
public class Utils {
	
	public static Method[] getMapColumnMethod(Object obj, 
			AnnotationReader reader) throws Exception {
		
		List<Method> result = new ArrayList<Method>();
		Method[] methods = obj.getClass().getMethods();
		for(int i=0;i<methods.length;i++){
			if(methods[i].getName().startsWith("set") &&
					methods[i].getParameterTypes().length==1){
				Annotation[] anns = reader.getAnnotations(obj.getClass(), methods[i]);
				
				for(int j=0;j<anns.length;j++){
					if(anns[j] instanceof MapColumns){
						result.add(methods[i]);
					}
				}
			}
		}
		return result.toArray(new Method[result.size()]);
	}
	
	public static Method[] getColumnMethod(
			Object obj, String name, AnnotationReader reader) throws Exception {
		
		List<Method> result = new ArrayList<Method>();
		Method[] methods = obj.getClass().getMethods();
		for(int i=0;i<methods.length;i++){
			if(methods[i].getName().startsWith("set") &&
					methods[i].getParameterTypes().length==1){
				Annotation[] anns = reader.getAnnotations(obj.getClass(), methods[i]);
				for(int j=0;j<anns.length;j++){
					if(anns[j] instanceof Column){
						Column column = (Column)anns[j];
						String columnName = column.columnName();
						if(name==null){
							result.add(methods[i]);
						} else if(columnName.equals(name)){
							result.add(methods[i]);
						}
					}
				}
			}
		}
		return result.toArray(new Method[result.size()]);
	}
	
	public static Cell getCell(Sheet sheet, String label, int from) throws XLSBeansException {
		int rows = sheet.getColumns();
		for(int i=0;i<rows;i++){
			Cell[] columns = sheet.getColumn(i);
			for(int j=from;j<columns.length;j++){
				if(columns[j].getContents().equals(label)){
					return columns[j];
				}
			}
		}
		throw new XLSBeansException("Cell '" + label + "' doesn't exist.");
	}
	
	/**
	 * Invokes the setter method using reflection.
	 * 
	 * @param setter the setter method
	 * @param obj the target object
	 * @param value the value which is String, primitive types or wrapper types
	 * @throws Exception
	 */
	public static void invokeSetter(Method setter,Object obj,String value) throws Exception {
		Class<?>[] types = setter.getParameterTypes();
		if(types.length!=1){
			return;
		}
		if(types[0].equals(String.class)){
			setter.invoke(obj, new Object[]{value});
		} else if(types[0].equals(Integer.TYPE) || types[0].equals(Integer.class)){
			if(value.length()==0){
				value = "0";
			}
			setter.invoke(obj, new Object[]{new Integer(value)});
		} else if(types[0].equals(Double.TYPE) || types[0].equals(Double.class)){
			if(value.length()==0){
				value = "0";
			}
			setter.invoke(obj, new Object[]{new Double(value)});
		} else if(types[0].equals(Short.TYPE) || types[0].equals(Short.class)){
			if(value.length()==0){
				value = "0";
			}
			setter.invoke(obj, new Object[]{new Short(value)});
		} else if(types[0].equals(Long.TYPE) || types[0].equals(Long.class)){
			if(value.length()==0){
				value = "0";
			}
			setter.invoke(obj, new Object[]{new Long(value)});
		} else if(types[0].equals(Float.TYPE) || types[0].equals(Float.class)){
			if(value.length()==0){
				value = "0";
			}
			setter.invoke(obj, new Object[]{new Float(value)});
		} else if(types[0].equals(Boolean.TYPE) || types[0].equals(Boolean.class)){
			if(value.length()==0){
				value = "false";
			}
			setter.invoke(obj, new Object[]{new Boolean(value)});
		} else if(types[0].equals(Character.TYPE) || types[0].equals(Character.class)){
			if(value.length()!=0){
				setter.invoke(obj, new Object[]{new Character(value.charAt(0))});
			}
		}
	}
	
	/**
     * Return cell object by using first argument sheet.
     * This cell will be found by label name in Excel sheet.
     * 
     * NOTICE: When the cell object is specified for the third argument,
     * a lower right cell is scanned from the cell.
	 * 
     * @param sheet JExcel Api sheet object.
     * @param label Target cell label.
     * @param after A lower right cell is scanned from this cell object.
     * 
     * @return Target JExcel Api cell object.
     * @throws XLSBeansException This occures when the cell is not found.
	 */
	public static Cell getCell(Sheet sheet, String label, Cell after)
			throws XLSBeansException {
		return getCell(sheet, label, after, false);
	}

    /**
     * Return cell object by using first argument sheet.
     * This cell will be found by label name in Excel sheet.
     * 
     * NOTICE: When the cell object is specified for the third argument,
     * a lower right cell is scanned from the cell.
     * 
     * @param sheet JExcel Api sheet object.
     * @param label Target cell label.
     * @param after A lower right cell is scanned from the cell object.
     * @param includeingAfter Is the third argument cell object scanned?
     * 
     * @return Target JExcel Api cell object.
     * @throws XLSBeansException This occures when the cell is not found.
     */
    public static Cell getCell(Sheet sheet, String label, Cell after,
            boolean includeAfter) throws XLSBeansException {
        return getCell(sheet, label, after, includeAfter, true);
    }
    
	/**
     * Return cell object by using first argument sheet.
     * This cell will be found by label name in Excel sheet.
     * 
     * NOTICE: When the cell object is specified for the third argument,
     * a lower right cell is scanned from the cell.
	 * 
	 * @param sheet JExcel Api sheet object.
	 * @param label Target cell label.
	 * @param after A lower right cell is scanned from the cell object.
	 * @param includeingAfter Is the third argument cell object scanned?
     * @param throwableWhenNotFound If this argument is true, throws XLSBeansException when 
     * we can't find target cell.
     * 
	 * @return Target JExcel Api cell object.
	 * @throws XLSBeansException This occures when the cell is not found.
	 */
	public static Cell getCell(Sheet sheet, String label, Cell after,
			boolean includeAfter, boolean throwableWhenNotFound) throws XLSBeansException {

		if (after == null) {
			// Call XLSBeans#getCell() - method if third argument is null.
			return Utils.getCell(sheet, label, 0);
		}

		int columnStart = after.getColumn();
		int rowStart = after.getRow();

		for (int col = columnStart; col < sheet.getColumns(); col++) {
			// getting cells each columns
			Cell[] columns = sheet.getColumn(col);
			for (int row = rowStart; row < columns.length; row++) {
				// except first cell if "includeAfter" is false.
				if (col == columnStart && row == rowStart && !includeAfter) {
					continue;
				}
				if (columns[row].getContents().equals(label)) {
					return columns[row];
				}
			}
		}
        if (throwableWhenNotFound) {
            // can't find cell
            throw new XLSBeansException("Cell '" + label + "' doesn't exist.");
        } else {
            return null;
        }
	}
	
	/**
	 * Return the method in object with annotation.
	 * 
	 * @param tableObj
	 *            target JavaBeans
	 * @param reader
	 *            AnnotationReader object
	 * @param clazz
     *            target annotation type class.
	 * @return setter method in argument object with annotation.
	 * @throws Exception unexpected exception.
	 */
	public static Method[] getMethodWithAnnotation(Object tableObj,
			AnnotationReader reader, Class<?> clazz) throws Exception {
		List<Method> result = new ArrayList<Method>();
		Method[] methods = tableObj.getClass().getMethods();

		for (Method method : methods) {
			// setter
			if (method.getName().startsWith("set")
					&& method.getParameterTypes().length == 1) {
				// find annotation
				Annotation[] ans = reader.getAnnotations(tableObj.getClass(),
						method);
				for (Annotation an : ans) {
					if (an.annotationType().isAssignableFrom(clazz)) {
						result.add(method);
						break;
					}
				}
			}
		}
		return result.toArray(new Method[result.size()]);
	}
	
	// TODO Javadoc
	public static void setPosition(int x, int y, Object obj, Method setter) throws Exception {
		String positionMethodName = setter.getName() + "Position";
		
		try {
			Method positionMethod = obj.getClass().getMethod(
					positionMethodName, Integer.TYPE, Integer.TYPE);
			positionMethod.invoke(obj, x, y);
			return;
		} catch(NoSuchMethodException ex){
		}
		try {
			Method positionMethod = obj.getClass().getMethod(
					positionMethodName, Point.class);
			positionMethod.invoke(obj, new Point(x, y));
			return;
		} catch(NoSuchMethodException ex){
		}
	}
}
