/*
 * Aipo is a groupware program developed by Aimluck,Inc.
 * Copyright (C) 2004-2011 Aimluck,Inc.
 * http://www.aipo.com
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.schedule.util;

import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;

import org.apache.cayenne.ObjectId;
import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.cayenne.query.Ordering;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.jetspeed.services.resources.JetspeedResources;

import com.aimluck.eip.cayenne.om.portlet.EipTSchedule;
import com.aimluck.eip.cayenne.om.portlet.EipTScheduleMap;
import com.aimluck.eip.cayenne.om.security.TurbineGroup;
import com.aimluck.eip.cayenne.om.security.TurbineUser;
import com.aimluck.eip.cayenne.om.security.TurbineUserGroupRole;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALEipManager;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.query.Operations;
import com.aimluck.eip.orm.query.ResultList;
import com.aimluck.eip.orm.query.SelectQuery;

/**
 * スケジュールのユーティリティクラスです。
 * 
 */
public class IdnetScheduleUtils {

  /** logger */
  private static final JetspeedLogger logger = JetspeedLogFactoryService
    .getLogger(IdnetScheduleUtils.class.getName());

  /** 管理者のID */
  public static final String ADMIN_USER_LOGIN_ID = JetspeedResources.getString(
    "idnet.admin",
    "");

  /** 講師【HE】の部署ID */
  protected static final String HE_POST_ID = JetspeedResources.getString(
    "idnet.post.he",
    "");

  /** 講師【接遇】の部署ID */
  protected static final String ENTERTAIN_POST_ID = JetspeedResources
    .getString("idnet.post.entertain", "");

  /** 重複スケジュールCSSクラス名 */
  protected static final String DUPLICATE_CLASS_NAME = "duplicate";

  /** 講師【HE】CSSクラス名 */
  protected static final String HE_CLASS_NAME = "he";

  /** 講師【接遇】CSSクラス名 */
  protected static final String ENTERTAIN_CLASS_NAME = "entertain";

  /** 講師混合CSSクラス名 */
  protected static final String MIXTURE_CLASS_NAME = "mixture";

  /**
   * HEのユーザー一覧を取得
   * 
   * @param rundata
   * @return
   */
  public static List<Integer> getHeUserList() {
    try {
      SelectQuery<TurbineUser> hequery = getSelectQuery(HE_POST_ID);
      ResultList<TurbineUser> helist = hequery.getResultList();
      List<Integer> heuidlist = new ArrayList<Integer>();
      TurbineUser heuser = null;
      int hesize = helist.size();
      for (int i = 0; i < hesize; i++) {
        heuser = helist.get(i);
        if (!heuidlist.contains(heuser.getUserId())) {
          heuidlist.add(heuser.getUserId());
        }
      }
      return heuidlist;
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
  }

  /**
   * 接遇のユーザー一覧を取得
   * 
   * @param rundata
   * @return
   */
  public static List<Integer> getEntertainUserList() {
    try {
      SelectQuery<TurbineUser> entertainquery =
        getSelectQuery(ENTERTAIN_POST_ID);
      ResultList<TurbineUser> entertainlist = entertainquery.getResultList();
      List<Integer> entertainidlist = new ArrayList<Integer>();
      TurbineUser entertainuser = null;
      int entertainsize = entertainlist.size();
      for (int i = 0; i < entertainsize; i++) {
        entertainuser = entertainlist.get(i);
        if (!entertainidlist.contains(entertainuser.getUserId())) {
          entertainidlist.add(entertainuser.getUserId());
        }
      }
      return entertainidlist;
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
  }

  /**
   * 講師の部署に応じたCSSのクラス名を取得する
   * 
   * @param rundata
   * @return
   */
  public static String getCssClassName(List<EipTScheduleMap> schedulemap,
      List<Integer> heIdList, List<Integer> entertainIdList) {
    try {

      String classname = null;

      EipTScheduleMap map = null;
      int hecount = 0;
      int entertaincount = 0;
      int mapsize = schedulemap.size();
      for (int i = 0; i < mapsize; i++) {
        map = schedulemap.get(i);
        if (heIdList.contains(map.getUserId())) {
          // HEのユーザー
          hecount++;
        }
        if (entertainIdList.contains(map.getUserId())) {
          // 接遇のユーザー
          entertaincount++;
        }
      }
      if (hecount > 0 && entertaincount > 0) {
        classname = MIXTURE_CLASS_NAME;
      } else if (hecount > 0 && entertaincount == 0) {
        classname = HE_CLASS_NAME;
      } else if (hecount == 0 && entertaincount > 0) {
        classname = ENTERTAIN_CLASS_NAME;
      }

      return classname;
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
  }

  /**
   * 
   * ユーザー検索のクエリを返します
   * 
   * @param post_id
   * @return
   */
  private static SelectQuery<TurbineUser> getSelectQuery(String post_id) {
    try {

      ObjectId oid =
        new ObjectId("TurbineUser", TurbineUser.USER_ID_PK_COLUMN, 3);
      Expression exp1 =
        ExpressionFactory.matchAllDbExp(
          oid.getIdSnapshot(),
          Expression.GREATER_THAN);
      Expression exp2 =
        ExpressionFactory.matchExp(TurbineUser.DISABLED_PROPERTY, "F");

      SelectQuery<TurbineUser> query = Database.query(TurbineUser.class);
      query.setQualifier(exp1.andExp(exp2));

      String groupName =
        (ALEipManager.getInstance().getPostMap().get(Integer.valueOf(post_id)))
          .getGroupName()
          .getValue();

      query.where(Operations.eq(TurbineUser.TURBINE_USER_GROUP_ROLE_PROPERTY
        + "."
        + TurbineUserGroupRole.TURBINE_GROUP_PROPERTY
        + "."
        + TurbineGroup.GROUP_NAME_PROPERTY, groupName));

      return query;
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
  }

  /**
   * 管理者のIDを取得します。
   * 
   * @param rundata
   * @param context
   * @return
   */
  public static Integer getAdminUserId() throws ALPageNotFoundException,
      ALDBErrorException {

    Integer id = null;
    try {
      SelectQuery<TurbineUser> userquery = Database.query(TurbineUser.class);
      Expression userexp =
        ExpressionFactory.matchDbExp(
          TurbineUser.LOGIN_NAME_COLUMN,
          ADMIN_USER_LOGIN_ID);
      userquery.setQualifier(userexp);
      List<Ordering> orders = new ArrayList<Ordering>();
      orders.add(new Ordering(TurbineUser.LAST_NAME_KANA_PROPERTY, true));
      orders.add(new Ordering(TurbineUser.FIRST_NAME_KANA_PROPERTY, true));
      userquery.getQuery().addOrderings(orders);

      List<TurbineUser> ulist = userquery.fetchList();

      TurbineUser tuser = ulist.get(0);
      id = tuser.getUserId().intValue();
    } catch (Exception e) {
      logger.error("[IdnetScheduleUtils]", e);
      throw new ALDBErrorException();

    }
    return id;
  }

  /**
   * 共有メンバーID一覧を取得します。
   * 
   * @param rundata
   * @param context
   * @return
   */
  public static ArrayList<Integer> getUserIds(
      List<EipTScheduleMap> schedulemaps, Integer admin_id)
      throws ALPageNotFoundException, ALDBErrorException {
    ArrayList<Integer> memberIds = new ArrayList<Integer>();
    try {
      List<Integer> memberList = getUsers(schedulemaps);

      Integer id = null;
      int size = memberList.size();
      for (int i = 0; i < size; i++) {
        id = memberList.get(i);
        if (!id.equals(Integer.valueOf(admin_id))) {
          memberIds.add(Integer.valueOf(id));
        }
      }
    } catch (Exception e) {
      logger.error("[ScheduleUtils]", e);
      throw new ALDBErrorException();

    }
    return memberIds;
  }

  /**
   * 共有メンバーを取得します。
   * 
   * @param rundata
   * @param context
   * @return
   */
  public static List<Integer> getUsers(List<EipTScheduleMap> schedulemaps)
      throws ALPageNotFoundException, ALDBErrorException {
    List<Integer> uidlist = new ArrayList<Integer>();

    try {
      EipTScheduleMap map = null;
      int mapsize = schedulemaps.size();
      for (int i = 0; i < mapsize; i++) {
        map = schedulemaps.get(i);
        if (!uidlist.contains(map.getUserId())) {
          uidlist.add(map.getUserId());
        }
      }
    } catch (Exception e) {
      logger.error("[IdnetScheduleUtils]", e);
      throw new ALDBErrorException();

    }
    return uidlist;
  }

  /**
   * メンバーの重複スケジュール
   * 
   * @param schedule
   * @param memberIdList
   * @param _old_scheduleid
   * @param _old_viewDate
   * @return
   */
  public static boolean isDuplicateMemberSchedule(EipTSchedule schedule,
      List<Integer> memberIdList, Integer _old_scheduleid, Date _old_viewDate) {
    if (isDuplicateMemberRepeatSchedule(
      schedule,
      memberIdList,
      _old_scheduleid,
      _old_viewDate)
      || isDuplicateMemberSpanSchedule(schedule, memberIdList)) {
      return true;
    }
    return false;
  }

  /**
   * メンバーの重複スケジュール
   * 
   * @param schedule
   * @param memberIdList
   * @param _old_scheduleid
   * @param _old_viewDate
   * @return
   */
  public static boolean isDuplicateMemberRepeatSchedule(EipTSchedule schedule,
      List<Integer> memberIdList, Integer _old_scheduleid, Date _old_viewDate) {
    /* ダミースケジュール検索用 */
    GregorianCalendar cald = new GregorianCalendar();

    boolean result = false;
    {

      Date start_date;
      Date end_date;
      String repeat_pattern;
      String repeat_type;
      boolean week_0;
      boolean week_1;
      boolean week_2;
      boolean week_3;
      boolean week_4;
      boolean week_5;
      boolean week_6;
      String limit_flag;
      int month_day = -1;
      Integer db_scheduleid = null;
      boolean[] week_array = new boolean[7];
      boolean unlimited_repeat = false;
      try {
        start_date = schedule.getStartDate();

        end_date = schedule.getEndDate();

        repeat_pattern = schedule.getRepeatPattern();

        repeat_type = repeat_pattern.substring(0, 0);

        limit_flag = repeat_pattern.substring(repeat_pattern.length() - 1);

        week_0 = repeat_pattern.matches("W1.......");

        week_1 = repeat_pattern.matches("W.1......");

        week_2 = repeat_pattern.matches("W..1.....");

        week_3 = repeat_pattern.matches("W...1....");

        week_4 = repeat_pattern.matches("W....1...");

        week_5 = repeat_pattern.matches("W.....1..");

        week_6 = repeat_pattern.matches("W......1.");

        if (repeat_pattern.startsWith("M")) {
          month_day = Integer.parseInt(repeat_pattern.substring(1, 3));
        }

        // 単体スケジュールは期限1日のみのスケジュールとして判定
        if (repeat_pattern.startsWith("N")) {
          Calendar cal = Calendar.getInstance();
          cal.setTime(start_date);
          int dow = cal.get(Calendar.DAY_OF_WEEK);
          week_0 = (dow == Calendar.SUNDAY);
          week_1 = (dow == Calendar.MONDAY);
          week_2 = (dow == Calendar.TUESDAY);
          week_3 = (dow == Calendar.WEDNESDAY);
          week_4 = (dow == Calendar.THURSDAY);
          week_5 = (dow == Calendar.FRIDAY);
          week_6 = (dow == Calendar.SATURDAY);
          month_day = cal.get(Calendar.DAY_OF_MONTH);
        } else if (repeat_pattern.endsWith("N")) {
          unlimited_repeat = true;
        }

        week_array[0] = week_0;
        week_array[1] = week_1;
        week_array[2] = week_2;
        week_array[3] = week_3;
        week_array[4] = week_4;
        week_array[5] = week_5;
        week_array[6] = week_6;

      } catch (Exception e) {
        return false;
      }

      // メンバーのスケジュール状況をチェックする
      if (memberIdList.size() > 0) {
        List<Integer> fids = memberIdList;
        SelectQuery<EipTScheduleMap> fquery =
          Database.query(EipTScheduleMap.class);
        Expression fexp1 =
          ExpressionFactory.inExp(EipTScheduleMap.USER_ID_PROPERTY, fids);
        fquery.setQualifier(fexp1);

        Expression fexp2 =
          ExpressionFactory.matchExp(
            EipTScheduleMap.TYPE_PROPERTY,
            ScheduleUtils.SCHEDULEMAP_TYPE_USER);
        fquery.andQualifier(fexp2);

        Expression oneexp = null;
        Expression rdexp = null;
        Expression rwexp = null;
        // Expression rwlexp = null;
        Expression rmexp = null;

        { // １日スケジュールの検索
          Expression exp100 =
            ExpressionFactory.matchExp(EipTScheduleMap.EIP_TSCHEDULE_PROPERTY
              + "."
              + EipTSchedule.REPEAT_PATTERN_PROPERTY, "N");

          try {
            if (!unlimited_repeat) {
              Expression exp101 =
                ExpressionFactory.lessOrEqualExp(
                  EipTScheduleMap.EIP_TSCHEDULE_PROPERTY
                    + "."
                    + EipTSchedule.START_DATE_PROPERTY,
                  end_date);
              Expression exp102 =
                ExpressionFactory.greaterExp(
                  EipTScheduleMap.EIP_TSCHEDULE_PROPERTY
                    + "."
                    + EipTSchedule.END_DATE_PROPERTY,
                  start_date);
              oneexp = exp100.andExp(exp101.andExp(exp102));
            } else {
              oneexp = exp100;
            }
          } catch (Exception e) {

          }
        }

        { // 繰り返しスケジュールの検索
          // char lim = 'N';
          if ("ON".equals(limit_flag)) {
            // lim = 'L';
          }

          { // "D".equals(repeat_type.getValue())
            Expression dexp01 =
              ExpressionFactory.likeExp(EipTScheduleMap.EIP_TSCHEDULE_PROPERTY
                + "."
                + EipTSchedule.REPEAT_PATTERN_PROPERTY, "D_");
            rdexp = dexp01;
          }

          { // "W".equals(repeat_type.getValue())
            Expression wexp = null;
            List<Expression> wexps = new ArrayList<Expression>();
            if (week_0 == true) {
              wexp =
                ExpressionFactory.likeExp(
                  EipTScheduleMap.EIP_TSCHEDULE_PROPERTY
                    + "."
                    + EipTSchedule.REPEAT_PATTERN_PROPERTY,
                  "W1_______");
              wexps.add(wexp);
            }
            if (week_1 == true) {
              wexp =
                ExpressionFactory.likeExp(
                  EipTScheduleMap.EIP_TSCHEDULE_PROPERTY
                    + "."
                    + EipTSchedule.REPEAT_PATTERN_PROPERTY,
                  "W_1______");
              wexps.add(wexp);
            }
            if (week_2 == true) {
              wexp =
                ExpressionFactory.likeExp(
                  EipTScheduleMap.EIP_TSCHEDULE_PROPERTY
                    + "."
                    + EipTSchedule.REPEAT_PATTERN_PROPERTY,
                  "W__1_____");
              wexps.add(wexp);
            }
            if (week_3 == true) {
              wexp =
                ExpressionFactory.likeExp(
                  EipTScheduleMap.EIP_TSCHEDULE_PROPERTY
                    + "."
                    + EipTSchedule.REPEAT_PATTERN_PROPERTY,
                  "W___1____");
              wexps.add(wexp);
            }
            if (week_4 == true) {
              wexp =
                ExpressionFactory.likeExp(
                  EipTScheduleMap.EIP_TSCHEDULE_PROPERTY
                    + "."
                    + EipTSchedule.REPEAT_PATTERN_PROPERTY,
                  "W____1___");
              wexps.add(wexp);
            }
            if (week_5 == true) {
              wexp =
                ExpressionFactory.likeExp(
                  EipTScheduleMap.EIP_TSCHEDULE_PROPERTY
                    + "."
                    + EipTSchedule.REPEAT_PATTERN_PROPERTY,
                  "W_____1__");
              wexps.add(wexp);
            }
            if (week_6 == true) {
              wexp =
                ExpressionFactory.likeExp(
                  EipTScheduleMap.EIP_TSCHEDULE_PROPERTY
                    + "."
                    + EipTSchedule.REPEAT_PATTERN_PROPERTY,
                  "W______1_");
              wexps.add(wexp);
            }
            if (wexps.size() > 0) {
              rwexp = wexps.get(0);
              int wexpssize = wexps.size();
              for (int k = 1; k < wexpssize; k++) {
                rwexp = rwexp.orExp(wexps.get(k));
              }
            } else {
              rwexp =
                ExpressionFactory.likeExp(
                  EipTScheduleMap.EIP_TSCHEDULE_PROPERTY
                    + "."
                    + EipTSchedule.REPEAT_PATTERN_PROPERTY,
                  "W________");
            }
          }

          { // "M".equals(repeat_type.getValue())
            if (month_day > 0) { // 毎月、もしくは単体の場合
              DecimalFormat exF = new DecimalFormat("00");
              String md_str = exF.format(month_day);
              rmexp =
                ExpressionFactory.likeExp(
                  EipTScheduleMap.EIP_TSCHEDULE_PROPERTY
                    + "."
                    + EipTSchedule.REPEAT_PATTERN_PROPERTY,
                  "M" + md_str + "_");
            } else {
              rmexp =
                ExpressionFactory.likeExp(
                  EipTScheduleMap.EIP_TSCHEDULE_PROPERTY
                    + "."
                    + EipTSchedule.REPEAT_PATTERN_PROPERTY,
                  "M___");
            }
          }

          Expression repeatexp = oneexp;
          if (rdexp != null) {
            repeatexp = repeatexp.orExp(rdexp);
          }
          if (rwexp != null) {
            repeatexp = repeatexp.orExp(rwexp);
          }
          if (rmexp != null) {
            repeatexp = repeatexp.orExp(rmexp);
          }
          fquery.andQualifier(repeatexp);
        }

        db_scheduleid = schedule.getScheduleId();
        if (db_scheduleid != null && db_scheduleid >= 0) {
          Expression exp00 =
            ExpressionFactory.noMatchDbExp(
              EipTScheduleMap.EIP_TSCHEDULE_PROPERTY
                + "."
                + EipTSchedule.SCHEDULE_ID_PK_COLUMN,
              db_scheduleid);
          fquery.andQualifier(exp00);
        }

        fquery.distinct(true);
        List<EipTScheduleMap> f_list = fquery.fetchList();
        if (f_list != null && f_list.size() > 0) {
          // 繰り返しスケジュール同士の時刻幅での比較
          boolean existMember = false;
          int f_list_size = f_list.size();
          for (int i = 0; i < f_list_size; i++) {
            EipTScheduleMap map = f_list.get(i);

            Date dbStartDate = map.getEipTSchedule().getStartDate();
            Date dbEndDate = map.getEipTSchedule().getEndDate();

            boolean containtsRs = false;
            // 繰り返し期限付きの処理
            String ptn = map.getEipTSchedule().getRepeatPattern();

            if (ptn.charAt(0) == 'N') { // 単体スケジュール
              if ("D".equals(repeat_type) || "N".equals(repeat_type)) { // 毎日 or
                // 単体
                try {
                  if ((dbStartDate.before(end_date) && dbEndDate
                    .after(start_date))
                    || unlimited_repeat) {
                    containtsRs = true;
                  }
                } catch (Exception e) {
                  containtsRs = false;
                }
              } else {
                if ((dbStartDate.before(end_date) && dbEndDate
                  .after(start_date))
                  || unlimited_repeat) {
                  containtsRs = true;
                }
              }

            } else if (ptn.charAt(0) == 'D') {
              if (ptn.charAt(1) == 'L') {
                try {
                  if ((dbStartDate.before(end_date) && dbEndDate
                    .after(start_date))
                    || unlimited_repeat) {
                    containtsRs = true;
                  }
                } catch (Exception e) {
                  containtsRs = false;
                }
              } else {
                containtsRs = true;
              }
            } else if (ptn.charAt(0) == 'W') {
              if (ptn.charAt(8) == 'L') {
                try {
                  if ((dbStartDate.before(end_date) && dbEndDate
                    .after(start_date))
                    || unlimited_repeat) {
                    containtsRs = true;
                  }
                } catch (Exception e) {
                  containtsRs = false;
                }
              } else {
                containtsRs = true;
              }
            } else if (ptn.charAt(0) == 'M') {
              if (ptn.charAt(3) == 'L') {
                try {
                  if ((dbStartDate.before(end_date) && dbEndDate
                    .after(start_date))
                    || unlimited_repeat) {
                    containtsRs = true;
                  }
                } catch (Exception e) {
                  containtsRs = false;
                }
              } else {
                containtsRs = true;
              }
            } else {
              containtsRs = true;
            }

            if (containtsRs) {
              int ss_flg = ScheduleUtils.compareTime(start_date, dbEndDate);
              int se_flg = ScheduleUtils.compareTime(end_date, dbStartDate);
              if (ss_flg > 0 && se_flg < 0) {
                /* 期限無しのスケジュール同士の場合は重複とみなす */
                if (!"N".equals(ptn) && ptn.endsWith("N") && unlimited_repeat) {
                  existMember = true;
                } else {
                  Date _start_date = null;
                  Date _end_date = null;

                  if (!"N".equals(ptn)
                    && ptn.endsWith("N")
                    && !unlimited_repeat) {
                    _start_date = (Date) start_date.clone();
                    _end_date = (Date) end_date.clone();
                  } else if (("N".equals(ptn) || !ptn.endsWith("N"))
                    && unlimited_repeat) {
                    _start_date = (Date) dbStartDate.clone();
                    _end_date = (Date) dbEndDate.clone();
                  } else if (("N".equals(ptn) || !ptn.endsWith("N"))
                    && !unlimited_repeat) {

                    if (dbStartDate.after(start_date)) {
                      _start_date = (Date) dbStartDate.clone();
                    } else {
                      _start_date = (Date) start_date.clone();
                    }

                    if (dbEndDate.before(end_date)) {
                      _end_date = (Date) dbEndDate.clone();
                    } else {
                      _end_date = (Date) end_date.clone();
                    }

                  }

                  if ((_start_date == null) || (_end_date == null)) {
                    continue;
                  }

                  /* 期限内の日付を全て比較 */
                  Expression dexp1 =
                    ExpressionFactory.matchExp(
                      EipTSchedule.NAME_PROPERTY,
                      "dummy");

                  Expression dexp2 =
                    ExpressionFactory.matchExp(
                      EipTSchedule.PARENT_ID_PROPERTY,
                      map.getScheduleId());

                  if (db_scheduleid != null) {
                    Expression dexp21 =
                      ExpressionFactory.matchExp(
                        EipTSchedule.PARENT_ID_PROPERTY,
                        db_scheduleid);
                    dexp2 = dexp2.orExp(dexp21);
                  }
                  Expression dexp3 = null;

                  cald.setTime(_start_date);
                  cald.set(Calendar.MILLISECOND, 0);
                  cald.set(Calendar.SECOND, 0);
                  cald.set(Calendar.MINUTE, 0);
                  cald.set(Calendar.HOUR_OF_DAY, 0);
                  Date ddate = cald.getTime();
                  List<EipTSchedule> temp = null;

                  if ("N".equals(repeat_pattern)) {
                    /* 繰り返しスケジュールのうちひとつだけを移動した場合の処理 */
                    if ((_old_scheduleid != null) && (_old_viewDate != null)) {
                      if ((_old_scheduleid.intValue() == map
                        .getScheduleId()
                        .intValue())
                        && ScheduleUtils.compareToDate(
                          _start_date,
                          _old_viewDate) == 0) {
                        continue;
                      }
                    }

                    try {
                      dexp3 =
                        ExpressionFactory.matchExp(
                          EipTSchedule.START_DATE_PROPERTY,
                          ddate);
                      temp =
                        Database.query(
                          EipTSchedule.class,
                          dexp1.andExp(dexp2).andExp(dexp3)).fetchList();
                      if (temp == null || temp.size() <= 0) {
                        existMember = true;
                        break;
                      }
                    } catch (Exception e) {
                      logger.error("[DuplicateMemberCheck]: ", e);
                      existMember = true;
                      break;
                    }
                  } else if (repeat_pattern.startsWith("D")) {
                    while (!ddate.after(_end_date)) {
                      if (ScheduleUtils.matchDay(cald, ptn)) {
                        try {
                          dexp3 =
                            ExpressionFactory.matchExp(
                              EipTSchedule.START_DATE_PROPERTY,
                              ddate);
                          temp =
                            Database.query(
                              EipTSchedule.class,
                              dexp1.andExp(dexp2).andExp(dexp3)).fetchList();
                          if (temp == null || temp.size() <= 0) {
                            existMember = true;
                            break;
                          }
                        } catch (Exception e) {
                          logger.error("[DuplicateMemberCheck]: ", e);
                          existMember = true;
                          break;
                        }
                      }
                      cald.add(Calendar.DATE, 1);
                      ddate = cald.getTime();
                    }
                  } else if (repeat_pattern.startsWith("W")) {
                    /* ダミースケジュールを探す */
                    int wlen = week_array.length;
                    if (wlen < 1) {
                      continue;
                    }
                    int k;
                    while (!ddate.after(_end_date)) {
                      k = (cald.get(Calendar.DAY_OF_WEEK) - 1) % wlen;
                      if ((week_array[k] == true)
                        && ScheduleUtils.matchDay(cald, ptn)) {
                        try {
                          dexp3 =
                            ExpressionFactory.matchExp(
                              EipTSchedule.START_DATE_PROPERTY,
                              ddate);
                          temp =
                            Database.query(
                              EipTSchedule.class,
                              dexp1.andExp(dexp2).andExp(dexp3)).fetchList();
                          if (temp == null || temp.size() <= 0) {
                            existMember = true;
                            break;
                          }
                        } catch (Exception e) {
                          logger.error("[DuplicateMemberCheck]: ", e);
                          existMember = true;
                          break;
                        }
                      }
                      cald.add(Calendar.DATE, 1);
                      ddate = cald.getTime();
                    }
                  } else if (repeat_pattern.startsWith("M")) {
                    /* 比較開始日までカレンダー移動 */
                    cald.setTime(dbStartDate);
                    cald.set(Calendar.MILLISECOND, 0);
                    cald.set(Calendar.SECOND, 0);
                    cald.set(Calendar.MINUTE, 0);
                    cald.set(Calendar.HOUR_OF_DAY, 0);

                    if (month_day > 0) {
                      cald.set(Calendar.DAY_OF_MONTH, month_day);
                    } else {
                      continue;
                    }
                    Date tmp_date = cald.getTime();
                    while (tmp_date.before(ddate)) {
                      cald.add(Calendar.MONTH, 1);
                      /* 月によって日にちがないときのための処理 */
                      while (month_day > cald
                        .getActualMaximum(Calendar.DAY_OF_MONTH)) {
                        cald.add(Calendar.MONTH, 1);
                        cald.set(Calendar.DAY_OF_MONTH, month_day);
                        if (tmp_date.before(tmp_date)) {
                          break;
                        }
                      }
                      tmp_date = cald.getTime();
                    }
                    ddate = tmp_date;
                    /* 比較開始 */
                    while (!ddate.after(_end_date)) {
                      if (ScheduleUtils.matchDay(cald, ptn)) {
                        try {
                          dexp3 =
                            ExpressionFactory.matchExp(
                              EipTSchedule.START_DATE_PROPERTY,
                              ddate);
                          temp =
                            Database.query(
                              EipTSchedule.class,
                              dexp1.andExp(dexp2).andExp(dexp3)).fetchList();
                          if (temp == null || temp.size() <= 0) {
                            existMember = true;
                            break;
                          }
                        } catch (Exception e) {
                          logger.error("[DuplicateMemberCheck]: ", e);
                          existMember = true;
                          break;
                        }
                      }
                      cald.add(Calendar.MONTH, 1);
                      /* 月によって日にちがないときのための処理 */
                      while (month_day > cald
                        .getActualMaximum(Calendar.DAY_OF_MONTH)) {
                        cald.add(Calendar.MONTH, 1);
                        cald.set(Calendar.DAY_OF_MONTH, month_day);
                        if (!ddate.after(_end_date)) {
                          break;
                        }
                      }
                      ddate = cald.getTime();
                    }
                  } else {
                    continue;
                  }
                }
                // existFacility = true;
              }
            }
            if (existMember) {
              break;
            }
          }
          if (existMember) {
            return existMember;
          }
        }
      }
    }
    return result;
  }

  /**
   * メンバーの期間重複スケジュール
   * 
   * @param schedule
   * @param memberIdList
   * @param _old_scheduleid
   * @param _old_viewDate
   * @return
   */
  public static boolean isDuplicateMemberSpanSchedule(EipTSchedule schedule,
      List<Integer> memberIdList) {

    boolean result = false;
    {
      Date start_date;
      Date end_date;
      String repeat_pattern;
      Integer db_scheduleid = null;
      try {
        start_date = schedule.getStartDate();
        end_date = schedule.getEndDate();
        repeat_pattern = schedule.getRepeatPattern();
      } catch (Exception e) {
        return false;
      }

      // 期間スケジュール
      if (repeat_pattern.equals("S")) {

        // メンバーのスケジュール状況をチェックする
        if (memberIdList.size() > 0) {
          List<Integer> fids = memberIdList;
          SelectQuery<EipTScheduleMap> fquery =
            Database.query(EipTScheduleMap.class);
          Expression fexp1 =
            ExpressionFactory.inExp(EipTScheduleMap.USER_ID_PROPERTY, fids);
          fquery.setQualifier(fexp1);

          Expression fexp2 =
            ExpressionFactory.matchExp(
              EipTScheduleMap.TYPE_PROPERTY,
              ScheduleUtils.SCHEDULEMAP_TYPE_USER);
          fquery.andQualifier(fexp2);
          {
            // 期間スケジュールの検索
            Expression exp100 =
              ExpressionFactory.matchExp(EipTScheduleMap.EIP_TSCHEDULE_PROPERTY
                + "."
                + EipTSchedule.REPEAT_PATTERN_PROPERTY, "S");
            fquery.andQualifier(exp100);

            db_scheduleid = schedule.getScheduleId();
            if (db_scheduleid != null && db_scheduleid >= 0) {
              Expression exp00 =
                ExpressionFactory.noMatchDbExp(
                  EipTScheduleMap.EIP_TSCHEDULE_PROPERTY
                    + "."
                    + EipTSchedule.SCHEDULE_ID_PK_COLUMN,
                  db_scheduleid);
              fquery.andQualifier(exp00);
            }

            fquery.distinct(true);
            List<EipTScheduleMap> f_list = fquery.fetchList();
            if (f_list != null && f_list.size() > 0) {

              int f_list_size = f_list.size();
              for (int i = 0; i < f_list_size; i++) {
                EipTScheduleMap map = f_list.get(i);

                Date dbStartDate = map.getEipTSchedule().getStartDate();
                Date dbEndDate = map.getEipTSchedule().getEndDate();

                // 繰り返し期限付きの処理
                String ptn = map.getEipTSchedule().getRepeatPattern();
                if (ptn.charAt(0) == 'S') {
                  try {
                    // 重複期間のチェック
                    if ((dbStartDate.before(end_date) || dbStartDate
                      .compareTo(end_date) == 0)
                      && (dbEndDate.after(start_date) || dbEndDate
                        .compareTo(start_date) == 0)) {
                      result = true;
                      break;
                    }
                  } catch (Exception e) {
                    logger.error("[DuplicateMemberCheck]: ", e);
                    result = false;
                    break;
                  }
                }
              }
            }
          }
        }
      }
    }
    return result;
  }
}
