/**
*** Program jdbcMysqlResult.java
***    in product twz1jdbcForMysql, 
***    Copyright 1997, 1998 by Terrence W. Zellers.
***   
***  All rights explicitly reserved.
***
***  See file "LICENSE" in this package for conditions of use.
**/

package twz1.jdbc.mysql;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.sql.*; 
import java.util.Vector;

/** This class is a class of convenience only acting as a quasi
*** isolated extension (but not a subclass) to jdbcMysqlStmt.  Its
*** purpose is solely to isolate and concentrate the code dealing
*** with query results from the connection and API management code.
**/

final class jdbcMysqlResult
{
/** Our parent statement */
jdbcMysqlStmt statement;

/** The grandparent connection */
jdbcMysqlConnex connex;

/** Current metadata */
RSMd metadata;

/** Creation catalog */
String catalog;

/** current rowcount */
int priorRowCount;  // Stash the prior value.
int rowCount;       // What we return from getRows()
int realRowCount;   // Count of cached or read rows.
int readRowCount;   // Rows actually returned from DB.

/** last status */
int lastStatus;

/** status message */
String sMessage;

/** Cache mode */
int cacheMode;
static final int CM_MEMORY = 0;
static final int CM_DISK   = 1;
static final int CM_NONE   = 2;
static final int CM_PHONEY = 3;

/** disk input/output streams */
BufferedInputStream dbis;
BufferedOutputStream dbos;

/** disk cache path */
String dcpath;

/** int statement ID */
int stid;

/** disk cache filename */
String dcfName;

/** The disk cache FILE */
File dcFile;

/** file actually written? */
boolean dCacheWritten;

/** The memory cache. */
Vector mcache;

/** Is nocache done */
boolean noCacheDone;

/** Has there been a data error */
boolean dataError;

/** Connection input Bag */
jdbcMysqlBag inBag;

/** Workbag */
jdbcMysqlBag wBag;

/** set for each action */
int OID;

/** The result sets */
Vector rSets;

static final String errs[] = {
              "E0700 Error in result initialization.",
              "E0701 Error in getResult().",
              "P0702 Unsupported type in getResult()",
              "E0703 Error synchronizing headers and data.",
              "E0704 Error caching results.", 
              "P0705 getRow() request for non memory caching.",
              "E0706 cacheClose() error",
              "E0707 error in buildColumn()",
              "E0708 Command results in error - ",
              "E0709 Error clearing prior results",
              "E0710 Error in data retrieval -",
              "E0711 Error. Unhandled cacheMode.",
                             };

/*===================================================================+
||                    Result constructor                            ||
+===================================================================*/

jdbcMysqlResult(jdbcMysqlStmt st, jdbcMysqlConnex cx)
         throws SQLException
    {
    this.statement = st;
    this.connex    = cx;
    this.metadata  = null;
    this.priorRowCount = -1;
    this.rowCount  = -1;
    this.lastStatus = -1;
    this.sMessage = null;
    this.cacheMode = CM_MEMORY;
    this.dbis = null;
    this.dbos = null;
    this.dcpath = null;
    this.stid   = jdbcMysqlBase.getOID();
    this.dcfName = null;
    this.dcFile = null;
    this.dCacheWritten = false;
    this.noCacheDone = true;
    this.mcache = null;
    this.rSets = null;
    this.realRowCount = 0;
    this.readRowCount = 0;
    this.dataError = false;
    this.wBag = new jdbcMysqlBag(10);

    String test;
    int testnum;
    
    try {

        this.inBag = connex.getInBag();
        test = connex.getProperty("cacheMode");
        if(test != null) 
            {
            test = test.trim();
            if(test.equalsIgnoreCase("disk")) cacheMode = CM_DISK;
            }
        dcpath = connex.getProperty("cachePath");
        if(dcpath != null) dcpath = dcpath.trim();
        else dcpath=".";
        if(dcpath.equals("")) dcpath = ".";

        if(!connex.isMultipleQuery()) cacheMode = CM_NONE;

        }
    catch(Exception e) { errHandlerE(0, e); }
    }

/*-------------------------------------------------------------------+
|                   Clear results of previous                        |
+-------------------------------------------------------------------*/

/** This method clears results from previous queries and sets up
*** new queries.
**/
void clear() throws SQLException
    {

    cacheClose(noCacheDone);

     /** Force nulls to encourage gc */
    int t, ts;
    if(rSets != null)
        {
        jdbcMysqlRSet rs;
        ts = rSets.size();
        for(t = 0; t < ts; t++)
            {
            rs = (jdbcMysqlRSet) rSets.elementAt(t);
            rs.iClose();
            }
        rSets = null;
        }
    if(cacheMode == CM_PHONEY) cacheMode = CM_MEMORY;
    mcache = null;
    metadata = null;
    catalog = null;
    rowCount = -1;
    realRowCount = 0;
    lastStatus = -1;
    dataError = false;
    }
      
                 
/*-------------------------------------------------------------------+
|                       get results!                                 |
+-------------------------------------------------------------------*/

void getResult(int type, int command) throws SQLException
    {
    int i, w;
    try {
        /** Read the primary result */
        if(type == statement.QTYPE_EU)
            {
            if(command == jdbcMysqlBase.COM_COMMENT)
               {
               rowCount = 0;
               return;
               }
            inBag.read();
            lastStatus = inBag.bToI(1);
            statement.lastInsertID = -1;
            if(lastStatus == 0) 
                {
                rowCount = inBag.viToI();
                if(inBag.at < inBag.maxAt)
                       statement.lastInsertID = inBag.viToL();
                }
               
            if(lastStatus == 255) 
                {
                rowCount = inBag.bToI(2);
                sMessage = inBag.getZTstring();
                }
            return;
            }

        inBag.read();
        if(inBag.peekAbyte() == 255)
            {
            lastStatus = inBag.bToI(1);
            int j = inBag.bToI(2);
            sMessage = new String(j + " " + inBag.getZTstring());
            jdbcMysqlBase.errMessage(errs[8] + sMessage);
            return;
            }

        /** read query type results! */
        w = inBag.viToI();
        metadata = new RSMd(statement);
        catalog = new String(connex.catalog);
        for(i = 0; i < w; i++) buildColumn();

        inBag.read();
        i = inBag.bToI(1);
        if(i != 254) jdbcMysqlBase.errMessage(errs[3]);
 
        switch(cacheMode)
             {
             case CM_DISK  :  cacheItDisk(); break;
             case CM_MEMORY: cacheItMem(); break;
             case CM_NONE :  noCacheDone = false; 
                             break;
             default: errHandlerM(11);
             }
        }
    catch(Exception e) { errHandlerE(1, e); }
    }
       
/*-------------------------------------------------------------------+
|                Write data to cache in memory                       |
+-------------------------------------------------------------------*/

private void cacheItMem() throws SQLException
    {
    byte[] u;
    boolean eod = false;
    try {
        mcache = new Vector();
        realRowCount = 0;
        readRowCount = 0;       
        for(eod = inEoD(inBag); !eod; eod = inEoD(inBag))
            {
            if(realRowCount < statement.maxRows) 
                {
                mcache.addElement(inBag.inside);
                inBag.inside = null;
                realRowCount++;
                }
            readRowCount++;
            }
        if(dataError) errHandlerNM(10, lastStatus, sMessage);
        }
    catch(Exception e) 
        {
        if(!eod) while(!inEoD(inBag));   // exhaust query.
        errHandlerE(4, e); 
        }
    }

/*-------------------------------------------------------------------+
|                Write data to cache on disk                         |
+-------------------------------------------------------------------*/

private void cacheItDisk() throws SQLException
    {
    boolean eod = false;
    try {
        dCacheWritten = true;
        dcfName = new String(dcpath + "/" + stid + ".cache");
        dcFile  = new File(dcfName);
        dbos = new BufferedOutputStream(new FileOutputStream(dcFile));      
        inBag.setOutput(dbos);
        realRowCount = 0;
        readRowCount = 0;
        for(eod = inEoD(inBag); !eod; eod = inEoD(inBag))
            {
            if(realRowCount < statement.maxRows) 
                {
                inBag.write();
                realRowCount++;
                }
            readRowCount++;
            }
        inBag.write();  // Write the EoD record!
        dbos.close();
        dbos = null;
        inBag.setOutput(null);
        if(dataError)
            {
            dcFile.delete();
            errHandlerNM(10, lastStatus, sMessage);
            }
        dbis = new BufferedInputStream(new FileInputStream(dcFile));
        inBag = new jdbcMysqlBag();
        inBag.setInput(dbis);
        }
    catch(Exception e) 
        {
        if(!eod) while(!inEoD(inBag));   // exhaust the query.
        errHandlerE(4, e); 
        }
    }


/*-------------------------------------------------------------------+
|                   Read the bag and see if an EOD                   |
+-------------------------------------------------------------------*/

/** Read data and return a true if it is the last bag.  Static because
*** it will also be ref'd by RSet.
**/
boolean inEoD(jdbcMysqlBag in) 
        throws SQLException
    {
    in.read();
    int s = in.getSize();
    int u = in.peekAbyte();
    if(u == 254 && s < 4) return true;
    if(u == 255)
        {
        in.bToI(1);
        dataError = true;
        lastStatus = 2;
        sMessage = "Data error!";   
        try {
            lastStatus = in.bToI(2);
            sMessage = in.getRstring();
            }
        catch(Exception e){} // Already set dataError.
        return true;  // We be done with data!
        }
    return false;
    }  

/*-------------------------------------------------------------------+
|                     Build a column from the data                   |
+-------------------------------------------------------------------*/

private void buildColumn() throws SQLException
    {
    int i, j;
    String work;
    jdbcMysqlField column;
    try {
        inBag.read();
        column = new jdbcMysqlField(metadata.cc);
        column.setColumnCatalog(catalog);
        i = inBag.viToI();
        work = inBag.getNstring(i);
        column.setColumnTable(work);
        i = inBag.viToI();
        work = inBag.getNstring(i);
        column.setColumnName(work);
        i = inBag.nbToI();
        column.setColumnSize(i);
        i = inBag.nbToI(); 
        column.setColumnType(i);
        j = inBag.bToI(1);
        if(j > 0)
            {
            i = inBag.bToI(1); 
            column.setColumnFlags(i);
            j--;
            }
        if(j > 0)
            { 
            i = inBag.bToI(1);
            column.setColumnDecimals(i);
            j--;
            } 
        metadata.addField(column);
        }
    catch(Exception e) { errHandlerE(7, e); }
    }

/*-------------------------------------------------------------------+
|                     Hand back results                              |
+-------------------------------------------------------------------*/

int getStatus() { return lastStatus; }
int getRows()   
     { 
     if(rowCount == -1) return rowCount;  
     priorRowCount = rowCount;
     rowCount = -1;
     return  priorRowCount;
     }

String getMessage() { return sMessage; }

/*-------------------------------------------------------------------+
|                      talk to the ResultSet                         |
+-------------------------------------------------------------------*/

void addRS(jdbcMysqlRSet rs) 
    { 
    if(rSets == null) rSets = new Vector();
    rSets.addElement(rs);
    }

void doneRS(int oid)
    {
    jdbcMysqlRSet rs;
    if(rSets != null) for(int i = 0; i < rSets.size(); i++)
        {
        rs = (jdbcMysqlRSet) rSets.elementAt(i);
        if(rs.myID() == oid) rSets.removeElementAt(i--);
        }
    }    

void cacheClose(boolean dataDone) throws SQLException
    {
    if(cacheMode == CM_MEMORY) return;
    if(cacheMode == CM_PHONEY) { cacheMode = CM_MEMORY; return; }
    try {
        if(cacheMode == CM_DISK)
            {
            if(dCacheWritten)
                {
                if(dbis != null)dbis.close();
                try{ dcFile.delete();} catch(Exception e){}
                inBag = connex.getInBag();
                }
            return;
            }
        // cacheMode is CM_NONE
        if(!dataDone && noCacheDone == false) while(!inEoD(inBag)); 
        if(!noCacheDone)
            {
            connex.lock(false, statement.myOID, statement.lTimeout);
            noCacheDone = true; 
            }
        }
    catch(Exception e)  { errHandlerE(6, e); }
    }

/** Return the next previously unretrieved result set. 
*** Under current implementations of this driver and MySQL
*** there should be only one RS so this may be seen as overkill
*** or prudent planning, take your pick.
**/
jdbcMysqlRSet getUnretrievedRS()
    {
    if(rSets == null) return null;
    jdbcMysqlRSet rs;
    int s = rSets.size();
    for(int i = 0; i < s; i++)
        {
        rs = (jdbcMysqlRSet) rSets.elementAt(i);
        if(!rs.isRetrieved())
            {
            rs.retrieved(true);
            return rs;
            }
        }
    return null;
    }


/** Are there unretrieved RS's? */

boolean areUnretrieved()
    {
    if(rSets == null) return false;
    jdbcMysqlRSet rs;
    int s = rSets.size();
    for(int i = 0; i < s; i++)
        {
        rs = (jdbcMysqlRSet) rSets.elementAt(i);
        if(!rs.isRetrieved()) return true;
        }
    return false;
    }

/*-------------------------------------------------------------------+
|                  create a pseudo result                            |
+-------------------------------------------------------------------*/

void pseudoResult(Vector r, RSMd m) throws SQLException
    {
    clear();
    cacheMode = CM_PHONEY;    
    catalog = new String(connex.catalog);
    metadata = m;
    if(r == null) mcache = new Vector();
    else mcache = r;
    realRowCount = readRowCount = mcache.size();
    }       
        
/*-------------------------------------------------------------------+
|                      Error handler                                 |
+-------------------------------------------------------------------*/


void errHandlerM(int n) throws SQLException
    {
    String o = "\n" + errs[n];
    jdbcMysqlBase.errMessage(o);
    }

void errHandlerE(int n, Exception e) throws SQLException
    {
    String o = "\n" + errs[n] + jdbcMysqlBase.eMessage(e);
    jdbcMysqlBase.errMessage(o);
    }

void errHandlerNM(int n, int u, String s) throws SQLException
    {
    String o = "\n" + errs[n] + u + " " + s;
    jdbcMysqlBase.errMessage(o);
    }

} 
