/**
*** Program jdbcMysqlRSet.java
***    in product twz1jdbcForMysql, 
***    Copyright 1997, 1998 by Terrence W. Zellers.
***   
***  All rights explicitly reserved.
***
***  See file "LICENSE" in this package for conditions of use.
**/

package twz1.jdbc.mysql;
import java.io.ByteArrayInputStream;
import java.io.ObjectInputStream;
import java.io.InputStream;
import java.math.BigDecimal;
import java.sql.*;
import java.util.StringTokenizer;
import java.util.Vector;

/** This class implements Result Set to return SQL data to the
*** application.  Since MySQL returns all numeric data
*** as text strings, we treat all columns as such regardless of 
*** their defined type and parse that string into the requested
*** type, only throwing an exception if the parse is impossible.
*** At a later time I may validate conversions strictly according
*** to the API, but for now, let it fly.
**/
public final class jdbcMysqlRSet implements ResultSet
{
/** My object id */
int myOID;

/** Access the result */
jdbcMysqlResult  result;

/** Access the statement */
jdbcMysqlStmt    statement;

/** Has this RS been retrieved */
boolean retrieved;

/** Access to the input */
jdbcMysqlBag     inBag;

/** Access to the mapping */
RSMd metadata;

/** Current row */
int currentRow;

/** Have we been closed */
boolean isClosed;

/** Have we exhaused data */
boolean isDone;

/** Have we even started yet */
boolean isStarted;

/** Are we done */
boolean noMoreData;

/** A bit of API silliness */
boolean lastNull;

/** do we lock before answering inquiries ? */
boolean rsLock;

/* Can we getXXX data? */
boolean gettable;

/** Do we throw exception in getDate before 1900? */
boolean rsDateXcept;

/** localize column count */
int columnCount;

/* localize cachemode */
boolean cached;

/* Localize statement lock timeout */
int lTimeout;

/* Localize maxRows */
int maxRows;

/* The row parse data */
Row xRow;

/* Localized ref to data cache */
Vector mcache;

/* And size */
int mcacheSize;

static final int GETSTRING        = 0;
static final int GETBOOLEAN       = 1;
static final int GETBYTE          = 2;
static final int GETSHORT         = 3;
static final int GETINT           = 4;
static final int GETLONG          = 5;
static final int GETFLOAT         = 6;
static final int GETDOUBLE        = 7;
static final int GETBIGDECIMAL    = 8;
static final int GETBYTES         = 9;
static final int GETDATE          = 10;
static final int GETTIME          = 11;
static final int GETTIMESTAMP     = 12;
static final int GETASCIISTREAM   = 13;
static final int GETBINARYSTREAM  = 14;
static final int GETUNICODESTREAM = 15;
static final int GETOBJECT        = 16;
static final int FINDCOLUMN       = 17;
static final int GETMETADATA      = 18;
static final int WASNULL          = 19;

static final String nameIt[] = {
                           "getString",
                           "getBoolean", 
                           "getByte",
                           "getShort",
                           "getInt", 
                           "getLong",
                           "getFloat",
                           "getDouble",
                           "getBigDecimal", 
                           "getBytes",
                           "getDate",
                           "getTime",
                           "getTimestamp",
                           "getAsciiStream",
                           "getBinaryStream",
                           "getUnicodeStream",
                           "getObject",
                           "findColumn",
                           "getMetaData",
                           "wasNull",
                               };

static final String errs[] = {
       "E1400 Result set has been closed.",
       "E1401 Results exhausted.",
       "E1402 Error from next()", 
       "E1403 A valid row has not been requested.",
       "E1404 Error in wasNull()", 
       "E1405 Error recognizing data for API", 
       "E1406 Data is not recognisable as numeric", 
       "E1407 Value exceeds the range for this type.", 
       "P1408 Program error",
       "E1409 getBigDecimal scaling error.",
       "E1410 Too many tokens!",
       "E1411 Error parsing date/time object.",
       "E1412 Status or synchronization error.",
       "E1413 Error in status retrieving metadata.",
       "E1414 Error retrieving column by name.",
       "E1415 Error in object conversion - ", 
       "E1416 Error in getObject converting MySQL type ",
       "E1417 Error retrieving data - ",
                             };


/*===================================================================+
||                           Constructor                            ||
+===================================================================*/

jdbcMysqlRSet(jdbcMysqlResult mresult) throws SQLException
    {
    this.myOID = jdbcMysqlBase.getOID();
    this.result = mresult;
    this.statement = result.statement;
    this.inBag = result.inBag;
    this.metadata = result.metadata;
    this.currentRow = -1;
    this.isClosed = false;
    this.isDone = false;
    this.noMoreData = false;
    this.isStarted = false;
    this.retrieved = false;
    this.gettable = false;
    this.rsLock = statement.rsLock;
    result.addRS(this);
    this.rsDateXcept = false;
    this.columnCount = metadata.cc;
    this.xRow = new Row(result);
    this.lTimeout = statement.lTimeout;
    this.maxRows  = statement.maxRows;

    if(result.cacheMode == result.CM_MEMORY 
           || result.cacheMode == result.CM_PHONEY) cached = true;
    else cached = false;

    if(cached){ mcache = result.mcache; mcacheSize = mcache.size(); }
    else { mcache = null;  mcacheSize = -1; }

    if(statement.cx != null)
        {
        String test = 
               statement.cx.curl.getProperty("rsDateXcept");
        if(jdbcMysqlBase.boolValue(test, 0, 0) == 1) 
               rsDateXcept = true;
        }
    }

/*===================================================================+
||                    Package cooperative methods                   ||
+===================================================================*/


/** We be closed! */
void iClose() throws SQLException
    {
    if(isClosed) return;
    result.cacheClose(noMoreData);
    result.doneRS(myOID);     
    isClosed = true;
    gettable = false; 
    result = null;
    statement = null;
    mcache = null;
    xRow = null;
    metadata = null;
    inBag = null;
    }

/** Tell others who I am. */
int myID() { return myOID; }


/*===================================================================+
||                        API methods                               ||
+===================================================================*/

/*-------------------------------------------------------------------+
|                      Next row of data                              |
+-------------------------------------------------------------------*/

/** See API.  Get next row */
public boolean next() throws SQLException
    {
    int reqid = jdbcMysqlBase.getOID();
    boolean lockset = false;
    try {
        lockset = statement.lock(true, reqid, lTimeout);
        if(isClosed) errHandlerM(0, " -- next()");       
        if(!isDone)
            {
            isStarted = true;
            gettable = true;
            currentRow++;
            if(currentRow < maxRows)
                {
                if(cached)
		    {
                    if(currentRow < mcacheSize)
                           xRow.init(mcache.elementAt(currentRow));
                    else noMoreData = true;
                    }
                else{
                    if(!result.inEoD(inBag)) xRow.init(inBag.inside);
                    else noMoreData = true;  
                    }
                }           
            if(noMoreData || currentRow >= maxRows) 
                { 
                isDone = true;
                gettable = false;
                result.cacheClose(noMoreData);
                noMoreData = true;
                }
            if(result.dataError) 
                    errHandlerNM(17, result.lastStatus, result.sMessage);
            }
        lockset = statement.lock(false, reqid, lTimeout);
        }
    catch(Exception e)
        {
        isDone = true;
        gettable = false;
        result.cacheClose(noMoreData);
        if(lockset) statement.lock(false, reqid, lTimeout);
        errHandlerE(2, e);
        }
    return !isDone;
    }

/*-------------------------------------------------------------------+
|                      close the RS                                  |
+-------------------------------------------------------------------*/

/** See API. */
public void close() throws SQLException
    {
    if(isClosed) return;
    jdbcMysqlStmt st = (jdbcMysqlStmt) statement;
    int reqid = jdbcMysqlBase.getOID();
    st.lock(true, reqid, lTimeout);
    iClose();
    st.lock(false, reqid, lTimeout);
    }

/*-------------------------------------------------------------------+
|                        Get warnings                                |
+-------------------------------------------------------------------*/

/** We don't support warnings at this time.  Do the sync just to
*** stay kosher
**/
public SQLWarning getWarnings() throws SQLException 
    {
    nopSynch("getWarnings()");
    return null;
    }

/*-------------------------------------------------------------------+
|                         clear warnings                             |
+-------------------------------------------------------------------*/

/** I said we don't support warnings! */
public void clearWarnings() throws SQLException 
    { nopSynch("clearWarnings()");}

/*-------------------------------------------------------------------+
|                      get cursor name                               |
+-------------------------------------------------------------------*/

/** MySQL doesn't do cursors.  Puke as required by the API. */
public String getCursorName() throws SQLException 
    {
    nopSynch("getCursorName()"); 
    errHandlerM(11, "");
    return null; 
    }



/*===================================================================+
||                    API get methods                               ||
+===================================================================*/

/**   A thoughtless API request.   There is no way to make
*** this threadsafe!
**/
public boolean wasNull() throws SQLException
    { return ((Integer)getIt(WASNULL, 0, null, 2)).intValue() == 1; }

public ResultSetMetaData getMetaData() throws SQLException 
    {
    return (ResultSetMetaData) getIt(GETMETADATA, 0, null, 2);
    }

/** Return the ordinal column */
public int findColumn(String name) throws SQLException
    { 
    Integer I = (Integer) getIt(FINDCOLUMN, 0, name, 2);
    return I.intValue();
    }
    
/** See the API */
public String getString(int c) throws SQLException
    { return (String) getIt(GETSTRING, c, null, 45); }

/** See the API */
public String getString(String c) throws SQLException
    { return (String) getIt(GETSTRING, 0, c, 53); }
    

/** See the API.  Nominally MySQL does not support booleans, but I will
*** interpret "true", "false", "yes", "no", "y", "n", "0" or "1" as boolean
*** data.
***/
public boolean getBoolean(int c) throws SQLException
    {
    Boolean t = (Boolean) getIt(GETBOOLEAN, c, null, 45); 
    return t.equals(Boolean.TRUE);
    }

/** See the API.  Nominally MySQL does not support booleans, but I will
*** interpret "true", "false", "yes", "no", "y", "n", "0" or "1" as boolean
*** data.
***/
public boolean getBoolean(String c) throws SQLException
    {
    Boolean t = (Boolean) getIt(GETBOOLEAN, 0, c, 53); 
    return t.equals(Boolean.TRUE);
    }


/** See API. */
public byte getByte(int c) throws SQLException
   { 
   Byte v = (Byte) getIt(GETBYTE, c, null, 45);
   return v.byteValue();
   }
    
/** See API. */
public byte getByte(String c) throws SQLException
   { 
   Byte v = (Byte) getIt(GETBYTE, 0, c, 53);
   return v.byteValue();
   }


/** See API. */
public short getShort(int c) throws SQLException
   { 
   Short v = (Short) getIt(GETSHORT, c, null, 45);
   return v.shortValue();
   }

/** See API. */
public short getShort(String c) throws SQLException
   { 
   Short v = (Short) getIt(GETSHORT, 0, c, 53);
   return v.shortValue();
   }
    

/** See API. */
public int getInt(int c) throws SQLException
   { 
   Integer v = (Integer) getIt(GETINT, c, null, 45);
   return v.intValue();
   }

/** See API. */
public int getInt(String c) throws SQLException
   { 
   Integer v = (Integer) getIt(GETINT, 0, c, 53);
   return v.intValue();
   }


/** See API. */
public long getLong(int c) throws SQLException
   { 
   Long v = (Long) getIt(GETLONG, c, null, 45);
   return v.longValue();
   }

/** See API. */
public long getLong(String c) throws SQLException
   { 
   Long v = (Long) getIt(GETLONG, 0, c, 53);
   return v.longValue();
   }


/** See API */
public float getFloat(int c) throws SQLException
    {
    Float v = (Float) getIt(GETFLOAT, c, null, 45);
    return v.floatValue();
    }

/** See API */
public float getFloat(String c) throws SQLException
    {
    Float v = (Float) getIt(GETFLOAT, 0, c, 53);
    return v.floatValue();
    }


/** See the API */
public double getDouble(int c) throws SQLException
   {
   Double v = (Double) getIt(GETDOUBLE, c, null, 45);
   return v.doubleValue();
   }

/** See the API */
public double getDouble(String c) throws SQLException
   {
   Double v = (Double) getIt(GETDOUBLE, 0, c, 53);
   return v.doubleValue();
   }


/** See API */
public BigDecimal getBigDecimal(int c, int s) throws  SQLException
    {
    BigDecimal bd =  (BigDecimal) getIt(GETBIGDECIMAL, c, null, 45);
    try { bd = bd.setScale(s);   }
    catch(Exception e) { errHandlerE(9, e); }
    return bd;
    }

/** See API */
public BigDecimal getBigDecimal(String c, int s) throws  SQLException
    {
    BigDecimal bd =  (BigDecimal) getIt(GETBIGDECIMAL, 0, c, 53);
    try { bd = bd.setScale(s);   }
    catch(Exception e) { errHandlerE(9, e); }
    return bd;
    }


/** See API */
public byte[] getBytes(int c) throws SQLException
    { return (byte[]) getIt(GETBYTES, c, null, 77); }

/** See API */
public byte[] getBytes(String c) throws SQLException
    { return (byte[]) getIt(GETBYTES, 0, c, 85); }
    
    
/** See API */
public Date getDate(int c) throws SQLException
    { return (Date) getIt(GETDATE, c, null, 45); }
    
/** See API */
public Date getDate(String c) throws SQLException
    { return (Date) getIt(GETDATE, 0, c, 53); }


/** See API */
public Time getTime(int c) throws SQLException
    { return (Time) getIt(GETTIME, c, null, 45); }

/** See API */
public Time getTime(String c) throws SQLException
    { return (Time) getIt(GETTIME, 0, c, 53); }


/** See API */
public Timestamp getTimestamp(int c) throws SQLException
    { return (Timestamp) getIt(GETTIMESTAMP, c, null, 45); }

/** See API */
public Timestamp getTimestamp(String c) throws SQLException
    { return (Timestamp) getIt(GETTIMESTAMP, 0, c, 53); }


/** See API.  Because MySQL does not distinguish non-ASCII characters
*** getAsciiInputStream and getBinaryInputStream are treated the same.
**/
public InputStream getAsciiStream(int c) throws SQLException
    { return (InputStream) getIt(GETASCIISTREAM, c, null, 77); }

/** See API.  Because MySQL does not distinguish non-ASCII characters
*** getAsciiInputStream and getBinaryInputStream are treated the same.
**/
public InputStream getAsciiStream(String c) throws SQLException
    { return (InputStream) getIt(GETASCIISTREAM, 0, c, 85); }


/** See API.  Because MySQL does not distinguish non-ASCII characters
*** getAsciiInputStream and getBinaryInputStream are treated the same.
**/
public InputStream getBinaryStream(int c) throws SQLException
    { return (InputStream) getIt(GETBINARYSTREAM, c, null, 77); }

/** See API.  Because MySQL does not distinguish non-ASCII characters
*** getAsciiInputStream and getBinaryInputStream are treated the same.
**/
public InputStream getBinaryStream(String c) throws SQLException
    { return (InputStream) getIt(GETBINARYSTREAM, 0, c, 85); }


/** See API.  As implemented here this really does assume that
*** String is unicode and that getBytes() returns a byte representation
*** of the unicode string.  Will fix this when/if Sun gets their act
*** together re encoding and if/when someone complains (they will!).
**/
public InputStream getUnicodeStream(int c) throws SQLException
    { return (InputStream) getIt(GETUNICODESTREAM, c, null, 45); }

/** See API.  As implemented here this really does assume that
*** String is unicode and that getBytes() returns a byte representation
*** of the unicode string.  Will fix this when/if Sun gets their act
*** together re encoding and if/when someone complains (they will!).
**/
public InputStream getUnicodeStream(String c) throws SQLException
    { return (InputStream) getIt(GETUNICODESTREAM, 0, c, 53); }


/** getObject returns a byte array as an object */
public Object getObject(int c) throws SQLException
    { return getIt(GETOBJECT, c, null, 45); }

/** getObject returns a byte array as an object */
public Object getObject(String c) throws SQLException
    { return getIt(GETOBJECT, 0, c, 53); }


/*===================================================================+
||           Do the dirty work of digging out a column value        ||
+===================================================================*/

private Object getIt(int huh, int cnum, String cnam, int tflag) 
         throws SQLException
    {
    int flag = tflag;
    int reqid = 0;
    boolean lockset = false;
    Object it = null;
    byte[] b = null;
    String work =  null; 
    String how = "(not-determined yet)";
    int what = huh;
    jdbcMysqlField f = null;
    int n = -1;
    try {
        if(rsLock)
            {
            reqid = jdbcMysqlBase.getOID();
            lockset = statement.lock(true, reqid, lTimeout);
            }
        if((flag & 1) > 0) if(!gettable)testStatus(nameIt[what]);
        if((flag & 2) > 0) if(isClosed) testStatus(nameIt[what]);
        if((flag & 4) > 0) lastNull = false;
        if((flag & 8) > 0) if(cnam == null) n = cnum - 1;
        if((flag & 16) > 0)
            {
            f = fByName(cnam);
            n = f.columnNumber;
	    }
        if(what == GETOBJECT)
            {
            if(f == null) f = metadata.getField(n+1);
            switch(f.ctype)
                {
                case f.FIELD_TYPE_TINY     : what = GETBYTE;       break;
                case f.FIELD_TYPE_SHORT    : what = GETSHORT;      break;
                case f.FIELD_TYPE_LONG     : what = GETINT;        break;
                case f.FIELD_TYPE_FLOAT    : what = GETFLOAT;      break;
                case f.FIELD_TYPE_DECIMAL  : what = GETBIGDECIMAL; break;
                case f.FIELD_TYPE_DOUBLE   : what = GETDOUBLE;     break;
                case f.FIELD_TYPE_TIMESTAMP: what = GETTIMESTAMP;  break;
                case f.FIELD_TYPE_INT24     :
                case f.FIELD_TYPE_LONGLONG : what = GETLONG;       break;
                case f.FIELD_TYPE_DATE     : what = GETDATE;       break;
                case f.FIELD_TYPE_TIME     : what = GETTIME;       break;
                case f.FIELD_TYPE_DATETIME : what = GETTIMESTAMP;  break;
                case f.FIELD_TYPE_ENUM     :
                case f.FIELD_TYPE_SET      : what = GETSTRING;     break;
                case f.FIELD_TYPE_STRING   :
                case f.FIELD_TYPE_BLOB     :
                case f.FIELD_TYPE_TINY_BLOB :
                case f.FIELD_TYPE_MEDIUM_BLOB:
                case f.FIELD_TYPE_LONG_BLOB :
                case f.FIELD_TYPE_VAR_STRING:
                    if((f.cflags & f.BINARY_FLAG) > 0) 
                        {
                        what = GETBYTES;
                        flag += 32;
                        }
                    else what = GETSTRING; 
                    break;
                default: errHandlerM(16, String.valueOf(what));
                }
            } 

        if((flag & 32) > 0)  work = xRow.getString(n);
        if((flag & 64) > 0)  b = xRow.getBytes(n);

        switch(what)
            {
            case GETASCIISTREAM:
            case GETBINARYSTREAM:
                if(b == null) { lastNull = true; it = null; }
                else it = (InputStream) new ByteArrayInputStream(b);
                break;

            case GETBYTES: 
                if(b == null) { lastNull = true; it = null; }
                else{
                    int bs = b.length;
                    byte[] u = new byte[bs];
                    System.arraycopy(b, 0, u, 0, bs);
                    it = u;
                    }
                break;

            case GETMETADATA: it = metadata; break;

            case FINDCOLUMN: it = new Integer(n+1); break;

            case GETSTRING:
                if(work == null){ lastNull = true; break; }
                it = work;
                break;

            case GETBOOLEAN:
                if(work == null) {lastNull = true; work = "false"; }
                it = new Boolean(testBool(work));
                break;

            case GETBYTE:
                if(work == null) { lastNull = true; work = "0"; }
                try { it = new Byte(work); }
                catch(NumberFormatException e) { errHandlerM(6, ""); }
                break;

            case GETSHORT:
                if(work == null) { lastNull = true; work = "0"; }
                try { it = new Short(work); }
                catch(NumberFormatException e) { errHandlerM(6, ""); }
                break;

            case GETINT:
                if(work == null) { lastNull = true; work = "0"; }
                try { it = new Integer(work); }
                catch(NumberFormatException e) { errHandlerM(6, ""); }
                break;

            case GETLONG:
                if(work == null) { lastNull = true; work = "0"; }
                try { it = new Long(work); }
                catch(NumberFormatException e) { errHandlerM(6, ""); }
                break;

            case GETFLOAT:
                if(work == null) { lastNull = true; work = "0"; }
                try { it = new Float(work); }
                catch(NumberFormatException e) { errHandlerM(6,""); }
                break;

            case GETDOUBLE:
                if(work == null) { lastNull = true; work = "0"; }
                try { it = new Double(work); }
                catch(NumberFormatException e) { errHandlerM(6,""); }
                break;

            case GETBIGDECIMAL:
                if(work == null) { lastNull = true; work = "0"; }
                try{ it = new BigDecimal(work); }
                catch(NumberFormatException e) { errHandlerM(6, ""); }
                break;

            case GETDATE:
                if(work == null) { lastNull = true; it = null; }
                else{
                    int[] tdate = nParse(work, 3);
                    if(!rsDateXcept && tdate[0] < 1900)
                        { lastNull = true; it = null; }
                    else{
                        try {
                            it = new Date(tdate[0]-1900, tdate[1]-1, 
                                  tdate[2]);
                            }
                        catch(Exception e)
                            { errHandlerE2b(15, what, work, e); }
                        }
                    }
                break;

            case GETTIME:
                if(work == null) { lastNull = true; it = null; }
                else{
                    try {
                        int[] ttime = nParse(work, 3);
                        it = new Time(ttime[0], ttime[1], ttime[2]);
                        }
                    catch(Exception e)
                        { errHandlerE2b(15, what, work, e); }
                    }
                break;

            case GETTIMESTAMP:
                if(work == null) { lastNull = true; it = null; }
                else{
                    int yy, mm, dd, hh, ii, ss;
                    try { 
                        String sw, sj;
                        sw = work;
                        int sl;
                        if(work.indexOf("-") < 0)
                            {
                            if(sw.length() < 14) sw = new String("19"+sw);
                            sl = sw.length();
                            if( sl < 14) sl = 14 - sl;
                            sj = "19000101000001".substring(sl);
                            sw = new String(sw + sj);
                            yy = Integer.parseInt(sw.substring(0, 4));
                            mm = Integer.parseInt(sw.substring(4, 6));
                            dd = Integer.parseInt(sw.substring(6, 8));
                            hh = Integer.parseInt(sw.substring(8, 10));
                            ii = Integer.parseInt(sw.substring(10,12));
                            ss = Integer.parseInt(sw.substring(12, 14));
                            }
                        else{
                            int[] td = nParse(work, 6);
                            yy = td[0];  mm = td[1];  dd = td[2];
                            hh = td[3];  ii = td[4];  ss = td[5];
                            if(yy < 1900) yy+= 1900;
                            if(mm < 1) mm = 1;
                            if(dd < 1) dd = 1;
                            }
                        it = new Timestamp(yy-1900, mm-1, dd, hh, 
                                           ii, ss, 0);
                        }
                    catch(Exception e) 
                        { errHandlerE2b(15, what, work, e); }
                    }
                break;

            case GETUNICODESTREAM:
                if(work == null) { lastNull = true; it = null; }
                else{
                    b = work.getBytes();
                    it = (InputStream) new ByteArrayInputStream(b);
                    }
                break;

            case WASNULL: it = new Integer( lastNull ? 1 : 0 ); break;
               
            default: errHandlerM(8, "");
            
            }
        if(rsLock)lockset = statement.lock(false, reqid, lTimeout);
        }
    catch(Exception e)
        {
        if(lockset) statement.lock(false, reqid, lTimeout);
        how = cnam != null ? "(String)" : "(int)";
        errHandlerE2(5, huh, how, e);
        }
    return it;
    }


/*-------------------------------------------------------------------+
|                     parse a date/time string                       |
+-------------------------------------------------------------------*/

private int[] nParse(String s, int tt) throws SQLException
    {
    int[] ia = new int[tt];
    int i;
    String w;
    for(i = 0; i < tt; i++) ia[i] = 0;
    try {
        StringTokenizer st = new StringTokenizer(s, " -:.");
        i = 0;
        while(st.hasMoreTokens() && i < tt)
            {
            if(i >= tt) errHandlerM(10, s);
            w = st.nextToken();
            ia[i++] = Integer.parseInt(w);
            }
        }
    catch(Exception e) { errHandlerE(11, e); }
    return ia;
    }


/*-------------------------------------------------------------------+
|              extract the boolean meaning of a string               |
+-------------------------------------------------------------------*/

private boolean testBool(String work) throws SQLException
    {
    if(  work.equalsIgnoreCase("true")
      || work.equalsIgnoreCase("t")
      || work.equalsIgnoreCase("yes") 
      || work.equalsIgnoreCase("y")
      || work.equals("1")
      ) return true;
    if(  work.equalsIgnoreCase("false")
      || work.equalsIgnoreCase("f")
      || work.equalsIgnoreCase("no")
      || work.equalsIgnoreCase("n")
      || work.equals("0")
      ) return false;
    errHandlerM(5, "");
    return false;
    }

/*-------------------------------------------------------------------+
|                    A nop status and synchronization                |
+-------------------------------------------------------------------*/

private void nopSynch(String n) throws SQLException
    {
    int reqid = jdbcMysqlBase.getOID();
    boolean lockset = false;
    try {
        lockset = statement.lock(true, reqid, statement.lTimeout);
        testStatus(n);
        lockset = statement.lock(false, reqid, statement.lTimeout);
        }
    catch(Exception e)
        {
        if(lockset) statement.lock(false, reqid, statement.lTimeout);
        errHandlerE(12, e);
        }
     }


/*-------------------------------------------------------------------+
|               Return a column based upon its name                  |
+-------------------------------------------------------------------*/

private jdbcMysqlField fByName(String cnam) throws SQLException
    {
    jdbcMysqlField f = null;
    int n;
    n = cnam.indexOf('.');
    if(n > -1)
        {
        String table = cnam.substring(0, n);
        String col = cnam.substring(n+1);
         f = metadata.getField(table, col);
         }
     else f = metadata.getField(cnam);
     return f;
     }


/*-------------------------------------------------------------------+
|                      package   methods                             |
+-------------------------------------------------------------------*/

void retrieved(boolean r) { retrieved = r; }
boolean isRetrieved() { return retrieved; }

/*===================================================================+
||                         common methods                           ||
+===================================================================*/

private void testStatus(String s) throws SQLException
    {
    if(isClosed) errHandlerM(0, s);
    if(isDone)   errHandlerM(1, s);
    if(!isStarted) errHandlerM(3, s);
    }

private void errHandlerE(int n, Exception e) throws SQLException
    {
    String o = jdbcMysqlBase.eMessage(e);
    errHandlerM(n, o);
    }

private void errHandlerE2(int n, int where, String how,  Exception e) 
        throws SQLException
    {
    String o = " " + nameIt[where] + how  + jdbcMysqlBase.eMessage(e);
    errHandlerM(n, o);
    }

private void errHandlerE2b(int n, int where, String how,  Exception e) 
        throws SQLException
    {
    String o = " " + nameIt[where] + " " + how  
             + jdbcMysqlBase.eMessage(e);
    errHandlerM(n, o);
    }

private void errHandlerM(int n, String s) throws SQLException
    {
    String o = new String("\n" + errs[n] + " " + s);
    jdbcMysqlBase.errMessage(o);
    }

private void errHandlerNM(int n, int u, String s) throws SQLException
    {
    String o = new String("\n" + errs[n] + u +" " + s);
    jdbcMysqlBase.errMessage(o);
    }

}
