/*
 * MySQL PMDA
 *
 * Copyright (C) 1999 Silicon Graphics, Inc.  All Rights Reserved.
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as published
 * by the Free Software Fondation.
 * 
 * This program is distributed in the hope that it would be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  Further, any license provided herein,
 * whether implied or otherwise, is limited to this program in accordance with
 * the express provisions of the GNU General Public License.  Patent licenses,
 * if any, provided herein do not apply to combinations of this program with
 * other product or programs, or any other product whatsoever.  This program is
 * distributed without any warranty that the program is delivered free of the
 * rightful claim of any third person by way of infringement or the like.  See
 * the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston MA 02111-1307, USA.
 */

#include <stdio.h>
#include <ctype.h>
#include <libgen.h>
#include <limits.h>
#include <time.h>
#include <string.h>
#include <syslog.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/times.h>
#include <sys/mman.h>
#include <mysql/mysql.h>
#include "pmapi.h"
#include "pmda.h"
#include "./domain.h"

/*
 * list of instances
 */

static pmdaIndom indomtab[] = {
#define PROCESS_INDOM	0
    { PROCESS_INDOM, 0, NULL },
};

/* Options */
static char *host;
static char *user;
static char *pass;


static pmdaMetric metrictab[] = {

/* status.aborted_clients */
    { "status.aborted_clients", 
      { PMDA_PMID(0,0), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_COUNTER, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* status.aborted_connects */
    { "status.aborted_connects", 
      { PMDA_PMID(0,1), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_COUNTER, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* status.created_tmp_tables */
    { "status.created_tmp_tables", 
      { PMDA_PMID(0,2), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_COUNTER, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* status.delayed_insert_threads */
    { "status.delayed_insert_threads", 
      { PMDA_PMID(0,3), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_COUNTER, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* status.delayed_writes */
    { "status.delayed_writes", 
      { PMDA_PMID(0,4), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_COUNTER, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* status.delayed_errors */
    { "status.delayed_errors", 
      { PMDA_PMID(0,5), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_COUNTER, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* status.flush_commands */
    { "status.flush_commands", 
      { PMDA_PMID(0,6), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_COUNTER, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* status.handler_delete */
    { "status.handler_delete", 
      { PMDA_PMID(0,7), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_COUNTER, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* status.handler_read_first */
    { "status.handler_read_first", 
      { PMDA_PMID(0,8), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_COUNTER, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* status.handler_read_key */
    { "status.handler_read_key", 
      { PMDA_PMID(0,9), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_COUNTER, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* status.handler_read_next */
    { "status.handler_read_next", 
      { PMDA_PMID(0,10), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_COUNTER, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* status.handler_read_rnd */
    { "status.handler_read_rnd", 
      { PMDA_PMID(0,11), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_COUNTER, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* status.handler_update */
    { "status.handler_update", 
      { PMDA_PMID(0,12), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_COUNTER, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* status.handler_write */
    { "status.handler_write", 
      { PMDA_PMID(0,13), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_COUNTER, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* status.key_blocks_used */
    { "status.key_blocks_used", 
      { PMDA_PMID(0,14), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_COUNTER, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* status.key_read_requests */
    { "status.key_read_requests", 
      { PMDA_PMID(0,15), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_COUNTER, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* status.key_reads */
    { "status.key_reads", 
      { PMDA_PMID(0,16), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_COUNTER, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* status.key_write_requests */
    { "status.key_write_requests", 
      { PMDA_PMID(0,17), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_COUNTER, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* status.key_writes */
    { "status.key_writes", 
      { PMDA_PMID(0,18), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* status.max_used_connections */
    { "status.max_used_connections", 
      { PMDA_PMID(0,19), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* status.not_flushed_key_blocks */
    { "status.not_flushed_key_blocks", 
      { PMDA_PMID(0,20), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* status.not_flushed_delayed_rows */
    { "status.not_flushed_delayed_rows", 
      { PMDA_PMID(0,21), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* status.open_tables */
    { "status.open_tables", 
      { PMDA_PMID(0,22), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* status.open_files */
    { "status.open_files", 
      { PMDA_PMID(0,23), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* status.open_streams */
    { "status.open_streams", 
      { PMDA_PMID(0,24), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_COUNTER, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* status.opened_tables */
    { "status.opened_tables", 
      { PMDA_PMID(0,25), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_COUNTER, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* status.questions */
    { "status.questions", 
      { PMDA_PMID(0,26), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_COUNTER, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* status.running_threads */
    { "status.running_threads", 
      { PMDA_PMID(0,27), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* status.slow_queries */
    { "status.slow_queries", 
      { PMDA_PMID(0,28), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_COUNTER, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* status.uptime */
    { "status.uptime", 
      { PMDA_PMID(0,29), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_COUNTER, 
        PMDA_PMUNITS(1,0,0,PM_TIME_SEC,0,0) }, },



/* variables.back_log */
    { "variables.back_log", 
      { PMDA_PMID(1,0), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* variables.connect_timeout */
    { "variables.connect_timeout", 
      { PMDA_PMID(1,1), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(1,0,0,PM_TIME_SEC,0,0) }, },
/* variables.basedir */
    { "variables.basedir", 
      { PMDA_PMID(1,2), PM_TYPE_STRING, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(0,0,0,0,0,0) }, },
/* variables.datadir */
    { "variables.datadir", 
      { PMDA_PMID(1,3), PM_TYPE_STRING, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(0,0,0,0,0,0) }, },
/* variables.delayed_insert_limit */
    { "variables.delayed_insert_limit", 
      { PMDA_PMID(1,4), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* variables.delayed_insert_timeout */
    { "variables.delayed_insert_timeout", 
      { PMDA_PMID(1,5), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(1,0,0,PM_TIME_SEC,0,0) }, },
/* variables.delayed_queue_size */
    { "variables.delayed_queue_size", 
      { PMDA_PMID(1,6), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* variables.join_buffer */
    { "variables.join_buffer", 
      { PMDA_PMID(1,7), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(1,0,0,PM_SPACE_BYTE,0,0) }, },
/* variables.flush_time */
    { "variables.flush_time", 
      { PMDA_PMID(1,8), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(1,0,0,PM_TIME_SEC,0,0) }, },
/* variables.key_buffer */
    { "variables.key_buffer", 
      { PMDA_PMID(1,9), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(1,0,0,PM_SPACE_BYTE,0,0) }, },
/* variables.language */
    { "variables.language", 
      { PMDA_PMID(1,10), PM_TYPE_STRING, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(0,0,0,0,0,0) }, },
/* variables.log */
    { "variables.log", 
      { PMDA_PMID(1,11), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(0,0,0,0,0,0) }, },
/* variables.log_update */
    { "variables.log_update", 
      { PMDA_PMID(1,12), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(0,0,0,0,0,0) }, },
/* variables.long_query_time */
    { "variables.long_query_time", 
      { PMDA_PMID(1,13), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(1,0,0,PM_TIME_SEC,0,0) }, },
/* variables.low_priority_updates */
    { "variables.low_priority_updates", 
      { PMDA_PMID(1,14), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(0,0,0,0,0,0) }, },
/* variables.max_allowed_packet */
    { "variables.max_allowed_packet", 
      { PMDA_PMID(1,15), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(1,0,0,PM_SPACE_BYTE,0,0) }, },
/* variables.max_connections */
    { "variables.max_connections", 
      { PMDA_PMID(1,16), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* variables.max_connect_errors */
    { "variables.max_connect_errors", 
      { PMDA_PMID(1,17), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* variables.max_delayed_insert_threads */
    { "variables.max_delayed_insert_threads", 
      { PMDA_PMID(1,18), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* variables.max_join_size */
    { "variables.max_join_size", 
      { PMDA_PMID(1,19), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(1,0,0,PM_SPACE_BYTE,0,0) }, },
/* variables.max_sort_length */
    { "variables.max_sort_length", 
      { PMDA_PMID(1,20), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(1,0,0,PM_SPACE_BYTE,0,0) }, },
/* variables.net_buffer_length */
    { "variables.net_buffer_length", 
      { PMDA_PMID(1,21), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(1,0,0,PM_SPACE_BYTE,0,0) }, },
/* variables.port */
    { "variables.port", 
      { PMDA_PMID(1,22), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(0,0,0,0,0,0) }, },
/* variables.protocol_version */
/* NOTE: MySQL sends dash, but I have converted it to underscore in pmns
   to be consistent with the other names. However,
   "user data" are used to match strings incoming from daemon, so 
   the string below must contain dash */
    { "variables.protocol-version", 
      { PMDA_PMID(1,23), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(0,0,0,0,0,0) }, },
/* variables.record_buffer */
    { "variables.record_buffer", 
      { PMDA_PMID(1,24), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(1,0,0,PM_SPACE_BYTE,0,0) }, },
/* variables.skip_locking */
    { "variables.skip_locking", 
      { PMDA_PMID(1,25), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(0,0,0,0,0,0) }, },
/* variables.socket */
    { "variables.socket", 
      { PMDA_PMID(1,26), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(0,0,0,0,0,0) }, },
/* variables.sort_buffer */
    { "variables.sort_buffer", 
      { PMDA_PMID(1,27), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(1,0,0,PM_SPACE_BYTE,0,0) }, },
/* variables.table_cache */
    { "variables.table_cache", 
      { PMDA_PMID(1,28), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* variables.thread_stack */
    { "variables.thread_stack", 
      { PMDA_PMID(1,29), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(1,0,0,PM_SPACE_BYTE,0,0) }, },
/* variables.tmp_table_size */
    { "variables.tmp_table_size", 
      { PMDA_PMID(1,30), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(0,0,1,0,0,PM_COUNT_ONE) }, },
/* variables.tmpdir */
    { "variables.tmpdir", 
      { PMDA_PMID(1,31), PM_TYPE_STRING, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(0,0,0,0,0,0) }, },
/* variables.version */
    { "variables.version", 
      { PMDA_PMID(1,32), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(0,0,0,0,0,0) }, },
/* variables.wait_timeout */
    { "variables.wait_timeout", 
      { PMDA_PMID(1,33), PM_TYPE_U32, PM_INDOM_NULL, PM_SEM_DISCRETE, 
				PMDA_PMUNITS(1,0,0,PM_TIME_SEC,0,0) }, }
#if 0				
				,



/* processlist.id */
    { NULL, 
      { PMDA_PMID(2,0), PM_TYPE_U32, PROCESS_INDOM, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(0,0,0,0,0,0) }, },
/* processlist.user */
    { NULL, 
      { PMDA_PMID(2,1), PM_TYPE_STRING, PROCESS_INDOM, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(0,0,0,0,0,0) }, },
/* processlist.host */
    { NULL, 
      { PMDA_PMID(2,2), PM_TYPE_STRING, PROCESS_INDOM, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(0,0,0,0,0,0) }, },
/* processlist.db */
    { NULL, 
      { PMDA_PMID(2,3), PM_TYPE_STRING, PROCESS_INDOM, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(0,0,0,0,0,0) }, },
/* processlist.command */
    { NULL, 
      { PMDA_PMID(2,4), PM_TYPE_STRING, PROCESS_INDOM, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(0,0,0,0,0,0) }, },
/* processlist.time */
    { NULL, 
      { PMDA_PMID(2,5), PM_TYPE_U32, PROCESS_INDOM, PM_SEM_DISCRETE, 
				PMDA_PMUNITS(1,0,0,PM_TIME_SEC,0,0) }, },
/* processlist.state */
    { NULL, 
      { PMDA_PMID(2,6), PM_TYPE_STRING, PROCESS_INDOM, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(0,0,0,0,0,0) }, },
/* processlist.info */
    { NULL, 
      { PMDA_PMID(2,7), PM_TYPE_STRING, PROCESS_INDOM, PM_SEM_DISCRETE, 
        PMDA_PMUNITS(0,0,0,0,0,0) }, }
#endif
};

/*
 * Hash table to convert name to item number
 */
struct HTItem {
	struct HTItem *next;
	char *name;
	int type;
	int id;
};
typedef struct HTItem HTItem;

/* The hash size is just guessed. It might be worthy to find modulo with best
   average list length... */
#define HASH_SIZE 127
static HTItem *hashName2ID[HASH_SIZE];

/*
 * Structure where we store result. Either string or u32 is stored,
 * they are distingushed by length (integers have it -1). The length
 * is stored because most of the strings we get are constant so we do not need
 * to reallocate them
 */
typedef struct {
	short length;
	union {
	  char *string;
	  __uint32_t u32;
	} u;
} Result;

#define STATUS_DATA_NUM 30
Result statusData[STATUS_DATA_NUM];

#define VARIABLES_DATA_NUM 34
Result variablesData[VARIABLES_DATA_NUM];

#define MAX_RETRIES 5
static int getDataErrors=0; /* Current nesting level of mysql_getData routine */

/*
 * MySQL connection
 */
static MYSQL *DB;

static int statusGot=0,variablesGot=0;

/*
 * String hashing function, copied from glib
 */
static __uint32_t g_str_hash(char *s,unsigned int M)
{
  const char *p;
  __uint32_t h=0, g;

  for(p = s; *p != '\0'; p += 1) {
    h = ( h << 4 ) + tolower(*p);
    if ( ( g = h & 0xf0000000 ) ) {
      h = h ^ (g >> 24);
      h = h ^ g;
    }
  }

  return(h % M);
}

/*
 * Initializes MySQL connection
 */
static int mysql_initDB()
{
    if (DB) {
  	mysql_close(DB);
	DB = NULL;
    }

//  fprintf(stderr,"mysql_initDB() called!\n");

  DB = mysql_init(DB);
  DB = mysql_real_connect(DB,host,user,pass,NULL,0,NULL,0);
  if (!DB) {
//  	fprintf(stderr,"Cannot connect to DB!\n");
	return(-1);
  }
//  fprintf(stderr,"Connect error=%d\n",mysql_error(DB));
//  return(mysql_error(DB)?-1:0);
    return(0);
}

/*
 * Does MySQL query and stores result to data table
 */
static int mysql_getData(char *query,Result *data,int dataNum)
{
  MYSQL_RES *res;
  MYSQL_ROW row;
  HTItem *item;
  Result *r;
  int len;

  if ((getDataErrors > 0) || (!DB)) {
    /* (Re)connect to db */
    if (mysql_initDB() < 0) {
      /* We do not set getDataErrors to 0 to ensure we will get here
         when the function is called again */
      return(-1);
    }
  }

  if (mysql_real_query(DB,query,strlen(query))) {
  	if (getDataErrors >= MAX_RETRIES) {
	  getDataErrors = 0;
	  return(-1);
	}
	else {
	  getDataErrors++;
	  return(mysql_getData(query,data,dataNum));
	}
  }

  if (!(res = mysql_store_result(DB))) {
  	if (getDataErrors >= MAX_RETRIES) {
	  getDataErrors = 0;
	  return(-1);
	}
	else {
	  getDataErrors++;
	  return(mysql_getData(query,data,dataNum));
	}
  }

  while((row = mysql_fetch_row(res))) {
  	item = hashName2ID[g_str_hash(row[0],HASH_SIZE)];
	while(item) {
	  if (strcasecmp(item->name,row[0]) == 0) break;
	  item = item->next;
	}

//	fprintf(stderr,"'%s'='%s'\n",row[0],row[1]);

	if (!item) {
	  fprintf(stderr,"Warning: Unknown field from '%s': %s\n",query,row[0]);
	}
	else {
	  if (item->id >= dataNum) {
	    fprintf(stderr,"Error: Field '%s' with index %d out of data array!\n",row[0],item->id);
	  }
	  else {
	  	r = &data[item->id];
		if (item->type == PM_TYPE_U32) {
		  r->length = -1;
		  r->u.u32 = atol(row[1]);
		}
		else {
		  len = strlen(row[1]);
		  if (r->length != len) {
		  	/* Reallocate string */
			if (r->u.string) free(r->u.string);
			r->u.string = strdup(row[1]);
			r->length = len;
		  }
		  else {
		    /* Just overwrite */
		    strcpy(r->u.string,row[1]);
		  }
		}
	  }
	}
  }

  mysql_free_result(res);

  if (mysql_errno(DB)) {
  	if (getDataErrors >= MAX_RETRIES) {
	  getDataErrors = 0;
	  return(-1);
	}
	else {
	  getDataErrors++;
	  return(mysql_getData(query,data,dataNum));
	}
  }

  getDataErrors = 0;
  return(0);
}

/*
 * callback provided to pmdaFetch
 */
static int
mysql_fetchCallBack(pmdaMetric *mdesc, unsigned int inst, pmAtomValue *atom)
{
    __pmID_int		*idp = (__pmID_int *)&(mdesc->m_desc.pmid);
    Result *r;

//    fprintf(stderr,"cluster=%d, item=%d\n",idp->cluster,idp->item);

    if (idp->cluster == 0) {
    	if (!statusGot) {
	   if (mysql_getData("show status",statusData,STATUS_DATA_NUM) < 0) return(0);
	   statusGot = 1;
	}

        if (idp->item >= STATUS_DATA_NUM) return(PM_ERR_PMID);
        r = &statusData[idp->item];
    }
    else if (idp->cluster == 1) {
    	if (!variablesGot) {
	   if (mysql_getData("show variables",variablesData,VARIABLES_DATA_NUM) < 0) return(0);
	   variablesGot = 1;
	}

        if (idp->item >= VARIABLES_DATA_NUM) return(PM_ERR_PMID);
        r = &variablesData[idp->item];
    }
    else return(PM_ERR_PMID);

    if ((idp->cluster == 0) || (idp->cluster == 1)) {
        /* Store result */
	if (r->length < 0) atom->ul = r->u.u32;
	else atom->cp = r->u.string;
    }

    return(1);
}

static int
mysql_fetch(int numpmid, pmID pmidlist[], pmResult **resp, pmdaExt *pmda)
{
    statusGot = 0;
    variablesGot = 0;
    return pmdaFetch(numpmid, pmidlist, resp, pmda);
}

/*
 * Function to initialize hash table directly from metrictab
 */
static void initHashTable()
{
	int max = sizeof(metrictab)/sizeof(metrictab[0]);
	int i;
	char *s;
	int id;
	__uint32_t hv;
	HTItem *item,**ptr,*it2;

	for(i=0;i<max;i++) {
		s = metrictab[i].m_user;

		/* Find first dot in the name - we use string after it */
		while((*s) && (*s != '.')) s++;
		if (*s) s++;
		hv = g_str_hash(s,HASH_SIZE);
		item = malloc(sizeof(*item));
		item->next = NULL;
		item->name = s;
		item->id = metrictab[i].m_desc.pmid & 255;
		item->type = metrictab[i].m_desc.type;
		ptr = &hashName2ID[hv];
		while(*ptr) ptr = &(*ptr)->next;
		*ptr = item;
	}
}

/*
 * Initialise the agent
 */
void 
mysqla_init(pmdaInterface *dp)
{
    if (dp->status != 0) return;

    initHashTable();

    dp->version.two.fetch = mysql_fetch;
    pmdaSetFetchCallBack(dp, mysql_fetchCallBack);

    pmdaInit(dp, indomtab, sizeof(indomtab)/sizeof(indomtab[0]), metrictab,
             sizeof(metrictab)/sizeof(metrictab[0]));
}

static void
usage(void)
{
    fprintf(stderr, "Usage: %s [options]\n\n", pmProgname);
    fputs("Options:\n"
	  "  -d domain    use domain (numeric) for metrics domain of PMDA\n"
	  "  -l logfile   write log into logfile rather than using default log name\n"
	  "  -h host      mysql host (defaults to localhost)\n"
	  "  -u user      mysql user\n"
	  "  -p password  mysql password (will be hidden in command line)\n",
	  stderr);		
    exit(1);
}

/*
 * Set up the agent if running as a daemon.
 */
int
main(int argc, char **argv)
{
    int	err = 0;
		char c;
    pmdaInterface	dispatch;

    /* trim cmd name of leading directory components */
    pmProgname = basename(argv[0]);

    pmdaDaemon(&dispatch, PMDA_INTERFACE_3, pmProgname, MYSQLD,
		"mysql.log", "/var/pcp/pmdas/mysql/help");

    while((c = pmdaGetOpt(argc, argv, "D:d:l:h:u:p:?", &dispatch, &err)) != EOF) {
	switch(c) {
	case 'h':
	    host = optarg;
	    break;
	case 'u':
	    user = optarg;
	    break;
	case 'p':
	    /* We must hide password, so we duplicate it */
	    pass = strdup(optarg);
	    if (*optarg) {
		optarg[0] = '*';
		optarg[1] = 0;
	    }
	    break;
	default:
	    err++;
	}
    }

    if (err)
    	usage();

    pmdaOpenLog(&dispatch);
    mysqla_init(&dispatch);
/*
    printf("pmdaExt = %d\n",dispatch.version.two.ext->e_io);
*/

    pmdaConnect(&dispatch);
		
    pmdaMain(&dispatch);


    exit(0);
    /*NOTREACHED*/
}
