#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <sys/types.h>

#include "udm_config.h"
#include "udm_utils.h"
#include "udm_common.h"
#include "udm_charset.h"
#include "udm_xmalloc.h"

/* Letters which appear in all character sets        */
/* This is actually US ASCII and digits              */
/* This characters are in not particular order       */
/* We'll just use ctype tolower and toupper for them */
#define WORDCHAR "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789"

/* All additional tables are formed             */
/* according the next rules:                    */
/* First all capital letters.                   */
/* Then all lower letters in the same order.    */
/* If some letter has not upper/lower case then */
/* it is specified twice. The example is ssharp */
/* in iso-8859-1, iso-8859-2, cp1250 charsets.  */
/* Table must have 2*x characters               */
/* If it is not so, tolower/toupper translation */
/* will not be possible.                        */

/************************************************/
/* US ASCII has not additional letters */
static unsigned char usascii[]="";

/************************************************/
/* All Cyrillic charsets */
/* Win Cyrillic */
static unsigned char cp1251[]={
/* 33 Capital letters */
0xC0,0xC1,0xC2,0xC3,0xC4,0xC5,0xA8,0xC6,0xC7,0xC8,0xC9,
0xCA,0xCB,0xCC,0xCD,0xCE,0xCF,0xD0,0xD1,0xD2,0xD3,0xD4,
0xD5,0xD6,0xD7,0xD8,0xD9,0xDA,0xDB,0xDC,0xDD,0xDE,0xDF,
/* 33 Small letters in the same order*/
0xE0,0xE1,0xE2,0xE3,0xE4,0xE5,0xB8,0xE6,0xE7,0xE8,0xE9,
0xEA,0xEB,0xEC,0xED,0xEE,0xEF,0xF0,0xF1,0xF2,0xF3,0xF4,
0xF5,0xF6,0xF7,0xF8,0xF9,0xFA,0xFB,0xFC,0xFD,0xFE,0xFF,0};
/* Unix Cyrillic */
static unsigned char koi8r[]={
/* 33 Capital letters */
0xE1,0xE2,0xF7,0xE7,0xE4,0xE5,0xB3,0xF6,0xFA,0xE9,0xEA,
0xEB,0xEC,0xED,0xEE,0xEF,0xF0,0xF2,0xF3,0xF4,0xF5,0xE6,
0xE8,0xE3,0xFE,0xFB,0xFD,0xFF,0xF9,0xF8,0xFC,0xE0,0xF1,
/* 33 Small letters in the same order*/
0xC1,0xC2,0xD7,0xC7,0xC4,0xC5,0xA3,0xD6,0xDA,0xC9,0xCA,
0xCB,0xCC,0xCD,0xCE,0xCF,0xD0,0xD2,0xD3,0xD4,0xD5,0xC6,
0xC8,0xC3,0xDE,0xDB,0xDD,0xDF,0xD9,0xD8,0xDC,0xC0,0xD1,0};
/* Dos Cyrillic */
static unsigned char cp866[]={
/* 33 Capital letters */
0x80,0x81,0x82,0x83,0x84,0x85,0xF0,0x86,0x87,0x88,0x89,
0x8A,0x8B,0x8C,0x8D,0x8E,0x8F,0x90,0x91,0x92,0x93,0x94,
0x95,0x96,0x97,0x98,0x99,0x9A,0x9B,0x9C,0x9D,0x9E,0x9F,
/* 33 Small letters in the same order*/
0xA0,0xA1,0xA2,0xA3,0xA4,0xA5,0xF1,0xA6,0xA7,0xA8,0xA9,
0xAA,0xAB,0xAC,0xAD,0xAE,0xAF,0xE0,0xE1,0xE2,0xE3,0xE4,
0xE5,0xE6,0xE7,0xE8,0xE9,0xEA,0xEB,0xEC,0xED,0xEE,0xEF,0};
/* ISO Cyrillic */
static unsigned char iso88595[]={
/* 33 Capital letters */
0xB0,0xB1,0xB2,0xB3,0xB4,0xB5,0xA1,0xB6,0xB7,0xB8,0xB9,
0xBA,0xBB,0xBC,0xBD,0xBE,0xBF,0xC0,0xC1,0xC2,0xC3,0xC4,
0xC5,0xC6,0xC7,0xC8,0xC9,0xCA,0xCB,0xCC,0xCD,0xCE,0xCF,
/* 33 Small letters in the same order*/
0xD0,0xD1,0xD2,0xD3,0xD4,0xD5,0xF1,0xD6,0xD7,0xD8,0xD9,
0xDA,0xDB,0xDC,0xDD,0xDE,0xDF,0xE0,0xE1,0xE2,0xE3,0xE4,
0xE5,0xE6,0xE7,0xE8,0xE9,0xEA,0xEB,0xEC,0xED,0xEE,0xEF,0};
/* Mac Cyrillic */
static unsigned char maccyr[]={
/* 33 Capital letters */
0x80,0x81,0x82,0x83,0x84,0x85,0xDD,0x86,0x87,0x88,0x89,
0x8A,0x8B,0x8C,0x8D,0x8E,0x8F,0x90,0x91,0x92,0x93,0x94,
0x95,0x96,0x97,0x98,0x99,0x9A,0x9B,0x9C,0x9D,0x9E,0x9F,
/* 33 Small letters in the same order*/
0xE0,0xE1,0xE2,0xE3,0xE4,0xE5,0xDE,0xE6,0xE7,0xE8,0xE9,
0xEA,0xEB,0xEC,0xED,0xEE,0xEF,0xF0,0xF1,0xF2,0xF3,0xF4,
0xF5,0xF6,0xF7,0xF8,0xF9,0xFA,0xFB,0xFC,0xFD,0xFE,0xDF,0};

/******************************************************/
/* ISO Western */
/* Note that ssharp appears twice to make correct */
static unsigned char iso88591[]={
/* 31 Capital letters */
0xC0,0xC1,0xC2,0xC3,0xC4,0xC5,0xC6,0xC7,0xC8,0xC9,0xCA,
0xCB,0xCC,0xCD,0xCE,0xCF,0xD0,0xD1,0xD2,0xD3,0xD4,0xD5,
0xD6,0xD8,0xD9,0xDA,0xDB,0xDC,0xDD,0xDE,0xDF,
/* 31 Small letters in the same order*/
0xE0,0xE1,0xE2,0xE3,0xE4,0xE5,0xE6,0xE7,0xE8,0xE9,0xEA,
0xEB,0xEC,0xED,0xEE,0xEF,0xF0,0xF1,0xF2,0xF3,0xF4,0xF5,
0xF6,0xF8,0xF9,0xFA,0xFB,0xFC,0xFD,0xFE,0xDF,0};

/*********************************************************/
/* Central Europe                                        */
/* 41 upper + 41 lower = 82 letters                      */
/* Note that ssharp &#223; appears twice to make correct */
/* tolower and toupper translation                       */
static unsigned char iso88592[] = {
	/* upper letters */
193,195,194,196,161,198,200,199,207,201,204,203,202,208,205,206,197,165,163,209,
210,211,212,214,213,192,216,166,169,170,171,222,218,220,219,217,221,175,172,174,223,
	/* lower letter in the same order */
225,227,226,228,177,230,232,231,239,233,236,235,234,240,237,238,229,181,179,241,
242,243,244,246,245,224,248,182,185,186,187,254,250,252,251,249,253,191,188,190,223,
0};
static unsigned char cp1250[] = {
	/* upper letters */
193,195,194,196,165,198,200,199,207,201,204,203,202,208,205,206,197,188,163,209,
210,211,212,214,213,192,216,140,138,170,141,222,218,220,219,217,221,175,143,142,223,
	/* lower letter in the same order */
225,227,226,228,185,230,232,231,239,233,236,235,234,240,237,238,229,190,179,241,
242,243,244,246,245,224,248,156,154,186,157,254,250,252,251,249,253,191,159,158,223,
0};

/**********************************************************/
/* Greek  */
/* 42 upper + 42 lower = 84 letters                       */
/* Note that accented lowercase letters are treated as    */
/* capital. Same for (lowercase) final sigma.             */

static unsigned char iso88597[] = {
	/* upper letters */
0xc1,0xb6,0xdc,0xc2,0xc3,0xc4,0xc5,0xb8,0xdd,0xc6,0xc7,
0xb9,0xde,0xc8,0xc9,0xba,0xda,0xdf,0xc0,0xca,0xcb,0xcc,
0xcd,0xce,0xcf,0xbc,0xfc,0xd0,0xd1,0xd3,0xf2,0xd4,0xd5,
0xbe,0xdb,0xfd,0xfb,0xe0,0xd6,0xd7,0xd8,0xd9,0xbf,0xfe,
	/* lower letters in the same order */
0xe1,0xe1,0xe1,0xe2,0xe3,0xe4,0xe5,0xe5,0xe5,0xe6,0xe7,
0xe7,0xe7,0xe8,0xe9,0xe9,0xe9,0xe9,0xe9,0xea,0xeb,0xec,
0xed,0xee,0xef,0xef,0xef,0xf0,0xf1,0xf3,0xf3,0xf4,0xf5,
0xf5,0xf5,0xf5,0xf5,0xf5,0xf6,0xf7,0xf8,0xf9,0xf9,0xf9,0};

static unsigned char cp1253[] = {
	/* upper letters */
0xc1,0xa2,0xdc,0xc2,0xc3,0xc4,0xc5,0xb8,0xdd,0xc6,0xc7,
0xb9,0xde,0xc8,0xc9,0xba,0xda,0xdf,0xc0,0xca,0xcb,0xcc,
0xcd,0xce,0xcf,0xbc,0xfc,0xd0,0xd1,0xd3,0xf2,0xd4,0xd5,
0xbe,0xdb,0xfd,0xfb,0xe0,0xd6,0xd7,0xd8,0xd9,0xbf,0xfe,
	/* lower letters in the same order */
0xe1,0xe1,0xe1,0xe2,0xe3,0xe4,0xe5,0xe5,0xe5,0xe6,0xe7,
0xe7,0xe7,0xe8,0xe9,0xe9,0xe9,0xe9,0xe9,0xea,0xeb,0xec,
0xed,0xee,0xef,0xef,0xef,0xf0,0xf1,0xf3,0xf3,0xf4,0xf5,
0xf5,0xf5,0xf5,0xf5,0xf5,0xf6,0xf7,0xf8,0xf9,0xf9,0xf9,0};

/**********************************************************/
/* Arabic */
/* Arabic has not upper/lower letters     */
/* UdmTolower and UdmToupper have special */
/* condition to take in account this fact */
static unsigned char cp1256[] ={
0x8d, 0x8e, 0x90, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 0xd0, 0xd1,
0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd8, 0xd9, 0xda, 0xdb, 0xdc,
0xde, 0xdf, 0xe1, 0xe3, 0xe4, 0xe5, 0xe6, 0xec, 0xed, 0 };

/************************************************/
/*   Hebrew                                     */
/* iso-8859-8 and cp-1255 are letters compatible*/
static unsigned char iso88598[]={
0xE0,  /*  U+05D0  HEBREW LETTER ALEF		*/
0xE1,  /*  U+05D1  HEBREW LETTER BET		*/
0xE2,  /*  U+05D2  HEBREW LETTER GIMEL		*/
0xE3,  /*  U+05D3  HEBREW LETTER DALET		*/
0xE4,  /*  U+05D4  HEBREW LETTER HE		*/
0xE5,  /*  U+05D5  HEBREW LETTER VAV		*/
0xE6,  /*  U+05D6  HEBREW LETTER ZAYIN		*/
0xE7,  /*  U+05D7  HEBREW LETTER HET		*/
0xE8,  /*  U+05D8  HEBREW LETTER TET		*/
0xE9,  /*  U+05D9  HEBREW LETTER YOD		*/
0xEA,  /*  U+05DA  HEBREW LETTER FINAL KAF	*/
0xEB,  /*  U+05DB  HEBREW LETTER KAF		*/
0xEC,  /*  U+05DC  HEBREW LETTER LAMED		*/
0xED,  /*  U+05DD  HEBREW LETTER FINAL MEM	*/
0xEE,  /*  U+05DE  HEBREW LETTER MEM		*/
0xEF,  /*  U+05DF  HEBREW LETTER FINAL NUN	*/
0xF0,  /*  U+05E0  HEBREW LETTER NUN		*/
0xF1,  /*  U+05E1  HEBREW LETTER SAMEKH		*/
0xF2,  /*  U+05E2  HEBREW LETTER AYIN		*/
0xF3,  /*  U+05E3  HEBREW LETTER FINAL PE	*/
0xF4,  /*  U+05E4  HEBREW LETTER PE		*/
0xF5,  /*  U+05E5  HEBREW LETTER FINAL TSADI	*/
0xF6,  /*  U+05E6  HEBREW LETTER TSADI		*/
0xF7,  /*  U+05E7  HEBREW LETTER QOF		*/
0xF8,  /*  U+05E8  HEBREW LETTER RESH		*/
0xF9,  /*  U+05E9  HEBREW LETTER SHIN		*/
0xFA,  /*  U+05EA  HEBREW LETTER TAV		*/
0x00};

/****************************************************************/
/* Baltic                                                       */
/* 8859-13 and cp1257 are letters compatible                    */
/* FIXME */
/* NO 0xAF capital letter 'AE'
 * NO 0xBD small letter 'AE'
 * NO 0xDF small letter sharp
*/

static unsigned char iso885913[]={
'A',   /*  ------  LATIN CAPITAL LETTER A WITH ACUTE		*/
'A',   /*  ------  LATIN CAPITAL LETTER A WITH CIRCUMFLEX	*/
0xC4,  /*  U+00C4  LATIN CAPITAL LETTER A WITH DIAERESIS	*/
0xC2,  /*  U+0100  LATIN CAPITAL LETTER A WITH MACRON		*/
0xC0,  /*  U+0104  LATIN CAPITAL LETTER A WITH OGONEK		*/
0xC5,  /*  U+00C5  LATIN CAPITAL LETTER A WITH RING ABOVE	*/
'A' ,  /*  ------  LATIN CAPITAL LETTER A WITH TILDE		*/
0xC3,  /*  U+0106, LATIN CAPITAL LETTER C WITH ACUTE		*/
0xC8,  /*  U+010C  LATIN CAPITAL LETTER C WITH CARON		*/
'D',   /*  ------  LATIN CAPITAL LETTER D WITH STROKE		*/
0xC9,  /*  U+00C9  LATIN CAPITAL LETTER E WITH ACUTE		*/
'E',   /*  ------  LATIN CAPITAL LETTER E WITH DIAERESIS	*/
0xCB,  /*  U+0116  LATIN CAPITAL LETTER E WITH DOT ABOVE	*/
0xC7,  /*  U+0112  LATIN CAPITAL LETTER E WITH MACRON		*/
0xC6,  /*  U+0118  LATIN CAPITAL LETTER E WITH OGONEK		*/
0xCC,  /*  U+0122  LATIN CAPITAL LETTER G WITH CEDILLA		*/
'I',   /*  ------  LATIN CAPITAL LETTER I WITH ACUTE		*/
'I',   /*  ------  LATIN CAPITAL LETTER I WITH CIRCUMFLEX	*/
0xCE,  /*  U+012A  LATIN CAPITAL LETTER I WITH MACRON		*/
0xC1,  /*  U+012E  LATIN CAPITAL LETTER I WITH OGONEK		*/
'I',   /*  ------  LATIN CAPITAL LETTER I WITH TILDE		*/
0xCD,  /*  U+0136  LATIN CAPITAL LETTER K WITH CEDILLA		*/
0xCF,  /*  U+013B  LATIN CAPITAL LETTER L WITH CEDILLA		*/
0xD9,  /*  U+0141  LATIN CAPITAL LETTER L WITH STROKE		*/
0xD1,  /*  U+0143  LATIN CAPITAL LETTER N WITH ACUTE		*/
0xD2,  /*  U+0145  LATIN CAPITAL LETTER N WITH CEDILLA		*/
0xD3,  /*  U+00D3  LATIN CAPITAL LETTER O WITH ACUTE		*/
'O',   /*  ------  LATIN CAPITAL LETTER O WITH CIRCUMFLEX	*/
0xD6,  /*  U+00D6  LATIN CAPITAL LETTER O WITH DIAERESIS	*/
0xD4,  /*  U+014C  LATIN CAPITAL LETTER O WITH MACRON		*/
0xA8,  /*  U+00D8  LATIN CAPITAL LETTER O WITH STROKE		*/
0xD5,  /*  U+00D5  LATIN CAPITAL LETTER O WITH TILDE		*/
0xAA,  /*  U+0156  LATIN CAPITAL LETTER R WITH CEDILLA		*/
0xDA,  /*  U+015A  LATIN CAPITAL LETTER S WITH ACUTE		*/
0xD0,  /*  U+0160  LATIN CAPITAL LETTER S WITH CARON		*/
'T',   /*  ------  LATIN CAPITAL LETTER T WITH STROKE		*/
'U',   /*  ------  LATIN CAPITAL LETTER U WITH ACUTE		*/
'U',   /*  ------  LATIN CAPITAL LETTER U WITH CIRCUMFLEX	*/
0xDC,  /*  U+00DC  LATIN CAPITAL LETTER U WITH DIAERESIS	*/
0xDB,  /*  U+016A  LATIN CAPITAL LETTER U WITH MACRON		*/
0xD8,  /*  U+0172  LATIN CAPITAL LETTER U WITH OGONEK		*/
'U',   /*  ------  LATIN CAPITAL LETTER U WITH TILDE		*/
0xCA,  /*  U+0179  LATIN CAPITAL LETTER Z WITH ACUTE		*/
0xDE,  /*  U+017D  LATIN CAPITAL LETTER Z WITH CARON		*/
0xDD,  /*  U+017B  LATIN CAPITAL LETTER Z WITH DOT ABOVE	*/
'a',   /*  ------  LATIN SMALL LETTER A WITH ACUTE		*/
'a',   /*  ------  LATIN SMALL LETTER A WITH CIRCUMFLEX		*/
0xE4,  /*  U+00E4  LATIN SMALL LETTER A WITH DIAERESIS		*/
0xE2,  /*  U+0101  LATIN SMALL LETTER A WITH MACRON		*/
0xE0,  /*  U+0105  LATIN SMALL LETTER A WITH OGONEK		*/
0xE5,  /*  U+00E5  LATIN SMALL LETTER A WITH RING ABOVE		*/
'a',   /*  ------  LATIN SMALL LETTER A WITH TILDE		*/
0xE3,  /*  U+0107  LATIN SMALL LETTER C WITH ACUTE		*/
0xE8,  /*  U+010D  LATIN SMALL LETTER C WITH CARON		*/
'd',   /*  ------  LATIN SMALL LETTER D WITH STROKE		*/
0xE9,  /*  U+00E9  LATIN SMALL LETTER E WITH ACUTE		*/
'e',   /*  ------  LATIN SMALL LETTER E WITH DIAERESIS		*/
0xEB,  /*  U+0117  LATIN SMALL LETTER E WITH DOT ABOVE		*/
0xE7,  /*  U+0113  LATIN SMALL LETTER E WITH MACRON		*/
0xE6,  /*  U+0119  LATIN SMALL LETTER E WITH OGONEK		*/
0xEC,  /*  U+0123  LATIN SMALL LETTER G WITH CEDILLA		*/
'i',   /*  ------  LATIN SMALL LETTER I WITH ACUTE		*/
'i',   /*  ------  LATIN SMALL LETTER I WITH CIRCUMFLEX		*/
0xEE,  /*  U+012B  LATIN SMALL LETTER I WITH MACRON		*/
0xE1,  /*  U+012F  LATIN SMALL LETTER I WITH OGONEK		*/
'i',   /*  ------  LATIN SMALL LETTER I WITH TILDE		*/
0xED,  /*  U+0137  LATIN SMALL LETTER K WITH CEDILLA		*/
0xEF,  /*  U+013C  LATIN SMALL LETTER L WITH CEDILLA		*/
0xF9,  /*  U+0142  LATIN SMALL LETTER L WITH STROKE		*/
0xF1,  /*  U+0144  LATIN SMALL LETTER N WITH ACUTE		*/
0xF2,  /*  U+0146  LATIN SMALL LETTER N WITH CEDILLA		*/
0xF3,  /*  U+00F3  LATIN SMALL LETTER O WITH ACUTE		*/
'o',   /*  ------  LATIN SMALL LETTER O WITH CIRCUMFLEX		*/
0xF6,  /*  U+00F6  LATIN SMALL LETTER O WITH DIAERESIS		*/
0xF4,  /*  U+014D  LATIN SMALL LETTER O WITH MACRON		*/
0xB8,  /*  U+00F8  LATIN SMALL LETTER O WITH STROKE		*/
0xF5,  /*  U+00F5  LATIN SMALL LETTER O WITH TILDE		*/
0xBA,  /*  U+0157  LATIN SMALL LETTER R WITH CEDILLA		*/
0xFA,  /*  U+015B  LATIN SMALL LETTER S WITH ACUTE		*/
0xF0,  /*  U+0161  LATIN SMALL LETTER S WITH CARON		*/
't',   /*  ------  LATIN SMALL LETTER T WITH STROKE		*/
't',   /*  ------  LATIN SMALL LETTER U WITH ACUTE		*/
'u',   /*  ------  LATIN SMALL LETTER U WITH CIRCUMFLEX		*/
0xFC,  /*  U+00FC  LATIN SMALL LETTER U WITH DIAERESIS		*/
0xFB,  /*  U+016B  LATIN SMALL LETTER U WITH MACRON		*/
0xF8,  /*  U+0173  LATIN SMALL LETTER U WITH OGONEK		*/
'u',   /*  ------  LATIN SMALL LETTER U WITH TILDE		*/
0xEA,  /*  U+017A  LATIN SMALL LETTER Z WITH ACUTE		*/
0xFE,  /*  U+017E  LATIN SMALL LETTER Z WITH CARON		*/
0xFD,  /*  U+017C  LATIN SMALL LETTER Z WITH DOT ABOVE		*/
0x00};

/*
FIXME ISO-8859-4 
NO 0xA2 small letter KRA
NO 0xBD capital letter ENG
NO 0xBF small letter ENG
NO 0xC6 capital letter AE
NO 0xDF small letter sharp
NO 0xE6 small letter AE
*/

static unsigned char iso88594[]= {
0xC1,  /*  U+00C1  LATIN CAPITAL LETTER A WITH ACUTE		*/
0xC2,  /*  U+00C2  LATIN CAPITAL LETTER A WITH CIRCUMFLEX	*/
0xC4,  /*  U+00C4  LATIN CAPITAL LETTER A WITH DIAERESIS	*/
0xC0,  /*  U+0100  LATIN CAPITAL LETTER A WITH MACRON		*/
0xA1,  /*  U+0104  LATIN CAPITAL LETTER A WITH OGONEK		*/
0xC5,  /*  U+00C5  LATIN CAPITAL LETTER A WITH RING ABOVE	*/
0xC3,  /*  U+00C3  LATIN CAPITAL LETTER A WITH TILDE		*/
'C',   /*  ------  LATIN CAPITAL LETTER C WITH ACUTE		*/
0xC8,  /*  U+010C  LATIN CAPITAL LETTER C WITH CARON		*/
0xD0,  /*  U+0110  LATIN CAPITAL LETTER D WITH STROKE		*/
0xC9,  /*  U+00C9  LATIN CAPITAL LETTER E WITH ACUTE		*/
0xCB,  /*  U+00CB  LATIN CAPITAL LETTER E WITH DIAERESIS	*/
0xCC,  /*  U+0116  LATIN CAPITAL LETTER E WITH DOT ABOVE	*/
0xAA,  /*  U+0112  LATIN CAPITAL LETTER E WITH MACRON		*/
0xCA,  /*  U+0118  LATIN CAPITAL LETTER E WITH OGONEK		*/
0xAB,  /*  U+0122  LATIN CAPITAL LETTER G WITH CEDILLA		*/
0xCD,  /*  U+00CD  LATIN CAPITAL LETTER I WITH ACUTE		*/
0xCE,  /*  U+00CE  LATIN CAPITAL LETTER I WITH CIRCUMFLEX	*/
0xCF,  /*  U+012A  LATIN CAPITAL LETTER I WITH MACRON		*/
0xC7,  /*  U+012E  LATIN CAPITAL LETTER I WITH OGONEK		*/
0xA5,  /*  U+0128  LATIN CAPITAL LETTER I WITH TILDE		*/
0xD3,  /*  U+0136  LATIN CAPITAL LETTER K WITH CEDILLA		*/
0xA6,  /*  U+013B  LATIN CAPITAL LETTER L WITH CEDILLA		*/
'L',   /*  ------  LATIN CAPITAL LETTER L WITH STROKE		*/
'N',   /*  ------  LATIN CAPITAL LETTER N WITH ACUTE		*/
0xD1,  /*  U+0145  LATIN CAPITAL LETTER N WITH CEDILLA		*/
'O',   /*  ------  LATIN CAPITAL LETTER O WITH ACUTE		*/
0xD4,  /*  U+00D4  LATIN CAPITAL LETTER O WITH CIRCUMFLEX	*/
0xD6,  /*  U+00D6  LATIN CAPITAL LETTER O WITH DIAERESIS	*/
0xD2,  /*  U+014C  LATIN CAPITAL LETTER O WITH MACRON		*/
0xD8,  /*  U+00D8  LATIN CAPITAL LETTER O WITH STROKE		*/
0xD5,  /*  U+00D5  LATIN CAPITAL LETTER O WITH TILDE		*/
0xA3,  /*  U+0156  LATIN CAPITAL LETTER R WITH CEDILLA		*/
'S',   /*  ------  LATIN CAPITAL LETTER S WITH ACUTE		*/
0xA9,  /*  U+0160  LATIN CAPITAL LETTER S WITH CARON		*/
0xAC,  /*  U+0166  LATIN CAPITAL LETTER T WITH STROKE		*/
0xDA,  /*  U+00DA  LATIN CAPITAL LETTER U WITH ACUTE		*/
0xDB,  /*  U+00DB  LATIN CAPITAL LETTER U WITH CIRCUMFLEX	*/
0xDC,  /*  U+00DC  LATIN CAPITAL LETTER U WITH DIAERESIS	*/
0xDE,  /*  U+016A  LATIN CAPITAL LETTER U WITH MACRON		*/
0xD9,  /*  U+0172  LATIN CAPITAL LETTER U WITH OGONEK		*/
0xDD,  /*  U+0168  LATIN CAPITAL LETTER U WITH TILDE		*/
'Z',   /*  ------  LATIN CAPITAL LETTER Z WITH ACUTE		*/
0xAE,  /*  U+017D  LATIN CAPITAL LETTER Z WITH CARON		*/
'Z',   /*  ------  LATIN CAPITAL LETTER Z WITH DOT ABOVE	*/
0xE1,  /*  U+00E1  LATIN SMALL LETTER A WITH ACUTE		*/
0xE2,  /*  U+00E2  LATIN SMALL LETTER A WITH CIRCUMFLEX		*/
0xE4,  /*  U+00E4  LATIN SMALL LETTER A WITH DIAERESIS		*/
0xE0,  /*  U+0101  LATIN SMALL LETTER A WITH MACRON		*/
0xB1,  /*  U+0105  LATIN SMALL LETTER A WITH OGONEK		*/
0xE5,  /*  U+00E5  LATIN SMALL LETTER A WITH RING ABOVE		*/
0xE3,  /*  U+00E3  LATIN SMALL LETTER A WITH TILDE		*/
'c',   /*  ------  LATIN SMALL LETTER C WITH ACUTE		*/
0xE8,  /*  U+010D  LATIN SMALL LETTER C WITH CARON		*/
0xF0,  /*  U+0111  LATIN SMALL LETTER D WITH STROKE		*/
0xE9,  /*  U+00E9  LATIN SMALL LETTER E WITH ACUTE		*/
0xEB,  /*  U+00EB  LATIN SMALL LETTER E WITH DIAERESIS		*/
0xEC,  /*  U+0117  LATIN SMALL LETTER E WITH DOT ABOVE		*/
0xBA,  /*  U+0113  LATIN SMALL LETTER E WITH MACRON		*/
0xEA,  /*  U+0119  LATIN SMALL LETTER E WITH OGONEK		*/
0xBB,  /*  U+0123  LATIN SMALL LETTER G WITH CEDILLA		*/
0xED,  /*  U+00ED  LATIN SMALL LETTER I WITH ACUTE		*/
0xEE,  /*  U+00EE  LATIN SMALL LETTER I WITH CIRCUMFLEX		*/
0xEF,  /*  U+012B  LATIN SMALL LETTER I WITH MACRON		*/
0xE7,  /*  U+012F  LATIN SMALL LETTER I WITH OGONEK		*/
0xB5,  /*  U+0129  LATIN SMALL LETTER I WITH TILDE		*/
0xF3,  /*  U+0137  LATIN SMALL LETTER K WITH CEDILLA		*/
0xB6,  /*  U+013C  LATIN SMALL LETTER L WITH CEDILLA		*/
'l',   /*  ------  LATIN SMALL LETTER L WITH STROKE		*/
'n',   /*  ------  LATIN SMALL LETTER N WITH ACUTE		*/
0xF1,  /*  U+0146  LATIN SMALL LETTER N WITH CEDILLA		*/
'o',   /*  ------  LATIN SMALL LETTER O WITH ACUTE		*/
0xF4,  /*  U+00F4  LATIN SMALL LETTER O WITH CIRCUMFLEX		*/
0xF6,  /*  U+00F6  LATIN SMALL LETTER O WITH DIAERESIS		*/
0xF2,  /*  U+014D  LATIN SMALL LETTER O WITH MACRON		*/
0xF8,  /*  U+00F8  LATIN SMALL LETTER O WITH STROKE		*/
0xF5,  /*  U+00F5  LATIN SMALL LETTER O WITH TILDE		*/
0xB3,  /*  U+0157  LATIN SMALL LETTER R WITH CEDILLA		*/
's',   /*  ------  LATIN SMALL LETTER S WITH ACUTE		*/
0xB9,  /*  U+0161  LATIN SMALL LETTER S WITH CARON		*/
0xBC,  /*  U+0167  LATIN SMALL LETTER T WITH STROKE		*/
0xFA,  /*  U+00FA  LATIN SMALL LETTER U WITH ACUTE		*/
0xFB,  /*  U+00FB  LATIN SMALL LETTER U WITH CIRCUMFLEX		*/
0xFC,  /*  U+00FC  LATIN SMALL LETTER U WITH DIAERESIS		*/
0xFE,  /*  U+016B  LATIN SMALL LETTER U WITH MACRON		*/
0xF9,  /*  U+0173  LATIN SMALL LETTER U WITH OGONEK		*/
0xFD,  /*  U+0169  LATIN SMALL LETTER U WITH TILDE		*/
'z',   /*  ------  LATIN SMALL LETTER Z WITH ACUTE		*/
0xBE,  /*  U+017E  LATIN SMALL LETTER Z WITH CARON		*/
'z',   /*  ------  LATIN SMALL LETTER Z WITH DOT ABOVE		*/
0x00};

/****************************************************************/

/* Turkish ISO-8859-9 and windows-1254   */
/* character sets are letters compatible */

static unsigned char iso88599[]={

/* 30 Capital letters */
/* small y-diaresis + small sharp s */

0xC0,	/* U+00C0	LATIN CAPITAL LETTER A WITH GRAVE	*/
0xC1,	/* U+00C1	LATIN CAPITAL LETTER A WITH ACUTE	*/
0xC2,	/* U+00C2	LATIN CAPITAL LETTER A WITH CIRCUMFLEX	*/
0xC3,	/* U+00C3	LATIN CAPITAL LETTER A WITH TILDE	*/
0xC4,	/* U+00C4	LATIN CAPITAL LETTER A WITH DIAERESIS	*/
0xC5,	/* U+00C5	LATIN CAPITAL LETTER A WITH RING ABOVE	*/
0xC6,	/* U+00C6	LATIN CAPITAL LETTER AE			*/
0xC7,	/* U+00C7	LATIN CAPITAL LETTER C WITH CEDILLA	*/
0xC8,	/* U+00C8	LATIN CAPITAL LETTER E WITH GRAVE	*/
0xC9,	/* U+00C9	LATIN CAPITAL LETTER E WITH ACUTE	*/
0xCA,	/* U+00CA	LATIN CAPITAL LETTER E WITH CIRCUMFLEX	*/
0xCB,	/* U+00CB	LATIN CAPITAL LETTER E WITH DIAERESIS	*/
0xCC,	/* U+00CC	LATIN CAPITAL LETTER I WITH GRAVE	*/
0xCD,	/* U+00CD	LATIN CAPITAL LETTER I WITH ACUTE	*/
0xCE,	/* U+00CE	LATIN CAPITAL LETTER I WITH CIRCUMFLEX	*/
0xCF,	/* U+00CF	LATIN CAPITAL LETTER I WITH DIAERESIS	*/
0xD0,	/* U+011E	LATIN CAPITAL LETTER G WITH BREVE	*/
0xD1,	/* U+00D1	LATIN CAPITAL LETTER N WITH TILDE	*/
0xD2,	/* U+00D2	LATIN CAPITAL LETTER O WITH GRAVE	*/
0xD3,	/* U+00D3	LATIN CAPITAL LETTER O WITH ACUTE	*/
0xD4,	/* U+00D4	LATIN CAPITAL LETTER O WITH CIRCUMFLEX	*/
0xD5,	/* U+00D5	LATIN CAPITAL LETTER O WITH TILDE	*/
0xD6,	/* U+00D6	LATIN CAPITAL LETTER O WITH DIAERESIS	*/
0xD8,	/* U+00D8	LATIN CAPITAL LETTER O WITH STROKE	*/
0xD9,	/* U+00D9	LATIN CAPITAL LETTER U WITH GRAVE	*/
0xDA,	/* U+00DA	LATIN CAPITAL LETTER U WITH ACUTE	*/
0xDB,	/* U+00DB	LATIN CAPITAL LETTER U WITH CIRCUMFLEX	*/
0xDC,	/* U+00DC	LATIN CAPITAL LETTER U WITH DIAERESIS	*/
0xDD,	/* U+0130	LATIN CAPITAL LETTER I WITH DOT ABOVE	*/
'I',	/*		LATIN CAPITAL LETTER DOTLESS I		*/
0xDE,	/* U+015E	LATIN CAPITAL LETTER S WITH CEDILLA	*/

/* We add small 'y' with diaresis and small sharp 's' */
/* to implement correct toupper and tolower operation */

0xFF,	/* U+00FF	LATIN SMALL LETTER Y WITH DIAERESIS	*/
0xDF,	/* U+00DF	LATIN SMALL LETTER SHARP S		*/

/* 32 Small letters in the same order */

0xE0,	/* U+00E0	LATIN SMALL LETTER A WITH GRAVE		*/
0xE1,	/* U+00E1	LATIN SMALL LETTER A WITH ACUTE		*/
0xE2,	/* U+00E2	LATIN SMALL LETTER A WITH CIRCUMFLEX	*/
0xE3,	/* U+00E3	LATIN SMALL LETTER A WITH TILDE		*/
0xE4,	/* U+00E4	LATIN SMALL LETTER A WITH DIAERESIS	*/
0xE5,	/* U+00E5	LATIN SMALL LETTER A WITH RING ABOVE	*/
0xE6,	/* U+00E6	LATIN SMALL LETTER AE			*/
0xE7,	/* U+00E7	LATIN SMALL LETTER C WITH CEDILLA	*/
0xE8,	/* U+00E8	LATIN SMALL LETTER E WITH GRAVE		*/
0xE9,	/* U+00E9	LATIN SMALL LETTER E WITH ACUTE		*/
0xEA,	/* U+00EA	LATIN SMALL LETTER E WITH CIRCUMFLEX	*/
0xEB,	/* U+00EB	LATIN SMALL LETTER E WITH DIAERESIS	*/
0xEC,	/* U+00EC	LATIN SMALL LETTER I WITH GRAVE		*/
0xED,	/* U+00ED	LATIN SMALL LETTER I WITH ACUTE		*/
0xEE,	/* U+00EE	LATIN SMALL LETTER I WITH CIRCUMFLEX	*/
0xEF,	/* U+00EF	LATIN SMALL LETTER I WITH DIAERESIS	*/
0xF0,	/* U+011F	LATIN SMALL LETTER G WITH BREVE		*/
0xF1,	/* U+00F1	LATIN SMALL LETTER N WITH TILDE		*/
0xF2,	/* U+00F2	LATIN SMALL LETTER O WITH GRAVE		*/
0xF3,	/* U+00F3	LATIN SMALL LETTER O WITH ACUTE		*/
0xF4,	/* U+00F4	LATIN SMALL LETTER O WITH CIRCUMFLEX	*/
0xF5,	/* U+00F5	LATIN SMALL LETTER O WITH TILDE		*/
0xF6,	/* U+00F6	LATIN SMALL LETTER O WITH DIAERESIS	*/
0xF8,	/* U+00F8	LATIN SMALL LETTER O WITH STROKE	*/
0xF9,	/* U+00F9	LATIN SMALL LETTER U WITH GRAVE		*/
0xFA,	/* U+00FA	LATIN SMALL LETTER U WITH ACUTE		*/
0xFB,	/* U+00FB	LATIN SMALL LETTER U WITH CIRCUMFLEX	*/
0xFC,	/* U+00FC	LATIN SMALL LETTER U WITH DIAERESIS	*/
'i',	/*		LATIN SMALL LETTER I WITH DOT ABOVE	*/
0xFD,	/* U+0131	LATIN SMALL LETTER DOTLESS I		*/
0xFE,	/* U+015F	LATIN SMALL LETTER S WITH CEDILLA	*/
0xFF,	/* U+00FF	LATIN SMALL LETTER Y WITH DIAERESIS	*/
0xDF,	/* U+00DF	LATIN SMALL LETTER SHARP S		*/
0x00
};


/****************************************************************/

/* Type to store letters set */
typedef unsigned char TABLE[256];

/* A structure for additional letters */
/* which appear in different charsets */
typedef struct charset_struct {
	int		charset;	/* Charset ID */
	unsigned	char *chars;	/* Letters array */
	int		has_case;	/* 1 if has upper/lower letters */
	TABLE		lower;		/* ToLower table */
	TABLE		upper;		/* ToUpper table */
	unsigned char	num_word_range;
	TABLE		word_range;	
} TCHARSET;

static TCHARSET Charsets[] ={
	/* Plain ASCII */
	{UDM_CHARSET_USASCII,	usascii,	1},
	/* Cyrlllic */
	{UDM_CHARSET_KOI8R,	koi8r,		1},
	{UDM_CHARSET_CP1251,	cp1251,		1},
	{UDM_CHARSET_CP866,	cp866,		1},
	{UDM_CHARSET_MACCYR,	maccyr,		1},
	{UDM_CHARSET_ISO88595,	iso88595,	1},
	/* Western */
	{UDM_CHARSET_ISO88591,	iso88591,	1},
	/* Central */
	{UDM_CHARSET_CP1250,	cp1250,		1},
	{UDM_CHARSET_ISO88592,	iso88592,	1},
	/* Arabic */
	{UDM_CHARSET_CP1256, 	cp1256,		0},
	/* Greek */
	{UDM_CHARSET_ISO88597, 	iso88597,      	1},
	{UDM_CHARSET_CP1253, 	cp1253,		1},
	/* Baltic */
	{UDM_CHARSET_CP1257,	iso885913,	1},
	{UDM_CHARSET_ISO885913,	iso885913,	1},
	{UDM_CHARSET_ISO88594,	iso88594,	1},
	/* Hebrew */
	{UDM_CHARSET_ISO88598,	iso88598,	0},
	{UDM_CHARSET_CP1255,	iso88598,	0},
	/* Turkish */
	{UDM_CHARSET_ISO88599,	iso88599,	1},
	{UDM_CHARSET_CP1254,	iso88599,	1},
	{-1,NULL,0}
};


/*****************************************************************/
/* Charsets may have different aliases   */
/* Let's define ALIAS structure for them */
typedef struct alias_struct {
	int		charset;
	const unsigned char *alias;
} ALIAS;

static const ALIAS Aliases[] = {
	/* As far as UdmCharsetStr seek in the order of appearence    */
	/* in this array, MAIN MOST COMMON alias should follow first  */
	/* 7 bit US ASCII*/
	{UDM_CHARSET_USASCII,"us-ascii"},
	/* All cyrillic */
	{UDM_CHARSET_KOI8R,"koi8-r"},
	{UDM_CHARSET_KOI8R,"koi8r"},
	{UDM_CHARSET_KOI8R,"koi-8-r"},
	{UDM_CHARSET_KOI8R,"koi8"},
	{UDM_CHARSET_KOI8R,"koi-8"},
	{UDM_CHARSET_KOI8R,"koi"},
	
	{UDM_CHARSET_CP1251,"windows-1251"},
	{UDM_CHARSET_CP1251,"cp-1251"},
	{UDM_CHARSET_CP1251,"cp1251"},
	{UDM_CHARSET_CP1251,"windows1251"},
	{UDM_CHARSET_CP1251,"x-cp1251"},
	
	{UDM_CHARSET_ISO88595,"iso-8859-5"},
	{UDM_CHARSET_ISO88595,"iso8859-5"},
	{UDM_CHARSET_ISO88595,"iso8859.5"},
	{UDM_CHARSET_ISO88595,"iso-8859.5"},
	{UDM_CHARSET_ISO88595,"iso_8859-5"},
	{UDM_CHARSET_ISO88595,"iso_8859-5:1988"},
	{UDM_CHARSET_ISO88595,"iso-ir-144"},
	{UDM_CHARSET_ISO88595,"cyrillic"},
	
	{UDM_CHARSET_CP866,"cp-866"},
	{UDM_CHARSET_CP866,"cp866"},
	{UDM_CHARSET_CP866,"csibm866"},
	{UDM_CHARSET_CP866,"866"},
	{UDM_CHARSET_CP866,"ibm866"},
	{UDM_CHARSET_CP866,"x-cp866"},
	{UDM_CHARSET_CP866,"x-ibm866"},
	{UDM_CHARSET_CP866,"alt"},

	{UDM_CHARSET_MACCYR,"x-mac-cyrillic"},
	{UDM_CHARSET_MACCYR,"mac"}, 
	/* cp1250 */
	{UDM_CHARSET_CP1250,"windows-1250"},
	{UDM_CHARSET_CP1250,"cp-1250"},
	{UDM_CHARSET_CP1250,"cp1250"},
	{UDM_CHARSET_CP1250,"windows1250"},
	{UDM_CHARSET_CP1250,"x-cp1250"},
	/* 8859-1 */
	{UDM_CHARSET_ISO88591,"iso-8859-1"},
	{UDM_CHARSET_ISO88591,"iso8859-1"},
	{UDM_CHARSET_ISO88591,"iso8859.1"},
	{UDM_CHARSET_ISO88591,"iso-8859.1"},
	{UDM_CHARSET_ISO88591,"iso_8859-1:1988"},
	{UDM_CHARSET_ISO88591,"iso_8859-1"},
	{UDM_CHARSET_ISO88591,"iso_8859.1"},
	{UDM_CHARSET_ISO88591,"latin1"},
	/* 8859-2 */
	{UDM_CHARSET_ISO88592,"iso-8859-2"},
	{UDM_CHARSET_ISO88592,"iso8859-2"},
	{UDM_CHARSET_ISO88592,"iso8859.2"},
	{UDM_CHARSET_ISO88592,"iso-8859.2"},
	{UDM_CHARSET_ISO88592,"iso_8859-2:1988"},
	{UDM_CHARSET_ISO88592,"iso_8859-2"},
	{UDM_CHARSET_ISO88592,"iso_8859.2"},
	{UDM_CHARSET_ISO88592,"latin2"},
	/* cp1256 */
	{UDM_CHARSET_CP1256,"windows-1256"},
	{UDM_CHARSET_CP1256,"cp1256"},
	{UDM_CHARSET_CP1256,"windows1256"},
	{UDM_CHARSET_CP1256,"x-cp1256"},
	{UDM_CHARSET_CP1256,"cp-1256"},
	/* 8859-7 */
	{UDM_CHARSET_ISO88597,"iso-8859-7"},
	{UDM_CHARSET_ISO88597,"iso8859-7"},
	{UDM_CHARSET_ISO88597,"iso8859.7"},
	{UDM_CHARSET_ISO88597,"iso-8859.7"},
	{UDM_CHARSET_ISO88597,"iso_8859-7:1987"},
	{UDM_CHARSET_ISO88597,"iso_8859-7"},
	{UDM_CHARSET_ISO88597,"iso_8859.7"},
	/* cp1253 */
	{UDM_CHARSET_CP1253,"windows-1253"},
	{UDM_CHARSET_CP1253,"cp1253"},
	{UDM_CHARSET_CP1253,"windows1253"},
	{UDM_CHARSET_CP1253,"x-cp1253"},
	{UDM_CHARSET_CP1253,"cp-1253"},
	/* 8859-13 */
	{UDM_CHARSET_ISO885913,"iso-8859-13"},
	{UDM_CHARSET_ISO885913,"iso8859-13"},
	{UDM_CHARSET_ISO885913,"iso8859.13"},
	{UDM_CHARSET_ISO885913,"iso-8859.13"},
	{UDM_CHARSET_ISO885913,"iso_8859-13"},
	{UDM_CHARSET_ISO885913,"iso_8859.13"},
	{UDM_CHARSET_ISO885913,"latin7"},
	/* 1257 */
	{UDM_CHARSET_CP1257,   "windows-1257"},
	{UDM_CHARSET_CP1257,   "cp1257"},
	{UDM_CHARSET_CP1257,   "windows1257"},
	{UDM_CHARSET_CP1257,   "x-cp1257"},	
	{UDM_CHARSET_CP1257,   "cp-1257"},
	/* 8859-4 */
	{UDM_CHARSET_ISO88594,"iso-8859-4"},
	{UDM_CHARSET_ISO88594,"iso8859-4"},
	{UDM_CHARSET_ISO88594,"iso8859.4"},	
	{UDM_CHARSET_ISO88594,"iso-8859.4"},
	{UDM_CHARSET_ISO88594,"iso_8859-4"},
	{UDM_CHARSET_ISO88594,"iso_8859.4"},
	{UDM_CHARSET_ISO88594,"latin4"},
	/* 8859-8 */
	{UDM_CHARSET_ISO88598,"iso-8859-8"},
	{UDM_CHARSET_ISO88598,"iso8859-8"},
	{UDM_CHARSET_ISO88598,"iso8859.8"},
	{UDM_CHARSET_ISO88598,"iso-8859.8"},
	{UDM_CHARSET_ISO88598,"iso_8859-8"},
	{UDM_CHARSET_ISO88598,"iso_8859.8"},
	/* 8859-9 */
	{UDM_CHARSET_ISO88599,"iso-8859-9"},
	{UDM_CHARSET_ISO88599,"iso8859-9"},
	{UDM_CHARSET_ISO88599,"iso8859.9"},
	{UDM_CHARSET_ISO88599,"iso-8859.9"},
	{UDM_CHARSET_ISO88599,"iso_8859-9"},
	{UDM_CHARSET_ISO88599,"iso_8859.9"},
	{UDM_CHARSET_ISO88599,"latin5"},
	/* 1254 */
	{UDM_CHARSET_CP1254,   "windows-1254"},
	{UDM_CHARSET_CP1254,   "cp1254"},
	{UDM_CHARSET_CP1254,   "windows1254"},
	{UDM_CHARSET_CP1254,   "x-cp1254"},
	{UDM_CHARSET_CP1254,   "cp-1254"},
	/* 1255 */
	{UDM_CHARSET_CP1255,   "windows-1255"},
	{UDM_CHARSET_CP1255,   "cp1255"},
	{UDM_CHARSET_CP1255,   "windows1255"},
	{UDM_CHARSET_CP1255,   "x-cp1255"},
	{UDM_CHARSET_CP1255,   "cp-1255"},
	
	/* EOF */
	{-1,""}
};

/****************************************************************/


#ifdef USE_CHARSET_GUESSER
/* Cyrillic charset guesser                    */
/* Use --enable-charset-guesser to activate it */
/* Encoding auto detection
 * rtable[key_last[last_char]][(key_new[new_char])>>2]>>(key_new[new_char]<<1))&3;
 * returns the index of biletter frequences in internal (iso) encoding:
 *
 *    3    extremally rare
 *    0    indifferent
 *    1    often
 *    2    very often
 *
 */

/*
 *      The "key_last" is the first key for the table of biletter frequences:
 */
static const unsigned char key_last[256]={0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
0, 0, 0, 1, 2, 3, 4, 5, 6, 4, 7, 8, 4, 9, 4, 10, 4, 11, 12, 6, 4, 13, 14,
4, 4, 6, 15, 6, 4, 0, 0, 0, 0, 0, 0, 1, 4, 16, 4, 17, 4, 4, 6, 18, 4, 19,
4, 4, 4, 20, 21, 19, 4, 22, 4, 23, 4, 19, 24, 25, 4, 0, 0, 0, 0, 0, 26,
27, 26, 28, 26, 26, 27, 29, 27, 26, 27, 29, 27, 27, 27, 27, 27, 29, 28,
27, 26, 29, 28, 29, 28, 28, 28, 27, 27, 26, 27, 28, 30, 31, 32, 29, 33,
29, 34, 33, 35, 36, 37, 38, 39, 30, 40, 41, 42, 43, 44, 45, 46, 47, 48,
49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66,
67, 68, 69, 70, 71, 72, 73, 42, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83,
84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99, 100, 101,
102, 103, 104, 30, 105, 106, 29, 107, 29, 108, 107, 109, 110, 111, 112,
111, 30, 113, 114};
/*
 *      The "key_new" is the second key for the table of biletter frequences:
 */
static const unsigned char key_new[256]={0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
0, 0, 1, 2, 3, 4, 5, 6, 4, 7, 8, 4, 9, 4, 10, 4, 11, 12, 6, 4, 13, 14,
4, 4, 6, 15, 16, 4, 0, 0, 0, 0, 0, 0, 17, 4, 18, 4, 19, 4, 4, 16, 8, 4,
16, 4, 4, 4, 20, 21, 16, 4, 22, 4, 23, 4, 16, 24, 25, 4, 0, 0, 0, 0, 0,
26, 26, 27, 26, 27, 27, 26, 27, 26, 28, 28, 26, 26, 26, 26, 26, 26, 28,
29, 29, 27, 28, 28, 29, 29, 29, 26, 29, 27, 26, 27, 29, 30, 31, 32, 28,
33, 34, 28, 35, 36, 37, 38, 39, 40, 41, 42, 28, 43, 44, 45, 46, 47, 48,
49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66,
67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84,
85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99, 100, 101, 102,
103, 104, 105, 106, 41, 107, 28, 108, 109, 110, 111, 112, 113, 108, 114,
108, 115, 41, 116, 117};

/*
 *      The "to_the[]" are the maps from ISO 8859-5 encoding:
 */
static const unsigned char to_the[4][128]={
/* koi */
{165, 144, 184, 130, 187,
190, 139, 181, 154, 177, 145, 146, 148, 156, 150, 140, 191, 129, 138, 131,
160, 161, 134, 135, 136, 137, 132, 133, 174, 171, 168, 128, 167, 179, 162,
143, 153, 166, 142, 147, 169, 170, 172, 173, 149, 164, 159, 175, 225, 226,
247, 231, 228, 229, 246, 250, 233, 234, 235, 236, 237, 238, 239, 240, 242,
243, 244, 245, 230, 232, 227, 254, 251, 253, 255, 249, 248, 252, 224, 241,
193, 194, 215, 199, 196, 197, 214, 218, 201, 202, 203, 204, 205, 206, 207,
208, 210, 211, 212, 213, 198, 200, 195, 222, 219, 221, 223, 217, 216, 220,
192, 209, 157, 163, 176, 178, 152, 180, 182, 155, 183, 185, 186, 188, 189,
158, 151, 141},
/* win */
{149, 133, 153, 130, 135, 136, 137, 139,
160, 155, 166, 183, 152, 176, 173, 164, 169, 172, 174, 177, 181, 182, 165,
180, 150, 151, 145, 146, 147, 148, 132, 134, 171, 168, 128, 129, 170, 189,
178, 175, 163, 138, 140, 142, 141, 187, 161, 143, 192, 193, 194, 195, 196,
197, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207, 208, 209, 210, 211,
212, 213, 214, 215, 216, 217, 218, 219, 220, 221, 222, 223, 224, 225, 226,
227, 228, 229, 230, 231, 232, 233, 234, 235, 236, 237, 238, 239, 240, 241,
242, 243, 244, 245, 246, 247, 248, 249, 250, 251, 252, 253, 254, 255, 185,
184, 144, 131, 186, 190, 179, 191, 188, 154, 156, 158, 157, 167, 162, 159},
/* dos */
{201, 176, 203, 218, 202, 206, 223, 185, 255, 204, 177, 178, 254, 248,
251, 253, 220, 179, 197, 191, 205, 186, 195, 180, 194, 193, 192, 217, 188,
200, 187, 196, 183, 240, 213, 222, 242, 184, 221, 244, 212, 211, 190, 189,
249, 214, 246, 198, 128, 129, 130, 131, 132, 133, 134, 135, 136, 137, 138,
139, 140, 141, 142, 143, 144, 145, 146, 147, 148, 149, 150, 151, 152, 153,
154, 155, 156, 157, 158, 159, 160, 161, 162, 163, 164, 165, 166, 167, 168,
169, 170, 171, 172, 173, 174, 175, 224, 225, 226, 227, 228, 229, 230, 231,
232, 233, 234, 235, 236, 237, 238, 239, 252, 241, 199, 181, 243, 182, 209,
245, 210, 207, 208, 216, 215, 250, 247, 219},
/* mac */
{165, 201, 170, 163, 173, 176, 198, 178, 202, 179, 214,
196, 197, 161, 195, 255, 169, 194, 168, 177, 181, 166, 162, 182, 208, 209,
212, 213, 210, 211, 215, 160, 199, 221, 171, 174, 184, 193, 167, 186, 183,
188, 190, 203, 205, 200, 216, 218, 128, 129, 130, 131, 132, 133, 134, 135,
136, 137, 138, 139, 140, 141, 142, 143, 144, 145, 146, 147, 148, 149, 150,
151, 152, 153, 154, 155, 156, 157, 158, 159, 224, 225, 226, 227, 228, 229,
230, 231, 232, 233, 234, 235, 236, 237, 238, 239, 240, 241, 242, 243, 244,
245, 246, 247, 248, 249, 250, 251, 252, 253, 254, 223, 220, 222, 172, 175,
185, 207, 180, 187, 192, 189, 191, 204, 206, 164, 217, 219}};

/*
 *      The "from_the[]" are the maps to ISO 8859-5 encoding:
 */
static const unsigned char from_the[4][128]={
/* koi */
{159, 145, 131, 147, 154,
155, 150, 151, 152, 153, 146, 134, 143, 255, 166, 163, 129, 138, 139, 167,
140, 172, 142, 254, 244, 164, 136, 247, 141, 240, 253, 174, 148, 149, 162,
241, 173, 128, 165, 160, 158, 168, 169, 157, 170, 171, 156, 175, 242, 137,
243, 161, 245, 135, 246, 248, 130, 249, 250, 132, 251, 252, 133, 144, 238,
208, 209, 230, 212, 213, 228, 211, 229, 216, 217, 218, 219, 220, 221, 222,
223, 239, 224, 225, 226, 227, 214, 210, 236, 235, 215, 232, 237, 233, 231,
234, 206, 176, 177, 198, 180, 181, 196, 179, 197, 184, 185, 186, 187, 188,
189, 190, 191, 207, 192, 193, 194, 195, 182, 178, 204, 203, 183, 200, 205,
201, 199, 202},
/* win */
{162, 163, 131, 243, 158, 129, 159, 132,
133, 134, 169, 135, 170, 172, 171, 175, 242, 154, 155, 156, 157, 128, 152,
153, 140, 130, 249, 137, 250, 252, 251, 255, 136, 174, 254, 168, 143, 150,
138, 253, 161, 144, 164, 160, 145, 142, 146, 167, 141, 147, 166, 246, 151,
148, 149, 139, 241, 240, 244, 173, 248, 165, 245, 247, 176, 177, 178, 179,
180, 181, 182, 183, 184, 185, 186, 187, 188, 189, 190, 191, 192, 193, 194,
195, 196, 197, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207, 208, 209,
210, 211, 212, 213, 214, 215, 216, 217, 218, 219, 220, 221, 222, 223, 224,
225, 226, 227, 228, 229, 230, 231, 232, 233, 234, 235, 236, 237, 238, 239},
/* dos */
{176, 177, 178, 179, 180, 181, 182, 183, 184, 185, 186, 187, 188, 189,
190, 191, 192, 193, 194, 195, 196, 197, 198, 199, 200, 201, 202, 203, 204,
205, 206, 207, 208, 209, 210, 211, 212, 213, 214, 215, 216, 217, 218, 219,
220, 221, 222, 223, 129, 138, 139, 145, 151, 243, 245, 160, 165, 135, 149,
158, 156, 171, 170, 147, 154, 153, 152, 150, 159, 146, 175, 242, 157, 128,
132, 130, 137, 148, 133, 249, 250, 246, 248, 169, 168, 162, 173, 252, 251,
155, 131, 255, 144, 166, 163, 134, 224, 225, 226, 227, 228, 229, 230, 231,
232, 233, 234, 235, 236, 237, 238, 239, 161, 241, 164, 244, 167, 247, 174,
254, 141, 172, 253, 142, 240, 143, 140, 136},
/* mac */
{176, 177, 178, 179, 180, 181, 182, 183, 184, 185, 186,
187, 188, 189, 190, 191, 192, 193, 194, 195, 196, 197, 198, 199, 200, 201,
202, 203, 204, 205, 206, 207, 159, 141, 150, 131, 253, 128, 149, 166, 146,
144, 130, 162, 242, 132, 163, 243, 133, 147, 135, 137, 246, 148, 151, 168,
164, 244, 167, 247, 169, 249, 170, 250, 248, 165, 145, 142, 139, 140, 134,
160, 173, 129, 136, 171, 251, 172, 252, 245, 152, 153, 156, 157, 154, 155,
138, 158, 174, 254, 175, 255, 240, 161, 241, 239, 208, 209, 210, 211, 212,
213, 214, 215, 216, 217, 218, 219, 220, 221, 222, 223, 224, 225, 226, 227,
228, 229, 230, 231, 232, 233, 234, 235, 236, 237, 238, 143}};

static const unsigned char rtable[115][30]={{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 64, 85,
81, 85, 85, 21, 81, 204, 81, 85, 81, 85, 85, 21, 81, 204, 17, 0, 0, 0},
{0, 0, 0, 0, 0, 0, 0, 16, 0, 0, 68, 85, 20, 85, 81, 5, 81, 0, 80, 85, 20,
85, 81, 5, 81, 0, 16, 0, 0, 0},
{0, 0, 0, 0, 0, 0, 0, 16, 192, 0, 116, 3, 65, 3, 69, 208, 0, 16, 64, 3,
65, 3, 69, 208, 0, 16, 0, 0, 3, 3},
{0, 0, 0, 0, 0, 0, 0, 16, 192, 0, 116, 0, 65, 23, 5, 21, 0, 80, 80, 0,
65, 23, 5, 21, 0, 80, 16, 0, 3, 3},
{0, 0, 0, 0, 0, 0, 240, 252, 255, 255, 255, 255, 255, 255, 255, 255, 255,
255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 15},
{0, 0, 0, 0, 0, 0, 0, 16, 252, 0, 52, 85, 16, 85, 81, 5, 65, 240, 0, 85,
16, 85, 81, 5, 65, 240, 0, 60, 3, 3},
{0, 0, 0, 0, 0, 0, 0, 192, 3, 255, 3, 0, 0, 0, 0, 0, 12, 252, 60, 0, 0,
0, 0, 0, 12, 252, 252, 3, 252, 12},
{0, 0, 0, 0, 0, 0, 0, 16, 0, 0, 68, 80, 65, 64, 5, 21, 16, 80, 81, 80,
65, 64, 5, 21, 16, 80, 17, 0, 0, 0},
{0, 0, 0, 0, 0, 0, 0, 192, 3, 0, 0, 0, 0, 3, 0, 0, 0, 255, 63, 0, 0, 3,
0, 0, 0, 255, 255, 0, 0, 0},
{0, 0, 0, 0, 0, 0, 0, 16, 204, 0, 116, 3, 65, 23, 4, 213, 0, 19, 112, 3,
65, 23, 4, 213, 0, 19, 48, 12, 3, 3},
{0, 0, 0, 0, 0, 0, 0, 208, 204, 3, 68, 0, 113, 3, 4, 16, 0, 31, 64, 0,
113, 3, 4, 16, 0, 31, 192, 12, 15, 0},
{0, 0, 0, 0, 0, 0, 0, 16, 0, 0, 4, 85, 16, 85, 81, 5, 65, 12, 0, 85, 16,
85, 81, 5, 65, 12, 0, 0, 0, 0},
{0, 0, 0, 0, 0, 0, 0, 16, 192, 0, 116, 80, 65, 84, 5, 21, 0, 16, 64, 80,
65, 84, 5, 21, 0, 16, 0, 0, 3, 3},
{0, 0, 0, 0, 0, 0, 0, 192, 3, 252, 3, 0, 0, 0, 0, 0, 0, 252, 0, 0, 0, 0,
0, 0, 0, 252, 192, 3, 240, 12},
{0, 0, 0, 0, 0, 0, 0, 16, 192, 0, 68, 4, 125, 23, 5, 20, 0, 83, 65, 4,
125, 23, 5, 20, 0, 83, 1, 0, 3, 0},
{0, 0, 0, 0, 0, 0, 0, 208, 204, 0, 68, 0, 76, 3, 4, 212, 0, 3, 76, 0, 76,
3, 4, 212, 0, 3, 204, 204, 3, 0},
{0, 0, 0, 0, 0, 0, 0, 208, 255, 255, 247, 255, 255, 255, 255, 255, 255,
255, 127, 0, 65, 23, 5, 21, 0, 80, 16, 0, 3, 3},
{0, 0, 0, 0, 0, 0, 0, 208, 255, 255, 247, 255, 255, 255, 255, 255, 255,
255, 63, 85, 16, 85, 81, 5, 65, 240, 0, 60, 3, 3},
{0, 0, 0, 0, 0, 0, 0, 192, 255, 255, 243, 255, 255, 255, 255, 255, 255,
255, 63, 0, 0, 3, 0, 0, 0, 255, 255, 0, 0, 0},
{0, 0, 0, 0, 0, 0, 0, 192, 255, 255, 243, 255, 255, 255, 255, 255, 255,
255, 63, 0, 0, 0, 0, 0, 12, 252, 252, 3, 252, 12},
{0, 0, 0, 0, 0, 0, 0, 208, 255, 255, 247, 255, 255, 255, 255, 255, 255,
255, 63, 85, 16, 85, 81, 5, 65, 12, 0, 0, 0, 0},
{0, 0, 0, 0, 0, 0, 0, 208, 255, 255, 247, 255, 255, 255, 255, 255, 255,
255, 127, 80, 65, 84, 5, 21, 0, 16, 0, 0, 3, 3},
{0, 0, 0, 0, 0, 0, 0, 192, 255, 255, 243, 255, 255, 255, 255, 255, 255,
255, 63, 0, 0, 0, 0, 0, 0, 252, 192, 3, 240, 12},
{0, 0, 0, 0, 0, 0, 0, 208, 255, 255, 247, 255, 255, 255, 255, 255, 255,
255, 63, 84, 16, 85, 65, 5, 80, 204, 16, 0, 0, 0},
{0, 0, 0, 0, 0, 0, 0, 208, 255, 255, 247, 255, 255, 255, 255, 255, 255,
255, 127, 0, 76, 3, 4, 212, 0, 3, 204, 204, 3, 0},
{0, 0, 0, 0, 0, 0, 0, 208, 255, 255, 247, 255, 255, 255, 255, 255, 255,
255, 63, 84, 16, 85, 81, 5, 81, 12, 192, 48, 3, 3},
{0, 3, 3, 0, 0, 0, 0, 0, 207, 204, 3, 0, 0, 0, 0, 0, 0, 204, 12, 0, 0,
0, 0, 0, 0, 204, 12, 12, 243, 12},
{0, 3, 3, 0, 0, 0, 0, 0, 207, 252, 3, 0, 0, 0, 0, 0, 0, 204, 12, 0, 0,
0, 0, 0, 0, 204, 12, 207, 243, 12},
{0, 3, 3, 0, 0, 0, 0, 0, 207, 60, 3, 0, 0, 0, 0, 0, 0, 204, 12, 0, 0, 0,
0, 0, 0, 204, 12, 207, 3, 12},
{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
0, 0, 0, 0, 0, 0},
{84, 71, 84, 81, 84, 69, 5, 64, 1, 0, 128, 170, 166, 170, 170, 106, 166,
220, 166, 170, 166, 170, 170, 106, 166, 220, 102, 0, 0, 0},
{0, 63, 3, 12, 195, 240, 252, 204, 255, 255, 51, 0, 195, 0, 65, 48, 1,
255, 51, 0, 195, 0, 65, 48, 1, 255, 243, 255, 255, 15},
{240, 243, 63, 255, 51, 60, 51, 220, 207, 207, 119, 255, 60, 255, 243,
207, 255, 255, 127, 255, 60, 255, 243, 207, 255, 255, 255, 60, 255, 15}
,
{0, 3, 0, 0, 0, 0, 240, 204, 3, 252, 3, 0, 0, 0, 0, 0, 0, 252, 0, 0, 0,
0, 0, 0, 0, 252, 192, 3, 240, 12},
{0, 3, 0, 0, 0, 0, 0, 192, 3, 0, 0, 0, 0, 3, 0, 0, 0, 255, 63, 0, 0, 3,
0, 0, 0, 255, 255, 0, 0, 0},
{0, 51, 0, 0, 3, 0, 0, 192, 204, 0, 112, 3, 0, 3, 48, 192, 12, 255, 127,
3, 0, 3, 48, 192, 12, 255, 255, 12, 3, 3},
{0, 51, 0, 12, 3, 48, 240, 192, 252, 0, 48, 0, 0, 0, 0, 0, 0, 255, 63,
0, 0, 0, 0, 0, 0, 255, 255, 12, 3, 3},
{0, 51, 0, 12, 3, 48, 240, 204, 252, 0, 48, 0, 0, 0, 0, 0, 0, 255, 63,
0, 0, 0, 0, 0, 0, 255, 255, 60, 3, 3},
{0, 51, 0, 204, 3, 48, 243, 204, 252, 3, 48, 243, 60, 3, 48, 192, 3, 255,
63, 243, 60, 3, 48, 192, 3, 255, 255, 60, 15, 3},
{0, 51, 0, 12, 3, 48, 240, 204, 255, 0, 48, 0, 0, 0, 0, 0, 0, 255, 63,
0, 0, 0, 0, 0, 0, 255, 255, 60, 3, 3},
{0, 3, 0, 0, 0, 0, 240, 204, 3, 252, 3, 0, 0, 0, 0, 0, 0, 204, 0, 0, 0,
0, 0, 0, 0, 204, 192, 3, 240, 12},
{240, 243, 63, 255, 51, 60, 51, 204, 207, 207, 51, 255, 60, 255, 243, 207,
255, 255, 63, 255, 60, 255, 243, 207, 255, 255, 255, 60, 255, 15},
{85, 67, 20, 81, 20, 4, 1, 96, 1, 0, 136, 170, 105, 170, 162, 90, 166,
0, 164, 170, 105, 170, 162, 90, 166, 0, 100, 0, 0, 0},
{4, 55, 64, 0, 71, 65, 4, 80, 0, 3, 132, 0, 130, 19, 72, 224, 192, 32,
145, 0, 130, 19, 72, 224, 192, 32, 81, 0, 12, 0},
{5, 119, 64, 1, 71, 69, 4, 96, 192, 0, 184, 3, 130, 23, 138, 229, 1, 32,
129, 3, 130, 23, 138, 229, 1, 32, 65, 0, 3, 3},
{5, 7, 64, 1, 68, 69, 4, 224, 192, 3, 184, 64, 190, 19, 137, 228, 204,
103, 129, 64, 190, 19, 137, 228, 204, 103, 193, 0, 15, 3},
{5, 119, 64, 0, 71, 65, 4, 96, 209, 3, 184, 84, 150, 19, 74, 225, 1, 99,
145, 84, 150, 19, 74, 225, 1, 99, 81, 16, 15, 3},
{81, 67, 20, 93, 16, 52, 1, 96, 252, 0, 56, 170, 37, 170, 162, 10, 150,
240, 0, 170, 37, 170, 162, 10, 150, 240, 64, 60, 3, 3},
{4, 51, 48, 204, 7, 112, 3, 192, 252, 3, 176, 0, 177, 195, 49, 208, 199,
31, 188, 0, 177, 195, 49, 208, 199, 31, 252, 60, 15, 3},
{5, 119, 0, 0, 71, 64, 4, 224, 204, 3, 136, 149, 130, 83, 6, 224, 204,
99, 128, 149, 130, 83, 6, 224, 204, 99, 192, 12, 15, 0},
{81, 67, 20, 17, 16, 4, 0, 96, 0, 0, 72, 169, 101, 170, 150, 74, 165, 204,
96, 169, 101, 170, 150, 74, 165, 204, 96, 0, 0, 0},
{1, 115, 0, 16, 3, 64, 0, 224, 207, 255, 123, 144, 129, 36, 70, 9, 0, 239,
67, 144, 129, 36, 70, 9, 0, 239, 195, 15, 255, 15},
{69, 55, 68, 16, 87, 65, 4, 96, 221, 0, 184, 7, 130, 43, 73, 234, 4, 39,
177, 7, 130, 43, 73, 234, 4, 39, 113, 28, 3, 3},
{5, 71, 80, 80, 68, 65, 5, 96, 17, 0, 136, 161, 150, 164, 10, 40, 2, 160,
166, 161, 150, 164, 10, 40, 2, 160, 102, 16, 0, 0},
{5, 55, 64, 0, 71, 65, 4, 224, 205, 3, 136, 65, 178, 23, 73, 33, 0, 47,
129, 65, 178, 23, 73, 33, 0, 47, 193, 12, 15, 0},
{69, 71, 80, 16, 84, 65, 4, 96, 17, 0, 136, 164, 130, 148, 10, 42, 36,
160, 162, 164, 130, 148, 10, 42, 36, 160, 98, 16, 0, 0},
{81, 115, 20, 81, 19, 4, 1, 96, 0, 0, 72, 170, 101, 170, 166, 10, 150,
13, 80, 170, 101, 170, 166, 10, 150, 13, 80, 0, 0, 0},
{53, 55, 64, 16, 71, 65, 4, 96, 192, 3, 187, 12, 190, 23, 72, 232, 0, 32,
140, 12, 190, 23, 72, 232, 0, 32, 76, 0, 207, 3},
{69, 71, 84, 16, 84, 65, 4, 96, 193, 0, 184, 164, 134, 168, 10, 106, 81,
32, 129, 164, 134, 168, 10, 106, 81, 32, 65, 0, 3, 3},
{69, 71, 68, 16, 84, 65, 4, 96, 209, 0, 184, 4, 130, 107, 26, 42, 17, 160,
161, 4, 130, 107, 26, 42, 17, 160, 97, 16, 3, 3},
{21, 119, 68, 16, 71, 65, 4, 96, 209, 0, 136, 72, 190, 107, 74, 41, 0,
167, 146, 72, 190, 107, 74, 41, 0, 167, 82, 16, 3, 0},
{81, 67, 20, 93, 16, 52, 1, 224, 240, 0, 120, 169, 100, 170, 162, 26, 162,
28, 84, 169, 100, 170, 162, 26, 162, 28, 212, 48, 3, 3},
{48, 51, 12, 192, 3, 0, 3, 192, 204, 3, 112, 255, 125, 15, 52, 4, 255,
207, 124, 255, 125, 15, 52, 4, 255, 207, 252, 12, 15, 3},
{5, 3, 67, 16, 4, 65, 4, 224, 204, 0, 136, 0, 141, 19, 9, 232, 0, 87, 141,
0, 141, 19, 9, 232, 0, 87, 205, 204, 3, 0},
{4, 243, 3, 0, 7, 64, 0, 16, 220, 3, 180, 3, 189, 0, 55, 208, 252, 15,
128, 3, 189, 0, 55, 208, 252, 15, 0, 220, 15, 3},
{4, 55, 0, 192, 71, 64, 7, 16, 204, 3, 180, 243, 190, 7, 53, 228, 3, 35,
177, 243, 190, 7, 53, 228, 3, 35, 49, 12, 15, 3},
{5, 55, 4, 16, 71, 64, 0, 224, 204, 0, 184, 0, 190, 107, 5, 216, 12, 35,
188, 0, 190, 107, 5, 216, 12, 35, 252, 12, 3, 3},
{240, 51, 63, 195, 51, 12, 243, 12, 207, 252, 51, 255, 124, 255, 240, 195,
255, 12, 0, 255, 124, 255, 240, 195, 255, 12, 0, 207, 243, 15},
{192, 51, 15, 12, 51, 48, 240, 12, 255, 255, 115, 0, 64, 15, 48, 211, 252,
220, 80, 0, 64, 15, 48, 211, 252, 220, 16, 255, 255, 15},
{65, 115, 215, 29, 19, 55, 0, 224, 255, 255, 59, 165, 21, 169, 174, 10,
145, 204, 3, 165, 21, 169, 174, 10, 145, 204, 195, 255, 255, 15},
{1, 3, 3, 0, 0, 0, 0, 96, 207, 255, 123, 0, 64, 20, 5, 21, 0, 220, 64,
0, 64, 20, 5, 21, 0, 220, 64, 207, 255, 15},
{0, 67, 23, 16, 0, 0, 240, 220, 207, 255, 55, 148, 0, 169, 82, 9, 17, 252,
13, 148, 0, 169, 82, 9, 17, 252, 205, 207, 255, 15},
{48, 51, 3, 12, 3, 48, 0, 16, 243, 255, 55, 12, 0, 16, 49, 196, 0, 60,
51, 12, 0, 16, 49, 196, 0, 60, 51, 243, 255, 15},
{81, 115, 195, 12, 19, 243, 12, 224, 243, 255, 59, 88, 208, 84, 78, 246,
1, 12, 3, 88, 208, 84, 78, 246, 1, 12, 195, 243, 255, 15},
{252, 255, 252, 255, 71, 65, 4, 208, 255, 255, 247, 255, 255, 255, 255,
255, 255, 255, 191, 0, 130, 19, 72, 224, 192, 32, 81, 0, 12, 0},
{253, 255, 252, 255, 71, 69, 4, 224, 255, 255, 251, 255, 255, 255, 255,
255, 255, 255, 191, 3, 130, 23, 138, 229, 1, 32, 65, 0, 3, 3},
{253, 255, 252, 255, 68, 69, 4, 224, 255, 255, 251, 255, 255, 255, 255,
255, 255, 255, 191, 64, 190, 19, 137, 228, 204, 103, 193, 0, 15, 3},
{253, 255, 252, 255, 71, 65, 4, 224, 255, 255, 251, 255, 255, 255, 255,
255, 255, 255, 191, 84, 150, 19, 74, 225, 1, 99, 81, 16, 15, 3},
{253, 255, 252, 255, 16, 52, 1, 224, 255, 255, 251, 255, 255, 255, 255,
255, 255, 255, 63, 170, 37, 170, 162, 10, 150, 240, 64, 60, 3, 3},
{252, 255, 252, 255, 7, 112, 3, 192, 255, 255, 243, 255, 255, 255, 255,
255, 255, 255, 191, 0, 177, 195, 49, 208, 199, 31, 252, 60, 15, 3},
{253, 255, 252, 255, 71, 64, 4, 224, 255, 255, 251, 255, 255, 255, 255,
255, 255, 255, 191, 149, 130, 83, 6, 224, 204, 99, 192, 12, 15, 0},
{253, 255, 252, 255, 16, 4, 0, 224, 255, 255, 251, 255, 255, 255, 255,
255, 255, 255, 127, 169, 101, 170, 150, 74, 165, 204, 96, 0, 0, 0},
{253, 255, 252, 255, 3, 64, 0, 224, 255, 255, 251, 255, 255, 255, 255,
255, 255, 255, 127, 144, 129, 36, 70, 9, 0, 239, 195, 15, 255, 15},
{253, 255, 252, 255, 87, 65, 4, 224, 255, 255, 251, 255, 255, 255, 255,
255, 255, 255, 191, 7, 130, 43, 73, 234, 4, 39, 113, 28, 3, 3},
{253, 255, 252, 255, 68, 65, 5, 224, 255, 255, 251, 255, 255, 255, 255,
255, 255, 255, 191, 161, 150, 164, 10, 40, 2, 160, 102, 16, 0, 0},
{253, 255, 252, 255, 71, 65, 4, 224, 255, 255, 251, 255, 255, 255, 255,
255, 255, 255, 191, 65, 178, 23, 73, 33, 0, 47, 193, 12, 15, 0},
{253, 255, 252, 255, 84, 65, 4, 224, 255, 255, 251, 255, 255, 255, 255,
255, 255, 255, 191, 164, 130, 148, 10, 42, 36, 160, 98, 16, 0, 0},
{253, 255, 252, 255, 19, 4, 1, 224, 255, 255, 251, 255, 255, 255, 255,
255, 255, 255, 127, 170, 101, 170, 166, 10, 150, 13, 80, 0, 0, 0},
{253, 255, 252, 255, 71, 65, 4, 224, 255, 255, 251, 255, 255, 255, 255,
255, 255, 255, 191, 12, 190, 23, 72, 232, 0, 32, 76, 0, 207, 3},
{253, 255, 252, 255, 84, 65, 4, 224, 255, 255, 251, 255, 255, 255, 255,
255, 255, 255, 191, 164, 134, 168, 10, 106, 81, 32, 65, 0, 3, 3},
{253, 255, 252, 255, 84, 65, 4, 224, 255, 255, 251, 255, 255, 255, 255,
255, 255, 255, 191, 4, 130, 107, 26, 42, 17, 160, 97, 16, 3, 3},
{253, 255, 252, 255, 71, 65, 4, 224, 255, 255, 251, 255, 255, 255, 255,
255, 255, 255, 191, 72, 190, 107, 74, 41, 0, 167, 82, 16, 3, 0},
{253, 255, 252, 255, 16, 52, 1, 224, 255, 255, 251, 255, 255, 255, 255,
255, 255, 255, 127, 169, 100, 170, 162, 26, 162, 28, 212, 48, 3, 3},
{252, 255, 252, 255, 3, 0, 3, 192, 255, 255, 243, 255, 255, 255, 255, 255,
255, 255, 127, 255, 125, 15, 52, 4, 255, 207, 252, 12, 15, 3},
{253, 255, 255, 255, 4, 65, 4, 224, 255, 255, 251, 255, 255, 255, 255,
255, 255, 255, 191, 0, 141, 19, 9, 232, 0, 87, 205, 204, 3, 0},
{252, 255, 255, 255, 7, 64, 0, 208, 255, 255, 247, 255, 255, 255, 255,
255, 255, 255, 191, 3, 189, 0, 55, 208, 252, 15, 0, 220, 15, 3},
{252, 255, 252, 255, 71, 64, 7, 208, 255, 255, 247, 255, 255, 255, 255,
255, 255, 255, 191, 243, 190, 7, 53, 228, 3, 35, 49, 12, 15, 3},
{253, 255, 252, 255, 71, 64, 0, 224, 255, 255, 251, 255, 255, 255, 255,
255, 255, 255, 191, 0, 190, 107, 5, 216, 12, 35, 252, 12, 3, 3},
{252, 255, 255, 255, 51, 12, 243, 204, 255, 255, 243, 255, 255, 255, 255,
255, 255, 255, 63, 255, 124, 255, 240, 195, 255, 12, 0, 207, 243, 15},
{252, 255, 255, 255, 51, 48, 240, 204, 255, 255, 243, 255, 255, 255, 255,
255, 255, 255, 127, 0, 64, 15, 48, 211, 252, 220, 16, 255, 255, 15},
{253, 255, 255, 255, 19, 55, 0, 224, 255, 255, 251, 255, 255, 255, 255,
255, 255, 255, 63, 165, 21, 169, 174, 10, 145, 204, 195, 255, 255, 15},
{253, 255, 255, 255, 0, 0, 0, 224, 255, 255, 251, 255, 255, 255, 255, 255,
255, 255, 127, 0, 64, 20, 5, 21, 0, 220, 64, 207, 255, 15},
{252, 255, 255, 255, 0, 0, 240, 220, 255, 255, 247, 255, 255, 255, 255,
255, 255, 255, 63, 148, 0, 169, 82, 9, 17, 252, 205, 207, 255, 15},
{252, 255, 255, 255, 3, 48, 0, 208, 255, 255, 247, 255, 255, 255, 255,
255, 255, 255, 63, 12, 0, 16, 49, 196, 0, 60, 51, 243, 255, 15},
{253, 255, 255, 255, 19, 243, 12, 224, 255, 255, 251, 255, 255, 255, 255,
255, 255, 255, 63, 88, 208, 84, 78, 246, 1, 12, 195, 243, 255, 15},
{252, 255, 255, 255, 195, 240, 252, 204, 255, 255, 243, 255, 255, 255,
255, 255, 255, 255, 63, 0, 195, 0, 65, 48, 1, 255, 243, 255, 255, 15},
{252, 255, 255, 255, 51, 60, 243, 192, 255, 63, 247, 255, 255, 255, 255,
255, 255, 255, 127, 255, 60, 255, 243, 207, 255, 255, 255, 207, 3, 15},
{252, 255, 252, 255, 0, 0, 240, 204, 255, 255, 243, 255, 255, 255, 255,
255, 255, 255, 63, 0, 0, 0, 0, 0, 0, 252, 192, 3, 240, 12},
{252, 255, 252, 255, 0, 0, 0, 192, 207, 207, 243, 255, 255, 255, 255, 255,
255, 255, 63, 0, 0, 3, 0, 0, 0, 255, 255, 0, 0, 0},
{252, 255, 252, 255, 3, 0, 0, 192, 255, 63, 243, 255, 255, 255, 255, 255,
255, 255, 127, 3, 0, 3, 48, 192, 12, 255, 255, 12, 3, 3},
{252, 255, 252, 255, 3, 48, 48, 204, 207, 207, 243, 255, 255, 255, 255,
255, 255, 255, 63, 0, 0, 0, 0, 0, 0, 255, 255, 60, 3, 3},
{252, 255, 252, 255, 3, 48, 240, 192, 255, 63, 243, 255, 255, 255, 255,
255, 255, 255, 63, 0, 0, 0, 0, 0, 0, 255, 255, 12, 3, 3},
{252, 255, 252, 255, 3, 48, 51, 204, 207, 207, 243, 255, 255, 255, 255,
255, 255, 255, 63, 243, 60, 3, 48, 192, 3, 255, 255, 60, 15, 3},
{252, 255, 252, 255, 0, 0, 240, 204, 255, 255, 243, 255, 255, 255, 255,
255, 255, 255, 63, 0, 0, 0, 0, 0, 0, 204, 192, 3, 240, 12},
{252, 255, 255, 255, 51, 60, 243, 204, 255, 255, 243, 255, 255, 255, 255,
255, 255, 255, 63, 255, 60, 255, 243, 207, 255, 255, 255, 255, 255, 15}
};


#define from_to_char(argf,argt,argd) \
                ( argd < 128 )? argd :\
                (( argf ==UDM_CHARSET_ISO88595)&&( argt ==UDM_CHARSET_ISO88595))?  argd :\
                ( argf ==UDM_CHARSET_ISO88595)? to_the [ argt ][ argd & 127 ] :\
                ( argt ==UDM_CHARSET_ISO88595)? from_the [ argf ][ argd & 127 ] :\
         to_the [ argt ][(from_the [ argf ][ argd & 127]) & 127 ]

#define from_char(argf,argd) ((argd<128)||(argf==UDM_CHARSET_ISO88595))? argd :\
                             from_the [argf][argd & 127]

static int biletter_index (int encoding, int last_char, int new_char){
  int new;

  encoding--; /* offset to 0 el of arry */
  new=key_new[from_char(encoding,new_char)];
  new=(rtable[key_last[from_char(encoding,last_char)]][new>>2]>>((new & 3)<<1))&3;
  if (new == 3) new = -10;

  return(new);
}


int UdmGuessCharset(unsigned char *str, int charset){
  unsigned int lastchar, newchar;
  int i, j, from_encoding, code_num, statist[6];
  unsigned char *p = str;

  if (charset > UDM_CHARSET_ISO88595) return charset;
  from_encoding = charset > 0 ? charset : 1;

  lastchar = ' ';
  /*     */
  while ((newchar = *p++) && (newchar < 128))
    lastchar = newchar;

  str = p; j = 0;
  while (newchar)
  {
    /*    from_encoding   */
    i = biletter_index(from_encoding, lastchar, newchar);

    if (newchar > 128)
    {
      j = (i == 0) ? j+2:
          (i == 1) ? j++:
          (i == 2) ? 0:5;
      if (j > 3) break; /*    ! */
    }

    lastchar = newchar;
    newchar = *p++;
  }

  /*        -  */
  if (newchar == 0)
    return charset;

  /*   ,    */
  for(j = 1; j < 6; j++)
  {
    /*      j */
    statist[j] = 0;
    lastchar = ' ';
    p = str;
    while ((newchar = *p++))
    {
        statist[j] += biletter_index(j, lastchar, newchar);
        lastchar = newchar;
    }
  }

  /*    */
  code_num = from_encoding;
  for(i = 1; i < 6; i++)
    if (statist[i] > statist[code_num])
      code_num = i;

  /*   = code_num */
  return code_num;
}
#endif


/*********************************************************/

__INDLIB__ int UdmGetCharset(unsigned char *alias){
	if(alias){
		int i;
		for(i=0;Aliases[i].charset!=-1;i++)
			if(!strcasecmp(Aliases[i].alias,alias)){
				return(Aliases[i].charset);
			}
	}
	return(UDM_CHARSET_USASCII);
}

const char * UdmCharsetStr(int id){
	int i;
	
	for(i=0;Aliases[i].charset!=-1;i++)
		if(Aliases[i].charset == id )
			return(Aliases[i].alias);
	return(NULL);
}

char *UdmTolower(unsigned char *str, int charset){
	unsigned char *s;
	TABLE *t;

	if(!(s=str))return(NULL);
	if(Charsets[charset].has_case){
		t=&(Charsets[charset].lower);
		s=str;
		while(*s){
			*s=*(*t+*s);
			s++;
		}
	}else{
		while(*s){
			*s=tolower(*s);
			s++;
		}
	}
	return(str);
}


char * UdmToupper(unsigned char *str, int charset){
	unsigned char *s;
	TABLE *t;
	
	if(!(s=str))return(NULL);

	if(Charsets[charset].has_case){
		t=&(Charsets[charset].upper);
		s=str;
		while(*s){
			*s=*(*t+*s);
			s++;
		}
	}else{
		while(*s){
			*s=toupper(*s);
			s++;
		}
	}
	return(str);
}


#define SGML_CHAR_NUM 101
static const struct udm_sgml_chars
{
	const char	*sgml;
	unsigned char	equiv;
} SGMLChars[SGML_CHAR_NUM] =
  {
    { "lt",	      '<' } ,
    { "gt",	      '>' } ,
    { "amp",	      '&' } ,
    { "quot",	      '"' } ,
    { "nbsp",         0xA0 } , /* non breaking space */

    { "trade",	      153 } , /* trade mark */
    { "iexcl",        161 } , /* inverted exclamation mark */
    { "cent",         162 } , /* cent sign */
    { "pound",        163 } , /* pound sign */
    { "curren",       164 } , /* currency sign */
    { "yen",          165 } , /* yen sign */
    { "brvbar",       166 } , /* broken vertical bar, (brkbar) */
    { "sect",         167 } , /* section sign */
    { "uml",          168 } , /* spacing diaresis */
    { "copy",         169 } , /* copyright sign */
    { "ordf",         170 } , /* feminine ordinal indicator */
    { "laquo",        171 } , /* angle quotation mark, left */
    { "not",          172 } , /* negation sign */
    { "shy",          173 } , /* soft hyphen */
    { "reg",          174 } , /* circled R registered sign */
    { "hibar",        175 } , /* spacing macron */
    { "deg",          176 } , /* degree sign */
    { "plusmn",       177 } , /* plus-or-minus sign */
    { "sup2",         178 } , /* superscript 2 */
    { "sup3",         179 } , /* superscript 3 */
    { "acute",        180 } , /* spacing acute (96) */
    { "micro",        181 } , /* micro sign */
    { "para",         182 } , /* paragraph sign */
    { "middot",       183 } , /* middle dot */
    { "cedil",        184 } , /* spacing cedilla */
    { "sup1",         185 } , /* superscript 1 */
    { "ordm",         186 } , /* masculine ordinal indicator */
    { "raquo",        187 } , /* angle quotation mark, right */
    { "frac14",       188 } , /* fraction 1/4 */
    { "frac12",       189 } , /* fraction 1/2 */
    { "frac34",       190 } , /* fraction 3/4 */
    { "iquest",       191 } , /* inverted question mark */
    { "Agrave",       192 } , /* capital A, grave accent */ 
    { "Aacute",       193 } , /* capital A, acute accent */ 
    { "Acirc",        194 } , /* capital A, circumflex accent */ 
    { "Atilde",       195 } , /* capital A, tilde */ 
    { "Auml",         196 } , /* capital A, dieresis or umlaut mark */ 
    { "Aring",        197 } , /* capital A, ring */ 
    { "AElig",        198 } , /* capital AE diphthong (ligature) */ 
    { "Ccedil",       199 } , /* capital C, cedilla */ 
    { "Egrave",       200 } , /* capital E, grave accent */ 
    { "Eacute",       201 } , /* capital E, acute accent */ 
    { "Ecirc",        202 } , /* capital E, circumflex accent */ 
    { "Euml",         203 } , /* capital E, dieresis or umlaut mark */ 
    { "Igrave",       205 } , /* capital I, grave accent */ 
    { "Iacute",       204 } , /* capital I, acute accent */ 
    { "Icirc",        206 } , /* capital I, circumflex accent */ 
    { "Iuml",         207 } , /* capital I, dieresis or umlaut mark */ 
    { "ETH",          208 } , /* capital Eth, Icelandic (Dstrok) */ 
    { "Ntilde",       209 } , /* capital N, tilde */ 
    { "Ograve",       210 } , /* capital O, grave accent */ 
    { "Oacute",       211 } , /* capital O, acute accent */ 
    { "Ocirc",        212 } , /* capital O, circumflex accent */ 
    { "Otilde",       213 } , /* capital O, tilde */ 
    { "Ouml",         214 } , /* capital O, dieresis or umlaut mark */ 
    { "times",        215 } , /* multiplication sign */ 
    { "Oslash",       216 } , /* capital O, slash */ 
    { "Ugrave",       217 } , /* capital U, grave accent */ 
    { "Uacute",       218 } , /* capital U, acute accent */ 
    { "Ucirc",        219 } , /* capital U, circumflex accent */ 
    { "Uuml",         220 } , /* capital U, dieresis or umlaut mark */ 
    { "Yacute",       221 } , /* capital Y, acute accent */ 
    { "THORN",        222 } , /* capital THORN, Icelandic */ 
    { "szlig",        223 } , /* small sharp s, German (sz ligature) */ 
    { "agrave",       224 } , /* small a, grave accent */ 
    { "aacute",       225 } , /* small a, acute accent */ 
    { "acirc",        226 } , /* small a, circumflex accent */ 
    { "atilde",       227 } , /* small a, tilde */
    { "auml",         228 } , /* small a, dieresis or umlaut mark */ 
    { "aring",        229 } , /* small a, ring */
    { "aelig",        230 } , /* small ae diphthong (ligature) */ 
    { "ccedil",       231 } , /* small c, cedilla */ 
    { "egrave",       232 } , /* small e, grave accent */ 
    { "eacute",       233 } , /* small e, acute accent */ 
    { "ecirc",        234 } , /* small e, circumflex accent */ 
    { "euml",         235 } , /* small e, dieresis or umlaut mark */ 
    { "igrave",       236 } , /* small i, grave accent */ 
    { "iacute",       237 } , /* small i, acute accent */ 
    { "icirc",        238 } , /* small i, circumflex accent */ 
    { "iuml",         239 } , /* small i, dieresis or umlaut mark */ 
    { "eth",          240 } , /* small eth, Icelandic */ 
    { "ntilde",       241 } , /* small n, tilde */ 
    { "ograve",       242 } , /* small o, grave accent */ 
    { "oacute",       243 } , /* small o, acute accent */ 
    { "ocirc",        244 } , /* small o, circumflex accent */ 
    { "otilde",       245 } , /* small o, tilde */ 
    { "ouml",         246 } , /* small o, dieresis or umlaut mark */ 
    { "divide",       247 } , /* division sign */
    { "oslash",       248 } , /* small o, slash */ 
    { "ugrave",       249 } , /* small u, grave accent */ 
    { "uacute",       250 } , /* small u, acute accent */ 
    { "ucirc",        251 } , /* small u, circumflex accent */ 
    { "uuml",         252 } , /* small u, dieresis or umlaut mark */ 
    { "yacute",       253 } , /* small y, acute accent */ 
    { "thorn",        254 } , /* small thorn, Icelandic */ 
    { "yuml",         255 }  /* small y, dieresis or umlaut mark */
  };


char UdmSgmlToChar(char *sgml){
	int i;
	char res;

	for(i=0;i<SGML_CHAR_NUM;i++)
		if(!strncmp(sgml,SGMLChars[i].sgml,strlen(SGMLChars[i].sgml))){
			res=(char)SGMLChars[i].equiv;
			return(res);
		}
	return(0);
}

char * UdmUnescapeSgmlStr(char * str){
	char * s=str,*e,c;
	
	while(*s){
		if(*s=='&'){
			if(*(s+1)=='#'){
				for(e=s+2;(e-s<15)&&(*e<='9')&&(*e>='0');e++);
				if(*e==';'){
					int v=atoi(s+2);
					if(v>=0&&v<=255)*s=(char)v;
					memmove(s+1,e+1,strlen(e+1)+1);
				}
				s++;
			}else{
				for(e=s+1;(e-s<15)&&(((*e<='z')&&(*e>='a'))||((*e<='Z')&&(*e>='A')));e++);
				if((*e==';')&&(c=UdmSgmlToChar(s+1))){
					*s=c;
					memmove(s+1,e+1,strlen(e+1)+1);
					
				}
			}
		}
		s++;
	}
	return(str);
}


static int FillLowerString(unsigned char *str,TABLE table){
unsigned int i,len;

	len=strlen(str)/2;
	for(i=0;i<256;i++)table[i]=tolower((unsigned char)i);
	for(i=0;i<len;i++){
		table[str[i]]=str[i+len];
	}
	return(0);
}

static int FillUpperString(unsigned char *str,TABLE table){
unsigned int i,len;

	len=strlen(str)/2;
	for(i=0;i<256;i++)table[i]=toupper((unsigned char)i);
	for(i=0;i<len;i++){
		table[str[i+len]]=str[i];
	}
	return(0);
}

static int comp_char(const void *ch1, const void *ch2)
{
    return ( *((const unsigned char *)ch1) - *((const unsigned char *)ch2) );
}

static int cmprange(const void *v1, const void *v2){
int len1, len2;
	len1 = *((const char*)v1+1)- *((const char*)v1);
	len2 = *((const char*)v2+1)- *((const char*)v2);
	if(len1>len2)
		return -1;
	else if(len1<len2)
		return 1;
	else 
		return 0;
}

static void MakeWordRange(int charset)
{
    size_t nelem, i;
    unsigned char *buf;

    nelem = strlen(Charsets[charset].chars)+strlen(WORDCHAR);
    buf = (unsigned char *) UdmXmalloc(nelem+1);
    sprintf(buf, "%s%s", WORDCHAR, Charsets[charset].chars);
    qsort((void*)buf, nelem, sizeof(unsigned char), comp_char);
    Charsets[charset].num_word_range=0;
    Charsets[charset].word_range[0]=buf[0];
    for(i=0; i<nelem; i++)
    {
	if( buf[i+1] != buf[i]+1 )
	{
	    Charsets[charset].word_range[((Charsets[charset].num_word_range)*2)+1]=buf[i];
	    Charsets[charset].num_word_range+=1;
	    Charsets[charset].word_range[((Charsets[charset].num_word_range)*2)]=buf[i+1];	    
	}
    }
    /* Sort rage by range length */
    qsort(Charsets[charset].word_range, Charsets[charset].num_word_range, 2, cmprange);
    free(buf);
    return;
}

static int FillRecodeString(unsigned char *from,unsigned char *to,TABLE table){
unsigned int len,i;

	for(i=0;i<256;i++)table[i]=(unsigned char)i;
	len=strlen(from);
	if(len!=strlen(to))return(1);
	for(i=0;i<len;i++){
		if((from[i]!=to[i])&&(from[i]>=0x80)){
			table[(unsigned int)from[i]]=to[i];
		}
	}
	return(0);
}


char * UdmRecode(unsigned char *str,int from,int to){
	unsigned char *s;
	TABLE t;

#ifdef DEBUG_RECODE
	fprintf(stderr,"Recoding from=%d to=%d\n",from,to);
#endif

	if(!(s=str))return(str);
	if(from==to)return(str);
	FillRecodeString(Charsets[from].chars,Charsets[to].chars,t);
	while(*s){
		if(*s!=*(t+*s))
		*s=*(t+*s);
		s++;
	}
	return(str);
}

__INDLIB__ int UdmInitCharset(void){
int i;
	for(i=0;Charsets[i].charset!=-1;i++){
		if(Charsets[i].charset!=i){
			fprintf(stderr,"Internal Error in charset.c\n");
			fprintf(stderr,"Charset[i].charset !=i :\n");
			fprintf(stderr,"%d !=  %d\n",i,Charsets[i].charset);
			fprintf(stderr,"Badly formed Charset[i] variable\n");
			exit(1);
		}
		FillUpperString(Charsets[i].chars,Charsets[i].upper);
		FillLowerString(Charsets[i].chars,Charsets[i].lower);
		MakeWordRange(i);
	}
	return(0);
}

int UdmWordChar(int s, int charset){
	int i;
	for(i=0; i<Charsets[charset].num_word_range; i++){
		if( s >=(char)(Charsets[charset].word_range[i*2]) &&
        		s <=(char)(Charsets[charset].word_range[i*2+1]) )
		return(1);
	}
	return(0);
}

char * UdmGetWord(char *s, char **last, int charset)
{
    unsigned char *tok=NULL;
    int fl, i;
	
    if (s == NULL && (s = *last) == NULL)
        return NULL;

    /*	We find beginning of the word */
    fl = 1;
    while(fl)
    {
	if( *s==0 )
	{
	    *last=s;
	    return NULL ;
	}
	
        for(i=0; i<Charsets[charset].num_word_range; i++)
        {
	    if( *s >=(char)(Charsets[charset].word_range[i*2]) &&
                *s <=(char)(Charsets[charset].word_range[i*2+1]) )
            {
	        tok = s;
		fl = 0;
		break;
	    }
	}
	s++;
    }
    /*	We find end of the word	*/
    fl = 1;					    
    while(fl)
    {
	if( *s == 0 )
	{
	    *last=s;
	    return tok ;
	}
        for(i=0; i<Charsets[charset].num_word_range; i++)
        {
            if( *s >=(char)(Charsets[charset].word_range[i*2]) &&
                *s <=(char)(Charsets[charset].word_range[i*2+1]) )
	    {
	        fl = 1;
		break;
	    }else{
    		fl=0;
	    }
        }
	if( !fl ){
            *s='\0';
            *last = s+1;
	}
	s++;
    }
    return tok;
}

static int is_bool_lex(int c){
	switch(c){
        	case '|':return(UDM_STACK_OR);
		case '&':
		case '+':return(UDM_STACK_AND);
		case '(':return(UDM_STACK_LEFT);
		case ')':return(UDM_STACK_RIGHT);
		case '~':
		case '-':return(UDM_STACK_NOT);
        }
        return(-1);
}
/* 
The function return boolean identificator, the word stores in word variable
*/
int UdmGetLex(char **word, char **last, int charset)
{
    unsigned char *tok=NULL, *s;
    int fl, i, lex;
    size_t len = 0;

    if((s = *last) == NULL)
        return -1;

    /*	We find beginning of the word */
    fl = 1;
    while(fl)
    {
	if( *s==0 )
	{
	    *last=s;
	    return -1;
	}
	
	if((lex=is_bool_lex(*s))!=-1){
        	*last = s+1;
		return lex;
	}

        for(i=0; i<Charsets[charset].num_word_range; i++)
        {
	    if( *s >=(unsigned char)(Charsets[charset].word_range[i*2]) &&
                *s <=(unsigned char)(Charsets[charset].word_range[i*2+1]) )
            {
	        tok = s;
		fl = 0;
		break;
	    }
	}
	s++;
    }
    /*	We find end of the word	*/
    fl = 1;					    
    while(fl)
    {
	if( *s == 0 )
	{
	    len = s-tok;
	    *last=s;
	    break;
	}
        for(i=0; i<Charsets[charset].num_word_range; i++)
        {
            if( *s >=(unsigned char)(Charsets[charset].word_range[i*2]) &&
                *s <=(unsigned char)(Charsets[charset].word_range[i*2+1]) )
	    {
	        fl = 1;
		break;
	    }else{
    		fl=0;
	    }
        }
	if( !fl ){
	    len = s-tok;
            *last = s;
	}
	s++;
    }
    *word = UdmXmalloc(len+1);
    udm_snprintf(*word, len+1, "%s", tok);
    return (UDM_STACK_WORD);
}
