/*
** Copyright (c) 1998 by Timothy Gerard Endres
** <mailto:time@ice.com>  <http://www.ice.com>
** 
** This program is free software.
** 
** You may redistribute it and/or modify it under the terms of the GNU
** General Public License as published by the Free Software Foundation.
** Version 2 of the license should be included with this distribution in
** the file LICENSE, as well as License.html. If the license is not
** included	with this distribution, you may find a copy at the FSF web
** site at 'www.gnu.org' or 'www.fsf.org', or you may write to the
** Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139 USA.
**
** THIS SOFTWARE IS PROVIDED AS-IS WITHOUT WARRANTY OF ANY KIND,
** NOT EVEN THE IMPLIED WARRANTY OF MERCHANTABILITY. THE AUTHOR
** OF THIS SOFTWARE, ASSUMES _NO_ RESPONSIBILITY FOR ANY
** CONSEQUENCE RESULTING FROM THE USE, MODIFICATION, OR
** REDISTRIBUTION OF THIS SOFTWARE. 
** 
*/

package com.ice.sqlclient;

import java.sql.*;
import java.awt.*;
import java.awt.event.*;
import com.sun.java.swing.*;
import com.sun.java.swing.event.*;
import com.sun.java.swing.table.*;

import com.ice.util.AWTUtilities;
import com.ice.util.UserProperties;


public class
TableDefsPanel extends JPanel
		implements ActionListener, ChangeListener, ItemListener
	{
	private JComboBox		tableList;
	private JButton			refreshButton;

	private JPanel			tablePanel;
	private JSplitPane		splitPane;

	private JTable			resultTable;
	private MySQLAdapter	sqlAdapter;
	private JScrollPane		scroller;

	private boolean			isListFresh;
	private boolean			isTableFresh; // unused

	private int				autoResizeMode;


	public
	TableDefsPanel()
		{
		super();
		this.isListFresh = false;
		this.setDoubleBuffered( true );
		this.establishContents();
		}

	public void
	actionPerformed( ActionEvent event )
		{
		String command = event.getActionCommand();

		if ( command.equals( "REFRESH" ) )
			{
			this.refresh();
			}
		else
			{
			ErrMsgAdapter msgTable =
				new ErrMsgAdapter
					( "INTERNAL ERROR: unknown command: " + command );
			this.setTableModel( msgTable );
			}
		}

	public void
	refresh()
		{
		SQLClientHandler handler = this.getMyHandler();
		if ( handler != null )
			{
			handler.flushTableCaches();
			this.isListFresh = false;
			this.refreshTableList();
			}
		}

	public void
	itemStateChanged( ItemEvent event )
		{
		if ( event.getStateChange() == ItemEvent.SELECTED )
			{
			String tableName = (String)
				this.tableList.getSelectedItem();

			this.refreshTableDef( tableName );
			}
		}

	public void
	stateChanged( ChangeEvent event )
		{
		JTabbedPane pane =
			(JTabbedPane) event.getSource();

		if ( pane.getSelectedComponent() == this )
			{
			if ( ! this.isListFresh )
				{
				this.refreshTableList();
				}
			}
		}

	public void
	refreshTableDef( String tableName )
		{
		SQLClientHandler handler = this.getMyHandler();
		if ( handler == null ) return;

		try {
			this.sqlAdapter = (MySQLAdapter)
				handler.getTableColumnInfo( tableName );

			this.setTableModel( this.sqlAdapter );
			}
		catch ( SQLException ex )
			{
			String msg =
				"SQLException: retrieving table definition:\n"
					+ ex.getMessage();
			ErrMsgAdapter msgTable = new ErrMsgAdapter( msg );
			this.setTableModel( msgTable );
			}
		}

	public void
	refreshTableList()
		{
		SQLClientHandler handler = this.getMyHandler();
		if ( handler == null ) return;

		try {
			String[] list =
				handler.getDatabaseTableList();

			this.tableList.removeAllItems();

			for ( int i = 0 ; i < list.length ; ++i )
				{
				this.tableList.addItem( list[i] );
				}

			this.isListFresh = true;
			}
		catch ( SQLException ex )
			{
			String msg =
				"SQLException: retrieving table information:\n"
					+ ex.getMessage();
			ErrMsgAdapter msgTable = new ErrMsgAdapter( msg );
			this.setTableModel( msgTable );
			}
		}

	private void
	setTableModel( TableModel model )
		{
		if ( model instanceof ErrMsgAdapter )
			{
			this.resultTable.setAutoResizeMode
				( JTable.AUTO_RESIZE_ALL_COLUMNS );
			}
		else
			{
			this.resultTable.setAutoResizeMode
				( this.autoResizeMode );
			}

		this.resultTable.setModel( model );
		
		this.resultTable.tableChanged
			( new TableModelEvent
				( model, TableModelEvent.HEADER_ROW ) );
		}

	private SQLClientHandler
	getMyHandler()
		{
		SQLClientHandler handler =
			SQLClientHandler.getCurrentHandler();

		if ( handler == null )
			{
			ErrMsgAdapter msgTable =
				new ErrMsgAdapter
					( "INTERNAL ERROR: current handler is null." );
			this.setTableModel( msgTable );
			}
		else if ( ! handler.isOpen() )
			{
			ErrMsgAdapter msgTable =
				new ErrMsgAdapter
					( "INTERNAL ERROR: connection is not open." );
			this.setTableModel( msgTable );
			handler = null;
			}

		return handler;
		}

	private void 
	establishContents()
		{
		JLabel	lbl;
		int		row = 0;

		this.setLayout( new GridBagLayout() );

		this.refreshButton = new JButton( "Refresh" );
		this.refreshButton.addActionListener( this );
		this.refreshButton.setActionCommand( "REFRESH" );
		AWTUtilities.constrain(
			this, this.refreshButton,
			GridBagConstraints.NONE,
			GridBagConstraints.CENTER,
			0, row, 1, 1, 0.0, 0.0 );

		this.tableList = new JComboBox();
		this.tableList.setEditable( false );
		this.tableList.addItemListener( this );
		AWTUtilities.constrain(
			this, this.tableList,
			GridBagConstraints.HORIZONTAL,
			GridBagConstraints.CENTER,
			1, row++, 1, 1, 1.0, 0.0 );
		
		this.tablePanel = new JPanel();
		this.tablePanel.setLayout( new GridBagLayout() );

		int subRow = 0;

		lbl = new JLabel( "Table Definition:" );
		AWTUtilities.constrain(
			this.tablePanel, lbl,
			GridBagConstraints.NONE,
			GridBagConstraints.WEST,
			0, subRow++, 1, 1, 0.0, 0.0 );

		this.sqlAdapter =
			new MySQLAdapter( null );

		this.resultTable =
			new JTable( this.sqlAdapter );

		this.autoResizeMode =
			UserProperties.getClassConstant
				( "tableDefsAutoResizeMode",
					JTable.AUTO_RESIZE_ALL_COLUMNS );

		this.resultTable.setAutoResizeMode
			( this.autoResizeMode );

        this.scroller =
			JTable.createScrollPaneForTable
				( this.resultTable );
		Dimension sz = new Dimension( 300, 50 );
		this.scroller.setMinimumSize( sz );

		this.scroller.setVerticalScrollBarPolicy
			( ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS );

		this.scroller.setHorizontalScrollBarPolicy
			( ScrollPaneConstants.HORIZONTAL_SCROLLBAR_ALWAYS );

		AWTUtilities.constrain(
			this.tablePanel, scroller,
			GridBagConstraints.BOTH,
			GridBagConstraints.CENTER,
			0, subRow++, 1, 1, 1.0, 1.0 );

		AWTUtilities.constrain(
			this, this.tablePanel,
			GridBagConstraints.BOTH,
			GridBagConstraints.NORTH,
			0, row++, 2, 1, 1.0, 1.0 );
		}

	}


