/*-
 * See the file LICENSE for redistribution information.
 *
 * Copyright (c) 1999, 2000
 *	Sleepycat Software.  All rights reserved.
 */

#include "db_config.h"

#ifndef lint
static const char revid[] = "$Id: qam_method.c,v 11.6 2000/10/20 23:31:15 ubell Exp $";
#endif /* not lint */

#ifndef NO_SYSTEM_INCLUDES
#include <sys/types.h>
#include <errno.h>
#include <string.h>
#endif

#include "db_int.h"
#include "db_page.h"
#include "db_int.h"
#include "db_shash.h"
#include "db_am.h"
#include "qam.h"
#include "db.h"
#include "mp.h"
#include "lock.h"
#include "log.h"

static int __qam_set_extentsize __P((DB *, u_int32_t));
static int __qam_remove_callback __P((DB *, void *));

struct __qam_cookie {
	DB_LSN lsn;
	QUEUE_FILELIST *filelist;
};

/*
 * __qam_db_create --
 *	Queue specific initialization of the DB structure.
 *
 * PUBLIC: int __qam_db_create __P((DB *));
 */
int
__qam_db_create(dbp)
	DB *dbp;
{
	QUEUE *t;
	int ret;

	/* Allocate and initialize the private queue structure. */
	if ((ret = __os_calloc(dbp->dbenv, 1, sizeof(QUEUE), &t)) != 0)
		return (ret);
	dbp->q_internal = t;
	dbp->set_q_extentsize = __qam_set_extentsize;

	t->re_pad = ' ';

	return (0);
}

/*
 * __qam_db_close --
 *	Queue specific discard of the DB structure.
 *
 * PUBLIC: int __qam_db_close __P((DB *));
 */
int
__qam_db_close(dbp)
	DB *dbp;
{
	DB_MPOOLFILE *mpf, **mpfp;
	QUEUE *t;
	u_int32_t i;
	int ret;

	t = dbp->q_internal;

	mpfp = t->mpfarray;
	if (mpfp != NULL)
		for (i = t->low_extent; i <= t->hi_extent; i++, mpfp++) {
			mpf = *mpfp;
			*mpfp = NULL;
			if (mpf != NULL && (ret = memp_fclose(mpf)) != 0)
				return (ret);
		}
			

	__os_free(t, sizeof(QUEUE));
	dbp->q_internal = NULL;

	return (0);
}

static int
__qam_set_extentsize(dbp, extentsize)
	DB *dbp;
	u_int32_t extentsize;
{
	DB_ILLEGAL_AFTER_OPEN(dbp, "set_extentsize");

	if (extentsize < 1) {
		__db_err(dbp->dbenv, "Extent size must be at least 1.");
		return (EINVAL);
	}

	((QUEUE*)dbp->q_internal)->page_ext = extentsize;

	return (0);
}

/*
 * __db_prqueue --
 *	Print out a queue
 *
 * PUBLIC: int __db_prqueue __P((DB *, u_int32_t));
 */
int
__db_prqueue(dbp, flags)
	DB *dbp;
	u_int32_t flags;
{
	PAGE *h;
	QMETA *meta;
	db_pgno_t i, last, pg_ext;
	int ret;

	/* Find out the page number of the last page in the database. */
	i = PGNO_BASE_MD;
	if ((ret = memp_fget(dbp->mpf, &i, 0, &meta)) != 0)
		return (ret);

	last = QAM_RECNO_PAGE(dbp, meta->start, meta->cur_recno);

	if ((ret = __db_prpage(dbp, (PAGE *)meta, flags)) != 0)
		return (ret);
	if ((ret = memp_fput(dbp->mpf, meta, 0)) != 0)
		return (ret);
	
	if (meta->cur_recno < meta->start)
		return (0);

	/* Dump each page. */
	for (i = 1; i <= last; ++i) {
		if ((ret = __qam_fget(dbp, &i, 0, &h)) != 0) {
			pg_ext = ((QUEUE *)dbp->q_internal)->page_ext;
			if (ret == ENOENT) {
				i += pg_ext - 1;
				continue;
			}
			if (ret == EINVAL) {
				i += pg_ext - ((i - 1) % pg_ext) - 1;
				continue;
			}
			return (ret);
		}
		(void)__db_prpage(dbp, h, flags);
		if ((ret = __qam_fput(dbp, i, h, 0)) != 0)
			return (ret);
	}

	return (0);
}

/*
 * __qam_remove
 *	Remove method for a Queue.
 *
 * PUBLIC: int __qam_remove __P((DB *, const char *,
 * PUBLIC:      const char *, DB_LSN *, int (**)(DB *, void*), void **));
 */
int
__qam_remove(dbp, name, subdb, lsnp, callbackp, cookiep)
	DB *dbp;
	const char *name, *subdb;
	DB_LSN *lsnp;
	int (**callbackp) __P((DB *, void *));
	void **cookiep;
{
	DBT namedbt;
	DB_ENV *dbenv;
	DB_LSN lsn, newlsn;
	QUEUE *qp;
	int ret;
	char *backup, buf[256], *real_back, *real_name;
	QUEUE_FILELIST *filelist, *fp;
	struct __qam_cookie *qam_cookie;

	dbenv = dbp->dbenv;
	ret = 0;
	backup = real_back = real_name = NULL;
	filelist = NULL;

	PANIC_CHECK(dbenv);

	/*
	 * Subdatabases.
	 */
	if (subdb != NULL) {
		__db_err(dbenv,
		    "Queue does not support multiple databases per file.");
		ret = EINVAL;
		goto done;
	}

	qp = (QUEUE *)dbp->q_internal;

	if (qp->page_ext != 0 &&
	    (ret = __qam_gen_filelist(dbp, &filelist)) != 0)
		goto done;

	if (filelist == NULL)
		goto done;

	for (fp = filelist; fp->mpf != NULL; fp++) {
		snprintf(buf,
		    sizeof(buf), QUEUE_EXTENT, qp->dir, qp->name, fp->id);
		if ((ret = __db_appname(dbenv,
		    DB_APP_DATA, NULL, buf, 0, NULL, &real_name)) != 0)
			goto done;
		if (LOGGING_ON(dbenv)) {
			memset(&namedbt, 0, sizeof(namedbt));
			namedbt.data = (char *)buf;
			namedbt.size = strlen(buf) + 1;

			if ((ret =
			    __qam_delete_log(dbenv, dbp->open_txn,
			    &lsn, DB_FLUSH, &namedbt, &newlsn)) != 0) {
				__db_err(dbenv,
				    "%s: %s", name, db_strerror(ret));
				goto done;
			}
		}
		(void)__memp_fremove(fp->mpf);
		if ((ret = memp_fclose(fp->mpf)) != 0)
			goto done;
		qp->mpfarray[fp->id - qp->low_extent] = NULL;
		/* Create name for backup file. */
		if (TXN_ON(dbenv)) {
			if ((ret = __db_backup_name(dbenv,
			    buf, &backup, &newlsn)) != 0)
				goto done;
			if ((ret = __db_appname(dbenv, DB_APP_DATA,
			     NULL, backup, 0, NULL, &real_back)) != 0)
				goto done;
			if ((ret = __os_rename(dbenv,
			     real_name, real_back)) != 0)
				goto done;
			__os_freestr(real_back);
			real_back = NULL;
		}
		else
			if ((ret = __os_unlink(dbenv, real_name)) != 0)
				goto done;
		__os_freestr(real_name);
		real_name = NULL;
	}
	if ((ret= __os_malloc(dbenv,
	    sizeof(struct __qam_cookie), NULL, &qam_cookie)) != 0)
		goto done;
	qam_cookie->lsn = *lsnp;
	qam_cookie->filelist = filelist;
	*cookiep = qam_cookie;
	*callbackp = __qam_remove_callback;

done:
	if (real_back != NULL)
		__os_freestr(real_back);
	if (real_name != NULL)
		__os_freestr(real_name);
	if (backup != NULL)
		__os_freestr(backup);

	return (ret);
}

static int
__qam_remove_callback(dbp, cookie)
	DB *dbp;
	void *cookie;
{
	DB_ENV *dbenv;
	DB_LSN *lsnp;
	QUEUE *qp;
	QUEUE_FILELIST *filelist, *fp;
	char *backup, buf[256], *real_back;
	int ret;

	qp = (QUEUE *)dbp->q_internal;
	if (qp->page_ext == 0)
		return (__os_unlink(dbp->dbenv, cookie));

	dbenv = dbp->dbenv;
	lsnp = &((struct __qam_cookie *)cookie)->lsn;
	filelist = fp = ((struct __qam_cookie *)cookie)->filelist;
	real_back = backup = NULL;
	if ((ret =
	    __db_backup_name(dbenv, qp->name, &backup, lsnp)) != 0)
		goto err;
	if ((ret = __db_appname(dbenv,
	    DB_APP_DATA, NULL, backup, 0, NULL, &real_back)) != 0)
		goto err;
	if ((ret = __os_unlink(dbp->dbenv, real_back)) != 0)
		goto err;

	__os_freestr(backup);
	__os_freestr(real_back);

	if (fp == NULL)
		return (0);

	for (; fp->mpf != NULL; fp++) {
		snprintf(buf,
		    sizeof(buf), QUEUE_EXTENT, qp->dir, qp->name, fp->id);
		real_back = backup = NULL;
		if ((ret = __db_backup_name(dbenv, buf, &backup, lsnp)) != 0)
			goto err;
		if ((ret = __db_appname(dbenv,
		    DB_APP_DATA, NULL, backup, 0, NULL, &real_back)) != 0)
			goto err;
		ret = __os_unlink(dbenv, real_back);
		__os_freestr(real_back);
		__os_freestr(backup);
	}
	__os_free(filelist, 0);
	__os_free(cookie, sizeof (struct __qam_cookie));

	return (0);

err:
	if (backup != NULL)
		__os_freestr(backup);

	if (real_back != NULL)
		__os_freestr(real_back);

	return (ret);
}

/*
 * __qam_rename
 *	Rename method for Queue.
 *
 * PUBLIC: int __qam_rename __P((DB *,
 * PUBLIC:     const char *, const char *, const char *));
 */
int
__qam_rename(dbp, filename, subdb, newname)
	DB *dbp;
	const char *filename, *subdb, *newname;
{
	DBT namedbt, newnamedbt;
	DB_ENV *dbenv;
	DB_LSN newlsn;
	QUEUE *qp;
	QUEUE_FILELIST *fp, *filelist;
	char buf[256], nbuf[256], *namep, *real_name, *real_newname;
	int ret;

	dbenv = dbp->dbenv;
	ret = 0;
	real_name = real_newname = NULL;
	filelist = NULL;

	qp = (QUEUE *)dbp->q_internal;

	if (subdb != NULL) {
		__db_err(dbenv,
		    "Queue does not support multiple databases per file.");
		ret = EINVAL;
		goto err;
	}
	if (qp->page_ext != 0 &&
	    (ret = __qam_gen_filelist(dbp, &filelist)) != 0)
		goto err;
	namep = rindex(newname, PATH_SEPARATOR[0]);
	if (namep != NULL)
		newname = namep + 1;
	
	for (fp = filelist; fp != NULL && fp->mpf != NULL; fp++) {
		if ((ret = __memp_fremove(fp->mpf)) != 0)
			goto err;
		if ((ret = memp_fclose(fp->mpf)) != 0)
			goto err;
		qp->mpfarray[fp->id - qp->low_extent] = NULL;
		snprintf(buf,
		    sizeof(buf), QUEUE_EXTENT, qp->dir, qp->name, fp->id);
		if ((ret = __db_appname(dbenv,
		    DB_APP_DATA, NULL, buf, 0, NULL, &real_name)) != 0)
			goto err;
		snprintf(nbuf,
		     sizeof(nbuf), QUEUE_EXTENT, qp->dir, newname, fp->id);
		if ((ret = __db_appname(dbenv,
		    DB_APP_DATA, NULL, nbuf, 0, NULL, &real_newname)) != 0)
			goto err;
		if (LOGGING_ON(dbenv)) {
			memset(&namedbt, 0, sizeof(namedbt));
			namedbt.data = (char *)buf;
			namedbt.size = strlen(buf) + 1;

			memset(&newnamedbt, 0, sizeof(namedbt));
			newnamedbt.data = (char *)nbuf;
			newnamedbt.size = strlen(nbuf) + 1;

			if ((ret =
			    __qam_rename_log(dbenv,
			    dbp->open_txn, &newlsn, 0,
			    &namedbt, &newnamedbt)) != 0) {
				__db_err(dbenv, "%s: %s", filename, db_strerror(ret));
				goto err;
			}

			if ((ret = __log_filelist_update(dbenv, dbp,
			    dbp->log_fileid, newname, NULL)) != 0)
				goto err;
		}
		if ((ret = __os_rename(dbenv, real_name, real_newname)) != 0)
			goto err;
		__os_freestr(real_name);
		__os_freestr(real_newname);
		real_name = real_newname = NULL;
	}


err:
	if (real_name != NULL)
		__os_freestr(real_name);
	if (real_newname != NULL)
		__os_freestr(real_newname);
	if (filelist != NULL)
		__os_free(filelist, 0);

	return (ret);
}
