/*
 * Copyright (c) 2007, 2015 Borland Software Corporation, Christian W. Damus, and others
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *    Dmitry Stadnik (Borland) - initial API and implementation
 *    Ansgar Radermacher (CEA LIST) - added support for EMF validation
 *    	bug fix and re-factoring (separating common class)
 *      specific version for Papyrus
 *      Amine EL KOUHEN (CEA LIST) - Added decoration Service
 *    Christian W. Damus - bug 466629
 */
package org.eclipse.papyrus.uml.diagram.common.providers;

import static org.eclipse.papyrus.uml.diagram.common.Activator.log;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Observable;
import java.util.Observer;

import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.emf.transaction.util.TransactionUtil;
import org.eclipse.gef.EditPart;
import org.eclipse.gmf.runtime.common.core.service.AbstractProvider;
import org.eclipse.gmf.runtime.common.core.service.IOperation;
import org.eclipse.gmf.runtime.diagram.core.util.ViewUtil;
import org.eclipse.gmf.runtime.diagram.ui.parts.IDiagramEditDomain;
import org.eclipse.gmf.runtime.diagram.ui.services.decorator.AbstractDecorator;
import org.eclipse.gmf.runtime.diagram.ui.services.decorator.IDecorator;
import org.eclipse.gmf.runtime.diagram.ui.services.decorator.IDecoratorProvider;
import org.eclipse.gmf.runtime.diagram.ui.services.decorator.IDecoratorTarget;
import org.eclipse.gmf.runtime.notation.Edge;
import org.eclipse.gmf.runtime.notation.View;
import org.eclipse.papyrus.infra.core.services.ServicesRegistry;
import org.eclipse.papyrus.infra.services.decoration.DecorationChange;
import org.eclipse.papyrus.infra.services.decoration.DecorationChange.DecorationChangeKind;
import org.eclipse.papyrus.infra.services.decoration.DecorationService;
import org.eclipse.papyrus.infra.services.decoration.util.Decoration;
import org.eclipse.papyrus.infra.services.decoration.util.IPapyrusDecoration;
import org.eclipse.papyrus.uml.diagram.common.util.ServiceUtilsForGMF;
import org.eclipse.ui.PlatformUI;
import org.eclipse.uml2.uml.Constraint;
import org.eclipse.uml2.uml.ValueSpecification;

/**
 * Generic validation decorator provider (for the case application == null)
 */
public abstract class ValidationDecoratorProvider extends AbstractProvider implements IDecoratorProvider {

	protected static final String KEY = "validationStatus"; //$NON-NLS-1$

	private static Map<String, IDecorator> allDecorators = new HashMap<String, IDecorator>();

	/**
	 * Refined by generated class
	 *
	 * @see org.eclipse.gmf.runtime.diagram.ui.services.decorator.IDecoratorProvider#createDecorators(org.eclipse.gmf.runtime.diagram.ui.services.decorator.IDecoratorTarget)
	 *
	 * @param decoratorTarget
	 */
	@Override
	public abstract void createDecorators(IDecoratorTarget decoratorTarget);

	/**
	 * Refined by generated class
	 *
	 * @see org.eclipse.gmf.runtime.common.core.service.IProvider#provides(org.eclipse.gmf.runtime.common.core.service.IOperation)
	 *
	 * @param operation
	 * @return
	 */
	@Override
	public abstract boolean provides(IOperation operation);

	/**
	 * Refresh the decorators of a specific view
	 *
	 * @param view
	 */
	public static void refreshDecorators(View view) {
		refreshDecorators(ViewUtil.getIdStr(view), TransactionUtil.getEditingDomain(view));
	}

	/**
	 * Refresh the decorators of a specific view
	 */
	private static void refreshDecorators(String viewId, final TransactionalEditingDomain domain) {
		final IDecorator decorator = viewId != null ? allDecorators.get(viewId) : null;
		if (decorator == null || domain == null) {
			return;
		}
		PlatformUI.getWorkbench().getDisplay().asyncExec(new Runnable() {

			@Override
			public void run() {
				try {
					domain.runExclusive(new Runnable() {

						@Override
						public void run() {
							decorator.refresh();
						}
					});
				} catch (Exception e) {
					log.error("Decorator refresh failure", e); //$NON-NLS-1$
				}
			}
		});
	}

	public static class StatusDecorator extends AbstractDecorator implements Observer {

		/**
		 * The ID of the view
		 */
		private String viewId;

		/**
		 * Store a copy of the editingDomain;
		 */
		private TransactionalEditingDomain editingDomain;

		/**
		 * The decoration service that manages me.
		 */
		private DecorationService decorationService;

		private Object element;

		/**
		 * Diagram Decorator
		 */
		private final DiagramDecorationAdapter diagramDecorationAdapter;

		/**
		 * @generated
		 */
		public StatusDecorator(IDecoratorTarget decoratorTarget) {
			super(decoratorTarget);
			diagramDecorationAdapter = new DiagramDecorationAdapter(decoratorTarget);
		}

		/**
		 * Completely refresh the decorators of a view
		 *
		 * @see org.eclipse.gmf.runtime.diagram.ui.services.decorator.IDecorator#refresh()
		 */
		@Override
		public void refresh() {
			View view = (View) getDecoratorTarget().getAdapter(View.class);
			if (view == null || view.eResource() == null) {
				return;
			}
			if ((view.getElement() != null) && (decorationService != null)) {
				diagramDecorationAdapter.removeDecorations();
				List<IPapyrusDecoration> semanticDecorations = new BasicEList<IPapyrusDecoration>();
				for (Object decoratedElement : getDecoratedElements(element)) {
					semanticDecorations.addAll(decorationService.getDecorations(decoratedElement, false));
				}
				List<IPapyrusDecoration> graphicalDecorations = decorationService.getDecorations(view, false);

				List<IPapyrusDecoration> decorations = new LinkedList<IPapyrusDecoration>(semanticDecorations);
				decorations.addAll(graphicalDecorations);
				if (view instanceof Edge) {
					diagramDecorationAdapter.setDecorationsEdge(decorations, 50, true);
				} else {
					diagramDecorationAdapter.setDecorationsNode(decorations, 0, true);
				}
			}
		}

		/**
		 * Refresh the decorators of a view when given a DecorationChange information.
		 *
		 * @param change
		 *            A decoration change, e.g. addition or removal
		 */
		public void refresh(DecorationChange change) {

			if (change.getChangeKind() == DecorationChangeKind.DecorationRemoved || change.getChangeKind() == DecorationChangeKind.DecorationModified || change.getChangeKind() == DecorationChangeKind.RefreshAll) {
				// always recreate all decorations, in case of a deletion (would require recalculation of positions) or
				// if all decorations should be refreshed
				refresh();
				return;
			}
			View view = (View) getDecoratorTarget().getAdapter(View.class);
			if (view == null || view.eResource() == null) {
				return;
			}
			if (view instanceof Edge) {
				// always recreate all decorations for an edge (since the position of all changes, if one is added or removed)
				refresh();
				return;
			}
			// add decoration
			if (view.getElement() != null) {
				if (change.getChangeKind() == DecorationChangeKind.DecorationAdded) {
					diagramDecorationAdapter.addDecorationNode(change.getDecoration(), -1, true);
				}
			}
		}

		/**
		 * activate the decorators of this view.
		 * Register a listener for editing domain of the view
		 */
		@Override
		public void activate() {
			try {
				final View view = (View) getDecoratorTarget().getAdapter(View.class);
				element = view.getElement();
				EditPart editPart = (EditPart) getDecoratorTarget().getAdapter(EditPart.class);
				IDiagramEditDomain domain = (IDiagramEditDomain) editPart.getViewer().getEditDomain();
				ServicesRegistry serviceRegistry = ServiceUtilsForGMF.getInstance().getServiceRegistry(domain);
				decorationService = serviceRegistry.getService(DecorationService.class);
				// Register As an Decoration service customer
				decorationService.addListener(this);
				TransactionUtil.getEditingDomain(view).runExclusive(new Runnable() {

					@Override
					public void run() {
						StatusDecorator.this.viewId = view != null ? ViewUtil.getIdStr(view) : null;
						StatusDecorator.this.editingDomain = TransactionUtil.getEditingDomain(view);
					}
				});
			} catch (Exception e) {
				log.error("ViewID access failure", e); //$NON-NLS-1$
			}

			if (viewId != null) {
				// add self to global decorators registry
				IDecorator decorator = allDecorators.get(viewId);
				if (decorator == null) {
					allDecorators.put(viewId, this);
				}
			}
		}

		/**
		 * deactivate the decorators of this view
		 */
		@Override
		public void deactivate() {
			diagramDecorationAdapter.removeDecorations();

			if (decorationService != null) {
				decorationService.deleteListener(this);
				decorationService = null;
			}

			if (viewId != null) {
				// remove self from global decorators registry
				allDecorators.remove(viewId);
			}

			super.deactivate();
		}

		// Refresh when the decoration service adds a decoration
		@Override
		public void update(Observable o, Object arg) {
			// check whether update is for this view
			if (arg instanceof DecorationChange) {
				DecorationChange change = (DecorationChange) arg;
				if ((change.getChangeKind() == DecorationChangeKind.RefreshAll) || decorationMatches(element, change.getDecoration())) {
					refresh(change);
				}
			}
		}

		/**
		 * Return the list of elements that correspond to a viewElement. In most cases, this list
		 * contains just the view element. But some decorations need to apply to multiple elements.
		 * In particular, decorations for a value specification of a constraint need to apply for the
		 * constraint as well - see bug 427863 - Constraint does not show ValueSpecification error marker
		 *
		 * @param viewElement
		 * @return the list of observed decorations
		 */
		public EList<Object> getDecoratedElements(Object viewElement) {
			EList<Object> decoratedElements = new BasicEList<Object>();
			decoratedElements.add(viewElement);
			/**
			 * Add decorations of specification to constraint.
			 */
			if (element instanceof Constraint) {
				ValueSpecification vs = ((Constraint) element).getSpecification();
				if (vs != null) {
					decoratedElements.add(vs);
				}
			}
			return decoratedElements;
		}

		/**
		 * Does the viewElement match a decoration?
		 *
		 * @param viewElement
		 *            the element behind a view
		 * @param decoration
		 *            a decoration
		 * @return true, if matches
		 */
		public boolean decorationMatches(Object viewElement, Decoration decoration) {
			return getDecoratedElements(viewElement).contains(decoration.getElement());
		}
	}
}
