package org.mandas.docker.client.messages;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

/**
 * Immutable implementation of {@link Platform}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutablePlatform.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutablePlatform implements Platform {
  private final String architecture;
  private final String os;
  private final String osVersion;
  private final List<String> osFeatures;
  private final String variant;
  private final List<String> features;

  private ImmutablePlatform(
      String architecture,
      String os,
      String osVersion,
      List<String> osFeatures,
      String variant,
      List<String> features) {
    this.architecture = architecture;
    this.os = os;
    this.osVersion = osVersion;
    this.osFeatures = osFeatures;
    this.variant = variant;
    this.features = features;
  }

  /**
   * @return The value of the {@code architecture} attribute
   */
  @JsonProperty("Architecture")
  @Override
  public String architecture() {
    return architecture;
  }

  /**
   * @return The value of the {@code os} attribute
   */
  @JsonProperty("OS")
  @Override
  public String os() {
    return os;
  }

  /**
   * @return The value of the {@code osVersion} attribute
   */
  @JsonProperty("OSVersion")
  @Override
  public String osVersion() {
    return osVersion;
  }

  /**
   * @return The value of the {@code osFeatures} attribute
   */
  @JsonProperty("OSFeatures")
  @Override
  public List<String> osFeatures() {
    return osFeatures;
  }

  /**
   * @return The value of the {@code variant} attribute
   */
  @JsonProperty("Variant")
  @Override
  public String variant() {
    return variant;
  }

  /**
   * @return The value of the {@code features} attribute
   */
  @JsonProperty("Features")
  @Override
  public List<String> features() {
    return features;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Platform#architecture() architecture} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for architecture
   * @return A modified copy of the {@code this} object
   */
  public final ImmutablePlatform withArchitecture(String value) {
    String newValue = Objects.requireNonNull(value, "architecture");
    if (this.architecture.equals(newValue)) return this;
    return new ImmutablePlatform(newValue, this.os, this.osVersion, this.osFeatures, this.variant, this.features);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Platform#os() os} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for os
   * @return A modified copy of the {@code this} object
   */
  public final ImmutablePlatform withOs(String value) {
    String newValue = Objects.requireNonNull(value, "os");
    if (this.os.equals(newValue)) return this;
    return new ImmutablePlatform(this.architecture, newValue, this.osVersion, this.osFeatures, this.variant, this.features);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Platform#osVersion() osVersion} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for osVersion
   * @return A modified copy of the {@code this} object
   */
  public final ImmutablePlatform withOsVersion(String value) {
    String newValue = Objects.requireNonNull(value, "osVersion");
    if (this.osVersion.equals(newValue)) return this;
    return new ImmutablePlatform(this.architecture, this.os, newValue, this.osFeatures, this.variant, this.features);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Platform#osFeatures() osFeatures}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutablePlatform withOsFeatures(String... elements) {
    List<String> newValue = createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutablePlatform(this.architecture, this.os, this.osVersion, newValue, this.variant, this.features);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Platform#osFeatures() osFeatures}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of osFeatures elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutablePlatform withOsFeatures(Iterable<String> elements) {
    if (this.osFeatures == elements) return this;
    List<String> newValue = createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutablePlatform(this.architecture, this.os, this.osVersion, newValue, this.variant, this.features);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Platform#variant() variant} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for variant
   * @return A modified copy of the {@code this} object
   */
  public final ImmutablePlatform withVariant(String value) {
    String newValue = Objects.requireNonNull(value, "variant");
    if (this.variant.equals(newValue)) return this;
    return new ImmutablePlatform(this.architecture, this.os, this.osVersion, this.osFeatures, newValue, this.features);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Platform#features() features}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutablePlatform withFeatures(String... elements) {
    List<String> newValue = createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutablePlatform(this.architecture, this.os, this.osVersion, this.osFeatures, this.variant, newValue);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Platform#features() features}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of features elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutablePlatform withFeatures(Iterable<String> elements) {
    if (this.features == elements) return this;
    List<String> newValue = createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutablePlatform(this.architecture, this.os, this.osVersion, this.osFeatures, this.variant, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutablePlatform} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutablePlatform
        && equalTo(0, (ImmutablePlatform) another);
  }

  private boolean equalTo(int synthetic, ImmutablePlatform another) {
    return architecture.equals(another.architecture)
        && os.equals(another.os)
        && osVersion.equals(another.osVersion)
        && osFeatures.equals(another.osFeatures)
        && variant.equals(another.variant)
        && features.equals(another.features);
  }

  /**
   * Computes a hash code from attributes: {@code architecture}, {@code os}, {@code osVersion}, {@code osFeatures}, {@code variant}, {@code features}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + architecture.hashCode();
    h += (h << 5) + os.hashCode();
    h += (h << 5) + osVersion.hashCode();
    h += (h << 5) + osFeatures.hashCode();
    h += (h << 5) + variant.hashCode();
    h += (h << 5) + features.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code Platform} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "Platform{"
        + "architecture=" + architecture
        + ", os=" + os
        + ", osVersion=" + osVersion
        + ", osFeatures=" + osFeatures
        + ", variant=" + variant
        + ", features=" + features
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link Platform} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable Platform instance
   */
  public static ImmutablePlatform copyOf(Platform instance) {
    if (instance instanceof ImmutablePlatform) {
      return (ImmutablePlatform) instance;
    }
    return ImmutablePlatform.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutablePlatform ImmutablePlatform}.
   * <pre>
   * ImmutablePlatform.builder()
   *    .architecture(String) // required {@link Platform#architecture() architecture}
   *    .os(String) // required {@link Platform#os() os}
   *    .osVersion(String) // required {@link Platform#osVersion() osVersion}
   *    .osFeature|addAllOsFeatures(String) // {@link Platform#osFeatures() osFeatures} elements
   *    .variant(String) // required {@link Platform#variant() variant}
   *    .feature|addAllFeatures(String) // {@link Platform#features() features} elements
   *    .build();
   * </pre>
   * @return A new ImmutablePlatform builder
   */
  public static ImmutablePlatform.Builder builder() {
    return new ImmutablePlatform.Builder();
  }

  /**
   * Builds instances of type {@link ImmutablePlatform ImmutablePlatform}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private static final long INIT_BIT_ARCHITECTURE = 0x1L;
    private static final long INIT_BIT_OS = 0x2L;
    private static final long INIT_BIT_OS_VERSION = 0x4L;
    private static final long INIT_BIT_VARIANT = 0x8L;
    private long initBits = 0xfL;

    private String architecture;
    private String os;
    private String osVersion;
    private List<String> osFeatures = new ArrayList<String>();
    private String variant;
    private List<String> features = new ArrayList<String>();

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code Platform} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(Platform instance) {
      Objects.requireNonNull(instance, "instance");
      this.architecture(instance.architecture());
      this.os(instance.os());
      this.osVersion(instance.osVersion());
      addAllOsFeatures(instance.osFeatures());
      this.variant(instance.variant());
      addAllFeatures(instance.features());
      return this;
    }

    /**
     * Initializes the value for the {@link Platform#architecture() architecture} attribute.
     * @param architecture The value for architecture 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Architecture")
    public final Builder architecture(String architecture) {
      this.architecture = Objects.requireNonNull(architecture, "architecture");
      initBits &= ~INIT_BIT_ARCHITECTURE;
      return this;
    }

    /**
     * Initializes the value for the {@link Platform#os() os} attribute.
     * @param os The value for os 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("OS")
    public final Builder os(String os) {
      this.os = Objects.requireNonNull(os, "os");
      initBits &= ~INIT_BIT_OS;
      return this;
    }

    /**
     * Initializes the value for the {@link Platform#osVersion() osVersion} attribute.
     * @param osVersion The value for osVersion 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("OSVersion")
    public final Builder osVersion(String osVersion) {
      this.osVersion = Objects.requireNonNull(osVersion, "osVersion");
      initBits &= ~INIT_BIT_OS_VERSION;
      return this;
    }

    /**
     * Adds one element to {@link Platform#osFeatures() osFeatures} list.
     * @param element A osFeatures element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder osFeature(String element) {
      this.osFeatures.add(Objects.requireNonNull(element, "osFeatures element"));
      return this;
    }

    /**
     * Adds elements to {@link Platform#osFeatures() osFeatures} list.
     * @param elements An array of osFeatures elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder osFeatures(String... elements) {
      for (String element : elements) {
        this.osFeatures.add(Objects.requireNonNull(element, "osFeatures element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link Platform#osFeatures() osFeatures} list.
     * @param elements An iterable of osFeatures elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("OSFeatures")
    public final Builder osFeatures(Iterable<String> elements) {
      this.osFeatures.clear();
      return addAllOsFeatures(elements);
    }

    /**
     * Adds elements to {@link Platform#osFeatures() osFeatures} list.
     * @param elements An iterable of osFeatures elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllOsFeatures(Iterable<String> elements) {
      for (String element : elements) {
        this.osFeatures.add(Objects.requireNonNull(element, "osFeatures element"));
      }
      return this;
    }

    /**
     * Initializes the value for the {@link Platform#variant() variant} attribute.
     * @param variant The value for variant 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Variant")
    public final Builder variant(String variant) {
      this.variant = Objects.requireNonNull(variant, "variant");
      initBits &= ~INIT_BIT_VARIANT;
      return this;
    }

    /**
     * Adds one element to {@link Platform#features() features} list.
     * @param element A features element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder feature(String element) {
      this.features.add(Objects.requireNonNull(element, "features element"));
      return this;
    }

    /**
     * Adds elements to {@link Platform#features() features} list.
     * @param elements An array of features elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder features(String... elements) {
      for (String element : elements) {
        this.features.add(Objects.requireNonNull(element, "features element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link Platform#features() features} list.
     * @param elements An iterable of features elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Features")
    public final Builder features(Iterable<String> elements) {
      this.features.clear();
      return addAllFeatures(elements);
    }

    /**
     * Adds elements to {@link Platform#features() features} list.
     * @param elements An iterable of features elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllFeatures(Iterable<String> elements) {
      for (String element : elements) {
        this.features.add(Objects.requireNonNull(element, "features element"));
      }
      return this;
    }

    /**
     * Builds a new {@link ImmutablePlatform ImmutablePlatform}.
     * @return An immutable instance of Platform
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutablePlatform build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutablePlatform(
          architecture,
          os,
          osVersion,
          createUnmodifiableList(true, osFeatures),
          variant,
          createUnmodifiableList(true, features));
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_ARCHITECTURE) != 0) attributes.add("architecture");
      if ((initBits & INIT_BIT_OS) != 0) attributes.add("os");
      if ((initBits & INIT_BIT_OS_VERSION) != 0) attributes.add("osVersion");
      if ((initBits & INIT_BIT_VARIANT) != 0) attributes.add("variant");
      return "Cannot build Platform, some of required attributes are not set " + attributes;
    }
  }

  private static <T> List<T> createSafeList(Iterable<? extends T> iterable, boolean checkNulls, boolean skipNulls) {
    ArrayList<T> list;
    if (iterable instanceof Collection<?>) {
      int size = ((Collection<?>) iterable).size();
      if (size == 0) return Collections.emptyList();
      list = new ArrayList<>(size);
    } else {
      list = new ArrayList<>();
    }
    for (T element : iterable) {
      if (skipNulls && element == null) continue;
      if (checkNulls) Objects.requireNonNull(element, "element");
      list.add(element);
    }
    return list;
  }

  private static <T> List<T> createUnmodifiableList(boolean clone, List<T> list) {
    switch(list.size()) {
    case 0: return Collections.emptyList();
    case 1: return Collections.singletonList(list.get(0));
    default:
      if (clone) {
        return Collections.unmodifiableList(new ArrayList<>(list));
      } else {
        if (list instanceof ArrayList<?>) {
          ((ArrayList<?>) list).trimToSize();
        }
        return Collections.unmodifiableList(list);
      }
    }
  }
}
