/*
 * Decompiled with CFR 0.152.
 */
package io.fabric8.kubernetes.client.utils;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import io.fabric8.kubernetes.api.model.AuthInfo;
import io.fabric8.kubernetes.api.model.AuthProviderConfig;
import io.fabric8.kubernetes.api.model.NamedAuthInfo;
import io.fabric8.kubernetes.client.Config;
import io.fabric8.kubernetes.client.KubernetesClientException;
import io.fabric8.kubernetes.client.http.HttpClient;
import io.fabric8.kubernetes.client.http.HttpRequest;
import io.fabric8.kubernetes.client.internal.KubeConfigUtils;
import io.fabric8.kubernetes.client.internal.SSLUtils;
import io.fabric8.kubernetes.client.utils.Serialization;
import io.fabric8.kubernetes.client.utils.URLUtils;
import io.fabric8.kubernetes.client.utils.Utils;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.security.spec.InvalidKeySpecException;
import java.time.Instant;
import java.util.Base64;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import javax.net.ssl.KeyManager;
import javax.net.ssl.TrustManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OpenIDConnectionUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(OpenIDConnectionUtils.class);
    public static final String ID_TOKEN_KUBECONFIG = "id-token";
    public static final String ISSUER_KUBECONFIG = "idp-issuer-url";
    public static final String REFRESH_TOKEN_KUBECONFIG = "refresh-token";
    private static final String REFRESH_TOKEN_PARAM = "refresh_token";
    public static final String GRANT_TYPE_PARAM = "grant_type";
    public static final String CLIENT_ID_PARAM = "client_id";
    public static final String CLIENT_SECRET_PARAM = "client_secret";
    public static final String CLIENT_ID_KUBECONFIG = "client-id";
    public static final String CLIENT_SECRET_KUBECONFIG = "client-secret";
    private static final String IDP_CERT_DATA = "idp-certificate-authority-data";
    private static final String WELL_KNOWN_OPENID_CONFIGURATION = ".well-known/openid-configuration";
    private static final String GRANT_TYPE_REFRESH_TOKEN = "refresh_token";
    private static final String JWT_TOKEN_EXPIRY_TIMESTAMP_KEY = "exp";
    private static final String JWT_PARTS_DELIMITER_REGEX = "\\.";
    private static final int TOKEN_EXPIRY_DELTA = 10;

    private OpenIDConnectionUtils() {
    }

    public static CompletableFuture<String> resolveOIDCTokenFromAuthConfig(Config currentConfig, Map<String, String> currentAuthProviderConfig, HttpClient.Builder clientBuilder) {
        String originalToken = currentAuthProviderConfig.get(ID_TOKEN_KUBECONFIG);
        String idpCert = currentAuthProviderConfig.getOrDefault(IDP_CERT_DATA, OpenIDConnectionUtils.getClientCertDataFromConfig(currentConfig));
        if (OpenIDConnectionUtils.isTokenRefreshSupported(currentAuthProviderConfig)) {
            HttpClient httpClient = OpenIDConnectionUtils.initHttpClientWithPemCert(idpCert, clientBuilder);
            CompletionStage result = ((CompletableFuture)((CompletableFuture)OpenIDConnectionUtils.getOpenIdConfiguration(httpClient, currentAuthProviderConfig).thenCompose(openIdConfiguration -> OpenIDConnectionUtils.refreshOpenIdToken(httpClient, currentAuthProviderConfig, openIdConfiguration))).thenApply(oAuthToken -> OpenIDConnectionUtils.persistOAuthToken(currentConfig, oAuthToken, null))).thenApply(oAuthToken -> {
                if (oAuthToken == null || Utils.isNullOrEmpty(((OAuthToken)oAuthToken).idToken)) {
                    LOGGER.warn("token response did not contain an id_token, either the scope \\\"openid\\\" wasn't requested upon login, or the provider doesn't support id_tokens as part of the refresh response.");
                    return originalToken;
                }
                return ((OAuthToken)oAuthToken).idToken;
            });
            ((CompletableFuture)result).whenComplete((s, t) -> httpClient.close());
            return result;
        }
        return CompletableFuture.completedFuture(originalToken);
    }

    static boolean isTokenRefreshSupported(Map<String, String> currentAuthProviderConfig) {
        return Utils.isNotNull(currentAuthProviderConfig.get(REFRESH_TOKEN_KUBECONFIG));
    }

    private static CompletableFuture<OpenIdConfiguration> getOpenIdConfiguration(HttpClient client, Map<String, String> authProviderConfig) {
        HttpRequest request = client.newHttpRequestBuilder().uri(OpenIDConnectionUtils.resolveWellKnownUrlForOpenIDIssuer(authProviderConfig)).build();
        return client.sendAsync(request, String.class).thenApply(response -> {
            try {
                if (response.isSuccessful() && response.body() != null) {
                    return Serialization.unmarshal((String)response.body(), OpenIdConfiguration.class);
                }
                String responseBody = (String)response.body();
                LOGGER.warn("oidc: failed to query metadata endpoint: {} {}", (Object)response.code(), (Object)responseBody);
            }
            catch (Exception e) {
                LOGGER.warn("Could not refresh OIDC token, failure in getting refresh URL", (Throwable)e);
            }
            return null;
        });
    }

    private static CompletableFuture<OAuthToken> refreshOpenIdToken(HttpClient httpClient, Map<String, String> authProviderConfig, OpenIdConfiguration openIdConfiguration) {
        if (openIdConfiguration == null || Utils.isNullOrEmpty(openIdConfiguration.tokenEndpoint)) {
            LOGGER.warn("oidc: discovery object doesn't contain a valid token endpoint: {}", (Object)openIdConfiguration);
            return CompletableFuture.completedFuture(null);
        }
        HttpRequest request = OpenIDConnectionUtils.initTokenRefreshHttpRequest(httpClient, authProviderConfig, openIdConfiguration.tokenEndpoint);
        return httpClient.sendAsync(request, String.class).thenApply(r -> {
            String body = (String)r.body();
            if (body != null) {
                if (r.isSuccessful()) {
                    try {
                        return Serialization.unmarshal(body, OAuthToken.class);
                    }
                    catch (Exception e) {
                        LOGGER.warn("Failure in fetching refresh token: ", (Throwable)e);
                    }
                } else {
                    LOGGER.warn("Response: {}", (Object)body);
                }
            }
            return null;
        });
    }

    public static OAuthToken persistOAuthToken(Config currentConfig, OAuthToken oAuthToken, String token) {
        HashMap<String, String> authProviderConfig = new HashMap<String, String>();
        if (oAuthToken != null) {
            authProviderConfig.put(ID_TOKEN_KUBECONFIG, oAuthToken.idToken);
            authProviderConfig.put(REFRESH_TOKEN_KUBECONFIG, oAuthToken.refreshToken);
            Optional.of(currentConfig).map(Config::getAuthProvider).map(AuthProviderConfig::getConfig).ifPresent(c -> c.putAll(authProviderConfig));
        }
        if (currentConfig.getFile() != null && currentConfig.getCurrentContext() != null) {
            try {
                io.fabric8.kubernetes.api.model.Config kubeConfig = KubeConfigUtils.parseConfig(currentConfig.getFile());
                String userName = currentConfig.getCurrentContext().getContext().getUser();
                NamedAuthInfo namedAuthInfo = kubeConfig.getUsers().stream().filter(n -> n.getName().equals(userName)).findFirst().orElseGet(() -> {
                    NamedAuthInfo result = new NamedAuthInfo(userName, new AuthInfo());
                    kubeConfig.getUsers().add(result);
                    return result;
                });
                if (namedAuthInfo.getUser() == null) {
                    namedAuthInfo.setUser(new AuthInfo());
                }
                if (namedAuthInfo.getUser().getAuthProvider() == null) {
                    namedAuthInfo.getUser().setAuthProvider(new AuthProviderConfig());
                }
                namedAuthInfo.getUser().getAuthProvider().getConfig().putAll(authProviderConfig);
                if (Utils.isNotNullOrEmpty(token)) {
                    namedAuthInfo.getUser().setToken(token);
                }
                KubeConfigUtils.persistKubeConfigIntoFile(kubeConfig, currentConfig.getFile().getAbsolutePath());
            }
            catch (IOException ex) {
                LOGGER.warn("oidc: failure while persisting new tokens into KUBECONFIG", (Throwable)ex);
            }
        }
        return oAuthToken;
    }

    private static String resolveWellKnownUrlForOpenIDIssuer(Map<String, String> authProviderConfig) {
        return URLUtils.join(authProviderConfig.get(ISSUER_KUBECONFIG), "/", WELL_KNOWN_OPENID_CONFIGURATION);
    }

    private static HttpClient initHttpClientWithPemCert(String idpCert, HttpClient.Builder clientBuilder) {
        String pemCert = new String(Base64.getDecoder().decode(idpCert));
        try {
            TrustManager[] trustManagers = SSLUtils.trustManagers(pemCert, null, false, null, null);
            KeyManager[] keyManagers = SSLUtils.keyManagers(pemCert, null, null, null, null, null, null, null);
            clientBuilder.sslContext(keyManagers, trustManagers);
            return clientBuilder.build();
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | UnrecoverableKeyException | CertificateException | InvalidKeySpecException e) {
            throw KubernetesClientException.launderThrowable("Could not import idp certificate", (Throwable)e);
        }
    }

    private static HttpRequest initTokenRefreshHttpRequest(HttpClient client, Map<String, String> authProviderConfig, String tokenRefreshUrl) {
        String clientId = authProviderConfig.get(CLIENT_ID_KUBECONFIG);
        String clientSecret = authProviderConfig.getOrDefault(CLIENT_SECRET_KUBECONFIG, "");
        HttpRequest.Builder httpRequestBuilder = client.newHttpRequestBuilder().uri(tokenRefreshUrl);
        String credentials = Base64.getEncoder().encodeToString((clientId + ':' + clientSecret).getBytes(StandardCharsets.UTF_8));
        httpRequestBuilder.header("Authorization", "Basic " + credentials);
        LinkedHashMap<String, String> requestBody = new LinkedHashMap<String, String>();
        requestBody.put("refresh_token", authProviderConfig.get(REFRESH_TOKEN_KUBECONFIG));
        requestBody.put(GRANT_TYPE_PARAM, "refresh_token");
        requestBody.put(CLIENT_ID_PARAM, clientId);
        requestBody.put(CLIENT_SECRET_PARAM, clientSecret);
        httpRequestBuilder.post(requestBody);
        return httpRequestBuilder.build();
    }

    public static boolean idTokenExpired(Config config) {
        Map authProviderConfig;
        String accessToken;
        if (config.getAuthProvider() != null && config.getAuthProvider().getConfig() != null && OpenIDConnectionUtils.isValidJwt(accessToken = (String)(authProviderConfig = config.getAuthProvider().getConfig()).get(ID_TOKEN_KUBECONFIG))) {
            try {
                String[] jwtParts = accessToken.split(JWT_PARTS_DELIMITER_REGEX);
                String jwtPayload = jwtParts[1];
                String jwtPayloadDecoded = new String(Base64.getDecoder().decode(jwtPayload));
                Map jwtPayloadMap = Serialization.unmarshal(jwtPayloadDecoded, Map.class);
                int expiryTimestampInSeconds = (Integer)jwtPayloadMap.get(JWT_TOKEN_EXPIRY_TIMESTAMP_KEY);
                return Instant.ofEpochSecond(expiryTimestampInSeconds).minusSeconds(10L).isBefore(Instant.now());
            }
            catch (Exception e) {
                return true;
            }
        }
        return true;
    }

    private static boolean isValidJwt(String token) {
        if (token != null && !token.isEmpty()) {
            String[] jwtParts = token.split(JWT_PARTS_DELIMITER_REGEX);
            return jwtParts.length == 3;
        }
        return false;
    }

    private static String getClientCertDataFromConfig(Config config) {
        if (config.getCaCertData() != null && !config.getCaCertData().isEmpty()) {
            return config.getCaCertData();
        }
        try {
            if (config.getCaCertFile() != null) {
                return Base64.getEncoder().encodeToString(Files.readAllBytes(Paths.get(config.getCaCertFile(), new String[0])));
            }
        }
        catch (IOException e) {
            LOGGER.debug("Failure in reading certificate data from {}", (Object)config.getCaCertFile());
        }
        return null;
    }

    @JsonIgnoreProperties(ignoreUnknown=true)
    public static final class OAuthToken {
        @JsonProperty(value="id_token")
        private String idToken;
        @JsonProperty(value="refresh_token")
        private String refreshToken;

        @JsonProperty(value="id_token")
        public void setIdToken(String idToken) {
            this.idToken = idToken;
        }

        @JsonProperty(value="refresh_token")
        public void setRefreshToken(String refreshToken) {
            this.refreshToken = refreshToken;
        }
    }

    @JsonIgnoreProperties(ignoreUnknown=true)
    public static final class OpenIdConfiguration {
        @JsonProperty(value="token_endpoint")
        private String tokenEndpoint;

        @JsonProperty(value="token_endpoint")
        public void setTokenEndpoint(String tokenEndpoint) {
            this.tokenEndpoint = tokenEndpoint;
        }
    }
}

