/*
 * ====================================================================
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */
package org.apache.hc.client5.http.impl.classic;

import org.apache.hc.client5.http.HttpRoute;
import org.apache.hc.client5.http.classic.ExecChain;
import org.apache.hc.client5.http.classic.ExecRuntime;
import org.apache.hc.client5.http.config.RequestConfig;
import org.apache.hc.client5.http.entity.DecompressingEntity;
import org.apache.hc.client5.http.entity.EntityBuilder;
import org.apache.hc.client5.http.entity.GzipDecompressingEntity;
import org.apache.hc.client5.http.protocol.HttpClientContext;
import org.apache.hc.core5.http.ClassicHttpRequest;
import org.apache.hc.core5.http.ClassicHttpResponse;
import org.apache.hc.core5.http.HttpEntity;
import org.apache.hc.core5.http.HttpException;
import org.apache.hc.core5.http.HttpHost;
import org.apache.hc.core5.http.Method;
import org.apache.hc.core5.http.io.entity.StringEntity;
import org.apache.hc.core5.http.message.BasicClassicHttpRequest;
import org.apache.hc.core5.http.message.BasicClassicHttpResponse;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;

class TestContentCompressionExec {

    @Mock
    private ExecRuntime execRuntime;
    @Mock
    private ExecChain execChain;
    @Mock
    private ClassicHttpRequest originaRequest;

    private HttpClientContext context;
    private HttpHost host;
    private ExecChain.Scope scope;
    private ContentCompressionExec impl;

    @BeforeEach
    void setup() {
        MockitoAnnotations.openMocks(this);
        host = new HttpHost("somehost", 80);
        context = HttpClientContext.create();
        scope = new ExecChain.Scope("test", new HttpRoute(host), originaRequest, execRuntime, context);
        impl = new ContentCompressionExec();
    }


    @Test
    void testContentEncodingNoEntity() throws Exception {
        final ClassicHttpRequest request = new BasicClassicHttpRequest(Method.GET, host, "/");
        final ClassicHttpResponse response = new BasicClassicHttpResponse(200, "OK");

        Mockito.when(execChain.proceed(request, scope)).thenReturn(response);

        impl.execute(request, scope, execChain);

        final HttpEntity entity = response.getEntity();
        Assertions.assertNull(entity);
    }

    @Test
    void testNoContentEncoding() throws Exception {
        final ClassicHttpRequest request = new BasicClassicHttpRequest(Method.GET, host, "/");
        final ClassicHttpResponse response = new BasicClassicHttpResponse(200, "OK");
        final StringEntity original = new StringEntity("plain stuff");
        response.setEntity(original);

        Mockito.when(execChain.proceed(request, scope)).thenReturn(response);

        impl.execute(request, scope, execChain);

        final HttpEntity entity = response.getEntity();
        Assertions.assertNotNull(entity);
        Assertions.assertTrue(entity instanceof StringEntity);
    }

    @Test
    void testGzipContentEncoding() throws Exception {
        final ClassicHttpRequest request = new BasicClassicHttpRequest(Method.GET, host, "/");
        final ClassicHttpResponse response = new BasicClassicHttpResponse(200, "OK");
        final HttpEntity original = EntityBuilder.create().setText("encoded stuff").setContentEncoding("GZip").build();
        response.setEntity(original);

        Mockito.when(execChain.proceed(request, scope)).thenReturn(response);

        impl.execute(request, scope, execChain);

        final HttpEntity entity = response.getEntity();
        Assertions.assertNotNull(entity);
        Assertions.assertTrue(entity instanceof DecompressingEntity);
    }

    @Test
    void testGzipContentEncodingZeroLength() throws Exception {
        final ClassicHttpRequest request = new BasicClassicHttpRequest(Method.GET, host, "/");
        final ClassicHttpResponse response = new BasicClassicHttpResponse(200, "OK");
        final HttpEntity original = EntityBuilder.create().setText("").setContentEncoding("GZip").build();
        response.setEntity(original);

        Mockito.when(execChain.proceed(request, scope)).thenReturn(response);

        impl.execute(request, scope, execChain);

        final HttpEntity entity = response.getEntity();
        Assertions.assertNotNull(entity);
        Assertions.assertTrue(entity instanceof StringEntity);
    }

    @Test
    void testXGzipContentEncoding() throws Exception {
        final ClassicHttpRequest request = new BasicClassicHttpRequest(Method.GET, host, "/");
        final ClassicHttpResponse response = new BasicClassicHttpResponse(200, "OK");
        final HttpEntity original = EntityBuilder.create().setText("encoded stuff").setContentEncoding("x-gzip").build();
        response.setEntity(original);

        Mockito.when(execChain.proceed(request, scope)).thenReturn(response);

        impl.execute(request, scope, execChain);

        final HttpEntity entity = response.getEntity();
        Assertions.assertNotNull(entity);
        Assertions.assertTrue(entity instanceof DecompressingEntity);
    }

    @Test
    void testDeflateContentEncoding() throws Exception {
        final ClassicHttpRequest request = new BasicClassicHttpRequest(Method.GET, host, "/");
        final ClassicHttpResponse response = new BasicClassicHttpResponse(200, "OK");
        final HttpEntity original = EntityBuilder.create().setText("encoded stuff").setContentEncoding("deFlaTe").build();
        response.setEntity(original);

        Mockito.when(execChain.proceed(request, scope)).thenReturn(response);

        impl.execute(request, scope, execChain);

        final HttpEntity entity = response.getEntity();
        Assertions.assertNotNull(entity);
        Assertions.assertTrue(entity instanceof DecompressingEntity);
    }

    @Test
    void testIdentityContentEncoding() throws Exception {
        final ClassicHttpRequest request = new BasicClassicHttpRequest(Method.GET, host, "/");
        final ClassicHttpResponse response = new BasicClassicHttpResponse(200, "OK");
        final HttpEntity original = EntityBuilder.create().setText("encoded stuff").setContentEncoding("identity").build();
        response.setEntity(original);

        Mockito.when(execChain.proceed(request, scope)).thenReturn(response);

        impl.execute(request, scope, execChain);

        final HttpEntity entity = response.getEntity();
        Assertions.assertNotNull(entity);
        Assertions.assertTrue(entity instanceof StringEntity);
    }

    @Test
    void testBrotliContentEncoding() throws Exception {
        final ClassicHttpRequest request = new BasicClassicHttpRequest(Method.GET, host, "/");
        final ClassicHttpResponse response = new BasicClassicHttpResponse(200, "OK");
        final HttpEntity original = EntityBuilder.create().setText("encoded stuff").setContentEncoding("br").build();
        response.setEntity(original);

        Mockito.when(execChain.proceed(request, scope)).thenReturn(response);

        impl.execute(request, scope, execChain);

        final HttpEntity entity = response.getEntity();
        Assertions.assertNotNull(entity);
        Assertions.assertTrue(entity instanceof DecompressingEntity);
    }

    @Test
    void testUnknownContentEncoding() throws Exception {
        final ClassicHttpRequest request = new BasicClassicHttpRequest(Method.GET, host, "/");
        final ClassicHttpResponse response = new BasicClassicHttpResponse(200, "OK");
        final HttpEntity original = EntityBuilder.create().setText("encoded stuff").setContentEncoding("whatever").build();
        response.setEntity(original);

        impl = new ContentCompressionExec(false);

        Mockito.when(execChain.proceed(request, scope)).thenReturn(response);

        Assertions.assertThrows(HttpException.class, () ->
                impl.execute(request, scope, execChain));
    }

    @Test
    void testContentEncodingRequestParameter() throws Exception {
        final ClassicHttpRequest request = new BasicClassicHttpRequest(Method.GET, host, "/");
        final ClassicHttpResponse response = new BasicClassicHttpResponse(200, "OK");
        final HttpEntity original = EntityBuilder.create().setText("encoded stuff").setContentEncoding("GZip").build();
        response.setEntity(original);

        final RequestConfig config = RequestConfig.custom()
                .setContentCompressionEnabled(false)
                .build();

        context.setRequestConfig(config);

        Mockito.when(execChain.proceed(request, scope)).thenReturn(response);

        impl.execute(request, scope, execChain);

        final HttpEntity entity = response.getEntity();
        Assertions.assertNotNull(entity);
        Assertions.assertFalse(entity instanceof GzipDecompressingEntity);
    }

}
