/* @(#)ywho.c,v 1.8 2000/12/17 17:03:10 kim Exp */
/*
 * ywho.c: Use rpc to obtain information about remote users
 *	   if it is invoked as yusers, present info in short format
 *
 *
 * Original Author:
 *	James P. Lewis		| FREE TO DISTRIBUTE WITH THIS HEADER
 *	York University		| REMAINS INTACT.
 *	4700 Keele Street	|
 *	Downsview, Ontario	|
 *	Canada			|
 *	M3J-1P3			|
 *
 *	...yunexus!james
 *
 * yuptime, yusers, and .yhosts added by Christos Zoulas
 * christos@ee.cornell.edu
 *
 */
#ifndef lint
static const char rcsid[] = "@(#)ywho.c,v 1.8 2000/12/17 17:03:10 kim Exp";
#endif /* lint */

#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/param.h>
#include <stdio.h>
#include <signal.h>
#include <string.h>
#define	_NO_UT_TIME	/* linux hack */
#include <utmp.h>
#include <ctype.h>
#include <setjmp.h>
#if defined(__svr4__) || defined(__SVR4)
#define PORTMAP
#endif
#ifdef __NetBSD__
# undef FSHIFT		/* Use protocol's shift and scale values */
# undef FSCALE
#endif
#include <rpc/rpc.h>
#include <rpcsvc/rstat.h>
#ifdef __NetBSD__
# include <rpcsvc/rnusers.h>
#endif
#include <rpcsvc/rusers.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netdb.h>
#ifdef __NetBSD__
# include <netgroup.h>
#endif

#ifdef MAXHOSTNAMELEN
# define HST_LEN MAXHOSTNAMELEN
#else
# define HST_LEN 64
#endif
#if !defined(hpux) || defined(__hpux)
#define UTHOST
#endif

#ifdef lint
/*
 * Make lint think Malloc's value is well aligned.
 */
typedef union {
    double a; long b; int c; short d; char e;
} *align_t;
#else
typedef char *align_t;
#endif

#define NIL(a) ((a *) 0)
#define NEW(a) ((a *) Malloc((unsigned) sizeof(a)))
#define NEWN(a, n) ((a *) Malloc((unsigned) (sizeof(a) * (n))))
#define Strdup(a) \
	((char *) strcpy((char *) Malloc((unsigned) (strlen(a) + 1)), a))
#define SEP "\t \n,="
#ifndef FSCALE
#define FSCALE (1 << 8)
#endif

#define RWHO 	0
#define RUSERS 	1
#define RUPTIME 2

#ifndef SYSHOSTS
#define SYSHOSTS "/etc/yhosts"
#endif

typedef struct hosts_t {
    struct hostent hp;
    struct hosts_t *next;
} hosts_t;

static align_t Malloc();
static void do_host();
static void do_timeout();
static char *pname;
static int aflag = FALSE;
static int  timeout;
static int  what;
static jmp_buf goback;
static int interrupted;
static hosts_t *hosts = NIL(hosts_t);
static char nick_name[HST_LEN];


static void
set_signals()
{
#ifdef SIG_SETMASK
    struct sigaction sa;

    sa.sa_flags = SA_NODEFER|SA_RESTART;
    sigemptyset(&sa.sa_mask);
    sa.sa_handler = do_timeout;
    (void) sigaction(SIGALRM, &sa, NULL);
    (void) sigaction(SIGINT, &sa, NULL);
#else
    (void) signal(SIGALRM, do_timeout);
    (void) signal(SIGINT, do_timeout);
#endif
}

static void
addhost(h)
char *h;
{
    struct hostent *hp;
    struct hosts_t *hh, *hptr = NIL(hosts_t);
    int i;

    if (h == NIL(char))
	return;

    if ((hp = gethostbyname(h)) == NIL(struct hostent)) {
	(void) fprintf(stderr,
	    "%s: gethostbyname: can't get addr for %s\n", pname, h);
	exit(1);
    }
    if (hosts == NIL(hosts_t)) {
	hosts = hptr = NEW(hosts_t);
	hh = NIL(hosts_t);
    } else {
	for (hh = hosts; hh != NIL(hosts_t); hptr = hh, hh = hh->next) {
	    if ((i = strcmp(hp->h_name, hh->hp.h_name)) < 0)
		break;
	}
	if (i == 0)
	    return;
	if (hh == hosts) {
	    hptr = NEW(hosts_t);
	    hosts = hptr;
	} else {
	    hptr->next = NEW(hosts_t);
	    hptr = hptr->next;
	}
    }
    hptr->hp = *hp;
    hptr->hp.h_name = Strdup(hp->h_name);
#ifndef h_addr
    hptr->hp.h_addr = NEWN(char, hp->h_length);
    memcpy(hptr->hp.h_addr, hp->h_addr, hp->h_length);
#else
    for (i = 0; hp->h_addr_list[i] != NIL(char); i++);
    hptr->hp.h_addr_list = NEWN(char *, i + 1);
    hptr->hp.h_addr_list[i] = NIL(char);
    for (i--;i >= 0; i--) {
	hptr->hp.h_addr_list[i] = NEWN(char, hp->h_length);
	memcpy(hptr->hp.h_addr_list[i], hp->h_addr_list[i], hp->h_length);
    }
#endif

    for (h = hptr->hp.h_name; *h; h++)
	*h = isupper((int)*h) ? tolower((int)*h) : *h;
    hptr->next = hh;
} /* end addhost */


static void
addnetgroup(ng)
char *ng;
{
#ifdef	linux
    /* glibc provides a return value to indicate success, so use it */
    if (setnetgrent(ng) == 0) {
	(void) fprintf(stderr, "%s: Invalid netgroup: %s\n", pname, ng);
	exit(1);
    }
#else
    /* traditional SunOS/Solaris (and NetBSD) implementations just
       return void, and hence no way to distinguish success/failure */
    setnetgrent(ng);
#endif	/* linux */
    do {
#ifdef __NetBSD__
	const
#endif
	char *hp = NIL(char), *gp = NIL(char), *dp = NIL(char);
	if (getnetgrent(&hp, &gp, &dp) == 0)
	    break;
	if (hp && (isalnum((int)hp[0]) || (hp[0] == '_')))
	    addhost(hp);
    } while (1);
#ifndef __SVR4
    endnetgrent();
#endif
}


static int
parseyhosts()
{
    char buffer[BUFSIZ];
    char *ptr, *eptr;
    FILE *fp;
    int err1 = 0, err2 = 0;

    ptr = getenv("HOME");
    (void) sprintf(buffer, "%s%s.yhosts",
		   ptr ? ptr : "", ptr ? "/" : "");
    fp = fopen(buffer, "r");

    if (fp == NIL(FILE)) {
	err1 = errno;
	fp = fopen(SYSHOSTS, "r");
    }

    if (fp == NIL(FILE)) {
	err2 = errno;
	(void) fprintf(stderr,
	    "%s: Could not open \n\t`%s' (%s) or \n\t`%s' (%s).\n",
	    pname, buffer, strerror(err1), SYSHOSTS, strerror(err2));
	return -1;
    }

    err1 = 0;

    while (fgets(buffer, BUFSIZ, fp) != 0) {

	if ((ptr = strtok(buffer, SEP)) == NIL(char))
	    continue;

	if (ptr[0] == '#')
	    continue;

	if (strcmp(ptr, "timeout") == 0) {
	    if ((ptr = strtok(NIL(char), "= \t\n")) == NIL(char)) {
		(void) fprintf(stderr,
		    "%s: Missing timeout value.\n", pname);
		err1 = -1;
		break;
	    }
	    timeout = strtol(ptr, &eptr, 0);
	    if ((ptr == eptr) || (*eptr != '\0')) {
		(void) fprintf(stderr,
		    "%s: Invalid timeout value (%s).\n", pname, ptr);
		err1 = -1;
		break;
	    }
	    if (timeout <= 0) {
		(void) fprintf(stderr,
		    "%s: Negative or zero timeout value (%d).\n", pname,
		    timeout);
		err1 = -1;
		break;
	    }
	    continue;
	}

	if (strcmp(ptr, "netgroup") == 0) {
	    if ((ptr = strtok(NIL(char), "= \t\n")) == NIL(char)) {
		(void) fprintf(stderr,
		    "%s: Missing netgroup.\n", pname);
		err1 = -1;
		break;
	    }
	    addnetgroup(ptr);
	    continue;
	}

	addhost(ptr);
    }
    (void) fclose(fp);
    return err1;
}


int
main(argc, argv)
int     argc;
char    *argv[];
{
    int i;
    char *s;
    /* the following is static to prevent register allocation,
       and potential clobbering by longjmp() */
    static hosts_t *hptr;

    if ((pname = strrchr(*argv, '/')) == NIL(char))
	pname = *argv;
    else
	pname++;

    what = RWHO;
    what += (pname[1] == 'u'); /* y(u)sers */
    what += (pname[2] == 'p'); /* yu(p)time */
    timeout = 2;

    for (i = 1; i < argc; i++) {
	if (argv[i][0] == '-')
	    for (s = &argv[i][1]; *s; s++)
	    switch (*s) {
	    case 'a':
		aflag = TRUE;
		break;

	    case 't':
		if ((*++s == '\0') && ((s = argv[++i]) == NIL(char))) {
		    (void) fprintf(stderr, "%s: Missing timeout.\n", pname);
		    exit(1);
		}
		{
		    char *eptr;
		    timeout = strtol(s, &eptr, 0);
		    if ((s == eptr) || (*eptr != '\0') || (timeout <= 0)) {
			(void) fprintf(stderr, "%s: Invalid timeout.\n", pname);
			exit(1);
		    }
		}
		goto nextarg;

	    case 'g':
		if ((*++s == '\0') && ((s = argv[++i]) == NIL(char))) {
		    (void) fprintf(stderr, "%s: Missing netgroup.\n", pname);
		    exit(1);
		}
		addnetgroup(s);
		goto nextarg;

	    default:
		(void) fprintf(stderr,
		   "Usage: %s [-a] [-t timeout] [host|-g netgroup] ...\n",
		   pname);
		exit(1);
	    }
	else
	    addhost(argv[i]);
nextarg:
    }

    if (hosts == NIL(hosts_t)) {
	if (parseyhosts() == -1)
	    exit(1);
	if (hosts == NIL(hosts_t)) {
	    (void) fprintf(stderr, "%s: No hosts.\n", pname);
	    exit(1);
	}
    }

    set_signals();

    (void) fflush(stdout);
    interrupted = 0;
    for (hptr = hosts; hptr != NIL(hosts_t); hptr = hptr->next) {
	if (!setjmp(goback)) {
	    alarm(timeout+1);
	    interrupted = 0;
	    do_host(hptr);
	}
	alarm(0);
    }
    return(0);
}

static struct utmpidlearr  	cutmpidlearr;
static struct statstime		sttime;

static void
do_banner()
{
    static int didbanner = FALSE;
    if (didbanner)
	return;
    (void) fprintf(stdout,
#ifdef UTHOST
	"%-16.16s %-8.8s %-8.8s %-12.12s %6.6s  %-16.16s\n",
	"Host",  "User", "tty", "login at ", "idle", "from host");
#else
	"%-16.16s %-8.8s %-8.8s %-12.12s %6.6s\n",
	"Host",  "User", "tty", "login at ", "idle");
#endif
    didbanner = TRUE;
}
/* do_host():
 *	Somehow I am corrupting memory. If I don't declare utmpidlearray
 *	and sttime as globals, I core dump *sometimes*. Allocation for
 *	these might be wrong. Anyway it is working now. Till it breaks
 *	again.
 */
static void
do_host(hst)
hosts_t *hst;
{
    char                tmp[BUFSIZ];
    int                 ss, dd, hh, mm, printed = FALSE;
    int                 i, sock;
    register CLIENT     *client, *client_st;
    struct hostent      *hp;
    struct timeval      pertry_timeout, total_timeout;
    struct sockaddr_in  server_addr;
    enum clnt_stat      clnt_stat, clnt_stat_st;
    char                *ptr, *host;
    long		j;

    (void) strncpy(nick_name, hst->hp.h_name, HST_LEN);

    hp = &hst->hp;

    if (! isdigit((int)nick_name[0]))
	if ((ptr = strchr(nick_name, '.')) != NIL(char))
	    *ptr = '\0';

    if (what == RUPTIME) {
	(void) fprintf(stdout, "%-16.16s ", nick_name);
	(void) fflush(stdout);
    }
    if (what == RUPTIME) {
	sock = RPC_ANYSOCK;
	pertry_timeout.tv_sec = timeout;
	pertry_timeout.tv_usec = 0;
	total_timeout.tv_sec = timeout;
	total_timeout.tv_usec = 0;
	memcpy((caddr_t) &server_addr.sin_addr, hp->h_addr, hp->h_length);
	server_addr.sin_family = AF_INET;
	server_addr.sin_port = 0;
	client_st = clntudp_create(&server_addr, RSTATPROG, RSTATVERS_TIME,
				   pertry_timeout, &sock);
	if (client_st == NULL) {
	    (void) fprintf(stdout, "down\n");
	    return;
	}
	clnt_stat_st = clnt_call(client_st, RSTATPROC_STATS,
	    (xdrproc_t)xdr_void, 0, (xdrproc_t)xdr_statstime,
	    (caddr_t) &sttime, total_timeout);
	if (clnt_stat_st != RPC_SUCCESS) {
	    switch (what) {
	    case RWHO:
	    case RUPTIME:
	    case RUSERS:
		(void) fprintf(stderr, "%-16.16s ", nick_name);
		clnt_perror(client_st, "RSTAT");
		break;
	    }
	    return;
	}
	clnt_freeres(client_st, (xdrproc_t)xdr_statstime, (caddr_t) &sttime);
	clnt_destroy(client_st);
    }

    sock = RPC_ANYSOCK;
    pertry_timeout.tv_sec = timeout;
    pertry_timeout.tv_usec = 0;
    total_timeout.tv_sec = timeout;
    total_timeout.tv_usec = 0;
    memcpy((caddr_t) &server_addr.sin_addr, hp->h_addr, hp->h_length);
    server_addr.sin_family = AF_INET;
    server_addr.sin_port = 0;
    client = clntudp_create(&server_addr, RUSERSPROG,
			    RUSERSVERS_IDLE, pertry_timeout, &sock);
    if (client == NULL) {
	switch (what) {
	    case RWHO:
		if (aflag) {
		    do_banner();
		    (void) fflush(stdout);
		    (void) fprintf(stderr, "%-16.16s ", nick_name);
		    clnt_pcreateerror("No answer" /* "clntudp_create" */ );
		}
		break;
	    case RUSERS:
		break;
	    case RUPTIME:
		break;
	}
	return;
    }

    clnt_stat = clnt_call(client, RUSERSPROC_NAMES,
	(xdrproc_t)xdr_void, 0, (xdrproc_t)xdr_utmpidlearr,
	(caddr_t) &cutmpidlearr, total_timeout);
    if (clnt_stat != RPC_SUCCESS) {
	switch (what) {
	case RWHO:
	case RUSERS:
	    (void) fprintf(stderr, "%-16.16s ", nick_name);
	case RUPTIME:
	    clnt_perror(client, "RUSERS");
	    break;
	}
	return;
    }


    switch (what) {

    case RWHO:
	if (cutmpidlearr.uia_cnt == 0 && aflag) {
	    do_banner();
	    (void) fprintf(stdout, "%-16.16s ", nick_name);
	    (void) fprintf(stdout, "Nobody logged on.\n");
	    (void) fflush(stdout);
	    break;
	}
	for (i = 0; i < cutmpidlearr.uia_cnt; i++) {
	    j = cutmpidlearr.uia_arr[i]->ui_utmp.ut_time;

	    dd = hh = 0;
	    mm = cutmpidlearr.uia_arr[i]->ui_idle;

	    if (mm >= 60 || mm < 0) {
		if (!aflag)
		    continue;
		hh = mm / 60;
		mm = mm % 60;
	    }

	    do_banner();
	    (void) fprintf(stdout, "%-16.16s ", nick_name);
	    (void) fflush(stdout);

	    if (hh >= 24) {
		dd = hh / 24;
		hh = hh % 24;
	    }

	    if (dd == 0)
		if (hh == 0)
		    if (mm == 0)
			    *tmp = 0;
		    else
			(void) sprintf(tmp, mm < 0 ? "     -" : "    %2d", mm);
		else
		    (void) sprintf(tmp, " %2d:%-2.2d", hh, mm);
	    else
		(void) sprintf(tmp, "%2dd%-2.2dh", dd, hh);

	    (void) fprintf(stdout, "%-8.8s %-8.8s %-12.12s %-6.6s ",
		   cutmpidlearr.uia_arr[i]->ui_utmp.ut_name,
		   cutmpidlearr.uia_arr[i]->ui_utmp.ut_line,
		   ctime(&j)+4, tmp);

#ifdef UTHOST

#define SIZ_UT_HOST sizeof(cutmpidlearr.uia_arr[i]->ui_utmp.ut_host)

	    if (*(host = cutmpidlearr.uia_arr[i]->ui_utmp.ut_host)) {
	        char *dot, *col;
	        (void) strncpy(tmp, host, SIZ_UT_HOST);
		tmp[SIZ_UT_HOST] = '\0';
		if (!isdigit((int)*tmp)
		    && (dot = strchr(tmp, '.')) != NIL(char)) {
		    if ((col = strchr(tmp, ':')) != NIL(char)) {
			if (col > dot)
			    (void) strcpy(dot, col);
		    }
		    else
			*dot = '\0';
		}
		for (ptr = tmp; *ptr; ptr++)
		     *ptr = isupper((int)*ptr) ? tolower((int)*ptr) : *ptr;
		(void) fprintf(stdout, " %-16.16s\n", tmp);
	    }
	    else

#undef SIZ_UT_HOST

#endif /* UTHOST */
		(void) fprintf(stdout, "\n");
	}
	break;
    case RUSERS:
	/* print in rusers format */
	for (i = 0; i < cutmpidlearr.uia_cnt; i++) {
	    printed = FALSE;
	    if (i == 0)
		(void) sprintf(tmp, "%-16.16s:", nick_name);
	    (void) strcat(tmp, " ");
	    (void) strncat(tmp, cutmpidlearr.uia_arr[i]->ui_utmp.ut_name, 8);

	    if (strlen(tmp) > (size_t) 60) {
		(void) fprintf(stdout, "%s\n", tmp);
		(void) strcpy(tmp, "                 "); /* 17 spcs */
		printed = TRUE;
	    }
	}
	if (i > 0 && ! printed)
	    (void) fprintf(stdout, "%s\n", tmp);
	(void) fflush(stdout);
	break;
    case RUPTIME:
	ss = sttime.curtime.tv_sec - sttime.boottime.tv_sec;
	mm = hh = dd = 0;
	if (ss >= 60) {
	    mm = ss / 60;
	    ss = ss % 60;
	}
	if (mm >= 60) {
	    hh = mm / 60;
	    mm = mm % 60;
	}
	if (hh >= 24) {
	    dd = hh / 24;
	    hh = hh % 24;
	}
	if (dd > 0)
	    (void) sprintf(tmp, "%3d+%.2d:%.2d", dd, hh, mm);
	else if (hh > 0)
	    (void) sprintf(tmp, "    %2d:%.2d", hh, mm);
	else
	    (void) sprintf(tmp, "       %2d", mm);

	(void) fprintf(stdout, " up  %s, %3d %5s, load %3.2f, %3.2f, %3.2f\n",
	    tmp,
	    cutmpidlearr.uia_cnt,
	    cutmpidlearr.uia_cnt == 1 ? " user" : "users",
	    (double) sttime.avenrun[0] / FSCALE,
	    (double) sttime.avenrun[1] / FSCALE,
	    (double) sttime.avenrun[2] / FSCALE);
	break;
    }

    clnt_freeres(client, (xdrproc_t)xdr_utmpidlearr, (caddr_t) &cutmpidlearr);
    clnt_destroy(client);

    return;
} /* end do_host */

/* Malloc():
 *	Memory checked malloc
 */
static align_t
Malloc(nth)
unsigned nth;
{
#ifndef lint
    char *ptr;
    extern char *malloc();

    if ((ptr = malloc(nth)) == NIL(char)) {
	(void) fprintf(stderr, "%s: Out of memory.\n", pname);
	exit(1);
    }
    return(ptr);
#else
    return(nth ? (align_t) 0 : (align_t) 0);
#endif
} /* end Malloc */

/* do_timeout():
 *	Since the select timeout does not work right, we use ours
 */
static void
do_timeout(signum)
int signum;
{
    switch (what) {
    case RWHO:
	if (signum == SIGALRM) {
	    if (aflag) {
		do_banner();
		(void) fprintf(stdout, "%-16.16s No answer.\n", nick_name);
	    }
	}
	else if (signum == SIGINT) {
	    do_banner();
	    if (! interrupted) {
		interrupted = 1;
		(void) fprintf(stdout, "%-16.16s Interrupted.\n", nick_name);
		sleep(1);
	    }
	    else {
		(void) fprintf(stdout, "%-16.16s Exiting.\n", nick_name);
		exit(0);
	    }
	}
	(void) fflush(stdout);
	longjmp(goback, 1);
	break;
    case RUPTIME:
	if (signum == SIGALRM)
	    (void) fprintf(stdout, "down\n");
	else if (signum == SIGINT) {
	    if (! interrupted) {
		interrupted = 1;
		(void) fprintf(stdout, "Interrupted.\n");
		sleep(1);
	    }
	    else {
		(void) fprintf(stdout, "Exiting.\n");
		exit(0);
	    }
	}
	longjmp(goback, 1);
	break;
    case RUSERS:
	if (signum == SIGALRM) ;
	else if (signum == SIGINT) {
	    if (! interrupted) {
		interrupted = 1;
		(void) fprintf(stdout, "Interrupted.\n");
		sleep(1);
	    }
	    else {
		(void) fprintf(stdout, "Exiting.\n");
		exit(0);
	    }
	}
	(void) fflush(stdout);
	longjmp(goback, 1);
	break;
    }
} /* end do_timeout */

#ifdef hpux
void (*
signal(s, a))()
	int s;
void (*a)();
{
	struct sigvec osv, sv;

	sigvector(s, 0, &osv);
	sv = osv;
	sv.sv_handler = a;
	sv.sv_flags = SV_BSDSIG;
	if (sigvector(s, &sv, 0) < 0)
	    return (BADSIG);
	return (osv.sv_handler);
}
#endif
