/*
 * Copyright (c) 1997, 1999, 2000, 2001, Mark Buser.
 * Copyright (c) 2001, 2003, 2004, Danny Backx.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the names the authors (see above), nor the names of other
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Header: /pack/anoncvs/xinvest/src/tape.c,v 1.10 2004/11/05 18:31:48 danny Exp $
 */
#include <Xm/XmAll.h>

#include "color.h"
#include "opttick.h"
#include "pixmap.h"
#include "server.h"
#include "xutil.h"

/* Globals */
static Widget DrawingArea;
static Pixmap Drawpixmap[2];
static int which = 0;
static Dimension height, width;
static Dimension pixheight, pixwidth;
static GC gc;

static int running = True;
static int speed = 12;
static int offset = 2;

#define DELAY ((long)(500/speed*offset)) 

/* Draw the ticker tape, reinstall timer if this is not just a refresh */
/* ARGSUSED */
void scrollTape ( int refresh, XtIntervalId timer)
{
  static Dimension pos = 0;
  Display *dpy = XtDisplay (DrawingArea);

  /* Refresh display */
  if (pixwidth <= width) {
    pos = pos % width;

    XCopyArea ( dpy, Drawpixmap[which], XtWindow (DrawingArea),
                gc, pos, 0, pixwidth-pos, pixheight, 0, 0 );
    XCopyArea ( dpy, Drawpixmap[which], XtWindow (DrawingArea),
                gc, 0, 0, pos, pixheight, width-pos, 0 );
  } else {
    pos = pos % pixwidth;

    if ( (int)(pixwidth - pos) < (int)width ) {
      XCopyArea ( dpy, Drawpixmap[which], XtWindow (DrawingArea), 
  		  gc, pos, 0, pixwidth-pos, pixheight, 0, 0 );
      XCopyArea ( dpy, Drawpixmap[which], XtWindow (DrawingArea), 
		  gc, 0, 0, width-(pixwidth-pos), pixheight, pixwidth-pos, 0 );
    } else
      XCopyArea ( dpy, Drawpixmap[which], XtWindow (DrawingArea), 
		  gc, pos, 0, width, pixheight, 0, 0 );
  }

  /* Scrolling or stopping */
  if (running && !refresh) {

    if (Drawpixmap[1-which]) {                 /* Switch to new pixmap */
      XFreePixmap (dpy, Drawpixmap[which]);    /* Created in makeTextTape */
      Drawpixmap[which] = (Pixmap)NULL;
      which = 1-which;
    }
    pos += offset; 

    XtAppAddTimeOut ( XtWidgetToApplicationContext(DrawingArea),
                      DELAY, (XtTimerCallbackProc)scrollTape,
  		      (XtPointer) False );
  }
}


#define LARGE_FONT \
if (per->large) { \
  cur_font = per->large; \
  XSetFont ( XtDisplay(DrawingArea), gc, per->large->fid ); \
}

void makeTextTape ()  
{
  Display *dpy = XtDisplay (DrawingArea);

  extern PerDisplay *per;
  XFontStruct *cur_font;

  Pixel fg, bg, shadow, color;
  int x = 0, y;

  QUERY_STRUCT *data;
  int tick, tapewidth;
  char *string;

  XtVaGetValues (DrawingArea, XmNheight,     &height,
                              XmNwidth,      &width,
                              XmNforeground, &fg,
                              XmNbackground, &bg,
                 NULL );

  /* Contrasting shadow */
  if ( isColorVisual(dpy) ) {
    Pixel top, bot;

    XmGetColors ( XtScreen (DrawingArea), GetColormap(), bg,
                  NULL, &top, &bot, NULL );
    if ( isDarkPixel( dpy, fg ) )
      shadow = top;
    else
      shadow = bot;
  } else
    shadow = bg;

  /* 
  ** Determine length of text.
  */
  tapewidth = 32;                                 /* icon width */
  for (tick=0; tick < tickGetNum(); tick++) {
    data = tickGetDetail(tick);
    
    if ((string = tickGetNickName(tick)) == NULL)
      string = tickGetName(tick);
    LARGE_FONT
    tapewidth += XTextWidth ( cur_font, string, strlen(string) );
    tapewidth += XTextWidth ( cur_font, " ", strlen(" ") );

    if (data && data->values[DETAIL_PRICE])
      string = data->values[DETAIL_PRICE];
    else
      string = "N/A";
    tapewidth += XTextWidth ( cur_font, string, strlen(string) );
    tapewidth += XTextWidth ( cur_font, " ", strlen(" ") );

    if (data && data->values[DETAIL_CHANGE])
      string = data->values[DETAIL_CHANGE];
    else
      string = "N/A";
    tapewidth += XTextWidth ( cur_font, string, strlen(string) );
    tapewidth += XTextWidth ( cur_font, "   ", strlen("   ") );
  }

  if ((int)tapewidth < (int)width)
    tapewidth = width;

  /* Freed in scrollTape, unless tape never on, then freed here. */
  if (Drawpixmap[1-which]) {
    XFreePixmap (dpy, Drawpixmap[1-which]);
    Drawpixmap[1-which] = (Pixmap)NULL;
  }
  Drawpixmap[1-which] = XCreatePixmap ( dpy,
                               RootWindowOfScreen( XtScreen(DrawingArea)),
                               tapewidth, height,
                               DefaultDepthOfScreen( XtScreen(DrawingArea)) );

  pixheight = height; pixwidth = tapewidth;

  /*  Clear pixmap with background color */
  XSetForeground ( dpy, gc, bg );
  XFillRectangle ( dpy, Drawpixmap[1-which], gc,
		   0, 0, pixwidth, pixheight );

  /* Copy icon onto tape */
  XSetClipMask ( dpy, gc, GetPixmap(PSICON,MASK, dpy));
  XSetClipOrigin ( dpy, gc, 0, (height>=32)?(height/2 - 16):0 );
  XCopyArea ( dpy, GetPixmap(PSICON, NORMAL, dpy), Drawpixmap[1-which],
	      gc, 0, 0, 32, 32, 0, (height>=32)?(height/2-16):0 );
  XSetClipMask ( dpy, gc, None);
  x = 32;

  for (tick=0; tick < tickGetNum(); tick++) {
    data = tickGetDetail(tick);
    
    if ((string = tickGetNickName(tick)) == NULL)
      string = tickGetName(tick);
    y = height/2 + cur_font->descent;
    LARGE_FONT
    XSetForeground ( dpy, gc, shadow );
    XDrawString ( dpy, Drawpixmap[1-which], gc, 
                  x+1, y+1, string, strlen(string));
    XSetForeground ( dpy, gc, fg );
    XDrawString ( dpy, Drawpixmap[1-which], gc, 
                  x, y, string, strlen(string) );
    x += XTextWidth ( cur_font, string, strlen(string) );
    x += XTextWidth ( cur_font, " ", strlen(" ") );

    if (data && data->values[DETAIL_PRICE])
      string = data->values[DETAIL_PRICE];
    else
      string = "N/A";
    y += cur_font->descent;
    XSetForeground ( dpy, gc, shadow );
    XDrawString ( dpy, Drawpixmap[1-which], gc, 
                  x+1, y+1, string, strlen(string));
    XSetForeground ( dpy, gc, fg );
    XDrawString ( dpy, Drawpixmap[1-which], gc, 
                  x, y, string, strlen(string));
    x += XTextWidth ( cur_font, string, strlen(string) );
    x += XTextWidth ( cur_font, " ", strlen(" ") );

    if (data && data->values[DETAIL_CHANGE]) {
      string = data->values[DETAIL_CHANGE];
      if (strchr (string, '-') )
        color = GetColor(RED);
      else if (strpbrk (string, "123456789") )
        color = GetColor(GREEN);
      else
        color = GetColor(GREY);
    } else {
      string = "N/A";
      color = fg;
    }
    XSetForeground ( dpy, gc, shadow );
    XDrawString ( dpy, Drawpixmap[1-which], gc, 
                  x+1, y+1, string, strlen(string) );
    XSetForeground ( dpy, gc, color );
    XDrawString ( dpy, Drawpixmap[1-which], gc, 
                  x, y, string, strlen(string) );
    x += XTextWidth ( cur_font, string, strlen(string) );
    x += XTextWidth ( cur_font, "   ", strlen("   ") );

  }
}

void initTape ( Widget Drawing, GC Gc ) {
  DrawingArea = Drawing;
  gc = Gc;
}

/*
** Session save/restore functions
*/
void tapeSetScroll (int smooth, int delay)
{
  if (smooth)
    offset = 1;
  else
    offset = 2;

  if (delay)
    speed = delay;    
}

void tapeGetScroll (int *smooth, int *delay)
{
  if (offset == 1)
    *smooth = 1;
  else
    *smooth = 0;

  *delay = speed;
}

/* ARGSUSED */
void drawTape (Widget Drawing, XtPointer client_data, XtPointer call_data)
{
  static int firsttime = 1;

  if (firsttime) {
    /* Turn off drawing area widget gravity so smaller windows generate expose
    ** events.
    */
    XSetWindowAttributes attrs;
    unsigned long mask = CWBitGravity;
    attrs.bit_gravity = ForgetGravity;
    XChangeWindowAttributes ( XtDisplay (Drawing), XtWindow (Drawing),
                              mask, &attrs );

    XtVaSetValues (Drawing, XmNunitType, XmPIXELS, NULL);
    firsttime = 0;

    /* Refresh display, install timer */
    which = 1;          /* makeTextTape always makes the "other" one */
    makeTextTape ();    /* make '0' */
    which = 0;          /* use '0' */
    scrollTape ( False, (XtIntervalId) NULL); 

  } else {
    /* Refresh display, don't install timer */
    makeTextTape ();
    scrollTape ( True, (XtIntervalId) NULL); 
  }
}

/*
** Tape option support
*/

/* Control for tape display stop, play, smooth, speed controls */
/* ARGSUSED */
void procTape (Widget w, int which, XtPointer call_data)
{
  XmToggleButtonCallbackStruct *state =
         (XmToggleButtonCallbackStruct *) call_data;

  switch (which) {

	  case 0: /* popdown */
		  XtPopdown (GetTopShell(w));
		  break;

	  case 1: /* play */
		  if (!running) {
		    running = True;
		    scrollTape ( False, (XtIntervalId) NULL);
		  }
		  break;

	  case 2: /* stop */
		  if (running)
		    running = False;
		  break;

	  case 3: /* smooth */
		  if (state->set && offset == 2) {
		    offset = 1;
                  } else if (offset == 1) {
		    offset = 2;
                  }
		  break;

	  case 4: /* speed */
  		  XmScaleGetValue ( w, &speed );
		  if (speed == 0)
		    speed = 1;
		  break;

	  default: break;
  }
}
