/*
 * Copyright (c) 1997, 2001, Mark Buser.
 * Copyright (c) 2001, 2003, 2004, Danny Backx.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the names the authors (see above), nor the names of other
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Header: /pack/anoncvs/xinvest/src/optupdate.c,v 1.15 2004/05/01 14:09:52 danny Exp $
 */
#include <sys/types.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <time.h>

#include <Xm/XmAll.h>

#include "optnet.h"
#include "opttick.h"
#include "optupdate.h"
#include "status.h"
#include "xutil.h"
#include "xquote.h"

#ifndef XQUOTE
#include "xinvest.h"
#endif

/* Given we're not updating today, when is the next day and
** start time for an update.
*/ 
#define NEXT_UPDATE_DAY(days,start,curr,next)          \
        next->tm_year = curr->tm_year;                 \
        next->tm_mon = curr->tm_mon;                   \
        if (days == 1) {                               \
          /* Next update day is tomorrow */            \
          next->tm_mday = curr->tm_mday +1;            \
        } else {                                       \
          /* Next update day is Monday */              \
          if (curr->tm_wday == 5)      /* Fri */       \
            next->tm_mday = curr->tm_mday +3;          \
	  else if (curr->tm_wday == 6) /* Sat */       \
            next->tm_mday = curr->tm_mday +2;          \
          else                         /* Sun-Thur */  \
            next->tm_mday = curr->tm_mday +1;          \
        }                                              \
        next->tm_hour = start;                         \
        next->tm_min = 0;                              \
        next->tm_sec = 0;

extern XtWorkProcId work;
extern int netFinished;

/* Not sure if this belongs here FIX ME */
char time_msg[80];

/* Local Globals */
static XtIntervalId timer;

static Widget optmenus[3];

static int autocycle = XmUNSET;
static int running = False;
static int minutes = 0;
static int days = 0;    /* M-F */
static int start = 9;   /* 9 AM */
static int end = 17;    /* 5 PM */

/* Pending vars */
static int pDays = 0;
static int pStart = 9;
static int pEnd = 17;

/*
** Update options support
*/


/*
** Session management functions
*/
void updateSetTime (int d, int s, int e)
{
  if (d == 0 || d == 1)
    days = pDays = d;

  if (s >= 0 && s < 24)
    start = pStart = s;

  if (e >= 0 && e < 24)
    end = pEnd = e;
}

void updateGetTime (int *d, int *s, int *e)
{
  *d = days;
  *s = start;
  *e = end;
}

void updateSetAuto (int autoUpdate, int freq)
{
  if (autoUpdate) {
    running = True;
    autocycle = XmSET;
  } else {
    running = False;
    autocycle = XmUNSET;
  }

  if (freq <= 0)
    minutes = 1;
  else
    minutes = freq;
}

void updateGetAuto (int *autoUpdate, int *freq)
{
  *autoUpdate = running;
  *freq = minutes;
}

/*
** Callbacks 
*/

/* Just make automatic frequncy text sensitive or not */

/* ARGSUSED */
static void autoUpdate (Widget w, XtPointer client_data, XtPointer call_data)
{
  XmToggleButtonCallbackStruct *state =
         (XmToggleButtonCallbackStruct *) call_data;

  if ( state->set )
    XtSetSensitive ((Widget)client_data, True);
  else
    XtSetSensitive ((Widget)client_data, False);
}

/* Process option menu changes; not official until 'ok' pressed. */

/* ARGSUSED */
static void timeUpdate (Widget w, XtPointer client_data, XtPointer call_data)
{
  Widget menu, cascade;
  WidgetList buttons;
  XmString label;

  int mnum;
  int item = (int) client_data;
  
  /* Which option menu */
  for (mnum=0; mnum < XtNumber(optmenus); mnum++) {
    XtVaGetValues (optmenus[mnum], XmNsubMenuId, &menu, NULL);
    if (menu == XtParent(w))
      break;
  }
  if (mnum == 0) pDays = item;
  if (mnum == 1) pStart = item;
  if (mnum == 2) pEnd = item;

  if (pEnd < pStart && mnum > 0) {
    if (mnum == 1) {        /* Make end at least start */
      pEnd = pStart;
      mnum = 2;
    } else if (mnum == 2) { /* Make start at least end */
      pStart = pEnd;
      mnum = 1;
    }
    item = pStart +1;       /* darned option menu titles */
    if (item > 12)
      item++;

    /* Change menu history */
    XtVaGetValues (optmenus[mnum], XmNsubMenuId, &menu, NULL);
    XtVaGetValues (menu, XmNchildren, &buttons, NULL);
    XtVaSetValues (optmenus[mnum], XmNmenuHistory, buttons[item], NULL);

    /* Change cascade button label */
    cascade = XmOptionButtonGadget (optmenus[mnum]);
    XtVaGetValues (buttons[item], XmNlabelString, &label, NULL);
    XtVaSetValues (cascade, XmNlabelString, label, NULL);

    XmUpdateDisplay (cascade);
  }
}

/*
** Real update processing.
*/

/* Print time of next update, if any */
void timeMsgUpdate (int command, time_t nextUpdate)
{
  time_t t = time ((time_t *)NULL);
  struct tm *now = localtime(&t);

  switch (command) {
    case TRIGGER_START:
         now = localtime (&nextUpdate);
         strftime (time_msg, XtNumber(time_msg),"Next update : %x %X", now);
         break;

    case TRIGGER_STOP:
         strcpy (time_msg, "  Canceled");
         break;

    default:
         strcpy (time_msg, "");
  }
}

/* Schedule next update, considering user selected valid days/times */
static time_t scheduleUpdate () {

  time_t t, nextUpdate;
  struct tm *curr, *next;

  /* Current time */
  t = time ((time_t *)NULL);
  curr = localtime(&t);

  if (days == 1 ||                                              /* M-S or */
      (days == 0 && curr->tm_wday > 0 && curr->tm_wday < 6)) {  /* M-F */
    /* Today is an update day */
    next = curr;
    if (start == end) {
      /* Always update */
      t += (time_t)(minutes * 60);
      next = localtime(&t);
      nextUpdate = t;
    } else if (curr->tm_hour < start) {
      /* Next update is at start time today */
      next->tm_hour = start;
      next->tm_min = 0;
      next->tm_sec = 0;
      nextUpdate = mktime (next);
    } else if (curr->tm_hour >= end) {
      /* Next update is start time on next update day */
      NEXT_UPDATE_DAY(days,start,curr,next)
      nextUpdate = mktime (next);
    } else {
      /* Current time in range, check update time */
      t += (time_t)(minutes * 60);
      next = localtime(&t);
      if (next->tm_hour >= end) {
	/* Next update is start time on next update day */
        NEXT_UPDATE_DAY(days,start,curr,next)
        nextUpdate = mktime (next);
      } else {
        nextUpdate = t;
      }
    }
  } else {
    next = curr;
    /* Next update is start time on next update day */
    NEXT_UPDATE_DAY(days,start,curr,next)
    nextUpdate = mktime (next);
  }

  return (nextUpdate);
}


/*
 * Record the time of last update
 */
static time_t	last_update = (time_t) 0;
void set_current_time()
{
	last_update = time((time_t *)0);
	
}

/* Start net access, no auto if command is refresh */

/* ARGSUSED */
void triggerUpdate (int command, XtIntervalId z)
{
  time_t t = time ((time_t *)NULL), nextUpdate;

  /* Busy, reschedule in 1 minute */
  if (command != TRIGGER_STOP && netFinished == False) {
    timer = XtAppAddTimeOut (XtWidgetToApplicationContext(per->Toplevel),
                 1L*60L*1000L, (XtTimerCallbackProc) triggerUpdate, 
		 (XtPointer) command);
    return;
  }

  nextUpdate = scheduleUpdate();

  switch (command) {
    case TRIGGER_STOP:
      if (work) {
        XtRemoveWorkProc (work);
        workNet (WORK_CLEAN);              /* clean up work proc */
        if (timer) {
          timer = (XtIntervalId) NULL;
          timeMsgUpdate (command, nextUpdate);
        }
      }
      break;

    case TRIGGER_START:
      if (tickGetNum() == 0) 
        write_status ("There are no ticker symbols to fetch.", ERR);
      else {
        if (timer && timer != z)    /* should be no outstanding timers */
          XtRemoveTimeOut (timer);
        timer = (XtIntervalId) NULL;
        work = XtAppAddWorkProc (XtWidgetToApplicationContext(per->Toplevel), 
                    (XtWorkProc)workNet, (XtPointer)WORK_NORMAL);

        if (running) {
          timer = XtAppAddTimeOut (XtWidgetToApplicationContext(per->Toplevel),
                    (nextUpdate-t)*1000L, (XtTimerCallbackProc)triggerUpdate, 
		    (XtPointer)TRIGGER_START);
          timeMsgUpdate (command, nextUpdate);
        }
      }
      break;

    case TRIGGER_REFRESH:
      if (tickGetNum() == 0) 
        write_status ("There are no ticker symbols to fetch.", ERR);
      else
        work = XtAppAddWorkProc (XtWidgetToApplicationContext(per->Toplevel), 
                              (XtWorkProc)workNet, (XtPointer)WORK_NORMAL);
      break;

    case TRIGGER_NEWS:
      work = XtAppAddWorkProc (XtWidgetToApplicationContext(per->Toplevel), 
                               (XtWorkProc)workNet, (XtPointer)WORK_NEWS);
      break;

    default: break;
  }
  strcpy(errmsg, time_msg);
  write_status_line (errmsg);
}


/* Verify entered params are good, kick off automatic quotes if enabled */
/* ARGSUSED */
static void procUpdate (Widget w, XtPointer which, XtPointer call_data)
{
  static Widget cycle = (Widget) NULL, min;
  int i;

  if (cycle == (Widget)NULL) {
    cycle = XtNameToWidget (GetTopShell(w), 
                         "UpdatePane.UpdateRow.UpdateAuto.form_0.button_0");
    min = XtNameToWidget (GetTopShell(w), 
                         "UpdatePane.UpdateRow.UpdateAuto.form_0.UpdateScale");
  }

  switch ((int)which) {

    case 0: /* Ok */
	  {
            time_t nextUpdate;

            XtPopdown (GetTopShell(w));

            /*
            ** Get New Params
            */
	    autocycle = XmToggleButtonGetState (cycle);
            XmScaleGetValue (min, &minutes);
            days = pDays;
            start = pStart;
            end = pEnd;

            /*
            ** Calculate next time
            */
            nextUpdate = scheduleUpdate();

            /*
            ** Configure timer
            */
	    if (running && timer) {         /* new config, halt current timer */
	      XtRemoveTimeOut (timer);
              timer = (XtIntervalId) NULL;
	    }

            if (autocycle) {
	      running = True;
              triggerUpdate (TRIGGER_START, (XtIntervalId)NULL);
            } else {
              if (running) {
	        running = False;
                timeMsgUpdate (TRIGGER_STOP, nextUpdate);
              }
            }
	    sprintf(errmsg, "");
            write_status_line (errmsg);

	  }
          break;

    case 1: /* Cancel */
            XtPopdown (GetTopShell(w));

	    /* Restore previous values */
	    XmScaleSetValue (min, minutes);
	    XmToggleButtonSetState (cycle, autocycle, True);

            for (i=0; i < XtNumber(optmenus); i++) { 
              Widget menu, cascade;
              WidgetList buttons;
              XmString label;
              int value;

              if (i==0) value = days;
              if (i==1) {
                value = start +1;
                if (value > 12)  /* darned option menu titles */
                  value++;
              }
              if (i==2) {
                value = end +1;
                if (value > 12)
                  value++;
              }

              /* Change menu history */
              XtVaGetValues (optmenus[i], XmNsubMenuId, &menu, NULL);
              XtVaGetValues (menu, XmNchildren, &buttons, NULL);
              XtVaSetValues (optmenus[i], XmNmenuHistory, buttons[value], NULL);

              /* Change cascade button label */
              cascade = XmOptionButtonGadget (optmenus[i]);
              XtVaGetValues (buttons[value], XmNlabelString, &label, NULL);
              XtVaSetValues (cascade, XmNlabelString, label, NULL);

              XmUpdateDisplay (cascade);
            }
            break;

    default: break;
  }
}

Widget createUpdateDialog ()
{
  char *frames[] =  {"UpdateAuto", "UpdateDay", "UpdateHour"};
  char *forms[] =   {"form_0", "form_1", "form_2"};
  char *buttons[] = {"button_0", "button_1"};

  Widget dialog, form, pane, row, frame, scale, button;
  int num;

  Dimension width, height, border;

  dialog = XtVaCreatePopupShell ("OptionUpdate", 
                           xmDialogShellWidgetClass,
                           GetTopShell(per->Toplevel),
                           NULL );

  pane = XtVaCreateWidget ("UpdatePane", xmPanedWindowWidgetClass, dialog,
                           XmNsashWidth, 1,
                           XmNsashHeight, 1,
                           NULL );

  row = XtVaCreateWidget ("UpdateRow", xmRowColumnWidgetClass, pane,
                           XmNnavigationType, XmNONE,
                           XmNtopAttachment, XmATTACH_FORM,
                           XmNleftAttachment, XmATTACH_FORM,
                           NULL );

  for (num = 0; num < XtNumber(frames); num ++) {
    
     frame =   XtVaCreateManagedWidget (frames[num],
                          xmFrameWidgetClass, row,
                          NULL);
     XtVaCreateManagedWidget (frames[num],
                          xmLabelGadgetClass, frame,
                          XmNchildType, XmFRAME_TITLE_CHILD,
                          XmNchildVerticalAlignment, XmALIGNMENT_CENTER,
                          NULL);
     form = XtVaCreateWidget (forms[num], 
                          xmFormWidgetClass, frame, 
                          XmNfractionBase, 5,
                          NULL );

     switch (num) {

        case 0: 
                button = XtVaCreateManagedWidget ( "button_0",
                            xmToggleButtonWidgetClass, form,
			    XmNset, (running == True),
                            XmNleftAttachment, XmATTACH_FORM,
                            XmNtopAttachment, XmATTACH_FORM,
                            XmNbottomAttachment, XmATTACH_FORM,
                            NULL );
                scale = XtVaCreateManagedWidget ("UpdateScale", 
                            xmScaleWidgetClass, form, 
			    XmNtopAttachment, XmATTACH_FORM,
			    XmNleftAttachment, XmATTACH_WIDGET,
			    XmNleftWidget, button,
			    XmNrightAttachment, XmATTACH_FORM,
			    XmNbottomAttachment, XmATTACH_FORM,
                            NULL);
                XtAddCallback ( button, XmNvalueChangedCallback,
                            (XtCallbackProc) autoUpdate, (XtPointer) scale);
		if (autocycle == XmSET) {
                  XtSetSensitive (scale, True);
		  XmScaleSetValue (scale, minutes);
		} else {
                  XtSetSensitive (scale, False);
	          XmScaleGetValue (scale, &minutes);
		}
                break;

        case 1:
                optmenus[0] = XmVaCreateSimpleOptionMenu (
                     form,"UpdateDayMenu",
                     NULL, (KeySym)NULL , days, (XtCallbackProc) timeUpdate,
                     XmVaPUSHBUTTON, NULL, NULL, NULL, NULL,
                     XmVaPUSHBUTTON, NULL, NULL, NULL, NULL,
                     XmNtopAttachment,  XmATTACH_FORM,
                     XmNleftAttachment, XmATTACH_FORM,
                     NULL);
                XtManageChild (optmenus[0]);
                break;

        case 2: {
                  char *optlabels[] =  {"UpdateHS", "UpdateHE"};
                  int mnum;

                  for (mnum=1; mnum < XtNumber(optmenus); mnum++) {
                    optmenus[mnum] = XmVaCreateSimpleOptionMenu (
                       form, optlabels[mnum-1], NULL, (KeySym)NULL, 
                       (mnum==1)?start:end, (XtCallbackProc) timeUpdate,
                       XmVaTITLE, NULL, 
                       XmVaPUSHBUTTON, NULL, NULL, NULL, NULL,
                       XmVaPUSHBUTTON, NULL, NULL, NULL, NULL,
                       XmVaPUSHBUTTON, NULL, NULL, NULL, NULL,
                       XmVaPUSHBUTTON, NULL, NULL, NULL, NULL,
                       XmVaPUSHBUTTON, NULL, NULL, NULL, NULL,
                       XmVaPUSHBUTTON, NULL, NULL, NULL, NULL,
                       XmVaPUSHBUTTON, NULL, NULL, NULL, NULL,
                       XmVaPUSHBUTTON, NULL, NULL, NULL, NULL,
                       XmVaPUSHBUTTON, NULL, NULL, NULL, NULL,
                       XmVaPUSHBUTTON, NULL, NULL, NULL, NULL,
                       XmVaPUSHBUTTON, NULL, NULL, NULL, NULL,
                       XmVaPUSHBUTTON, NULL, NULL, NULL, NULL,
                       XmVaTITLE, NULL, 
                       XmVaPUSHBUTTON, NULL, NULL, NULL, NULL,
                       XmVaPUSHBUTTON, NULL, NULL, NULL, NULL,
                       XmVaPUSHBUTTON, NULL, NULL, NULL, NULL,
                       XmVaPUSHBUTTON, NULL, NULL, NULL, NULL,
                       XmVaPUSHBUTTON, NULL, NULL, NULL, NULL,
                       XmVaPUSHBUTTON, NULL, NULL, NULL, NULL,
                       XmVaPUSHBUTTON, NULL, NULL, NULL, NULL,
                       XmVaPUSHBUTTON, NULL, NULL, NULL, NULL,
                       XmVaPUSHBUTTON, NULL, NULL, NULL, NULL,
                       XmVaPUSHBUTTON, NULL, NULL, NULL, NULL,
                       XmVaPUSHBUTTON, NULL, NULL, NULL, NULL,
                       XmVaPUSHBUTTON, NULL, NULL, NULL, NULL,
                       NULL);
                    XtManageChild (optmenus[mnum]);

                    if (mnum==1)
                      XtVaSetValues (optmenus[mnum], 
                                     XmNleftAttachment, XmATTACH_FORM,
                                     NULL);
                    else
                      XtVaSetValues (optmenus[mnum], 
                                     XmNleftAttachment, XmATTACH_POSITION,
                                     XmNleftPosition, 3,
                                     NULL);
                  } /* for num menus */
		}
                break;

     }
     XtManageChild (form);

  }

  XtManageChild (row);

  /* Buttons to add delete cancel exit */
  form = XtVaCreateWidget ( "ButForm", xmFormWidgetClass, pane,
                            XmNfractionBase, 5,
                            NULL );

  for (num=0; num < XtNumber(buttons); num++) {
    button = XtVaCreateManagedWidget (buttons[num], 
                                      xmPushButtonWidgetClass, form,
                                      XmNtopAttachment, XmATTACH_FORM,
                                      XmNbottomAttachment, XmATTACH_FORM,
                                      XmNleftAttachment, XmATTACH_POSITION,
                                      XmNleftPosition, 2*num+1,
                                      XmNrightAttachment, XmATTACH_POSITION,
                                      XmNrightPosition, 2*num+2,
                                      XmNshowAsDefault, (num==0)?True:False,
                                      XmNdefaultButtonShadowThickness, 1,
                                      NULL );
    XtAddCallback (button, XmNactivateCallback,
                   (XtCallbackProc) procUpdate, (XtPointer)num );
  }
  XtManageChild (form);
  XtManageChild (pane);

  /* Prevent pane from changing size */
  XtVaGetValues (dialog, 
                 XmNwidth, &width,
                 XmNheight, &height,
                 XmNborderWidth, &border,
                 NULL );

  XtVaSetValues (dialog,
                 XmNminWidth,  width +  border,
                 XmNmaxWidth,  width +  border,
                 XmNminHeight, height + border,
                 XmNmaxHeight, height + border,
                 NULL );
  return (dialog);
}
