/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef SHARED_H
#define SHARED_H    1

#include "common.h"
#include "fork.h"

#include <complex.h>
#include <gtk/gtk.h>

#undef _Complex_I
#define _Complex_I  (__extension__ 1.0i)
#undef I
#define I _Complex_I

/*------------------------------------------------------------------------*/

/*** Structs encapsulating global ("common") variables ***/
/* common  /crnt/ */
typedef struct
{
  double
    *air,   /* Ai/lambda, real part */
    *aii,   /* Ai/lambda, imaginary part */
    *bir,   /* Bi/lambda, real part */
    *bii,   /* Bi/lambda, imaginary part */
    *cir,   /* Ci/lambda, real part */
    *cii;   /* Ci/lambda, imaginary part */

  complex double *cur; /* Amplitude of basis function */

  char newer; /* New data available */
  char valid; /* Available data valid */

} crnt_t;

/* common  /data/ (geometry data) */
typedef struct
{
  int
    n,      /* Number of wire segments */
    np,     /* Number of wire segments in symmetry cell */
    m,      /* Number of surface patches */
    mp,     /* Number of surface patches in symmetry cell */
    npm,    /* = n+m  */
    np2m,   /* = n+2m */
    np3m,   /* = n+3m */
    ipsym,  /* Symmetry flag */
    *icon1, /* Segments end 1 connection */
    *icon2, /* Segments end 2 connection */
    *itag;  /* Segments tag number */

  /* Wire segment data */
  double
    *x1, *y1, *z1,  /* End 1 coordinates of wire segments */
    *x2, *y2, *z2,  /* End 2 coordinates of wire segments */
    *x, *y, *z,     /* Coordinates of segment centers */
    *si, *bi,       /* Length and radius of segments  */
    *cab,           /* cos(a)*cos(b) */
    *sab,           /* cos(a)*sin(b) */
    *salp,          /* Z component - sin(a) */

    /* Surface patch data */
    *t1x, *t1y, *t1z,   /* Coordinates of t1 vector */
    *t2x, *t2y, *t2z,   /* Coordinates of t2 vector */
    *px, *py, *pz,      /* Coordinates of patch center */
    *pbi,               /* Patch surface area */
    *psalp,             /* Z component - sin(a) */

    /* Wavelength in meters */
    wlam;

  /* My addition, for plotting patches */
  double
    *px1, *py1, *pz1,
    *px2, *py2, *pz2;

} data_t;

/* common  /dataj/ */
typedef struct
{
  int
    iexk,
    ind1,
    indd1,
    ind2,
    indd2,
    ipgnd;

  double
    s,
    b,
    xj,
    yj,
    zj,
    cabj,
    sabj,
    salpj,
    rkh,
    t1xj,
    t1yj,
    t1zj,
    t2xj,
    t2yj,
    t2zj;

  complex double
    exk,
    eyk,
    ezk,
    exs,
    eys,
    ezs,
    exc,
    eyc,
    ezc;

} dataj_t;

/* common  /fpat/ */
typedef struct
{
  int
    near,
    nfeh,
    nrx,
    nry,
    nrz,
    nth,
    nph,
    ipd,
    iavp,
    inor,
    iax,
    ixtyp;

  /* xpr6 moved to exec_data_t */
  double
    thets,
    phis,
    dth,
    dph,
    rfld,
    gnor,
    clt,
    cht,
    epsr2,
    sig2,
    pinr,
    pnlr,
    ploss,
    xnr,
    ynr,
    znr,
    dxnr,
    dynr,
    dznr;

} fpat_t;

/*common  /ggrid/ */
typedef struct
{
  int
    *nxa,
    *nya;

  double
    *dxa,
    *dya,
    *xsa,
    *ysa;

  complex double
    epscf,
    *ar1,
    *ar2,
    *ar3;

} ggrid_t;

/* common  /gnd/ */
typedef struct
{
  int
    ksymp,  /* Ground flag */
    ifar,   /* Int flag in RP card, for far field calculations */
    iperf,  /* Type of ground flag */
    nradl,  /* Number of radials in ground screen */
    gpflag; /* The gpflag from the GE card */

  double
    t2,     /* Const for radial wire ground impedance */
    cl,     /* Distance in wavelengths of cliff edge from origin */
    ch,     /* Cliff height in wavelengths */
    scrwl,  /* Wire length in radial ground screen normalized to w/length */
    scrwr;  /* Radius of wires in screen in wavelengths */

  complex double
    zrati,  /* Ground medium [Er-js/wE0]^-1/2 */
    zrati2, /* As above for 2nd ground medium */
    t1,     /* Const for radial wire ground impedance */
    frati;  /* (k1^2-k2^2)/(k1^2+k2^2), k1=w(E0Mu0)^1/2, k1=k2/ZRATI */

} gnd_t;

/* common  /gwav/ */
typedef struct
{
  double
    r1, /* Distance from current element to point where field is evaluated  */
    r2, /* Distance from image of element to point where field is evaluated */
    zmh, /* Z-Z', Z is height of field evaluation point */
    zph; /* Z+Z', Z' is height of current element */

  complex double
    u,  /* (Er-jS/WE0)^-1/2 */
    u2, /* u^2 */
    xx1, /* G1*exp(jkR1*r[i])  */
    xx2; /* G2*exp(jkR2*r'[i]) */

} gwav_t;

/* common  /incom/ */
typedef struct
{
  int isnor;

  double
    xo,
    yo,
    zo,
    sn,
    xsn,
    ysn;

} incom_t;

/* common  /matpar/ (matrix parameters) */
typedef struct
{
  int
    icase,  /* Storage mode of primary matrix */
    npblk,  /* Num of blocks in first (NBLOKS-1) blocks */
    nlast,  /* Num of blocks in last block */
    imat;   /* Storage reserved in CM for primary NGF matrix A */

} matpar_t;

/* common  /netcx/ */
typedef struct
{
  int
    masym,  /* Matrix symmetry flags */
    neq,
    npeq,
    neq2,
    nonet,  /* Number of two-port networks */
    ntsol,  /* "Network equations are solved" flag */
    nprint, /* Print control flag */
    *iseg1, /* Num of seg to which port 1 of network is connected */
    *iseg2, /* Num of seg to which port 2 of network is connected */
    *ntyp;  /* Type of networks */

  double
    *x11r,  /* Real and imaginary parts of network impedances */
    *x11i,
    *x12r,
    *x12i,
    *x22r,
    *x22i,
    pin,    /* Total input power from sources */
    pnls;   /* Power lost in networks */

  complex double zped;

} netcx_t;

/* common  /save/ */
typedef struct
{
  int *ip;  /* Vector of indices of pivot elements used to factor matrix */

  double
    epsr,       /* Relative dielectric constant of ground */
    sig,        /* Conductivity of ground */
    scrwlt,     /* Length of radials in ground screen approximation */
    scrwrt,     /* Radius of wires in ground screen approximation */
    fmhz,       /* Saved frequency of operation */
    last_freq;  /* My addition, Last frequency used in calculations */

  /* My addition, replace xtemp, ytemp etc in freq. loop */
  double
    *xtemp,
    *ytemp,
    *ztemp,
    *sitemp,
    *bitemp;

  double *freq; /* My addition, frequencies used in freq loop */
  char *fstep;  /* My addition, freq loop steps that returned results */

} save_t;

/* common  /segj/ */
typedef struct
{
  int
    *jco,   /* Stores connection data */
    jsno,   /* Total number of entries in ax, bx, cx */
    maxcon; /* Max. no. connections */

  double
    *ax, *bx, *cx; /* Store constants A, B, C used in current expansion */

} segj_t;

/* common  /smat/ */
typedef struct
{
  int nop; /* My addition */

  complex double *ssx;

} smat_t;

/* common  /vsorc/ */
typedef struct
{
  int
    *isant, /* Num of segs on which an applied field source is located */
    *ivqd,  /* Num of segs on which a current-slope discontinuity source is located */
    *iqds,  /* Same as above (?) */
    nsant,  /* Number of applied field voltage sources */
    nvqd,   /* Number of applied current-slope discontinuity sources */
    nqds;   /* Same as above (?) */

  complex double
    *vqd,   /* Voltage of applied-current slope discontinuity sources */
    *vqds,  /* Same as above (?) */
    *vsant; /* Voltages of applied field voltage sources */

} vsorc_t;

/* common  /yparm/ */
/* CP card support has been dropped */

/* common  /zload/ */
typedef struct
{
  int
    *ldsegn,    /* My addition, loaded segment numbers */
    *ldtype,    /* My addition, loading type */
    nldseg,     /* My addition, number of loaded segs  */
    nload;      /* Number of loading networks */

  complex double *zarray;   /* = Zi/(Di/lambda) */

} zload_t;

/* Parameters for projecting points on screen */
typedef struct
{
  double
    Wi,         /* Angle (inclination) of Z axis to Screen, +ve to viewer */
    Wr,         /* Rotation of X-Y plane around Z axis, +ve clockwise */
    sin_wi,     /* sin(Wi) */
    cos_wi,     /* cos(Wi) */
    sin_wr,     /* sin(Wr) */
    cos_wr,     /* cos(Wr) */
    r_max,      /* Max distance from xyz origin of a point in antenna */
    xy_scale1,  /* Scale factor to fit structure into drawable window */
    xy_scale,   /* Scale factor incorporating zoom factor */
    xy_zoom,    /* Structure Zoom factor */
    x_center,   /* X co-ordinate of xyz axes origin in screen's drawable */
    dx_center,  /* Displacement of center x (move projection) */
    y_center,   /* X co-ordinate of xyz axes origin in screen's drawable */
    dy_center;  /* Displacement of center y (move projection) */

  int
    width,   /*  Width of drawable */
    height;  /* Height of drawable */

  char type; /* Type of projection parameters struct */

  gboolean reset;  /* Reset flag needed in some functions */

} projection_parameters_t;

/* My addition, a struct for frequency loop related data.
 * This is needed to implement multiple FR cards so
 * that a frequency sweep can be made in separate bands
 */
typedef struct
{
  int
    ifreq,       /* Frequency specification flag */
    freq_steps;  /* Number of frequency steps */

  double
    delta_freq,  /* Frequency step in freq loop */
    min_freq,    /* Min frequency in an FR card */
    max_freq;    /* Max frequency in an FR card */

} freq_loop_data_t;


/* My addition, struct to hold data needed
 * to execute various calculations requested
 * by commands like EX, RP, NH, NE etc.
 */
typedef struct
{
  int
    *ldtyp,
    *ldtag,
    *ldtagf,
    *ldtagt,
    nthi,
    nphi,
    iexk,
    iped,
    ngraph,     /* Number of graphs to be plotted */
    pol_type,   /* User-specified Polarization type for plots and patterns */
    num_jobs;   /* Number of child processes (jobs) to fork */

  double
    *zlr,
    *zli,
    *zlc,
    xpr1,
    xpr2,
    xpr3,
    xpr4,
    xpr5,
    xpr6,
    rkh,
    zpnorm,
    thetis,
    phiss,
    zo;     /* Characteristic impedance used in VSWR calcs */

  int
    FR_cards,     /* Number of FR cards in input file */
    FR_index,     /* Index to FR card data in use */
    freq_step,    /* Frequency step in frequency loop */
    steps_total,  /* Total number of frequency steps */
    last_step;    /* Last frequency step */

  double
    fmhz_save,  /* Saved value of frequency clicked on by user in plots window */
    freq_mhz;   /* Current Frequency in MHz, moved from save_t */

  freq_loop_data_t *freq_loop_data;

} calc_data_t;

/* Impedance data */
typedef struct
{
  double
    *zreal,     /* Real part of input impedance */
    *zimag,     /* Imag part of input impedance */
    *zmagn,     /* Magnitude of input impedance */
    *zphase;    /* Phase ang of input impedance */

} impedance_data_t;

/* Radiation pattern data */
typedef struct
{
  double
    *gtot,          /* Radiation pattern (total gain) buffer */
    *max_gain,      /* Maximum gain for each polarization type */
    *min_gain,      /* Minimum gain for each polarization type */
    *max_gain_tht,  /* Theta angle where maximum gain occurs */
    *max_gain_phi,  /*   Phi angle where maximum gain occurs */
    *tilt,          /* Tilt angle of polarization ellipse  */
    *axrt,          /* Elliptic axial ratio of pol ellipse */
    fbratio;        /* Front to Back gain ratio */

    int
    *max_gain_idx,  /* Where in rad_pattern.gtot the max value occurs */
    *min_gain_idx,  /* Where in rad_pattern.gtot the min value occurs */
    *sens;          /* Polarization sense (vertical, horizontal, elliptic etc) */

} rad_pattern_t;

/* Near E/H field data */
typedef struct
{
  double
    /* Magnitude and phase of E/H fields */
    *ex, *ey, *ez, *hx, *hy, *hz,
    *fex, *fey, *fez, *fhx, *fhy, *fhz,
    /* Real part of E and H field values */
    *erx, *ery, *erz, *er,
    *hrx, *hry, *hrz, *hr;

  /* Max of E/H field values */
  double
    max_er, max_hr;

  /* Co-ordinates of field points and
   * max distance from xyz origin */
  double
    *px, *py, *pz, r_max;

  /* Animation step in rads */
  double anim_step;

  char newer; /* New data available */
  char valid; /* Available data valid */

} near_field_t;

/* Forked processes data */
typedef struct
{
  pid_t child_pid;          /* Child pid */
  int pnt2child_pipe[2];    /* Parent-to-child write pipe */
  int child2pnt_pipe[2];    /* Child-to-parent write pipe */
  char busy;                /* Child process busy flag */
  int fstep;                /* Frequency step assigned to child */

  /* File descriptor sets for select() */
  fd_set read_fds;
  fd_set write_fds;

} forked_proc_data_t;

/* Runtime configuration data */
typedef struct
{
  /* Current working directory */
  char working_dir[FILENAME_LEN];

  char xnec2c_glade[FILENAME_LEN];

  /* Current NEC2 input file */
  char input_file[FILENAME_LEN];

  /* If set true, then use GTK loops for the frequency
     loop iteration instead of spawning a pthread */
  int disable_pthread_freqloop;

  /* Main (structure) window position and size */
  int
    main_x,
    main_y,
    main_width,
    main_height;

  /* Main (structure) window state of widgets */
  int
    main_currents_togglebutton,
    main_charges_togglebutton,
    main_total,
    main_horizontal,
    main_vertical,
    main_right_hand,
    main_left_hand,
    main_loop_start;

  int
    main_rotate_spinbutton,
    main_incline_spinbutton,
    main_zoom_spinbutton;

  /* Radiation pattern window position and size */
  int
    rdpattern_x,
    rdpattern_y,
    rdpattern_width,
    rdpattern_height,
    rdpattern_zoom_spinbutton;

  /* Radiation pattern window state of widgets */
  int
    rdpattern_gain_togglebutton,
    rdpattern_eh_togglebutton,
    rdpattern_e_field,
    rdpattern_h_field,
    rdpattern_poynting_vector;

  /* Frequency plots window state of widgetsposition and size */
  int
    plots_x,
    plots_y,
    plots_width,
    plots_height;

  /* Frequency plots window state of widgets */
  int
    plots_gmax_togglebutton,
    plots_gdir_togglebutton,
    plots_gviewer_togglebutton,
    plots_vswr_togglebutton,
    plots_zrlzim_togglebutton,
    plots_zmgzph_togglebutton,
    plots_smith_togglebutton,
    plots_min_max,
    plots_s11,
    plots_clamp_vswr,
    plots_net_gain;

  /* NEC2 editor window position and size */
  int
    nec2_edit_x,
    nec2_edit_y,
    nec2_edit_width,
    nec2_edit_height;

  /* Enable Quit dialog */
  int confirm_quit;

  /* Preferred mathlib index, if available */
  int mathlib_idx, mathlib_batch_idx;

  /* See enum GAIN_SCALE */
  int gain_style;

  /* If true, then call Fit_to_Scale() on the X-axis
   * fr_plot->min_fscale/fr_plot->max_fscale values. */
  int plots_round_x_axis;

  /* if true, exit after the first frequency loop iteration */
  int batch_mode;

  /* verbose and debug levels, see console.h */
  int verbose, debug;

  /* true if ~/.xnec2c/xnec2c.conf does not exist, false otherwise */
  int first_run;

  /* optimization file write flags */
  int opt_write_csv;
  int opt_write_s1p;
  int opt_write_s2p_max_gain;
  int opt_write_s2p_viewer_gain;

  /* If for example --write-csv <file> is added to the cmdline then write
   * the file after each frequency iteration */
  char *filename_csv;
  char *filename_s1p;
  char *filename_s2p_max_gain;
  char *filename_s2p_viewer_gain;

} rc_config_t;

/* Runtime config file */
extern rc_config_t rc_config;

extern char *orig_numeric_locale;

/* Editor windows */
extern GtkWidget
  *wire_editor,     /* Wire designer window  */
  *arc_editor,      /* Arc designer window   */
  *helix_editor,    /* Helix designer window */
  *patch_editor,    /* Patch designer window */
  *reflect_editor,  /* Reflect design window */
  *scale_editor,    /* Scale designer window */
  *cylinder_editor, /* Cylinder designer window  */
  *transform_editor,/* Transform designer window */
  *gend_editor;     /* Geom End designer window  */

/* Command windows */
extern GtkWidget
  *excitation_command,  /* Ecitation command editor */
  *frequency_command,   /* Frequency command editor */
  *ground_command,      /* Ground command editor (GN) */
  *ground2_command,     /* Ground command editor (GD) */
  *loading_command,     /* Loading command editor   */
  *network_command,     /* Network command editor   */
  *txline_command,      /* Tran Line command editor */
  *radiation_command,   /* Rad Pattern command editor */
  *nearfield_command,   /* Near Fields command editor */
  *kernel_command,      /* Thin-wire Kernel command editor  */
  *intrange_command,    /* Interaction Range command editor */
  *execute_command,     /* Execute (EX card) command editor */
  *zo_command;          /* Tx Line Zo (ZO card) command editor */

/* Frequency step entry */
extern GtkEntry *rdpattern_fstep_entry ;

/* Radiation pattern rotation, freq and zoom spin buttons */
extern GtkSpinButton
  *rotate_rdpattern,
  *incline_rdpattern,
  *rotate_structure,
  *incline_structure,
  *rdpattern_frequency,
  *rdpattern_zoom,
  *structure_zoom;

/* Commands between parent and child processes */
extern char *fork_commands[];

/* Forked process data */
extern forked_proc_data_t **forked_proc_data;

extern projection_parameters_t
  rdpattern_proj_params,
  structure_proj_params;

/* Number of forked child processes */
extern int num_child_procs;

/* Global lock to prevent nested execution of functions like
   Frequency_Loop(), Open_Input_File(), and possibly others. */
extern GMutex global_lock;

/* Lock for frequency data to prevent use of data
   populated by Get_Freq_Data() and New_Frequency()
   before it is done filling the data buffers.  */
extern GMutex freq_data_lock;

/* Program forked flag */
extern gboolean FORKED;

/* Main window freq spinbutton */
extern GtkSpinButton *mainwin_frequency;

/* Scroll adjustments of geometry
 * and command treeview windows */
extern GtkAdjustment
  *geom_adjustment,
  *cmnd_adjustment ;

extern GtkTreeView
  *cmnt_treeview,
  *geom_treeview,
  *cmnd_treeview ;

/* Main, frequency plots and radiation pattern windows */
extern GtkWidget
  *main_window,
  *plots_window,
  *rdpattern_window,
  *nec2_edit_window;

extern GtkBuilder
  *main_window_builder,
  *plots_window_builder,
  *rdpattern_window_builder,
  *animate_dialog_builder,
  *quit_dialog_builder,
  *excitation_editor_builder,
  *radiation_editor_builder,
  *frequency_editor_builder,
  *ground2_editor_builder,
  *ground_editor_builder,
  *loading_editor_builder,
  *network_editor_builder,
  *txline_editor_builder,
  *nearfield_editor_builder,
  *kernel_editor_builder,
  *intrange_editor_builder,
  *zo_editor_builder,
  *execute_editor_builder,
  *wire_editor_builder,
  *arc_editor_builder,
  *helix_editor_builder,
  *patch_editor_builder,
  *reflect_editor_builder,
  *scale_editor_builder,
  *cylinder_editor_builder,
  *transform_editor_builder,
  *gend_editor_builder,
  *nec2_editor_builder,
  *nec2_save_dialog_builder;

/* Drawing area widgets */
extern GtkWidget
  *structure_drawingarea,
  *plots_drawingarea,
  *rdpattern_drawingarea;

extern GtkWidget *saveas_drawingarea;

/* Sizes of above */
extern int
  structure_width,
  structure_height,
  plots_width,
  plots_height,
  rdpattern_width,
  rdpattern_height;

/* Dialog widgets */
extern GtkWidget *quit_dialog;
extern GtkWidget *animate_dialog;
extern GtkWidget *error_dialog;
extern GtkWidget *nec2_save_dialog;

/* File chooser/select widgets */
extern GtkWidget *file_chooser;

/* Tree view clicked on by user */
extern GtkTreeView *selected_treeview;

/* Used to kill window deleted by user */
extern GtkWidget *kill_window ;

/* Animation timeout callback tag */
extern guint anim_tag;

/* Frequency loop idle function tag */
extern guint floop_tag;

/* Radiation pattern data */
extern rad_pattern_t *rad_pattern ;

/* Near E/H field data */
extern near_field_t near_field;

typedef struct Segment
{
  gint x1;
  gint x2;
  gint y1;
  gint y2;
} Segment_t;

/* Segments for drawing structure */
extern Segment_t *structure_segs ;

/* Global tag number for geometry editors */
extern gint tag_num;

/* Tree list stores */
extern GtkListStore
  *cmnt_store,
  *geom_store,
  *cmnd_store ;

typedef struct
{
  void (*callback)(char *);
  char *extension;
} filechooser_t;

extern filechooser_t *filechooser_callback;

extern complex double *cm;

/* Frequency step entry widget */
extern GtkEntry *structure_fstep_entry;

/* pointers to input/output files */
extern FILE *input_fp, *output_fp, *plot_fp;

/* Magnitude of seg/patch current/charge */
extern double *cmag, *ct1m, *ct2m;

/* Needed data */
extern impedance_data_t impedance_data;

/* Data for various calculations */
extern calc_data_t calc_data;

/* common  /crnt/ */
extern crnt_t crnt;

/* common  /dataj/ */
extern dataj_t dataj;

/* common  /data/ */
extern data_t data;

/* common  /fpat/ */
extern fpat_t fpat;

/*common  /ggrid/ */
extern ggrid_t ggrid;

/* common  /gnd/ */
extern gnd_t gnd;

/* common  /gwav/ */
extern gwav_t gwav;

/* common  /incom/ */
extern incom_t incom;

/* common  /matpar/ */
extern matpar_t matpar;

/* common  /netcx/ */
extern netcx_t netcx;

/* common  /save/ */
extern save_t save;

/* common  /segj/ */
extern segj_t segj;

/* common  /smat/ */
extern smat_t smat;

/* common  /vsorc/ */
extern vsorc_t vsorc;

/* common  /zload/ */
extern zload_t zload;

/* Segments for drawing structure */
extern Segment_t *structure_segs;

#define MOTION_EVENTS_COUNT   8

#endif
