/*
 * xmms-mad - mp3 plugin for xmms
 * Copyright (C) 2001-2002 Sam Clegg - See COPYING
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $superduper: decoder.c,v 1.9 2004/05/16 17:08:05 sam Exp $
 */
#include <assert.h>
#include <pthread.h>
#include <signal.h>

#include "xmms-mad.h"
#include "input.h"

#define BUFFER_SIZE 16*1024
/*#define DEBUG*/

/**
 * Scale PCM data
 */
static inline signed int
scale (mad_fixed_t sample)
{
  /* round */
  sample += (1L << (MAD_F_FRACBITS - 16));

  /* clip */
  if (sample >= MAD_F_ONE)
    sample = MAD_F_ONE - 1;
  else if (sample < -MAD_F_ONE)
    sample = -MAD_F_ONE;

  /* quantize */
  return sample >> (MAD_F_FRACBITS + 1 - 16);
}

void
write_output (struct mad_info_t *info, struct mad_pcm *pcm,
              struct mad_header *header)
{
  unsigned int nsamples;
  mad_fixed_t const *left_ch, *right_ch;
  char *output;
  int olen = 0;
  int pos = 0;

  nsamples = pcm->length;
  left_ch = pcm->samples[0];
  right_ch = pcm->samples[1];
  olen = nsamples * MAD_NCHANNELS (header) * 2;
  output = (char *) g_malloc (olen * sizeof (char));

  while (nsamples--)
    {
      signed int sample;
      /* output sample(s) in 16-bit signed little-endian PCM */
      sample = scale (*left_ch++);
      output[pos++] = (sample >> 0) & 0xff;
      output[pos++] = (sample >> 8) & 0xff;

      if (MAD_NCHANNELS (header) == 2)
        {
          sample = scale (*right_ch++);
          output[pos++] = (sample >> 0) & 0xff;
          output[pos++] = (sample >> 8) & 0xff;
        }
    }
  assert (pos == olen);
  mad_plugin.add_vis_pcm (mad_plugin.output->written_time (),
                          FMT_S16_LE, MAD_NCHANNELS (header), olen, output);
  while (mad_plugin.output->buffer_free () < olen && !info->stop)
    usleep (10000);
  mad_plugin.output->write_audio (output, olen);
  g_free (output);
}

/**
 * Decode all headers in the file and fill in stats
 * @return FALSE if scan failed.
 */
gboolean
scan_file (struct mad_info_t *info, gboolean fast)
{
  struct mad_stream stream;
  struct mad_header header;
  int remainder = 0;
  int len = 0;
  unsigned char buffer[BUFFER_SIZE];
  struct mad_frame frame; /* to read xing data */
  gboolean has_xing = FALSE;

  mad_stream_init (&stream);
  mad_header_init (&header);
  mad_frame_init (&frame);
  xing_init (&info->xing);

  info->bitrate = 0;
  info->pos = mad_timer_zero;

#ifdef DEBUG
  g_message ("f: scan_file");
#endif /* DEBUG */

  while (1)
    {
      remainder = stream.bufend - stream.next_frame;
      memcpy (buffer, stream.this_frame, remainder);
      len = input_get_data (info, buffer + remainder,
                            BUFFER_SIZE - remainder);
      if (len <= 0)
        break;

      mad_stream_buffer (&stream, buffer, len + remainder);

      while (1)
        {
	  if (mad_header_decode (&header, &stream) == -1)
	    {
	      if (stream.error == MAD_ERROR_BUFLEN)
		{
		  break;
		}
	      if (!MAD_RECOVERABLE (stream.error))
		{
#ifdef DEBUG
		  g_message ("unrecoverable error decoding header %d: %s",
		              info->frames, mad_stream_errorstr (&stream));
		  g_message ("remainder = %d", remainder);
		  g_message ("len = %d", len);
#endif /* DEBUG */
		  break;
		}
	      if (stream.error == MAD_ERROR_LOSTSYNC)
		{
		  /* ignore LOSTSYNC due to ID3 tags */
		  int tagsize = id3_tag_query (stream.this_frame,
		      stream.bufend -
		      stream.this_frame);
		  if (tagsize > 0)
		    {
		      mad_stream_skip (&stream, tagsize);
		      continue;
		    }
		}
#ifdef DEBUG
	      g_message ("error decoding header %d: %s",
		         info->frames, mad_stream_errorstr (&stream));
	      g_message ("remainder = %d", remainder);
	      g_message ("len = %d", len);
#endif /* DEBUG */
	      continue;
	    }
	  info->frames++;
          mad_timer_add (&info->duration, header.duration);
          if (info->frames == 1) 
	    {
	      /* most of these *should* remain constant */
	      info->bitrate = header.bitrate;
	      info->freq = header.samplerate;
	      info->channels = MAD_NCHANNELS(&header);
	      info->mpeg_layer = header.layer;
	      info->mode = header.mode;

	      if (xmmsmad_config.use_xing)
	        {
                  frame.header = header;
                  if (mad_frame_decode(&frame, &stream) == -1)
                    break;
	          if (xing_parse (&info->xing, stream.anc_ptr, stream.anc_bitlen) == 0) 
	            {
#ifdef DEBUG
	              g_message ("found xing header");
#endif /* DEBUG */
                      has_xing = TRUE;
                      info->vbr = TRUE; /* otherwise xing header would have been 'Info' */
		      info->frames = info->xing.frames;
		      mad_timer_multiply (&info->duration, info->frames);
                      info->bitrate = 8.0 * info->xing.bytes / mad_timer_count(info->duration, MAD_UNITS_SECONDS); 
                      break;
                    }
	        }

	    }
	  else 
	    {
	      /* perhaps we have a VRB file */
	      if (info->bitrate != header.bitrate)
		info->vbr = TRUE;
	      if (info->vbr)
		info->bitrate += header.bitrate;
	      /* check for changin layer/samplerate/channels */
	      if (info->mpeg_layer != header.layer)
	 	g_warning ("layer varies!!");
	      if (info->freq != header.samplerate)
		g_warning ("samplerate varies!!");
	      if (info->channels != MAD_NCHANNELS(&header))
		g_warning ("number of channels varies!!");
	    }

	  if (fast && info->frames > 10)
	    {
#ifdef DEBUG
	       g_message ("using fast playtime calculation");
#endif /* DEBUG */
	       info->frames = info->size / (stream.next_frame - stream.this_frame);
	       mad_timer_multiply (&info->duration, info->frames);
	       break;
	    }
	}
      if (stream.error != MAD_ERROR_BUFLEN)
	break;
    }

  if (info->vbr && !has_xing)
    info->bitrate = info->bitrate / info->frames;

  mad_frame_finish (&frame);
  mad_header_finish (&header);
  mad_stream_finish (&stream);
  xing_finish (&info->xing);

#ifdef DEBUG
  g_message ("e: scan_file");
#endif /* DEBUG */
  return info->frames != 0;
}

void *
decode (void *arg)
{
  char buffer[BUFFER_SIZE];
  int len;
  int seek_skip = 0;
  int remainder = 0;

  /* mad structs */
  struct mad_stream stream;
  struct mad_frame frame;
  struct mad_synth synth;

  /* track info is passed in as thread argument */
  struct mad_info_t *info = (struct mad_info_t *) arg;

#ifdef DEBUG
  g_message ("f: decode");
#endif /* DEBUG */

  /* init mad stuff */
  mad_frame_init (&frame);
  mad_stream_init (&stream);
  mad_synth_init (&synth);

  if (!mad_plugin.output->open_audio (info->fmt, info->freq, info->channels))
    {
      xmmsmad_error ("failed to open audio output: %s",
                     mad_plugin.output->description);
      g_message ("failed to open audio output: %s", mad_plugin.output->description);
      return 0;
    }

  /* main loop */
  do
    {
      if (info->stop)
        break;
      if (seek_skip)
        remainder = 0;
      else
        {
          remainder = stream.bufend - stream.next_frame;
          memcpy (buffer, stream.this_frame, remainder);
        }
      len = input_get_data (info, buffer + remainder,
                            BUFFER_SIZE - remainder);
      if (len <= 0)
        {
#ifdef DEBUG
          g_message ("finished decoding");
#endif /* DEBUG */
          break;
        }
      len += remainder;
      if (len < MAD_BUFFER_GUARD)
        {
          int i;
          for (i = len; i < MAD_BUFFER_GUARD; i++)
            buffer[i] = 0;
          len = MAD_BUFFER_GUARD;
        }

      mad_stream_buffer (&stream, buffer, len);

      if (seek_skip)
        {
          int skip = 2;
          do
            {
              if (mad_frame_decode (&frame, &stream) == 0)
                {
                  mad_timer_add (&info->pos, frame.header.duration);
                  if (--skip == 0)
                    mad_synth_frame (&synth, &frame);
                }
              else if (!MAD_RECOVERABLE (stream.error))
                break;
            }
          while (skip);
          seek_skip = 0;
        }

      while (!info->stop)
        {
          if (info->seek != -1 && !info->remote)
            {
              int new_position;
              int seconds = mad_timer_count (info->duration, MAD_UNITS_SECONDS);
              if (info->seek >= seconds)
                info->seek = seconds;

              mad_timer_set (&info->pos, info->seek, 0, 0);
              new_position =
                  ((double) info->seek / (double) seconds) * info->size;
#ifdef DEBUG
              g_message ("seeking to: %d bytes", new_position);
#endif
              if (lseek (info->infile, new_position, SEEK_SET) == -1)
                xmmsmad_error ("failed to seek to: %d", new_position);
              mad_frame_mute (&frame);
              mad_synth_mute (&synth);
              stream.error = MAD_ERROR_BUFLEN;
              mad_plugin.output->flush (mad_timer_count (info->pos, MAD_UNITS_MILLISECONDS));
              stream.sync = 0;
              info->seek = -1;
              seek_skip = 1;
              break;
            }

          if (mad_header_decode (&frame.header, &stream) == -1)
            {
              if (!MAD_RECOVERABLE (stream.error))
                break;
              if (stream.error == MAD_ERROR_LOSTSYNC)
                {
                  /* ignore LOSTSYNC due to ID3 tags */
                  int tagsize = id3_tag_query (stream.this_frame,
                                               stream.bufend -
                                               stream.this_frame);
		      tagsize = 100;
                  if (tagsize > 0)
                    {
                      mad_stream_skip (&stream, tagsize);
                      continue;
                    }
                }
#ifdef DEBUG
              g_message ("error decoding header %d: %s",
                      info->current_frame, mad_stream_errorstr (&stream));
#endif /* DEBUG */
              continue;
            }

          if (mad_frame_decode (&frame, &stream) == -1)
            {
              if (!MAD_RECOVERABLE (stream.error))
                break;
#ifdef DEBUG
              g_message ("error decoding frame %d: %s",
                      info->current_frame, mad_stream_errorstr (&stream));
#endif /* DEBUG */
            }

          info->current_frame++;
          if (info->current_frame % 32 == 0)
            {
              mad_plugin.set_info (info->title,
                                   (int) mad_timer_count (info->duration, MAD_UNITS_MILLISECONDS), frame.header.bitrate,
                                   frame.header.samplerate, MAD_NCHANNELS(&frame.header));
            }
          if (info->freq != frame.header.samplerate 
              || info->channels != (guint) MAD_NCHANNELS(&frame.header))
            {
#ifdef DEBUG
              g_message ("re-opening audio due to change in audio type");
#endif /* DEBUG */
              info->freq = frame.header.samplerate;
              info->channels = MAD_NCHANNELS(&frame.header);
              mad_plugin.output->close_audio ();
              if (!mad_plugin.output->open_audio (info->fmt, info->freq, info->channels))
                {
                  xmmsmad_error ("failed to re-open audio output: %s",
                    mad_plugin.output->description);      
                }
            }
          mad_synth_frame (&synth, &frame);
          mad_stream_sync (&stream);
          write_output (info, &synth.pcm, &frame.header);
          mad_timer_add (&info->pos, frame.header.duration);
        }
    }
  while (stream.error == MAD_ERROR_BUFLEN);

  /* free mad stuff */
  mad_frame_finish (&frame);
  mad_stream_finish (&stream);
  mad_synth_finish (&synth);

  if (!info->stop)
    {
      mad_plugin.output->buffer_free ();
      mad_plugin.output->buffer_free ();
      while (mad_plugin.output->buffer_playing ())
        usleep (10000);
    }

#ifdef DEBUG
  g_message ("e: decode");
#endif /* DEBUG */

  mad_plugin.output->close_audio ();
  info->stop = 1;
  pthread_exit (0);
}
