/*  $Id: date.c,v 4.5 1996/01/14 16:34:08 lupus Exp lupus $  */

/*
 * date.c
 * edited in vi w/ ts=4.
 *
 * format-independant date conversion and arrithmatic code.
 *
 * year, month, day are passed around as ints.
 * date's are pushed around as Date (typedef in date.h).
 *
 * code written with an eye toward making it 64-bit clean.  since i
 * don't [yet] have an alpha we'll have to wait for the acid test...
 *
 * notes on modular math:
 *
 * to get next item on 1...N -> (x % N)+1;
 * to get prev item on 1...N -> ((x + (2N-2)) % N)+1;
 *
 * everything but makedate else uses const arguments.  this makes it 
 * easier to convert this stuff to a struct-based date at some point.
 *
 * the only things that know what a Date is are: year(), month(), 
 * day() and date().  all of the increment/decrement functions
 * simply return makedate() w/ offset values of year(), etc.  e.g.,
 * incr() => return makedate( year(date), month(date), day(date)+1 );
 * the performance cost on my machine (bogomips=39) is < 1% and this
 * does make the code quite a bit simpler... :-)
 *
 * note: customers should call makedate() *not* date()!  makedate()
 * handles all of the overflow conditions (e.g., 32-Jan) and 
 * normalises the values.
 */

#include <time.h>		/* time(), localtime(), time_t, struct tm */
#include <stdlib.h>		/* NULL, exit() */
#include <stdio.h>		/* NULL on some systems */

#include "date.h"		/* prototypes */
#include "xfinans.h"		/* constants  */

#define MAX_YEAR    3000	/* unless your mortgage is over 1000 years.. */
#define MIN_YEAR    1000	/* ditto */


/* 
	used by year(), month(), day(), makedate().

	note: masks defined for `and then shift' use.  
	this is less error prone than counting zeros to get the
	correct bits for masking.  also saves changing the masks
	if the shift values are changed later on.

	to extract:   element = (date >> elementshift) & elementmask );
	to construct: result  = (element & element mask) << elementshift );
*/

#define YEARMASK  ((unsigned long)(0xfff))		/*  12 bits */
#define MONTHMASK ((unsigned long)(0x0ff))		/*   8 bits */
#define DAYMASK   ((unsigned long)(0x0ff))		/* + 8 bits */
												/* -------  */
#define DATEMASK  ((unsigned long)(0x0fffffff))	/*  28 bits */

#define DAYSHIFT   ((int) 0)	/* skip over nothing */
#define MONTHSHIFT ((int) 8)	/* skip over day == 8 bits */
#define YEARSHIFT  ((int)16)	/* skip over day + month = 8 + 8 bits */

/*
	use: monthdays[leapyear(year)][month]
*/
static int monthdays[2][13] =
	{
		{ 0, 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31},
		{ 0, 31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31},
	};

/*
	gcc unhappy w/ non-const init value for max_date...
*/

static Date min_date =  (((MIN_YEAR & YEARMASK)  << YEARSHIFT  ) |
			 ((       1 & MONTHMASK) << MONTHSHIFT ) |
			 ((       1 & DAYMASK)   << DAYSHIFT   ) );


static Date max_date =  (((MAX_YEAR & YEARMASK)  << YEARSHIFT  ) |
			 ((      12 & MONTHMASK) << MONTHSHIFT ) |
			 ((      31 & DAYMASK)   << DAYSHIFT   ) );

/*
 * assemble and disassemble Date from year, month, day.
 * 
 * there are some extra operations in year() and day().
 * they are minor, however, and i figured making them more alike
 * left the code easier to read.  ditto date().
 *
 * these four functions are the only ones which know the internal 
 * structure of a Date.
 */

int year ( const Date date ) { return ( (date >> YEARSHIFT)  & YEARMASK ); }
int month( const Date date ) { return ( (date >> MONTHSHIFT) & MONTHMASK); }
int day  ( const Date date ) { return ( (date >> DAYSHIFT)   & DAYMASK  ); }

/*
 *	this should *not* be called externally
 */

static Date date ( const int yr, const int mo, const int dy )
{
	extern Date min_date;
	extern Date max_date;

	/*
		looks prettier in debugger built step-by-step.
	*/
	register Date tmp = (Date)0;

	tmp |= ((yr	& YEARMASK)  << YEARSHIFT  );
	tmp |= ((mo	& MONTHMASK) << MONTHSHIFT );
	tmp |= ((dy	& DAYMASK)   << DAYSHIFT   );

	return tmp < min_date || tmp > max_date ? (Date)0 : tmp ;
}


/*
 *	makedate() is cleanup function for dates. it does *not* know
 *	what the internal structure of a date is -- it only validates
 *	that date ranges are valid, takes care of days and months
 *	over-/under-flowing into the previous periods.
 *
 *	this puts all the validation code in one place, makes it simpler
 *	to maintain.
 *
 *	notes:
 *
 *	adds current century (via time()) to years < 100.
 *
 *	makedate( year, month, 0 ) => end of previous month w/ year wrap
 *  e.g.,
 *		makedate( 1995,  1, 0 ) => 31-Dec-1994,
 *		makedate( 1995, 12, 0 ) => 30-Nov-1995
 *	
 *	called w/ day > monthdays[leapyear(year)][month] => increment month
 *	w/ year wrap until the day is w/in the current month.
 *	e.g., 
 *		makedate( 1995, 1, 35 ) => 4-Feb-1995
 *	
 *	called w/ month > 12 => increment year until month <= 12.
 *	e.g., 
 *		makedate( 1994, 15, 3 ) => 3-Mar-1995
 *
 *	year validation is done by date().
 *
 */
Date makedate( int yr, int mo, int dy )
{
	int mdays;

	/*
		yr assumed w/in +/- 50 years from current time if < 100.
		gives clean method for handling 2-digit year input.
	*/
	if( yr < 100 ) {

	  if( yr <= this_year() - 50 )
	    yr += (this_century() + 100);
	  else if( yr > this_year() + 50 )
	    yr += (this_century() - 100);
	  else
	    yr += this_century();
	}

	/*
		handle months in the next or previous year(s)
	*/

	while( mo > 12 )
	{
		mo -= 12;
		++yr;
	}

	while( mo <= 0 )
	{
		mo += 12;
		--yr;
	}

	/* 
		handle odditys like 45-January by subtracting off
		days in months until the days are w/in the current
		month.
	*/

	while( dy > (mdays = monthdays[leapyear(yr)][mo]) )
	{
		dy -= mdays;
		mo = ( mo % 12 ) + 1;
		yr += mo == 1;
	}

	/* 
		handle negative dates: must be in a previous month,
		so count down and add the current negative value to
		the days in previous month.  usually will only go one
		pass.

		e.g.,
			-3-Feb-1995 will become (31-3)-Jan-1995 == 28-Jan-1995.
			-45-Feb-1995 goes thru (31-45)-Jan == -14-Jan which
			then goes to (31-14)-Dec == 17-Dec-1994.

		0-anymonth = last day of previous month.
	*/

	while( dy <= 0 )
	{
		mo = ( (mo + 22) % 12 ) + 1;		/* previous month */
		yr -= mo == 12;						/* wrap year in december */
		dy += monthdays[leapyear(yr)][mo];	/* offset days */
	}

	/*
		at this point:
			the month is 1 <= mo <= 12 and the date
			the day is 1 <= dy <= monthdays[leapyear(year)][month]
			the CPU knows what the year is.

		now the date can be built.  date() returns (Date)0 if
		the date is out of range.
	*/

	return date( yr, mo, dy );
}

/*
	these don't know what a date is made of, they only adjust their
	respective fields w/in the date.

	note on months:

	same or last day of next/previous month.
	decrMon w/ 31-July will give 30-June, etc.  with 31-March
	will give either 28- or 29-Feb, depending on leapyear,

	incrMon w/ 30-Jun will give 30-Jul -- which probably won't be
	what the person wants but does save 30 clicks of the [+] key...

	Current list -- prepend "incr" or "decr" for direction:

	  Day        1 day
	  Week       7 day
	  Payday     1st of next month, 15th of current month
	  Month      same date next month w/ rounding down to last day.
	  Ending     ending date of next month (i.e., 31, 2[89], 30, 31...)
	             handles month-end payments more gracefully than Month.
	  Halfyear   semiannual w/ rounding down to last day of month.
	  Halfending semiannual month ending -- handles end of month payments
	             more gracefully than Semiyear.
	  Year       one year w/ rounding down to last day of month (feb 29->28)
	             for the one day every just-about 4 years i figured it
				 wasn't worth adding a Yearending function.
*/

/*
 	these are the simple ones: just add to what's there, no logic
	required.
*/

Date incrDay( const Date date)
	{ return makedate( year(date), month(date), (day(date)+1) ); }

Date decrDay(const Date date)
	{ return makedate( year(date), month(date), (day(date)-1) ); }

Date incrWk(const Date date)
	{ return makedate( year(date), month(date), (day(date)+7) ); }

Date decrWk(const Date date)
	{ return makedate( year(date), month(date), (day(date)-7) ); }

Date incrMon(const Date date)
	{ return makedate( year(date), month(date)+1, day(date) ); }

Date decrMon(const Date date)
	{ return makedate( year(date), month(date)-1, day(date) ); }

Date incrEnding( const Date date)
	{ return makedate( year(date), month(date)+2, 0 ); }

Date decrEnding( const Date date)
	{ return makedate( year(date), month(date), 0 ); }

Date incrQuarter(const Date date)
	{ return makedate( year(date), month(date)+3, day(date) ); }

Date decrQuarter(const Date date)
	{ return makedate( year(date), month(date)-3, day(date) ); }

Date incrQending(const Date date)
	{ return makedate( year(date), month(date)+4, 0 ); }

Date decrQending(const Date date)
	{ return makedate( year(date), month(date)-2, 0 ); }

Date incrHalfyear(const Date date)
	{ return makedate( year(date), month(date)+6, day(date) ); }

Date decrHalfyear(const Date date)
	{ return makedate( year(date), month(date)-6, day(date) ); }

Date incrHalfending(const Date date)
	{ return makedate( year(date), month(date)+7, 0 ); }

Date decrHalfending(const Date date)
	{ return makedate( year(date), month(date)-5, 0 ); }

Date incrYear( const Date date)
	{ return makedate( year(date)+1, month(date), day(date) ); }

Date decrYear( const Date date)
	{ return makedate( year(date)-1, month(date), day(date) ); }

/*
	functions w/ internal logic: gotta make decisions based on what the
	values are.
*/
	
Date incrPayday( const Date date )
{
	if( day(date) < 15 )
		/* first half of the month => 15th of this month */
		return makedate( year(date), month(date), 15 );
	else
		/* second half of the month => 1st of next month */
		return makedate( year(date), month(date)+1, 1 );
}

Date decrPayday( const Date date )
{
	if( day(date) < 15 )
		/* first half of this month => 15th of last month */
		return makedate( year(date), month(date)-1, 15 );
	else
		/* second half of this month => 1st of this month */
		return makedate( year(date), month(date), 1 );
}


/*
 * useful functions: today(), this_century(), leapyear(), min().
 * these don't depend on Date format.
 *
 *	today(), this_century() assume things like TZ are set, unless you happen
 *  to live in on the date line...
 */

Date today( void )
{
	time_t now = time( (time_t *)NULL );
	struct tm *today = localtime( &now );

	return makedate( today->tm_year, today->tm_mon+1, today->tm_mday );
}

/* this_year() returns the current offset from the current century, ie. 
   0 <= this_year() <= 99
*/
int this_year( void )
{
	time_t now = time( (time_t *)NULL );
	struct tm *today = localtime( &now );

	return today->tm_year % 100;
}

/*
	this_century() returns current century (e.g., 1900, 2000).  

	note: minor bug w/ use of static variable requires user to exit
	program on 31-dec every 100 years...  it also fails if time()
	doesn't return the full value (i.e., recompile this on a 64-bit
	machine sometime before 2080).  seemed minor enough to leave as-is.
*/
int this_century( void )
{
  static int century = 0;
  
  if( !century )
    {
      time_t now = time( (time_t *)NULL );
      /*
	 time() on unix == offset to 01-Jan-1970:00:00:00
	 cent = time * hr/3600sec * dy/24hr * yr/365.25dy * cent/100yr
      */
	  
      now /= 3600;                        /* convert to hours */
      now /= 24;			      /* convert to days */
      now /= 365.25;		      /* convert to years */
      now /= 100;			      /* convert to centurys */
      
      now += 1970;			      /* from offset to absolute */
      century = now - (now % 100);	      /* round to nearest 100 */
    }
  
  return century;
}

int leapyear( const int year )
{
  if( year > 0 )
    {
      /* valid year => test it and return {0,1} */
      return ( (!(year % 4) && (year % 100)) || !(year % 400) );
    }
  else
    {
      /*
	 assume invalid years are not leap years, simplifies
	 use with monthdays[leapyear(year(date))][m] if date
	 is zero.
      */
      return 0;
    }
}

int lastday(const Date date) /* the last day in the month of "date" */
{
  return monthdays[leapyear(year(date))][month(date)];
}

Date nextDate(Date date, int periodUnit, int periodLength) {
  int i;

  for(i=0; i<periodLength; i++) 
    switch(periodUnit) {
    case P_DAY:
      date=incrDay(date);
      break;
    case P_WEEK:
      date=incrWk(date);
      break;
    case P_PAYDAY:
      date=incrPayday(date);
      break;
    case P_MONTH:
      date=incrMon(date);
      break;
    case P_MONTH_ENDING:
      date=incrEnding(date);
      break;
    case P_YEAR:
      date=incrYear(date);
      break;
    }

  return date;
}

Date prevDate(Date date, int periodUnit, int periodLength) {
  int i;

  for(i=0; i<periodLength; i++) 
    switch(periodUnit) {
    case P_DAY:
      date=decrDay(date);
      break;
    case P_WEEK:
      date=decrWk(date);
      break;
    case P_PAYDAY:
      date=decrPayday(date);
      break;
    case P_MONTH:
      date=decrMon(date);
      break;
    case P_MONTH_ENDING:
      date=decrEnding(date);
      break;
    case P_YEAR:
      date=decrYear(date);
      break;
    }

  return date;
}

static long min( long a, long b )
  { return ( a < b ? a : b ); }

/* 
 * end of date.c
 */


/* 
	use this to generate an executable test function via:

		gcc -DDEBUG_DATE ${CFLAGS} -o test_date date.c

	uses stderr for section output to allow `test_date > /dev/null'
	for time trials, etc.  headers tell you if things are looping
	forever.
*/

#ifdef DEBUG_DATE

#include <stdio.h>	/* printf(), puts(), fputs() */

/*
	quick routine to check out dates...

	printing w/ dd-Mmm-YYYY makes it easier to read streaming
	by on the screen.  using 0x%08X makes it easy to see the 
	different parts of the date changing.
*/

void printdate( Date date )
{
	static unsigned long count = 0;
	static char monthname[13][4] =
	{
		"",
		"Jan", "Feb", "Mar", "Apr", "May", "Jun",
		"Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
	};
	int yr = year( date );
	int mo = month( date );
	int dy = day( date );

	printf( "%9lu : 0x%08lX : %02d-%3.3s-%04d\n",
		++count, date, dy, monthname[mo], yr);
}

/*
 * test full range of dates for each generator.
 * using 1...31 for incr-/decrMonth() checks that the dates are
 * cut to 28 passing through feb properly.
 *
 * be forwarned: this generates about 12Mb of output!
 */
int main( void )
{
    Date (*datefuncs[])(const Date) =	/* array of date functions */
    {
	incrDay, decrDay, incrWk, decrWk,
	incrMon, decrMon, incrEnding, decrEnding,
	incrQuarter, decrQuarter, incrQending, decrQending,
	incrHalfyear, decrHalfyear, incrHalfending, decrHalfending,
	incrYear, decrYear,
	incrPayday, decrPayday,
	(void *)NULL
    };

    char *funcnames[] = 				/* array of functio names */
    {
		"incrDay", "decrDay", "incrWk", "decrWk",
		"incrMon", "decrMon", "incrEnding", "decrEnding",
		"incrQuarter", "decrQuarter", "incrQending", "decrQending",
		"incrHalfyear", "decrHalfyear", "incrHalfending", "decrHalfending",
		"incrYear", "decrYear",
		"incrPayday", "decrPayday",
		(char *)NULL
	};

	int i;									/* temporary variable */
	Date tmp;								/* temporary variable */
	Date (*thisfunc)(const Date);			/* pointer to date function */
	Date test1 = makedate( 1970, 1, 1 );
	Date test2 = makedate(   70, 1, 1 );	/* century works? */

	puts( "\nmin_date, max_date:" );
	printdate( min_date );
	printdate( max_date );

	puts( "\ntoday's date:" );
	printdate( today() );

	puts( "\ntest1, test2.  test1 =?= test2" );
	printdate( test1 );
	printdate( test2 );

	puts( "Sequence Tests:" );

	/*
		i starts at -1, first use goes to 0
	*/

	for( i = -1 ; funcnames[i] ;  )
	{
		/*
			count up from min date
		*/

		thisfunc = datefuncs[++i];
		printf( "\n%s():\n", funcnames[i] );

		for( tmp = min_date ; tmp ; tmp = (*thisfunc)(tmp) )
			printdate( tmp );


		/*
			down down from max date
		*/

		thisfunc = datefuncs[++i];	/* count down */
		printf( "\n%s():\n", funcnames[i] );

		for( tmp = max_date ; tmp ; tmp = (*thisfunc)(tmp) )
			printdate( tmp );
	}

	printf( "\nDone (whew!)" );

	exit( 0 );
}

#endif
