/*
 * num-prim.c -- Implementation of numeric primitives
 *
 * (C) m.b (Matthias Blume); Jun 1992, HUB; Jan 1993 PU/CS
 *         Humboldt-University of Berlin
 *         Princeton University, Dept. of Computer Science
 *
 * $Id: num-prim.c,v 2.7 1994/11/12 22:22:52 blume Exp $
 */

# include "rcs.h"
RCSID ("$Id: num-prim.c,v 2.7 1994/11/12 22:22:52 blume Exp $")

# include <string.h>
# include <math.h>

# include "Cont.h"
# include "Numeric.h"
# include "Boolean.h"
# include "Cons.h"
# include "String.h"
# include "storext.h"
# include "type.h"
# include "except.h"
# include "reader.h"

# include "builtins.tab"

# define is_scm_number(obj) (ScmUPred (SCM_COMPLEX_PRED, (obj)))

static unsigned number_type_pred (enum unary_pred p)
{
  void *x = PEEK ();
  SET_TOP (ScmUPred (p, x) ? &ScmTrue : &ScmFalse);
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveNumberP (unsigned argcnt)
{
  return number_type_pred (SCM_COMPLEX_PRED);
}

unsigned ScmPrimitiveComplexP (unsigned argcnt)
{
  return number_type_pred (SCM_COMPLEX_PRED);
}

/*ARGSUSED*/
unsigned ScmPrimitiveRealP (unsigned argcnt)
{
  return number_type_pred (SCM_REAL_PRED);
}

/*ARGSUSED*/
unsigned ScmPrimitiveRationalP (unsigned argcnt)
{
  return number_type_pred (SCM_RATIONAL_PRED);
}

/*ARGSUSED*/
unsigned ScmPrimitiveIntegerP (unsigned argcnt)
{
  return number_type_pred (SCM_INTEGER_PRED);
}

/*ARGSUSED*/
unsigned ScmPrimitiveExactP (unsigned argcnt)
{
  return number_type_pred (SCM_EXACT_PRED);
}

/*ARGSUSED*/
unsigned ScmPrimitiveInexactP (unsigned argcnt)
{
  void *x = PEEK ();
  SET_TOP (ScmUPred (SCM_EXACT_PRED, x) ? &ScmFalse : &ScmTrue);
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveZeroP (unsigned argcnt)
{
  return number_type_pred (SCM_ZERO_PRED);
}

/*ARGSUSED*/
unsigned ScmPrimitivePositiveP (unsigned argcnt)
{
  return number_type_pred (SCM_POSITIVE_PRED);
}

/*ARGSUSED*/
unsigned ScmPrimitiveNegativeP (unsigned argcnt)
{
  return number_type_pred (SCM_NEGATIVE_PRED);
}

/*ARGSUSED*/
unsigned ScmPrimitiveOddP (unsigned argcnt)
{
  void *x = PEEK ();
  SET_TOP (ScmNumberIsEven (x, "odd?") ? &ScmFalse : &ScmTrue);
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveEvenP (unsigned argcnt)
{
  void *x = PEEK ();
  SET_TOP (ScmNumberIsEven (x, "even?") ? &ScmTrue : &ScmFalse);
  return 0;
}

static unsigned cmp_seq (unsigned argcnt, int (* cmp) (void *, void *))
{
  void *x, *y;
  int res = 1;

  if (argcnt < 2)
    error ("too few args (%u) to primitive comparision", (unsigned) argcnt);
  if (is_scm_number (PEEK ()) == 0) {
type_error:
    error ("non-numeric arg to primitive comparision: %w", PEEK());
    /*NOTREACHED*/
  }
  while (argcnt-- >= 2) {
    x = POP ();
    y = PEEK ();
    if (is_scm_number (y) == 0)
      goto type_error;
    if ((* cmp) (x, y) == 0) {
      res = 0;
      break;
    }
  }
  while (argcnt-- >= 2)
    (void) POP ();
  SET_TOP (res ? &ScmTrue : &ScmFalse);
  return 0;
}

static int num_eq (void *x, void *y)
{
  return ScmBinPred (SCM_EQ_PRED, x, y);
}

unsigned ScmPrimitiveNumEqual (unsigned argcnt)
{
  return cmp_seq (argcnt, num_eq);
}

static int num_lt (void *x, void *y)
{
  return ScmBinPred (SCM_CMP_PRED, x, y) < 0;
}

unsigned ScmPrimitiveLess (unsigned argcnt)
{
  return cmp_seq (argcnt, num_lt);
}

static int num_gt (void *x, void *y)
{
  return ScmBinPred (SCM_CMP_PRED, x, y) > 0;
}

unsigned ScmPrimitiveGreater (unsigned argcnt)
{
  return cmp_seq (argcnt, num_gt);
}

static int num_le (void *x, void *y)
{
  return ScmBinPred (SCM_CMP_PRED, x, y) <= 0;
}

unsigned ScmPrimitiveNotGreater (unsigned argcnt)
{
  return cmp_seq (argcnt, num_le);
}

static int num_ge (void *x, void *y)
{
  return ScmBinPred (SCM_CMP_PRED, x, y) >= 0;
}

unsigned ScmPrimitiveNotLess (unsigned argcnt)
{
  return cmp_seq (argcnt, num_ge);
}

static unsigned fn_seq (unsigned argcnt, void *(* fn) (void *, void *))
{
  void *x, *y;
  while (argcnt-- > 0) {
    x = POP ();
    y = PEEK ();
    x = (* fn) (x, y);
    SET_TOP (x);
  }
  return 0;
}

static void op_seq (unsigned argcnt, enum binary_op op)
{
  void *x, *y;
  while (argcnt-- > 0) {
    x = POP ();
    y = PEEK ();
    x = ScmBinOp (op, x, y);
    SET_TOP (x);
  }
}

static void *max_min_fn (void *x, void *y, int max)
{
  void *z;
  if ((ScmBinPred (SCM_CMP_PRED, x, y) < 0) == max)
    z = x, x = y, y = z;
  if (!ScmUPred (SCM_EXACT_PRED, y))
    x = ScmUOp (SCM_EXACT_TO_INEXACT_OP, x);
  return x;
}

static void *max_fn (void *x, void *y)
{
  return max_min_fn (x, y, 1);
}

unsigned ScmPrimitiveMax (unsigned argcnt)
{
  if (argcnt < 1)
    error ("too few args (%u) to primitive procedure max", (unsigned) argcnt);
  return fn_seq (argcnt - 1, max_fn);
}

static void *min_fn (void *x, void *y)
{
  return max_min_fn (x, y, 0);
}

unsigned ScmPrimitiveMin (unsigned argcnt)
{
  if (argcnt < 1)
    error ("too few args (%u) to primitive procedure min", (unsigned) argcnt);
  return fn_seq (argcnt - 1, min_fn);
}

unsigned ScmPrimitiveAdd (unsigned argcnt)
{
  if (argcnt == 0)
    Push (ScmLongToNumber (0));
  else if (argcnt > 1)
    op_seq (argcnt - 1, SCM_ADD_OP);
  return 0;
}

unsigned ScmPrimitiveMultiply (unsigned argcnt)
{
  if (argcnt == 0)
    Push (ScmLongToNumber (1));
  else if (argcnt > 1)
    op_seq (argcnt - 1, SCM_MUL_OP);
  return 0;
}

unsigned ScmPrimitiveSubtract (unsigned argcnt)
{
  void *tmp;

  if (argcnt < 1)
    error ("too few args (%u) to primitive procedure -", (unsigned) argcnt);
  if (argcnt > 1)
    op_seq (argcnt - 1, SCM_SUB_OP);
  else {
    tmp = ScmUOp (SCM_NEGATE_OP, PEEK());
    SET_TOP (tmp);
  }
  return 0;
}

unsigned ScmPrimitiveDiv (unsigned argcnt)
{
  void *tmp;

  if (argcnt < 1)
    error ("too few args (%u) to primitive procedure /", (unsigned) argcnt);
  if (argcnt > 1)
    op_seq (argcnt - 1, SCM_DIV_OP);
  else {
    tmp = ScmUOp (SCM_INVERSE_OP, PEEK ());
    SET_TOP (tmp);
  }
  return 0;
}

static unsigned prim_u_op (enum unary_op op)
{
  void *x = ScmUOp (op, PEEK ());
  SET_TOP (x);
  return 0;
}

/*ARGSUSED*/
unsigned  ScmPrimitiveAbs (unsigned argcnt)
{
  return prim_u_op (SCM_ABS_OP);
}

/*ARGSUSED*/
unsigned ScmPrimitiveNumerator (unsigned argcnt)
{
  return prim_u_op (SCM_NUMERATOR_OP);
}

/*ARGSUSED*/
unsigned ScmPrimitiveDenominator (unsigned argcnt)
{
  return prim_u_op (SCM_DENOMINATOR_OP);
}

/*ARGSUSED*/
unsigned ScmPrimitiveFloor (unsigned argcnt)
{
  return prim_u_op (SCM_FLOOR_OP);
}

/*ARGSUSED*/
unsigned ScmPrimitiveCeiling (unsigned argcnt)
{
  return prim_u_op (SCM_CEILING_OP);
}

/*ARGSUSED*/
unsigned ScmPrimitiveTruncate (unsigned argcnt)
{
  return prim_u_op (SCM_TRUNCATE_OP);
}

/*ARGSUSED*/
unsigned ScmPrimitiveRound (unsigned argcnt)
{
  return prim_u_op (SCM_ROUND_OP);
}

/*ARGSUSED*/
unsigned ScmPrimitiveExp (unsigned argcnt)
{
  return prim_u_op (SCM_EXP_OP);
}

/*ARGSUSED*/
unsigned ScmPrimitiveLog (unsigned argcnt)
{
  return prim_u_op (SCM_LOG_OP);
}

/*ARGSUSED*/
unsigned ScmPrimitiveSin (unsigned argcnt)
{
  return prim_u_op (SCM_SIN_OP);
}

/*ARGSUSED*/
unsigned ScmPrimitiveCos (unsigned argcnt)
{
  return prim_u_op (SCM_COS_OP);
}

/*ARGSUSED*/
unsigned ScmPrimitiveTan (unsigned argcnt)
{
  return prim_u_op (SCM_TAN_OP);
}

/*ARGSUSED*/
unsigned ScmPrimitiveAsin (unsigned argcnt)
{
  return prim_u_op (SCM_ASIN_OP);
}

/*ARGSUSED*/
unsigned ScmPrimitiveAcos (unsigned argcnt)
{
  return prim_u_op (SCM_ACOS_OP);
}

unsigned ScmPrimitiveAtan (unsigned argcnt)
{
  if (argcnt == 1)
    return prim_u_op (SCM_ATAN_OP);
  else if (argcnt == 2) {
    void *x, *y;
    x = POP ();
    y = PEEK ();
    SET_TOP (ScmAtan2 (x, y));
  } else
    error ("wrong argcnt (%u) to primitive procedure atan", (unsigned) argcnt);
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveSqrt (unsigned argcnt)
{
  return prim_u_op (SCM_SQRT_OP);
}

/*ARGSUSED*/
unsigned ScmPrimitiveMagnitude (unsigned argcnt)
{
  return prim_u_op (SCM_MAGNITUDE_OP);
}

/*ARGSUSED*/
unsigned ScmPrimitiveAngle (unsigned argcnt)
{
  return prim_u_op (SCM_ANGLE_OP);
}

/*ARGSUSED*/
unsigned ScmPrimitiveRealPart (unsigned argcnt)
{
  return prim_u_op (SCM_REAL_PART_OP);
}

/*ARGSUSED*/
unsigned ScmPrimitiveImagPart (unsigned argcnt)
{
  return prim_u_op (SCM_IMAG_PART_OP);
}

/*ARGSUSED*/
unsigned ScmPrimitiveExactToInexact (unsigned argcnt)
{
  return prim_u_op (SCM_EXACT_TO_INEXACT_OP);
}

/*ARGSUSED*/
unsigned ScmPrimitiveInexactToExact (unsigned argcnt)
{
  return prim_u_op (SCM_INEXACT_TO_EXACT_OP);
}

unsigned ScmPrimitiveStringToNumber (unsigned argcnt)
{
  void *tmp;
  unsigned base;
  ScmString *s;

  if (argcnt < 1 || argcnt > 2)
    error ("too %s args (%u) to primitive string->number",
	   argcnt < 1 ? "few" : "many", (unsigned) argcnt);
  if (argcnt == 1) {
    tmp = PEEK ();
    base = 10;
  } else {
    tmp = POP ();
    base = ScmNumberToUShort (PEEK (), "string->number");
    if (base != 2 && base != 8 && base != 10 && base != 16)
      error ("wrong base specification for primitive string->number: %w",
	     PEEK ());
  }
  if (ScmTypeOf (tmp) != ScmType (String))
    error ("wrong arg to primitive string->number: %w", tmp);
  s = tmp;
  if ((tmp = ScmParseNumberString (s->array, s->length, base)) == NULL)
    tmp = &ScmFalse;
  SET_TOP (tmp);
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveQuotient (unsigned argcnt)
{
  void *x, *y, *z;

  x = POP ();
  y = PEEK ();
  ScmQuotRem (x, y, &z, NULL, "quotient");
  SET_TOP (z);
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveRemainder (unsigned argcnt)
{
  void *x, *y, *z;

  x = POP ();
  y = PEEK ();
  ScmQuotRem (x, y, NULL, &z, "remainder");
  SET_TOP (z);
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveDivide (unsigned argcnt)
{
  void *x, *y, *q, *r;

  if (ScmMultiCont (ScmCC) == 0)
    error ("divide: continuation does not accept multiple values");
  x = POP ();
  y = PEEK ();
  ScmQuotRem (x, y, &q, &r, "divide");
  SET_TOP (r);
  PUSH (q);
  return 0;
}

unsigned ScmPrimitiveNumberToString (unsigned argcnt)
{
  void *x;
  unsigned base;
  int l;
  char *s;
  ScmString *strg;

  if (argcnt < 1 || argcnt > 2)
    error ("wrong argcnt(%u) to primitive number->string", (unsigned) argcnt);
  if (argcnt == 1) {
    base = 10;
    x = PEEK ();
  } else {
    x = POP ();
    base = ScmNumberToUShort (PEEK (), "number->string");
  }
  s = ScmUnparseNumber (x, base);
  l = strlen (s);
  SCM_VNEW (strg, String, l, char);
  strg->length = l;
  memcpy (strg->array, s, l);
  SET_TOP (strg);
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveMakeRectangular (unsigned argcnt)
{
  double re = ScmGetReal (POP ()), im = ScmGetReal (PEEK ());
  void *x = ScmNewRealComplex (re, im);
  SET_TOP (x);
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveMakePolar (unsigned argcnt)
{
  double m = ScmGetReal (POP ()), a = ScmGetReal (PEEK ());
  void *x = ScmNewRealComplex (m * cos (a), m * sin (a));
  SET_TOP (x);
  return 0;
}
