# include <stdio.h>
# include <stdlib.h>
# include <assert.h>

/* definitions of unsigned integral data type for a ``digit'' in bignums */
typedef unsigned short bigdig;

/* definition of an unsigned integral data type for a ``big register'' that */
/* can hold at least two ``bigdig''s */
typedef unsigned long bigreg;

/* number of bits in bigdig */
# define BIGBITS 16

/* ``bigreg'' 0 & 1 */
# define B0 ((bigreg)0)
# define B1 ((bigreg)1)

/* the base of the number system */
# define BASE (B1<<BIGBITS)

/* a bitmask with all bits of a ``bigdig'' on */
# define BIGMASK (~((~B0)<<BIGBITS))

/* the most significant bit of a ``bigdig'' */
# define BIGMSB (B1<<(BIGBITS-1))

/* an upper bound on the number of decimal digits for n bigdig's */
# define ndig_to_ndec(n) (5*(n))

/* an upper bound on the number of bigdig's per decimal digit */
# define ndec_to_ndig(n) ((21*(n)+99)/100)

/* number of bits in an unsigned long (lower bound) */
# define LONGBITS 32

extern void bigdig_copy (bigdig *from, bigdig *to, int l);
extern void shift_left (bigdig *a, int l, int b);
extern void shift_right (bigdig *a, int l, int b);

extern void sdiv (bigdig *a, bigdig *b, bigdig *q, bigdig *r, int la, int lb);

/* a proof that the following works: see Knuth Vol. 2 */
void sdiv (bigdig *a, bigdig *b, bigdig *q, bigdig *r, int la, int lb)
/* lq = la - lb + 1, lr = la + 1 */
/* q == NULL -> only the remainder will be computed */
/* r == a -> in place computation */
{
  int j, k, lshb;
  bigreg tmp, qh, b1, b2, tmp2;

  assert (la >= lb && lb >= 2);

  /* Find shift amount needed to normalize */
  for (lshb = 0, b1 = b [lb - 1]; (b1 & BIGMSB) == 0; b1 <<= 1)
    lshb++;

  /* initialize remainder */
  if (a != r)
    bigdig_copy (a, r, la);
  r [la] = 0;

  /* normalize */
  if (lshb) {
    shift_left (b, lb, lshb);
    shift_left (r, la + 1, lshb);
  }

  /* first two digits of (normalized) divisor */
  b1 = b [lb - 1];
  b2 = b [lb - 2];

  for (j = la; j >= lb; j--) {
    /* Educated guess... */
    tmp = (((bigreg) r [j]) << BIGBITS) | r [j - 1];
    if (r [j] == b1)
      qh = BASE - 1;
    else
      qh = tmp / b1;

    /* Fast verify by looking at next digit -> improving the guess*/
    tmp -= qh * b1; tmp2 = qh * b2;
    while (tmp < BASE && tmp2 > ((tmp << BIGBITS) | r [j - 2])) {
      qh = qh - 1;
      tmp2 = tmp2 - b2;
      tmp = tmp + b1;
    }

    /* multiply & subtract */
    tmp = 0;
    for (k = 0; k < lb; k++) {
      tmp += b [k] * qh;
      tmp2 = tmp & BIGMASK;
      tmp >>= BIGBITS;
      if (tmp2 > r [j - lb + k]) {
	r [j - lb + k] += BASE - tmp2;
	++tmp;
      } else
	r [j - lb + k] -= tmp2;
    }

    if (tmp > r [j]) {
      /* did not work out -- qh is still one too big */
      --qh;
      /* add back */
      tmp = 0;
      for (k = 0; k < lb; k++) {
	tmp += (bigreg) r [j - lb + k] + (bigreg) b [k];
	r [j - lb + k] = tmp & BIGMASK;
	tmp >>= BIGBITS;
      }
    }

    r [j] = 0;
    if (q != NULL)
      q [j - lb] = qh;
  }

  /* Denormalize */
  if (lshb) {
    shift_right (b, lb, lshb);
    shift_right (r, lb, lshb);
  }
}
