(define (envs-functor lgi-mod pmac-mod primv-mod aux-mod names-mod soft-mod)

  (module envs-sig ()

    (use scheme-sig)
    (use den-sig)
    (use lgi-mod lgi-sig)
    (use pmac-mod pmac-sig)
    (use primv-mod primv-sig pv)
    (use aux-mod aux-sig)
    (use names-mod names-sig)
    (use soft-mod soft-sig)

    (define i-car (integrable 'car))
    (define i-cdr (integrable 'cdr))

    ;; some utilities:
    (define (make-special-cons x) (cons x (special x)))
    (define (make-auxk-cons x) (cons x (aux-keyword x)))
    (define (make-integ-cons p) (cons (car p) (integrable (car p))))
    (define (make-soft-integ-cons x) (cons x (integrable x)))

    ;; make a procedure that checks if a given name is defined in a delta:
    (define (delta-checker delta)
      (delta (lambda (name) #f)))

    ;; a special ``name'' to denote ``lambda''
    ;; This is used in the compilation of ``define''.
    (define secret-name-for-lambda (generate 'lambda -1))

    ;; the empty environment, where nothing is defined:
    (define (empty-env name) (undefined name))

    ;; the r4rs (+ hygienic macros) environment:
    (define r4rs-macros '())

    (define r4rs-specials
      (cons
       (cons secret-name-for-lambda (special 'lambda))
       (append
	(map make-special-cons
	     '(define lambda quote if letrec begin case set! define-syntax
		delay let-syntax letrec-syntax))
	(map make-auxk-cons
	     '(else => unquote unquote-splicing ...)))))

    (define r4rs-values (map make-integ-cons (pv r4rs-values)))

    (define (r4rs-cadr name)

      (define (car/cdr-combination sym)

	(let* ((s (symbol->string sym))
	       (l-1 (- (string-length s) 1)))

	  (define (get-a-d-list i l)
	    (if (>= i l-1)
		l
		(let ((c (string-ref s i)))

		  (cond ((char=? c #\a)
			 (get-a-d-list (+ i 1) (cons 'car l)))
			((char=? c #\d)
			 (get-a-d-list (+ i 1) (cons 'cdr l)))
			(else #f)))))

	  (define (normalize l)
	    (cond ((not l) #f)
		  ((pair? (cdr l)) (integrable l))
		  ((eq? (car l) 'car) i-car)
		  (else i-cdr)))

	  (and (> l-1 1)
	       (char=? (string-ref s 0) #\c)
	       (char=? (string-ref s l-1) #\r)
	       (normalize (get-a-d-list 1 '())))))

      (cond ((not (symbol? name)) #f)
	    ((car/cdr-combination name))
	    (else #f)))

    (define (raw-r4rs-empty-env-delta env)
      (lambda (name)
	(cond
	 ((assq name r4rs-specials) => cdr)
	 ((assq name r4rs-macros) => cdr)
	 ((assq name r4rs-values) => cdr)
	 ((r4rs-cadr name))
	 (else (env name)))))

    (define soft-values (map make-soft-integ-cons soft-builtin-names))

    (define (r4rs-raw-r4rs-env-delta env)
      (lambda (name)
	(cond
	 ((assq name soft-values) => cdr)
	 (else (env name)))))

    (define r4rs-empty-env-delta
      (compose r4rs-raw-r4rs-env-delta raw-r4rs-empty-env-delta))

    ;; VSCM-specific additions to the user environment
    (define user-macros '())

    (define user-specials
      (append
       (map make-special-cons '(module primitive-transformer define-type))
       (map make-auxk-cons '(signature constant variable read-only))))

    (define user-values (map make-integ-cons (pv user-values)))

    (define user-types '())

    (define (user-r4rs-env-delta env)
      (lambda (name)
	(cond
	 ((assq name user-macros) => cdr)
	 ((assq name user-specials) => cdr)
	 ((assq name user-values) => cdr)
	 ((assq name user-types) => cdr)
	 (else (env name)))))

    ;; VSCM-specific additions to the system environment:

    (define system-values (map make-integ-cons (pv system-values)))

    (define (system-user-env-delta env)
      (lambda (name)
	(cond
	 ((assq name system-values) => cdr)
	 (else (env name)))))

    ;; various deltas for various signatures:

    (define user-empty-env-delta
      (compose user-r4rs-env-delta r4rs-empty-env-delta))
    (define system-empty-env-delta
      (compose system-user-env-delta user-empty-env-delta))

    (define raw-system-empty-env-delta
      (compose system-user-env-delta
	       (compose user-r4rs-env-delta raw-r4rs-empty-env-delta)))

    ;; POSIX support...
    (define posix-values (map make-integ-cons (pv posix-values)))

    (define (posix-empty-env-delta env)
      (lambda (name)
	(cond ((assq name posix-values) => cdr)
	      (else (env name)))))

    ;; the default ``system'' environment (used as the transformer environment)

    (define system-env (system-empty-env-delta empty-env))

    ;; the interactive toplevel environment (this is pretty gross hacking):

    (define toplevel-import #f)
    (define (register-toplevel-import-connector! conn)
      (set! toplevel-import (import conn)))

    (define global-definitions '())
    ;; register a global definition:
    (define (register-global-definition! symbol den)
      (if (or (macro? den)
	      (type? den)
	      (assq symbol r4rs-macros)
	      (assq symbol user-macros)
	      (assq symbol r4rs-specials)
	      (assq symbol user-specials)
	      (assq symbol user-types)
	      (assq symbol r4rs-values)
	      (assq symbol user-values)
	      (assq symbol soft-values)
	      (r4rs-cadr symbol))
	  (cond ((assq symbol global-definitions)
		 =>
		 (lambda (a) (set-cdr! a den)))
		(else (set! global-definitions
			    (cons (cons symbol den)
				  global-definitions))))
	  (drop-global-definition! symbol)))

    ;; Drop a global definition.
    ;; (This also re-installs the built-in definition if there was one.)
    (define (drop-global-definition! symbol)
      (define (remove-symbol-from l)
	(cond ((null? l) '())
	      ((eq? (caar l) symbol) (cdr l))
	      (else
	       (cons (car l) (remove-symbol-from (cdr l))))))
      (set! global-definitions (remove-symbol-from global-definitions)))

    ;; the toplevel env is special in that it always bottoms out in
    ;; a global denotation... (this is the main reason for all kinds of
    ;; peculiar behavior -- similar to non-monotonic reasoning)

    (define (tl-env-bottom name) (global name))

    (define tluser-env (user-empty-env-delta tl-env-bottom))

    (define (tlinitial-tluser-env-delta env)
      (lambda (name)
	(cond
	 ((assq name global-definitions) => cdr)
	 ((eq? name 'use) toplevel-import)
	 (else (env name)))))

    (define tlinitial-env (tlinitial-tluser-env-delta tluser-env))

    (define tlcurr-env tlinitial-env)
    (define (global-import! delta)
      (set! tlcurr-env (delta tlcurr-env)))

    (define (toplevel-env name) (tlcurr-env name))

    (define any-type
      (type (no-sgn 'any-type) empty-env)) ;***********************************
    (define r4rs-sig-type
      (type (built-in-sgn r4rs-empty-env-delta) empty-env))
    (define user-sig-type
      (type (built-in-sgn user-empty-env-delta) empty-env))
    (define system-sig-type
      (type (built-in-sgn system-empty-env-delta) empty-env))
    (define raw-system-sig-type
      (type (built-in-sgn raw-system-empty-env-delta) empty-env))
    (define posix-sig-type
      (type (built-in-sgn posix-empty-env-delta) empty-env))

    (set! r4rs-macros
	  (list
	   (cons 'let (macro let-macro system-env "Let"))
	   (cons 'let* (macro let*-macro system-env "Let*"))
	   (cons 'cond (macro cond-macro system-env "Cond"))
	   (cons 'quasiquote
		 (macro quasiquote-macro system-env "QuasiQuote"))
	   (cons 'or (macro or-macro system-env "Or"))
	   (cons 'and (macro and-macro system-env "And"))
	   (cons 'do (macro do-macro system-env "Do"))
	   (cons 'syntax-rules
		 (macro syntax-rules-macro system-env "Syntax-Rules"))))

    (set! user-types
	  (list
	   (cons 'any any-type)
	   (cons 'r4rs-sig r4rs-sig-type)
	   (cons 'user-sig user-sig-type)
	   (cons 'scheme-sig user-sig-type)
	   (cons 'system-sig system-sig-type)
	   (cons 'raw-system-sig raw-system-sig-type)
	   (cons 'posix-sig posix-sig-type)))))
