;;; The intermediate language: <type>:<body>
;;;
;;; constant:		quote:<constant>
;;; global variable:	global:<symbol>
;;; integrable:		integrable:<symbol>
;;; car/cdr integrable:	integrable:([car|cdr] ...)
;;; local variable:	local:(<unique-id> <orig-symbol> <update flag>)
;;; global assignment:	set!:(<global> <expr>)
;;; branch:		if:(<expr> <expr> <expr>)
;;; application:	app:(<expr> <expr> ...)
;;; let-form:		let:(((<local> <expr>)) <expr>)
;;; clambda-form:	clambda:(<local> <expr> "fname" (<local> ...) <expr>)
;;; cvlambda-form:	cvlambda:(<local> <expr>
;;;					"fname" (<local> ...) <local> <expr>)
;;; cdelay-form:	cdelay:(<local> <expr> "fname" <expr>)
;;;
;;; labeled expression:	label:(<label> ((<local> <expr>) ...) <exp>)
;;; goto:		goto:(<label> <expr> ...)
;;;
;;; integrables:
;;;	car, cdr, (cadr), cons, ref, deref, assign, closure,
;;;	get-closure, set-closure, not, +, -, *, <, >, <=, >=, zero?,
;;;	null?, pair?, boolean?, number?, string?, vector?, char?, append,
;;;	vector, list->vector, memv, eqv?


(define (cg-functor error-mod lgi-mod varset-mod aux-mod primv-mod hack-mod)

  (module cg-sig ()

    (use system-sig)
    (use ee-sig)
    (use lgi-mod lgi-sig)
    (use varset-mod varset-sig)
    (use error-mod error-sig)
    (use aux-mod aux-sig)
    (use lv-sig)
    (use primv-mod primv-sig)
    (use hack-mod hack-sig)

    (define voiding #f)
    (define showing #f)
    (define debug #t)
    (define show-branching #f)

    (define (impossible s)
      (error (string-append "codegen: " s)))

    (define (codegen exp closure-var name nargs has-rarg argl)

      (let* ((istk (do ((s (list (cons closure-var 0))
			   (cons (cons (car l) i) s))
			(l argl (cdr l))
			(i 1 (+ i 1)))
		       ((not (pair? l)) s)))
	     (hwm (length istk))		; highwater mark
	     (constants '())
	     (nconstants 0)
	     (nlab 0))

	(define (stack-top s)
	  ;; (if (null? s) 0 (+ (cdar s) 1))
	  (length s))

	(define (cut-stack s lv)
	  (do ((s s (cdr s)))
	      ((or (null? s)
		   (not (caar s))
		   (varset-in? (caar s) lv))
	       s)))

	(define (show-stack stk)
	  (display " STACK: ") (newline)
	  (do ((s stk (cdr s)))
	      ((null? s) 'done)
	    (display "    ")
	    (if (caar s)
		(display (hack (caar s)))
		(display "[dummy]"))
	    (display ": ")
	    (write (cdar s))
	    (newline)))

	(define (var-loc v stk)
	  (if debug
	      (cond ((assq v stk) => cdr)
		    (else
		     (display "VARIABLE ")
		     (write (hack v))
		     (display " NOT FOUND ON STACK")
		     (newline)
		     (show-stack stk)
		     (impossible "variable not on stack")))
	      (cdr (assq v stk))))

	(define (hwm! pos)
	  (if (>= pos hwm)
	      (set! hwm pos)))

	(define (new-constant c)
	  (let ((r nconstants))
	    (set! constants (cons c constants))
	    (set! nconstants (+ nconstants 1))
	    r))

	(define (data-constant x)
	  (let ((c (list 'quote x)))
	    (cond ((member c constants) => (lambda (l) (- (length l) 1)))
		  (else (new-constant c)))))

	(define (code-constant c) (new-constant c))

	(define (new-label)
	  (let ((l nlab))
	    (set! nlab (+ l 1))
	    l))

	(define (c-empty) '(c-empty))
	(define (c-list . l) (cons 'c-list l))
	(define (c-append . l) (cons 'c-append l))
	(define (c->list c)
	  (define (build cur rest)
	    (define (app l)
	      (if (not (pair? l))
		  rest
		  (build (car l) (app (cdr l)))))
	    (case (car cur)
	      ((c-empty) rest)
	      ((c-list) (append (cdr cur) rest))
	      ((c-append) (app (cdr cur)))
	      (else (impossible "c->list: funny thing"))))
	  (build c '()))

	(define (c-labels-only? c)
	  (case (car c)
	    ((c-empty) #t)
	    ((c-list) (not (contains pair? (cdr c))))
	    ((c-append)
	     (not (contains (lambda (x) (not (c-labels-only? x))) (cdr c))))
	    (else (impossible "c-labels-only?: funny thing"))))

	(define (take-i x)
	  (cond ((null? x) '(take-nil))
		((not x) '(take-false))
		((eq? x '#t) '(take-true))
		(else (list 'take (data-constant x)))))

	(define (exit-i) '(exit))
	(define (get-glob-i s) (list 'get-glob (data-constant s)))
	(define (take-primitive-i sy) (list 'take-primitive sy))
	(define (get-loc-i n) (list 'get-loc n))
	(define (get-loc-void-i n) (list 'get-loc-void n))
	(define (put-loc-i n) (list 'put-loc n))
	(define (put-loc-pop-i n) (list 'put-loc-pop n))
	(define (get-vec-i n i) (list 'get-vec n i))
	(define (get-vec-void-i n i) (list 'get-vec-void n i))
	(define (put-vec-pop-i n i) (list 'put-vec-pop n i))
	(define (multi-pop-i n) (if (= n 1) '(pop) (list 'multi-pop n)))
	(define (put-glob-i s) (list 'put-glob (data-constant s)))
	(define (put-glob-pop-i s) (list 'put-glob-pop (data-constant s)))
	(define (lambda-i c) (list 'lambda (code-constant c)))
	(define (delay-i c) (list 'delay (code-constant c)))
	(define (void-i n) (list 'void n))
	(define (jump-forward-i l) (list 'jump-forward l))
	(define (jump-backward-i l) (list 'jump-backward l))
	(define (label-i l) l)
	(define (call-i n) (list 'call n))
	(define (call-exit-i n) (list 'call-exit n))
	(define (primitive-i p) (list p))
	(define (make-closure-i e n) (list 'make-closure e n))
	(define (vector-i n) (list 'vector n))
	(define (true?jump-i lab) (list 'true?jump lab))
	(define (false?jump-i lab) (list 'false?jump lab))
	(define (true?jump+pop-i lab) (list 'true?jump+pop lab))
	(define (false?jump+pop-i lab) (list 'false?jump+pop lab))
	(define (true?jump:pop-i lab) (list 'true?jump:pop lab))
	(define (false?jump:pop-i lab) (list 'false?jump:pop lab))
	(define (pop-true?jump-i lab) (list 'pop-true?jump lab))
	(define (pop-false?jump-i lab) (list 'pop-false?jump lab))
	(define (check-i) (list 'check))
	(define (fetch-i op sy)
	  (list 'fetch
		(case op
		  ((fetch-constant) 0)
		  ((fetch-read-only) 1)
		  (else 2))
		(data-constant sy)))
	(define (module-i n) (list 'module n))
	(define (validate-i) (list 'validate))
	(define (vec-ref-i i) (list 'vec-ref i))
	(define (vec-set-i i) (list 'vec-set i))

	(define (compare-i t-jump cmp-sy t-retain e-retain c lab)

	  (define eq?-instr
	    '(((pop-neq?jump . neq?jump:pop)
	       .
	       (neq?jump+pop . neq?jump))
	      .
	      ((pop-eq?jump . eq?jump+pop)
	       .
	       (eq?jump:pop . eq?jump))))

	  (define eqv?-instr
	    '(((pop-neqv?jump . neqv?jump:pop)
	       .
	       (neqv?jump+pop . neqv?jump))
	      .
	      ((pop-eqv?jump . eqv?jump+pop)
	       .
	       (eqv?jump:pop . eqv?jump))))

	  (define memv-instr
	    '(((pop-nmemv?jump . nmemv?jump:pop)
	       .
	       (nmemv?jump+pop . nmemv?jump))
	      .
	      ((pop-memv?jump . memv?jump+pop)
	       .
	       (memv?jump:pop . memv?jump))))

	  (let* ((instr (cond ((eq? cmp-sy 'eq?) eq?-instr)
			      ((eq? cmp-sy 'eqv?) eqv?-instr)
			      ((eq? cmp-sy 'memv) memv-instr)
			      (else (impossible
				     "bad comparison instruction"))))
		 (instr (if t-jump (cdr instr) (car instr)))
		 (instr (if t-retain (cdr instr) (car instr)))
		 (instr (if e-retain (cdr instr) (car instr))))
	    (list instr (data-constant c) lab)))

	(define (pop-c n)
	  (cond ((positive? n)
		 (c-list (multi-pop-i n)))
		((zero? n)
		 (c-empty))
		(else (impossible "pop: negative"))))

	(define (gen-cnd exp stk t-first t-lv e-lv labmap k)
	  ;; exp: condition
	  ;; stk: stack layout before exp
	  ;; t-first: #t: ``then->else'', #f: ``else->then''
	  ;; ?-lv:  live variables at t/e branches
	  ;; k: continuation function:
	  ;;   (k i-t-code i-e-code t-stk e-stk)

	  (define (general-case)
	    (let* ((lab (new-label))
		   (lv (lv-after exp))
		   (cut-stk (cut-stack stk lv))
		   (cut-top (stack-top cut-stk)))
	      (if t-first
		  (k (c-append
		      (gen exp stk 'use cut-top labmap)
		      (c-list (pop-false?jump-i lab)))
		     (c-list (label-i lab))
		     cut-stk cut-stk)
		  (k (c-list (label-i lab))
		     (c-append
		      (gen exp stk 'use cut-top labmap)
		      (c-list (pop-true?jump-i lab)))
		     cut-stk cut-stk))))

	  (define (categ e)
	    (if (eq? (ee-type e) 'wrap)
		(let ((we (ee-body e)))
		  (case (ee-type we)
		    ((quote) (if (ee-body we) 'true 'false))
		    ((integrable) 'true)
		    (else 'unknown)))
		'unknown))

	  (define (adapt stk1 stk2 fv k)
	    (let* ((cut-stk1 (cut-stack stk1 fv))
		   (cut-stk2 (cut-stack stk2 fv))
		   (top1 (stack-top stk1))
		   (top2 (stack-top stk2))
		   (cut-top1 (stack-top cut-stk1))
		   (cut-top2 (stack-top cut-stk2)))
	      (if (not (= cut-top1 cut-top2))
		  (impossible "adapt: stack top mismatch"))
	      (k (pop-c (- top1 cut-top1))
		 (pop-c (- top2 cut-top2))
		 cut-stk1)))

	  (define (jump-around c)
	    (if (c-labels-only? c)
		c
		(let ((lab (new-label)))
		  (c-append
		   (c-list (jump-forward-i lab))
		   c
		   (c-list (label-i lab))))))

	  (define (one-known-branch cc cu cc=t->cu non-cu->t)
	    ;; condition is an if, whith one of the branches being constant
	    ;; cc - inner condition
	    ;; cu - unknown branch
	    ;; cc=t->cu: a true outcome of cc needs consulting of cu
	    ;; non-cu->t: if cu is not needed then jump to t branch

	    (if show-branching
		(begin
		  (display "branch: ")
		  (display (if t-first "+" "-"))
		  (display (if cc=t->cu "+" "-"))
		  (display (if non-cu->t "+" "-"))
		  (newline)))

	    (let ((cu-lv (lv-before cu))
		  (non-cu->1st (eq? non-cu->t t-first)))
	      (gen-cnd
	       cc stk
	       cc=t->cu
	       (if cc=t->cu cu-lv (if non-cu->t t-lv e-lv))
	       (if cc=t->cu (if non-cu->t t-lv e-lv) cu-lv)
	       labmap
	       (lambda (cc-t-code cc-e-code cc-t-stk cc-e-stk)
		 (gen-cnd
		  cu
		  (if cc=t->cu cc-t-stk cc-e-stk)
		  t-first t-lv e-lv labmap
		  (lambda (cu-t-code cu-e-code cu-t-stk cu-e-stk)
		    (let ((cc-na-stk (if cc=t->cu cc-e-stk cc-t-stk))
			  (cu-na-stk (if non-cu->t cu-t-stk cu-e-stk)))
		      (adapt
		       cc-na-stk cu-na-stk
		       (if non-cu->t t-lv e-lv)
		       (lambda (cc-a-code cu-a-code cc-cu-a-stk)
			 (let ((cc-join-code
				(c-append
				 (if cc=t->cu cc-e-code cc-t-code)
				 cc-a-code))
			       (cu-join-code
				(c-append
				 (if non-cu->t cu-t-code cu-e-code)
				 cu-a-code))
			       (top-code (if cc=t->cu cc-t-code cc-e-code)))
			   (if non-cu->1st
			       (if t-first
				   (k (c-append
				       top-code
				       cu-join-code
				       (jump-around cc-join-code))
				      cu-e-code
				      cc-cu-a-stk
				      cu-e-stk)
				   (k cu-t-code
				      (c-append
				       top-code
				       cu-join-code
				       (jump-around cc-join-code))
				      cu-t-stk
				      cc-cu-a-stk))
			       (let ((join-code
				      (if (c-labels-only? cc-join-code)
					  (c-append cu-join-code cc-join-code)
					  (c-append
					   cc-join-code
					   (jump-around cu-join-code)))))
				 (if t-first
				     (k (c-append top-code cu-t-code)
					join-code
					cu-t-stk
					cc-cu-a-stk)
				     (k join-code
					(c-append top-code cu-e-code)
					cc-cu-a-stk
					cu-t-stk))))))))))))))

	  (define (simple-comparison body)
	    (and (= (length body) 3)
		 (let ((f (car body))
		       (x (cadr body))
		       (y (caddr body)))
		   (and (eq? (ee-type f) 'wrap)
			(eq? (ee-type x) 'wrap)
			(eq? (ee-type y) 'wrap)
			(let ((wf (ee-body f))
			      (wx (ee-body x))
			      (wy (ee-body y)))
			  (and (integrable? wf)
			       (let ((sy (integrable-symbol wf)))
				 (or
				  (and (or (eq? sy 'eq?)
					   (eq? sy 'eqv?)
					   (eq? sy 'memv))
				       (eq? (ee-type wx) 'local)
				       (eq? (ee-type wy) 'quote)
				       (list sy wx (ee-body wy)))
				  (and (or (eq? sy 'eq?)
					   (eq? sy 'eqv?))
				       (eq? (ee-type wx) 'quote)
				       (eq? (ee-type wy) 'local)
				       (list sy wy (ee-body wx)))))))))))

	  (case (ee-type exp)
	    ((wrap)
	     (let ((wexp (ee-body exp)))
	       (if (eq? (ee-type wexp) 'local)
		   (let* ((lv (lv-before exp))
			  (cut-stk (cut-stack stk lv))
			  (cut-top (stack-top cut-stk))
			  (diff (- (stack-top stk) cut-top))
			  (lpos (var-loc wexp cut-stk)))
		     (if (= lpos (- cut-top 1))
			 (let ((t-use (varset-in? wexp t-lv))
			       (e-use (varset-in? wexp e-lv))
			       (lab (new-label)))

			   (define (t-k j ts es)
			     (k (c-append
				 (pop-c diff)
				 (c-list (j lab)))
				(c-list (label-i lab))
				ts es))

			   (define (e-k j ts es)
			     (k (c-list (label-i lab))
				(c-append
				 (pop-c diff)
				 (c-list (j lab)))
				ts es))

			   (if t-first
			       (if t-use
				   (if e-use
				       (t-k false?jump-i cut-stk cut-stk)
				       (t-k false?jump+pop-i cut-stk
					    (cdr cut-stk)))
				   (if e-use
				       (t-k false?jump:pop-i (cdr cut-stk)
					    cut-stk)
				       (t-k pop-false?jump-i (cdr cut-stk)
					    (cdr cut-stk))))
			       (if t-use
				   (if e-use
				       (e-k true?jump-i cut-stk cut-stk)
				       (e-k true?jump:pop-i cut-stk
					    (cdr cut-stk)))
				   (if e-use
				       (e-k true?jump+pop-i (cdr cut-stk)
					    cut-stk)
				       (e-k pop-true?jump-i (cdr cut-stk)
					    (cdr cut-stk))))))
			 (general-case)))
		   (general-case))))
	    ((if)
	     (let* ((body (ee-body exp))
		    (cc (car body))
		    (ct (cadr body))
		    (ce (caddr body))
		    (ct-cat (categ ct))
		    (ce-cat (categ ce)))
	       (case ct-cat
		 ((true)
		  (case ce-cat
		    ((true) (general-case)) ; shouldn't happen too often
		    ((false)
		     (gen-cnd cc stk t-first t-lv e-lv labmap k))
		    (else
		     (one-known-branch cc ce #f #t))))
		 ((false)
		  (case ce-cat
		    ((true)
		     (gen-cnd cc stk (not t-first) e-lv t-lv labmap k))
		    ((false) (general-case)) ; shouldn't happen too often
		    (else (one-known-branch cc ce #f #f))))
		 (else

		  (if show-branching
		      (begin
			(display "branch: *")
			(newline)))

		  (case ce-cat
		    ((true) (one-known-branch cc ct #t #t))
		    ((false) (one-known-branch cc ct #t #f))
		    (else
		     (let ((ct-lv (lv-before ct))
			   (ce-lv (lv-before ce))
			   (1lab (new-label)))
		       (gen-cnd
			cc stk #t ct-lv ce-lv labmap
			(lambda (cc-t-code cc-e-code cc-t-stk cc-e-stk)
			  (gen-cnd
			   ct cc-t-stk t-first t-lv e-lv labmap
			   (lambda (ct-t-code ct-e-code ct-t-stk ct-e-stk)
			     (gen-cnd
			      ce cc-e-stk t-first t-lv e-lv labmap
			      (lambda (ce-t-code ce-e-code ce-t-stk ce-e-stk)
				(adapt
				 ct-t-stk ce-t-stk t-lv
				 (lambda (ct-t-a-code ce-t-a-code t-a-stk)
				   (adapt
				    ct-e-stk ce-e-stk e-lv
				    (lambda (ct-e-a-code ce-e-a-code e-a-stk)
				      (if t-first
					  (k (c-append
					      cc-t-code ct-t-code ct-t-a-code
					      (c-list (jump-forward-i 1lab))
					      cc-e-code ce-t-code ce-t-a-code
					      (c-list (label-i 1lab)))
					     (let ((ce-join-code
						    (c-append
						     ce-e-code ce-e-a-code))
						   (ct-join-code
						    (c-append
						     ct-e-code ct-e-a-code)))
					       (if
						(c-labels-only? ct-join-code)
						(c-append
						 ce-join-code ct-join-code)
						(c-append
						 ct-join-code
						 (jump-around ce-join-code))))
					     t-a-stk
					     e-a-stk)
					  (k (let ((ct-join-code
						    (c-append
						     ct-t-code ct-t-a-code))
						   (ce-join-code
						    (c-append
						     ce-t-code ce-t-a-code)))
					       (if
						(c-labels-only? ct-join-code)
						(c-append
						 ce-join-code ct-join-code)
						(c-append
						 ct-join-code
						 (jump-around ce-join-code))))
					     (c-append
					      cc-t-code ct-e-code ct-e-a-code
					      (c-list (jump-forward-i 1lab))
					      cc-e-code ce-e-code ce-e-a-code
					      (c-list (label-i 1lab)))
					     e-a-stk
					     t-a-stk)))))))))))))))))))
	    ((app)
	     (cond ((simple-comparison (ee-body exp))
		    =>
		    (lambda (l)
		      (let* ((sy (car l))
			     (v (cadr l))
			     (c (caddr l))
			     (lv (lv-before exp))
			     (cut-stk (cut-stack stk lv))
			     (cut-top (stack-top stk))
			     (diff (- (stack-top stk) cut-top))
			     (lpos (var-loc v cut-stk)))
			(if (= lpos (- cut-top 1))
			    (let* ((lab (new-label))
				   (t-retain (varset-in? v t-lv))
				   (e-retain (varset-in? v e-lv))
				   (t-stk (if t-retain cut-stk (cdr cut-stk)))
				   (e-stk (if e-retain cut-stk (cdr cut-stk))))
			      (if t-first
				  (k (c-append
				      (pop-c diff)
				      (c-list
				       (compare-i
					#f sy t-retain e-retain c lab)))
				     (c-list (label-i lab))
				     t-stk
				     e-stk)
				  (k (c-list (label-i lab))
				     (c-append
				      (pop-c diff)
				      (c-list
				       (compare-i
					#t sy t-retain e-retain c lab)))
				     t-stk
				     e-stk)))
			    (general-case)))))
		   (else (general-case))))
	    (else
	     (general-case))))

	(define (gen exp stk usage dest labmap)
	  ;; usage:
	  ;;   finish
	  ;;     dest: #f -> exit
	  ;;           procedure stack-top -> code
	  ;;   use
	  ;;     dest: n: result at pos n, stack height = n + 1
	  ;;   ignore
	  ;;     dest: n: no result, stack height = n

	  (define (show)
	    (display "EXP: ") (newline) (write (hack exp)) (newline)
	    (display " USAGE: ") (write usage) (newline)
	    (display " DEST: ") (write dest) (newline)
	    (display " LV(B): ") (write (map hack (lv-before exp))) (newline)
	    (display " LV(A): ") (write (map hack (lv-after exp))) (newline)
	    (show-stack stk))

	  (define (simple push-i)
	    (if (eq? usage 'finish)
		(let ((ntop (+ (stack-top stk) 1)))
		  (hwm! ntop)
		  (c-append
		   (c-list (push-i))
		   (if dest (dest ntop) (c-list (exit-i)))))
		(let* ((top (stack-top stk))
		       (diff (- top dest)))
		  (if (eq? usage 'ignore)
		      (pop-c diff)
		      (cond ((zero? diff)
			     (hwm! (+ dest 1))
			     (c-list (push-i)))
			    ((negative? diff)
			     (impossible "simple: stack garble (2)"))
			    (else
			     (c-list (multi-pop-i diff)
				     (push-i))))))))

	  (define (suspension closure instr)
	    (if (eq? usage 'finish)
		(gen closure stk 'finish
		     (lambda (t)
		       (c-append
			(c-list instr)
			(if dest (dest t) (c-list (exit-i)))))
		     labmap)
		(let ((c (gen closure stk usage dest labmap)))
		  (if (eq? usage 'use)
		      (c-append c (c-list instr))
		      c))))

	  (define (build-call revl call-c tailcall-c)
	    (let* ((last (car revl))
		   (lv (lv-after last))
		   (cut-stk (cut-stack stk lv))
		   (cut-top (stack-top cut-stk))
		   (ntop (+ cut-top 1)))
	      (let loop
		  ((c (gen last stk 'use cut-top labmap))
		   (s (cons (cons #f cut-top) cut-stk))
		   (i (+ cut-top 1))
		   (el (cdr revl)))
		(if (not (pair? el))
		    (case usage
		      ((finish)
		       (if dest
			   (c-append c call-c (dest ntop))
			   (c-append c tailcall-c)))
		      ((ignore)
		       (c-append c call-c (pop-c (- ntop dest))))
		      (else
		       (c-append
			c call-c
			(let ((diff (- cut-top dest)))
			  (cond ((zero? diff) (c-empty))
				((= diff 1)
				 (c-list (put-loc-pop-i dest)))
				((negative? diff)
				 (impossible "build-call: negative pop"))
				(else (c-list (put-loc-i dest)
					      (multi-pop-i diff))))))))
		    (loop
		     (c-append c (gen (car el) s 'use i labmap))
		     (cons (cons #f i) s)
		     (+ i 1)
		     (cdr el))))))

	  (define (directly-integrable? f)
	    (and (eq? (ee-type f) 'wrap)
		 (let ((wf (ee-body f)))
		   (and (integrable? wf)
			(let ((sy (integrable-symbol wf)))
			  (and
			   (memq sy
				 '(car cdr cons ref deref assign
				       closure get-closure set-closure
				       append vector list->vector
				       fetch-constant fetch-read-only
				       fetch-variable verify module
				       vector-set! vector-ref))
			   sy))))))

	  (define (gen-integrable sy orig-op args)

	    (define (err)
	      ((semantic-error name)
	       "wrong # of arguments for integrable primitive: " sy))

	    (define (badint)
	      (impossible "bad integer arg to closure or module op"))

	    (define (badloc)
	      (impossible "bad local variable arg to closure op"))

	    (define (onearg)
	      (if (not (= (length args) 1)) (err))
	      (pure (car args) (primitive-i sy)))

	    (define (pure arg ci)
	      (if (eq? usage 'finish)
		  (gen arg stk 'finish
		       (lambda (t)
			 (c-append
			  (c-list ci)
			  (if dest (dest t) (c-list (exit-i)))))
		       labmap)
		  (let ((c (gen arg stk usage dest labmap)))
		    (if (eq? usage 'use)
			(c-append c (c-list ci))
			c))))

	    (define (int-exp e)
	      (and (eq? (ee-type e) 'wrap)
		   (let ((e (ee-body e)))
		     (and (eq? (ee-type e) 'quote)
			  (let ((v (ee-body e)))
			    (and (integer? v)
				 (exact? v)
				 (not (negative? v))
				 v))))))

	    (define (get-int e) (or (int-exp e) (badint)))

	    (define (get-sym e)
	      (define (err) (impossible "get-sym"))
	      (if (not (eq? (ee-type e) 'wrap))
		  (err)
		  (let ((e (ee-body e)))
		    (if (not (eq? (ee-type e) 'quote))
			(err)
			(let ((v (ee-body e)))
			  (if (not (symbol? v))
			      (err)
			      v))))))

	    (define (get-locvar e)
	      (if (not (eq? (ee-type e) 'wrap))
		  (badloc)
		  (let ((e (ee-body e)))
		    (if (not (local? e))
			(badloc)
			e))))

	    (define (twoarg)
	      (if (not (= (length args) 2)) (err))
	      (build-call
	       (reverse args)
	       (c-list (primitive-i sy))
	       (c-list (primitive-i sy) (exit-i))))

	    (case sy
	      ((car cdr ref deref list->vector not) (onearg))
	      ((fetch-constant fetch-read-only fetch-variable)
	       (pure (car args) (fetch-i sy (get-sym (cadr args)))))
	      ((verify)
	       ;; as a quick hack we will only evaluate the first arg
	       ;; and ignore the rest.
	       ;; The validate-instruction makes sure that the topmost stack
	       ;; element is a module -- no siganture comparison (yet).
	       (if (not (= (length args) 2))
		   (impossible "strange verify expression"))
	       (let ((ci (validate-i)))
		 (build-call
		  (list (car args))
		  (c-list ci)
		  (c-list ci (exit-i)))))
	      ((module)
	       (let ((len (length args)))
		 (let ((ci (module-i (- len 2))))
		   (build-call
		    (reverse args)
		    (c-list ci)
		    (c-list ci (exit-i))))))
	      ((cons assign) (twoarg))
	      ((append)
	       (let ((nargs (length args)))
		 (if (= nargs 2)
		     (twoarg)
		     (build-call
		      (reverse (cons orig-op args))
		      (c-list (call-i nargs))
		      (c-list (call-exit-i nargs))))))
	      ((closure)
	       (if (not (pair? args))
		   (err))
	       (let* ((eslots (get-int (car args)))
		      (other (cdr args))
		      (nother (length other))
		      (ci (make-closure-i eslots nother)))
		 (if (zero? nother)
		     (simple (lambda () ci))
		     (build-call
		      (reverse other)
		      (c-list ci)
		      (c-list ci (exit-i))))))
	      ((get-closure)
	       (if (not (= (length args) 2))
		   (err))
	       (let* ((v (get-locvar (car args)))
		      (i (get-int (cadr args)))
		      (lv (lv-after exp))
		      (lpos (var-loc v stk))
		      (top (stack-top stk)))
		 (case usage
		   ((finish)
		    (hwm! (+ top 1))
		    (c-append (c-list (get-vec-i lpos i))
			      (if dest (dest (+ top 1)) (c-list (exit-i)))))
		   ((ignore)
		    (let* ((diff (- top dest)))
		      (pop-c diff)))
		   (else
		    (let* ((diff (- lpos dest)))
		      (if (>= diff 0)
			  (begin
			    (hwm! (+ lpos 2))
			    (c-append
			     (pop-c (- top lpos 1))
			     (if (zero? diff)
				 (c-list
				  (get-vec-i lpos i)
				  (put-loc-pop-i lpos))
				 (c-list
				  (get-vec-i lpos i)
				  (put-loc-i dest)
				  (multi-pop-i (+ diff 1))))))
			  (simple
			   (lambda ()
			     ((if (or (not voiding)
				      (varset-in? v lv))
				  get-vec-i
				  get-vec-void-i)
			      lpos i)))))))))
	      ((set-closure)
	       (if (not (eq? usage 'ignore))
		   (impossible "result of set-closure used"))
	       (if (not (= (length args) 3))
		   (err))
	       (let* ((v (get-locvar (car args)))
		      (i (get-int (cadr args)))
		      (e (caddr args))
		      (lpos (var-loc v stk)))
		 (c-append
		  (gen e stk 'use dest labmap)
		  (c-list
		   (put-vec-pop-i lpos i)))))
	      ((vector)
	       (if (pair? args)
		   (let ((nargs (length args)))
		     (build-call
		      (reverse args)
		      (c-list (vector-i nargs))
		      (c-list (vector-i nargs) (exit-i))))
		   (simple (lambda () (vector-i 0)))))
	      ((vector-ref)
	       (let ((nargs (length args)))
		 (if (not (= nargs 2))
		     (err))
		 (let ((i (int-exp (cadr args))))
		   (if i
		       (pure (car args) (vec-ref-i i))
		       (build-call
			(reverse (cons orig-op args))
			(c-list (call-i nargs))
			(c-list (call-exit-i nargs)))))))
	      ((vector-set!)
	       (let ((nargs (length args)))
		 (if (not (= nargs 3))
		     (err))
		 (let ((i (int-exp (cadr args))))
		   (if i
		       (build-call
			(list (caddr args) (car args))
			(c-list (vec-set-i i))
			(c-list (vec-set-i i) (exit-i)))
		       (build-call
			(reverse (cons orig-op args))
			(c-list (call-i nargs))
			(c-list (call-exit-i nargs)))))))
	      (else (impossible "funny integrable"))))

	  (if showing (show))

	  (case (ee-type exp)
	    ((wrap)
	     (let ((wexp (ee-body exp)))
	       (case (ee-type wexp)
		 ((quote)
		  (simple (lambda ()
			    (take-i (ee-body wexp)))))
		 ((global)
		  (simple (lambda ()
			    (get-glob-i (global-symbol wexp)))))
		 ((integrable)
		  (simple (lambda ()
			    (take-primitive-i
			     (integrable-symbol wexp)))))
		 ((local)
		  (let ((lpos (var-loc wexp stk))
			(top (stack-top stk)))
		    (case usage
		      ((finish)
		       (if (= lpos (- top 1))
			   (if dest (dest top) (c-list (exit-i)))
			   (begin
			     (hwm! (+ top 1))
			     (c-append (c-list (get-loc-i lpos))
				       (if dest
					   (dest (+ top 1))
					   (c-list (exit-i)))))))
		      ((ignore) (pop-c (- top dest)))
		      (else
		       (let* ((diff (- dest lpos)))
			 (if (<= diff 0)
			     (c-append
			      (pop-c (- top lpos 1))
			      (cond ((zero? diff) (c-empty))
				    ((= diff -1)
				     (c-list (put-loc-pop-i dest)))
				    (else
				     (c-list
				      (put-loc-i dest)
				      (multi-pop-i (- diff))))))
			     (simple
			      (lambda ()
				((if (or (not voiding)
					 (varset-in? wexp (lv-after exp)))
				     get-loc-i
				     get-loc-void-i)
				 lpos)))))))))
		 (else (impossible "funny wrap")))))
	    ((set!)
	     (let* ((body (ee-body exp))
		    (gs (global-symbol (car body)))
		    (e (cadr body)))
	       (if (eq? usage 'finish)
		   (gen e stk
			'finish
			(lambda (t)
			  (c-list (put-glob-i gs) (exit-i)))
			labmap)
		   (c-append
		    (gen e stk 'use dest labmap)
		    (c-list ((if (eq? usage 'use)
				 put-glob-i
				 put-glob-pop-i)
			     gs))))))
	    ((let)
	     (let* ((body (ee-body exp))
		    (var (caaar body))
		    (init (cadaar body))
		    (e (cadr body))
		    (lv (lv-before e))
		    (cut-stk (cut-stack stk lv))
		    (cut-top (stack-top cut-stk))
		    (ignore (not (varset-in? var lv)))
		    (init-code
		     (gen init stk (if ignore 'ignore 'use) cut-top labmap))
		    (tmp-stk
		     (if ignore cut-stk (cons (cons var cut-top) cut-stk)))
		    (e-code
		     (gen e tmp-stk usage dest labmap)))
	       (c-append init-code e-code)))
	    ((clambda)
	     (let ((body (ee-body exp)))
	       (suspension
		(cadr body)
		(lambda-i
		 (let ((args (cadddr body)))
		   (codegen (caddddr body)
			    (car body)
			    (caddr body)
			    (length args)
			    #f args))))))
	    ((cvlambda)
	     (let ((body (ee-body exp)))
	       (suspension
		(cadr body)
		(lambda-i
		 (let ((args (cadddr body)))
		   (codegen (cadddddr body)
			    (car body)
			    (caddr body)
			    (length args)
			    #t
			    (append args (list (caddddr body)))))))))
	    ((cdelay)
	     (let ((body (ee-body exp)))
	       (suspension
		(cadr body)
		(delay-i
		 (codegen (cadddr body)
			  (car body)
			  (caddr body)
			  0 #f '())))))
	    ((if)
	     (let* ((body (ee-body exp))
		    (cnd (car body))
		    (t (cadr body))
		    (e (caddr body)))
	       (gen-cnd
		cnd stk #t (lv-before t) (lv-before e) labmap
		(lambda (i-t-code i-e-code t-stk e-stk)
		  (let ((t-code (c-append
				 i-t-code
				 (gen t t-stk usage dest labmap)))
			(e-code (c-append
				 i-e-code
				 (gen e e-stk usage dest labmap))))
		    (if (or (eq? usage 'finish)
			    (c-labels-only? e-code))
			(c-append t-code e-code)
			(let ((end-lab (new-label)))
			  (c-append
			   t-code (c-list (jump-forward-i end-lab))
			   e-code (c-list (label-i end-lab))))))))))
	    ((app)
	     (let* ((body (ee-body exp))
		    (f (car body)))
	       (cond
		((directly-integrable? f)
		 =>
		 (lambda (sy) (gen-integrable sy f (cdr body))))
		(else
		 (let* ((revl (reverse body))
			(nargs (- (length revl) 1)))
		   (build-call
		    revl
		    (c-list (call-i nargs))
		    (c-list (call-exit-i nargs))))))))
	    ((label)
	     (let* ((body (ee-body exp))
		    (lab (car body))
		    (clab (new-label))
		    (bl (cadr body))
		    (e (caddr body)))
	       (if (not (pair? bl))
		   (let* ((lv (lv-before exp))
			  (cut-stk (cut-stack stk lv))
			  (top (stack-top stk))
			  (cut-top (stack-top cut-stk))
			  (diff (- top cut-top)))
		     (c-append
		      (pop-c diff)
		      (c-list (label-i clab))
		      (gen e cut-stk usage dest
			   (cons (list lab clab cut-top) labmap))))
		   (let* ((first (cadar bl))
			  (lv (lv-before first))
			  (cut-stk (cut-stack stk lv))
			  (cut-top (stack-top cut-stk)))
		     (let loop
			 ((c (gen first stk 'use cut-top labmap))
			  (s (cons (cons (caar bl) cut-top) cut-stk))
			  (i (+ cut-top 1))
			  (bl (cdr bl)))
		       (if (not (pair? bl))
			   (c-append c
				     (c-list (label-i clab))
				     (gen e s usage dest
					  (cons
					   (list lab clab cut-top) labmap)))
			   (loop
			    (c-append c (gen (cadar bl) s 'use i labmap))
			    (cons (cons (caar bl) i) s)
			    (+ i 1)
			    (cdr bl))))))))
	    ((goto)
	     (let* ((body (ee-body exp))
		    (lab (car body))
		    (el (cdr body))
		    (labinfo (assv lab labmap))
		    (clab (cadr labinfo))
		    (start-idx (caddr labinfo))
		    (top (stack-top stk))
		    (jump (c-list (check-i) (jump-backward-i clab))))
	       (if (not (pair? el))
		   (c-append (pop-c (- top start-idx)) jump)
		   (let* ((first (car el))
			  (lv (lv-after first))
			  (cut-stk (cut-stack stk lv))
			  (cut-top (stack-top cut-stk)))

		     (define (copy-stuff from-low from-high to-low)
		       (let* ((len (- from-high from-low))
			      (to-high (+ len to-low))
			      (diff (- from-low to-high)))
			 (define (loop to)
			   (if (< to to-low)
			       (pop-c diff)
			       (c-append
				(c-list (put-loc-pop-i to))
				(loop (- to 1)))))
			 (loop (- to-high 1))))

		     (let loop
			 ((c (gen first stk 'use cut-top labmap))
			  (s (cons (cons #f cut-top) cut-stk))
			  (i (+ cut-top 1))
			  (el (cdr el)))
		       (if (not (pair? el))
			   (c-append c (copy-stuff cut-top i start-idx) jump)
			   (loop
			    (c-append c (gen (car el) s 'use i labmap))
			    (cons (cons #f i) s)
			    (+ i 1)
			    (cdr el))))))))
	    (else (impossible
		   (string-append
		    "funny expression type: "
		    (string-write (ee-type exp)))))))
	   
	`(,name
	  ,nargs
	  ,has-rarg
	  ,(list->vector (map (lambda (x) (local-symbol x)) argl))
	  ,(reverse constants)
	  ,hwm
	  ,nlab
	  ,@(c->list (c-append
		      (c-list (check-i))
		      (gen exp istk 'finish #f '()))))))


    (define (cg exp)

      (define (struct-err)
	(impossible "expression looks strange (no rts factorization)"))

      (let* ((b (ee-body exp))
	     (ign (if (not (eq? (ee-type exp) 'clambda))
		      (struct-err)))
	     (cv (car b))
	     (cexp (cadr b))
	     (fnam (caddr b))
	     (args (cadddr b))
	     (body (caddddr b))
	     (ign (if (not (and (eq? (ee-type cexp) 'wrap)
				(eq? (ee-type (ee-body cexp)) 'quote)
				(= (length args) 1)))
		      (struct-err))))

	(codegen body cv fnam 1 #f args)))))
