/* tn5250 -- an implentation of the 5250 telnet protocol.
 * Copyright (C) 1997 Michael Madore
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"
#include <stdlib.h>
#include <assert.h>
#include <ctype.h>
#include "buffer.h"
#include "displaybuf.h"
#include "stream5250.h"
#include "formattable.h"
#include "terminal.h"
#include "session.h"
#include "utility.h"

Session::Session ()
	: disp_buf (80, 24)
{
   term = NULL;

   inhibited = 0;
   invited = 1;

   for (int n = 0; n < sizeof (saved_bufs)/sizeof(DisplayBuffer*); n++)
   	saved_bufs[n] = NULL;
}

Session::~Session ()
{
   for (int n = 0; n < sizeof (saved_bufs)/sizeof(DisplayBuffer*); n++) {
      if (saved_bufs[n] != NULL)
	 delete saved_bufs[n];
   }
      
   if (strm != NULL)
      delete strm;
   if (term != NULL)
      delete term;
}

Terminal *Session::terminal () const
{
   return term;
}

Terminal *Session::terminal (Terminal *newterminal)
{
   Terminal *old = term;
   if ((term = newterminal) != NULL) {
      if (strm != NULL) {
	 if ((term->flags () & Terminal::HAS_COLOR) != 0)
	    strm->SetTerminalType ("IBM-3179-2");
	 else
	    strm->SetTerminalType ("IBM-5251-11");
      }
      term->update (disp_buf);
      term->update_indicators (disp_buf);
   }
   return old;
}

Stream5250 *Session::stream () const
{
   return strm;
}

Stream5250 *Session::stream (Stream5250 *newstream)
{
   Stream5250 *old = strm;
   if ((strm = newstream) != NULL) {
      if (term != NULL) {
	 if ((term->flags () & Terminal::HAS_COLOR) != 0)
	    strm->SetTerminalType ("IBM-3179-2");
	 else
	    strm->SetTerminalType ("IBM-5251-11");
      }
      term->update (disp_buf);
      term->update_indicators (disp_buf);
   }
   return old;
}

void Session::MainLoop ()
{
   int r;
   while (1) {
      r = term->waitevent ();
      if ((r & Terminal::EVENT_QUIT) != 0)
      	 return;
      if ((r & Terminal::EVENT_KEY) != 0)
	 HandleKey ();
      if ((r & Terminal::EVENT_DATA) != 0) {
	 if (!strm->HandleReceive ())
	    return;
	 HandleReceive ();
      }
   }
}

void Session::HandleKey ()
{
   int curfield;
   int shiftcount;
   int done = 0, send = 1;
   unsigned char aidcode;
   int cur_key;
   int attrib;
   unsigned char fieldtype;
   int valid_char;
   int X, Y;

   LOG (("HandleKey: entered.\n"));

   curfield = table.CurField();
   cur_key = term->getkey ();
   while(cur_key != -1) {

      X = disp_buf.cursor_x ();
      Y = disp_buf.cursor_y ();

      if(inhibited) {
#if 0
	 if(cur_key == Terminal::K_LEFT || cur_key == Terminal::K_RIGHT ||
	    cur_key == Terminal::K_DOWN || cur_key == Terminal::K_UP ||
	    cur_key == Terminal::K_TAB || cur_key == Terminal::K_BACKTAB) {
	    inhibited = 0;
	    disp_buf.clear_indicator (DisplayBuffer::IND_INHIBIT);
	 }
#endif
         if(cur_key == Terminal::K_RESET) 
         {
            inhibited = 0;
            disp_buf.clear_indicator (DisplayBuffer::IND_INHIBIT);
         }
	 else {
	    cur_key = term->getkey(); 
	    continue;
	 }
      }

	   switch(cur_key) {
 	      case Terminal::K_BACKSPACE:
 	      case Terminal::K_LEFT:
 	         disp_buf.left ();
		 curfield = table.FieldNum (disp_buf.cursor_x (),
		 	disp_buf.cursor_y ());
                 table.SetCurField (curfield);
                 break;
 	      case Terminal::K_RIGHT:
 	         disp_buf.right ();
		 curfield = table.FieldNum (disp_buf.cursor_x (),
		 	disp_buf.cursor_y ());
                 table.SetCurField (curfield);
                 break;
 	      case Terminal::K_UP:
 	         disp_buf.up ();
	         curfield = table.FieldNum (disp_buf.cursor_x (),
			 disp_buf.cursor_y ());
                 table.SetCurField (curfield);
                 break;
 	      case Terminal::K_DOWN:
 		 disp_buf.down ();
		 curfield = table.FieldNum (disp_buf.cursor_x (),
		         disp_buf.cursor_y ());
	         table.SetCurField (curfield);
                 break;
 	      case(Terminal::K_F1) :
		 done = 1;
		 aidcode = AID_F1;
		 break;
 	      case(Terminal::K_F2) :
		 done = 1;
		 aidcode = AID_F2;
		 break;
 	      case(Terminal::K_F3) :
		 done = 1;
		 aidcode = AID_F3; 
		 break;
 	      case(Terminal::K_F4) :
		 done = 1;
		 aidcode = AID_F4;
		 break;
 	      case(Terminal::K_F5) :
		 done = 1;
		 aidcode = AID_F5;
		 break;
 	      case(Terminal::K_F6) :
		 done = 1;
		 aidcode = AID_F6;
		 break;
 	      case(Terminal::K_F7) :
		 done = 1;
		 aidcode = AID_F7;
		 break;
 	      case(Terminal::K_F8) :
		 done = 1;
		 aidcode = AID_F8;
		 break;
 	      case(Terminal::K_F9) :
		 done = 1;
		 aidcode = AID_F9;
		 break;
 	      case(Terminal::K_F10) :
		 done = 1;
		 aidcode = AID_F10;
 		 LOG (("AID_F10\n"));
		 break;
 	      case(Terminal::K_F11) :
		 done = 1;
		 aidcode = AID_F11;
		 break;
 	      case(Terminal::K_F12) :
		 done = 1;
		 aidcode = AID_F12;
		 break;
 	      case(Terminal::K_F13) :
		 done = 1;
		 aidcode = AID_F13;
		 break;
 	      case(Terminal::K_F14) :
		 done = 1;
		 aidcode = AID_F14;
		 break; 
 	      case(Terminal::K_F15) :
		 done = 1;
		 aidcode = AID_F15;
		 break;
 	      case(Terminal::K_F16) :
		 done = 1;
		 aidcode = AID_F16;
		 break;
 	      case(Terminal::K_F17) :
		 done = 1;
		 aidcode = AID_F17;
		 break;
 	      case(Terminal::K_F18) :
		 done = 1;
		 aidcode = AID_F18;
		 break;
 	      case(Terminal::K_F19) :
		 done = 1;
		 aidcode = AID_F19;
		 break;
 	      case(Terminal::K_F20) :
		 done = 1;
		 aidcode = AID_F20;
		 break;
 	      case(Terminal::K_F21) :
		 done = 1;
		 aidcode = AID_F21;
		 break;
 	      case(Terminal::K_F22) :
		 done = 1;
		 aidcode = AID_F22;
		 break;
 	      case(Terminal::K_F23) :
		 done = 1;
		 aidcode = AID_F23;
		 break;
 	      case(Terminal::K_F24) :
		 done = 1;
		 aidcode = AID_F24;
		 break;
 	      case(Terminal::K_HOME) :
                 if(pending_insert) {
                    disp_buf.gotoIC();
                 }
                 else {
		    curfield = table.FirstField();
                    if(curfield > 0) {
 		       disp_buf.cursor (table.StartRow (table.CurField()) -1,
 		 	   table.StartCol (table.CurField ()) -1);
                    }
                    else {
                       disp_buf.cursor(0,0);
                    }
                 }
		 break;
 	      case(Terminal::K_END) :
		 curfield = table.FieldNum(X,Y);
		 if(curfield >= 0 && !table.IsBypass(curfield)) {
 		    disp_buf.cursor (table.StartRow (curfield) -1,
 		    	table.StartCol (curfield) -1);
 		    disp_buf.right (table.CountEOF (curfield));
		 }
		 break;
 	      case(Terminal::K_DELETE) :
		 curfield = table.FieldNum(X,Y);
		 if(curfield < 0 || table.IsBypass(curfield)) {
		    inhibited = 1;
 		    disp_buf.set_indicator (DisplayBuffer::IND_INHIBIT);
		 }
		 else {
		    table.DelChar(X, Y);
		    shiftcount = table.CountRight(curfield, X, Y);
 		    disp_buf.del (shiftcount);
		 }
		 break;
 	      case(Terminal::K_INSERT) :
		 if ((disp_buf.indicators () & DisplayBuffer::IND_INSERT)!=0) {
 		    disp_buf.clear_indicator (DisplayBuffer::IND_INSERT);
		 }
		 else {
 		    disp_buf.set_indicator (DisplayBuffer::IND_INSERT);
		 }
		 break;
 	      case(Terminal::K_TAB) :
		 LOG (("HandleKey: (K_TAB) curfield = %d\n", curfield));
		 curfield = table.NextField(disp_buf.cursor_x (),
		    disp_buf.cursor_y ()); 
 		 disp_buf.cursor (table.StartRow (curfield) -1,
	            table.StartCol (curfield) -1);
		 break;

 	      case(Terminal::K_BACKTAB) :
		 /* Backtab: Move to start of this field, or start of
		  * previous field if already there. */
		 LOG (("HandleKey: (K_BACKTAB) curfield = %d\n", curfield));

		 if (table.CountLeft(curfield, disp_buf.cursor_x(),
				     disp_buf.cursor_y()) == 0) {
		    curfield = table.PrevField(disp_buf.cursor_x (),
					       disp_buf.cursor_y ());
		 }
 		 disp_buf.cursor (table.StartRow (curfield) -1,
 		    table.StartCol (curfield) -1);
		 break;

 	      case(Terminal::K_ENTER) :
		 done = 1;
		 aidcode = AID_ENTER;
		 break;

 	      case(Terminal::K_ROLLDN) :
		 done = 1;
		 aidcode = AID_PGUP;
		 break;
 	      case(Terminal::K_ROLLUP) :
		 done = 1;
		 aidcode = AID_PGDN;
		 break;

	       case(Terminal::K_FIELDEXIT) :
		  curfield = table.FieldNum(X,Y);
		  if(curfield < 0 || table.IsBypass(curfield)) {
		     inhibited = 1;
 		     disp_buf.set_indicator (DisplayBuffer::IND_INHIBIT);
		  }
		  else {
		     table.FieldExit(disp_buf, X, Y);
                     curfield = table.NextField();
                     disp_buf.cursor(table.StartRow(curfield) - 1,
				     table.StartCol(curfield) - 1);
                     term->update(disp_buf);
		     if (table.IsAutoEnter(curfield)) done = 1;
		  }
		  break;

	      case Terminal::K_SYSREQ:
		 done = 1;
		 SystemRequest();
		 send = 0;
		 break;
      
              case Terminal::K_ATTENTION:
                 done = 1;
                 Attention();
                 send = 0;
                 break;

	      default:
 	         LOG (("HandleKey: cur_key = %c\n", cur_key));

		  /* Field Exit driven by '-' and '+', for Numeric Only and Signed Numbers */
		  curfield = table.FieldNum(X,Y);

		  if (cur_key == '+' || cur_key == '-') {
		    if(curfield < 0 || table.IsBypass(curfield)) {
		      inhibited = 1;
 		     disp_buf.set_indicator (DisplayBuffer::IND_INHIBIT);
		      break;
		    }
		    else {
		      if (table.IsNumOnly (curfield) || table.IsSignedNum(curfield)) {
		        table.FieldExit(disp_buf, X, Y);
			if (cur_key == '-') {
			  table.SetMinusZoneScreen(disp_buf, curfield);
			}
			  
                        curfield = table.NextField();
                        disp_buf.cursor(table.StartRow(curfield) - 1,
					table.StartCol(curfield) - 1);
			term->update(disp_buf);
			if (table.IsAutoEnter(curfield)) done = 1;
			break;
		      }
		    }
		  } /* End of '-' and '+' Field Exit Processing */


 		 if(cur_key >= Terminal::K_FIRST_SPECIAL || cur_key < ' ') {
		    break;
		 }

		 curfield = table.FieldNum(X,Y);
		 table.SetCurField(curfield);
		 if(curfield < 0 || table.IsBypass(curfield)) {
		    inhibited = 1;
		    disp_buf.set_indicator (DisplayBuffer::IND_INHIBIT);
		 }
		 else {

		    fieldtype = table.FieldType(curfield);

		    valid_char = 0;
		    LOG (("HandleKey: fieldtype = %d\n", fieldtype));
		    switch(fieldtype)
		    {
		       case ALPHA_SHIFT :
			  valid_char = 1;
			  break;

			case ALPHA_ONLY :
			   if (isalpha(cur_key) || 
			       cur_key == ',' ||
			       cur_key == '.' ||
			       cur_key == '-' ||
			       cur_key == ' ') {
			      valid_char = 1;
			   }

			   break;

                        case NUM_SHIFT :
			   valid_char = 1;
			   break;

                        case NUM_ONLY :
                           if(isnumeric(cur_key) ||
			      cur_key == '+' ||
			      cur_key == ',' ||
			      cur_key == '.' ||
			      cur_key == '-' ||
			      cur_key == ' ' ) {
                              valid_char = 1;
			  } else {
			     inhibited = 1;
			     disp_buf.set_indicator (
			     	DisplayBuffer::IND_INHIBIT);
			  }
			   break;

                        case KATA_SHIFT :
                           LOG(("KATAKANA not implemneted.\n"));
			   valid_char = 1;
			   break;
		 
		       case DIGIT_ONLY :
			  if(isdigit(cur_key)) {
			     valid_char = 1;
			  }
			  break;

                        case MAG_READER :
                           LOG(("MAG_READER not implemneted.\n"));
			   valid_char = 1;
			   break;

		        case SIGNED_NUM :
			  if(isdigit(cur_key) ||
			     cur_key == '+' ||
			     cur_key == '-') {
			     valid_char = 1;
			  }
		    }

		    if(valid_char)
		    {
		       if(table.IsMonocase(curfield)) {
			  cur_key = toupper(cur_key);
		       }
		       if((disp_buf.indicators () 
		       		& DisplayBuffer::IND_INSERT) != 0) {
			  if (table.IsFull(curfield)) {
			     inhibited = 1;
			     disp_buf.set_indicator (
			     	DisplayBuffer::IND_INHIBIT);
			     break;
			  }
			  table.InsChar(X, Y, ascii2ebcdic(cur_key));
		       }
		       else {
			  table.AddChar(X, Y, ascii2ebcdic(cur_key));
			  LOG (("HandleKey: curfield = %d\n", curfield));
		       }

		       attrib = table.Attribute(curfield);

		       if((disp_buf.indicators ()
		       		& DisplayBuffer::IND_INSERT) != 0) {
			  shiftcount = table.CountRight(curfield, X, Y);
			  disp_buf.ins (ascii2ebcdic (cur_key), shiftcount);
		       }
		       else
		          disp_buf.add (ascii2ebcdic (cur_key));

		       if( (X+1 == table.EndCol(curfield)) && (Y+1 == table.EndRow(curfield)) ) {
			  table.NextField();
			  disp_buf.cursor (table.StartRow (table.CurField ()) -1,
			  	table.StartCol (table.CurField ()) -1);
		       }
		       break;
		    }
		     else {
		        inhibited = 1;
   		        disp_buf.set_indicator (DisplayBuffer::IND_INHIBIT);
			break;
		     }

		 }
	   }
	cur_key = term->getkey ();
	}

	if (done && send) {
           disp_buf.set_indicator (DisplayBuffer::IND_X_SYSTEM);
	   term->update_indicators (disp_buf);
	   SendFields (cur_command, aidcode);
	}
	term->update (disp_buf);
	term->update_indicators (disp_buf);
}

// Tell the socket to receive as much data as possible, then, if there are
// full packets to handle, handle them.
void Session::HandleReceive ()
{
	int atn;

        LOG (("HandleReceive: entered.\n"));
	while (strm->GetPacketCount () > 0) {
		inpacket.Init();
		strm->GetPacket(&inpacket);
		cur_opcode = inpacket.GetOpcode();
		atn = inpacket.Attention();

		LOG (("HandleReceive: cur_opcode = 0x%02X %d\n", cur_opcode,
		  atn));

		switch(cur_opcode) {
		case Record::INVITE:
			Invite ();
			break;

		case Record::OUTPUT_ONLY:
			OutputOnly ();
			break;

		case Record::CANCEL_INVITE:
			CancelInvite ();
			break;

		case Record::MESSAGE_ON:
			MessageOn ();
			break;

		case Record::MESSAGE_OFF:
			MessageOff ();
			break;

		case Record::NO_OP:
		case Record::PUT_GET:
		case Record::SAVE_SCR:
		case Record::RESTORE_SCR:
		case Record::READ_IMMED:
		case Record::READ_SCR:
			break;

		default:
		      printf("Error: Unknown opcode %2.2X\n", cur_opcode);
		      exit (1);
		}
	      
		if(!inpacket.IsChainEnd())
			ProcessStream();
	}
}

void Session::Invite()
{
   LOG (("Invite: entered.\n"));
   invited = 1;
   disp_buf.clear_indicator (DisplayBuffer::IND_X_CLOCK);
   term->update_indicators (disp_buf);
}

void Session::CancelInvite()
{
   LOG (("CancelInvite: entered.\n"));
   disp_buf.set_indicator (DisplayBuffer::IND_X_CLOCK);
   term->update_indicators (disp_buf);
   strm->SendPacket(0, Record::DISPLAY, Record::H_NONE, Record::CANCEL_INVITE,
   	NULL);   
   invited = 0;
}

void Session::SendFields (int cur_command, int aidcode)
{
   int curfield, temp;
   Buffer field_buf;
   unsigned char c;
   int X, Y, size;

   X = disp_buf.cursor_x ();
   Y = disp_buf.cursor_y ();

   LOG (("SendFields: Number of fields: %d\n", table.NumFields ())); 

   field_buf.length (2);
   field_buf[0] = Y+1;
   field_buf[1] = X+1;

   if (aidcode)
      field_buf.append ((unsigned char)aidcode);

   LOG (("SendFields: row = %d; col = %d; aid = 0x%02x\n", Y, X, aidcode));

   if (cur_command == CMD_READ_INPUT_FIELDS) {
      for (curfield = 0; curfield < table.NumFields(); curfield++) {
	 unsigned char c;
	 size = table.Length(curfield);	
	 for (temp = 0; temp < size; temp++) {
	    c = table.GetChar (curfield, temp);
	    field_buf.append (c == 0 ? 0x40 : c);
	 }
      }
   } else { /* CMD_READ_IMMEDIATE or CMD_READ_MDT_FIELDS */
      for(curfield = 0; curfield < table.NumFields(); curfield++) {
	 LOG (("SendFields: curfield = %d; length = %d\n", curfield,
	    table.Length (curfield)));

	 if(table.IsModified(curfield)) {
            table.ClearModified(curfield);
	    LOG (("SendFields: Sending:\n"));
	    table.DumpField(curfield);

	    field_buf.append ((unsigned char)SBA);
	    field_buf.append ((unsigned char)table.StartRow(curfield));
	    field_buf.append ((unsigned char)table.StartCol(curfield));

	    size = table.Length(curfield);
	    LOG (("SendFields: size = %d\n", size));
	    for(temp = 0; temp < size; temp++) {
	       c = table.GetChar (curfield, temp);
	       field_buf.append (c == 0 ? 0x40 : c);
	    }
	 }

#ifndef NDEBUG
	 LOG (("SendFields: data = '"));
	 for(temp=0; temp < table.Length(curfield); temp++)
	    LOG (("%c", ebcdic2ascii(table.GetChar(curfield,temp))));
	 LOG (("'\n"));
#endif /* NDEBUG */
      }
   }

   disp_buf.clear_indicator (DisplayBuffer::IND_INSERT);
   term->update_indicators (disp_buf);
   
   strm->SendPacket(field_buf.length (), Record::DISPLAY, Record::H_NONE,
   	Record::PUT_GET, field_buf.get_data ());
}

void Session::ProcessStream()
{
   LOG (("ProcessStream: entered.\n"));
   pending_insert = 0;
   while(!inpacket.IsChainEnd() ) {

      cur_command = inpacket.GetByte();
      LOG (("ProcessStream: cur_command = 0x%02X\n", cur_command));
      ASSERT (cur_command == ESC);

      cur_command = inpacket.GetByte();
      switch(cur_command) {
	 case(CMD_WRITE_TO_DISPLAY) :
	    WriteToDisplay();
	    break;
	 case(CMD_CLEAR_UNIT) :
	    ClearUnit();
	    break;
	 case(CMD_CLEAR_FORMAT_TABLE) :
	    ClearFormatTable();
	    break;
	 case(CMD_READ_MDT_FIELDS) :
	    ReadMDTFields();
	    break;
	 case(CMD_READ_IMMEDIATE) :
	    ReadImmediate();
	    break;
	 case(CMD_READ_INPUT_FIELDS) :
	    ReadInputFields();
	    break;
	 case(CMD_READ_SCREEN_IMMEDIATE) :
	    ReadScreenImmediate();
	    break;
	 case(CMD_WRITE_STRUCTURED_FIELD) :
	    WriteStructuredField();
	    break;
	 case(CMD_SAVE_SCREEN) :
	    SaveScreen();
	    break;
	 case(CMD_RESTORE_SCREEN) :
	    RestoreScreen();
	    break;
	 case(CMD_WRITE_ERROR_CODE) :
	    WriteErrorCode();
	    break;
	 case(CMD_ROLL) :
	    Roll ();
	    break;
	 default :
	    printf("Error: Unknown command %2.2X\n", cur_command);
	    exit(1);
      }
   }
}

void Session::WriteErrorCode()
{
   unsigned char cur_order;
   int done;
   int curX, curY;

   LOG (("WriteErrorCode: entered.\n"));

   curX = disp_buf.cursor_x (); 
   curY = disp_buf.cursor_y ();

   disp_buf.cursor (23, 0);
   done = 0;
   while(!done) {
      if(inpacket.IsChainEnd()) {
         done = 1;
      }
      else {
         cur_order = inpacket.GetByte();
#ifndef NDEBUG
         if(cur_order > 0 && cur_order < 0x40)
	    LOG (("\n"));
#endif /* NDEBUG */
         if(cur_order == IC)
         {
            InsertCursor();
            pending_insert = 1;
         }
         else if(cur_order == ESC)
         {
            done = 1;
            inpacket.UnGetByte();
         }
         else if (Printable(cur_order))
	       disp_buf.add (cur_order); 
	 else {
               printf("Error: Unknown order -- %2.2X --\n", cur_order);
               inpacket.Dump();                 
         }
      }
   }
   LOG (("\n"));
   disp_buf.cursor (curY, curX);
   disp_buf.set_indicator (DisplayBuffer::IND_INHIBIT);
   inhibited = 1;
   term->update (disp_buf);
}

void Session::WriteToDisplay()
{
   unsigned char cur_order;
   unsigned char CC1;
   unsigned char CC2;
   int done;
   int count;
   int fieldnum;
   int Y, X;

   LOG (("WriteToDisplay: entered.\n"));

   CC1 = inpacket.GetByte();
   CC2 = inpacket.GetByte();
   LOG (("WriteToDisplay: 0x%02X:0x%02X\n", CC1, CC2));

   done = 0;
   while(!done) {
      if(inpacket.IsChainEnd()) {
         done = 1;
      }
      else {
         cur_order = inpacket.GetByte();
#ifndef NDEBUG
         if(cur_order > 0 && cur_order < 0x40)
	    LOG (("\n"));
#endif
         switch(cur_order) {
            case(IC) :
            case(IC2) :
               InsertCursor();
               pending_insert = 1;
               break;
            case(RA) :
               RepeatToAddress();
               break;
            case(SBA) :
               SetBufferAddress();
               break;
            case(SF) :
               StartOfField();
               break;
            case(SOH) :
               StartOfHeader();
               break;
            case(ESC) :
               done = 1;
               inpacket.UnGetByte();
               break;
            default :
               if(Printable(cur_order)) {

                  X = disp_buf.cursor_x ();
                  Y = disp_buf.cursor_y ();
                  if(table.FieldNum(X,Y) >= 0) {
                     table.PutChar(X,Y,cur_order);
                  }
		  disp_buf.add (cur_order);
#ifndef NDEBUG 
                  if(Attribute(cur_order)) {
                     LOG (("(0x%02X) ", cur_order));
                  }
                  else {
   		     LOG (("%c (0x%02X) ", ebcdic2ascii (cur_order),
			   cur_order));
                  }
#endif

               }
               else {
                  printf("Error: Unknown order -- %2.2X --\n", cur_order);
                  inpacket.Dump();                 
                  exit(1);
               }
            } /* end switch */
         } /* end else */
      } /* end while */
      LOG (("\n"));

      if(pending_insert) {
	 LOG (("WriteToDisplay: disp_buf.set_new_ic()\n"));
	 disp_buf.set_new_ic ();
	 X = disp_buf.cursor_x ();
	 Y = disp_buf.cursor_y ();
         fieldnum = table.FieldNum(X,Y);
	 if (fieldnum >= 0)
            table.SetCurField(fieldnum);
      }
      else
      {
         done = 0;
         count = 0;
         while(!done & count < table.NumFields()) {

            if(!table.IsBypass(count)) {
	       LOG (("WriteToDisplay: Position to field %d\n", count));
	       disp_buf.cursor (table.StartRow (count) -1,
	       	  table.StartCol (count) -1);
	       LOG (("WriteToDisplay: row = %d; col = %d\n",
		  table.StartRow (count), table.StartCol (count)));
	       X = disp_buf.cursor_x (); 
	       Y = disp_buf.cursor_y ();
               fieldnum = table.FieldNum(X,Y);
               if (fieldnum >= 0)
                  table.SetCurField(fieldnum);
               done = 1;
            }

            count++;
         }
      }

      if(!done) {
	 LOG (("WriteToDisplay: Moving IC to (1,1)\n"));
	 disp_buf.cursor (0, 0);
      }
      LOG (("WriteToDisplay: CTL = 0x%02X\n", CC2));
      if (CC2 & CTL_MESSAGE_ON)
         disp_buf.set_indicator (DisplayBuffer::IND_MESSAGE_WAITING);

      if ( (CC2 & CTL_MESSAGE_OFF) && !(CC2 & CTL_MESSAGE_ON) )
         disp_buf.clear_indicator (DisplayBuffer::IND_MESSAGE_WAITING);

      term->update (disp_buf);
      term->update_indicators (disp_buf);
}

void Session::ClearUnit ()
{
   LOG (("ClearUnit: entered.\n"));
   table.Clear();
   disp_buf._rename_me_clear ();
}

void Session::ClearFormatTable()
{
   LOG (("ClearFormatTable: entered.\n"));
   table.Clear();
   disp_buf.cursor (0, 0);
   disp_buf.set_indicator (DisplayBuffer::IND_X_SYSTEM);
}

void Session::ReadImmediate()
{
   LOG (("ReadImmediate: entered.\n"));
   SendFields (CMD_READ_IMMEDIATE, 0);
}

void Session::SystemRequest()
{
   LOG (("SystemRequest: entered.\n"));
   strm->SendPacket(0, Record::DISPLAY, Record::H_SRQ, Record::NO_OP, NULL);   
}

void Session::Attention()
{
   LOG (("Attention: entered.\n"));
   strm->SendPacket(0, Record::DISPLAY, Record::H_ATN, Record::NO_OP, NULL);
}

void Session::OutputOnly()
/*
   I'm not sure what the actual behavior of this opcode is supposed 
   to be.  I'm just sort-of fudging it based on empirical testing.
*/
{
   unsigned char temp[2];

   LOG (("OutputOnly: entered.\n"));

   /* 
      We get this if the user picks something they shouldn't from the
      System Request Menu - such as transfer to previous system.
   */
   if(inpacket.SysRequest()) {
      temp[0] = inpacket.GetByte();
      temp[1] = inpacket.GetByte();
      LOG (("OutputOnly: ?? = 0x02X; ?? = 0x%02X\n", temp[0], temp[1]));
   }
   else {
      /*  
         Otherwise it seems to mean the Attention key menu is being 
         displayed.
      */
   }
}

void Session::SaveScreen()
{
   unsigned char outbuf[4];
   int n;

   LOG (("SaveScreen: entered.\n"));

   for (n = 1; n < (int)(sizeof (saved_bufs)/sizeof (DisplayBuffer*)); n++) {
      if (saved_bufs[n] == NULL)
	 break;
   }

   ASSERT (saved_bufs[n] == NULL);
   saved_bufs[n] = new DisplayBuffer (disp_buf);

   outbuf[0] = 0x04;
   outbuf[1] = 0x12;
   outbuf[2] = (unsigned char)n;
   outbuf[3] = table.Save();

   LOG (("SaveScreen: display buffer = %d; format buffer = %d\n",
      outbuf[2], outbuf[3]));

   strm->SendPacket(4, Record::DISPLAY, Record::H_NONE, Record::SAVE_SCR,
   	outbuf);   
}

void Session::RestoreScreen()
{
   int screen, format;   

   LOG (("RestoreScreen: entered.\n"));

   screen = inpacket.GetByte();
   format = inpacket.GetByte(); 

   LOG (("RestoreScreen: screen = %d; format = %d\n"));

   ASSERT (screen < (int)(sizeof (saved_bufs)/sizeof (DisplayBuffer*)));
   ASSERT (saved_bufs[screen] != NULL);
   disp_buf = *saved_bufs[screen];
   delete saved_bufs[screen];
   saved_bufs[screen] = NULL;

   table.Restore (format);

   term->update(disp_buf);
}

void Session::MessageOn()
{
   LOG (("MessageOn: entered.\n"));
   disp_buf.set_indicator (DisplayBuffer::IND_MESSAGE_WAITING);
   term->update_indicators (disp_buf);
}

void Session::MessageOff()
{
   LOG (("MessageOff: entered.\n"));
   disp_buf.clear_indicator (DisplayBuffer::IND_MESSAGE_WAITING);
   term->update_indicators (disp_buf);
}

void Session::Roll()
{
	unsigned char direction, top, bot;
	int lines;

	direction = inpacket.GetByte ();	
	top = inpacket.GetByte ();
	bot = inpacket.GetByte ();

        LOG (("Roll: direction = 0x%02X; top = %d; bottom = %d\n",
           (int)direction, (int)top, (int)bot));

	lines = (direction & 0x1f);
	if ((direction & 0x80) == 0)
		lines = -lines;

        LOG (("Roll: lines = %d\n", lines));

	if (lines == 0)
		return;

	disp_buf.roll (top, bot, lines);
	term->update (disp_buf);
}

void Session::StartOfField()
{
   int Y, X;
   int done, curpos;
   unsigned char FFW1, FFW2, FCW1, FCW2;
   unsigned char Attr;
   unsigned char Length1, Length2, cur_char;
   int numfields;
   int input_field;
   int charcount;
   int endrow, endcol;

   LOG (("StartOfField: entered.\n"));

   cur_char = inpacket.GetByte();
   
   input_field = 0;
   if ((cur_char & 0xe0) != 0x20) {
      input_field = 1;
      FFW1 = cur_char;
      FFW2 = inpacket.GetByte();

      LOG (("StartOfField: field format word = 0x%02X%02X\n", FFW1, FFW2));
      cur_char = inpacket.GetByte();

      FCW1 = 0;
      FCW2 = 0;
      while ((cur_char & 0xe0) != 0x20) {
         FCW1 = cur_char;
         FCW2 = inpacket.GetByte();
	 
	 LOG (("StartOfField: field control word = 0x%02X%02X\n", FCW1, FCW2));
         cur_char = inpacket.GetByte();
      }
   } else {
      FFW1 = 0;
      FFW2 = 0;
   }

   assert ((cur_char & 0xe0) == 0x20);
   LOG (("StartOfField: attribute = 0x%02X\n", cur_char));
   Attr = cur_char;
   disp_buf.add (cur_char);

   cur_char = inpacket.GetByte();
   Length1 = cur_char;
   Length2 = inpacket.GetByte();

   X = disp_buf.cursor_x ();
   Y = disp_buf.cursor_y ();

   if(input_field) 
   {
      LOG (("StartOfField: Adding field.\n"));
      table.AddField(FFW1, FFW2, FCW1, FCW2, Attr, Length1, Length2, Y+1, X+1);
      numfields = table.NumFields();
      LOG (("StartOfField: length = %d; SR = %d; SC = %d; ER = %d; EC = %d\n",
	 table.Length (numfields-1), table.StartRow (numfields-1),
	 table.StartCol (numfields-1), table.EndRow (numfields-1),
	 table.EndCol (numfields-1)));
   }
   else
   {
      numfields = table.NumFields();
      LOG (("StartOfField: Output only field.\n"));
   }

   Display_FFW(numfields - 1);

   curpos = 0;
   done = 0;
   charcount = 0;
   while(!done) {

      if (inpacket.IsChainEnd ())
         break;

      cur_char = inpacket.GetByte();
      
      if(Printable(cur_char)) {
         if(Attribute(cur_char)) {
	    LOG (("StartOfField: attribute = 0x%02X\n", cur_char));
         }
         else {
            charcount++;
            if(input_field)
               table.FieldChar(numfields-1,curpos,cur_char);
            disp_buf.add (cur_char);
            curpos++;
	    LOG (("StartOfField: char = '%c' (0x%02X)\n",
	       ebcdic2ascii (cur_char), cur_char));
         }
      }
      else {
         done = 1;
         inpacket.UnGetByte();
      }
   }

  endrow = table.EndRow(numfields-1);
  endcol = table.EndCol(numfields-1);
  if(endcol == 80) {
     endcol = 1;
     if(endrow == 24) {
	endrow = 1;
     }
     else {
	endrow++;
     }
  }
  else {
     endcol++;
  }

   LOG (("StartOfField: endrow = %d; endcol = %d\n", endrow, endcol));
   disp_buf.cursor (endrow-1, endcol-1);
   disp_buf.add (0x20);

   if (input_field) {
      table.DumpField(numfields-1);
   }
}

void Session::StartOfHeader()
{
   int length;
   int count;
   unsigned char temp;

   LOG (("StartOfHeader: entered.\n"));

   inpacket.Dump();

   length = inpacket.GetByte();

   LOG (("StartOfHeader: length = %d\nStartOfHeader: data = ", length));
   for(count=0; count < length; count++) {
      temp = inpacket.GetByte();
      LOG (("0x%02X ", temp));
      if(count == 3)
      {
         table.SetMessageLine(temp);
      }
   }
   LOG (("\n"));

   table.Clear();
}

void Session::SetBufferAddress()
{
   int X,Y;
   
   Y = inpacket.GetByte();
   X = inpacket.GetByte();
  
   disp_buf.cursor (Y-1, X-1);

   LOG (("SetBufferAddress: row = %d; col = %d\n", Y, X));
}

void Session::RepeatToAddress()
{
   unsigned char temp[4];
   int X, Y, curcol, count;

   LOG (("RepeatToAddress: entered.\n"));

   temp[0] = inpacket.GetByte();
   temp[1] = inpacket.GetByte();
   temp[2] = inpacket.GetByte();

   LOG (("RepeatToAddress: row = %d; col = %d; char = 0x%02X\n",
      temp[0], temp[1], temp[2]));

   X = disp_buf.cursor_x () + 1;
   Y = disp_buf.cursor_y () + 1;

   LOG (("RepeatToAddress: currow = %d\n", Y));

   count = temp[1] - X+1;
   if(temp[0] - Y  > 0)
   {
      count = count + 80 - X;
      if(temp[0] - Y > 1)
      {
         count = count + 80 * (temp[0]-Y-1);
      }   
   }
   for(curcol = 0; curcol < count; curcol++)
      disp_buf.add (temp[2]);
}

void Session::InsertCursor()
{
   int cur_char1, cur_char2;

   LOG (("InsertCursor: entered.\n"));

   cur_char1 = inpacket.GetByte();
   cur_char2 = inpacket.GetByte();
  
   LOG (("InsertCursor: row = %d; col = %d\n", cur_char1, cur_char2));
   
   disp_buf.set_temp_ic (cur_char1-1, cur_char2-1);
}

void Session::WriteStructuredField()
{
   unsigned char temp[5];

   LOG (("WriteStructuredField: entered.\n"));

   temp[0] = inpacket.GetByte();
   temp[1] = inpacket.GetByte();
   temp[2] = inpacket.GetByte();
   temp[3] = inpacket.GetByte();
   temp[4] = inpacket.GetByte();

   LOG (("WriteStructuredField: length = %d\n",
      (temp[0] << 8) | temp[1]));  
   LOG (("WriteStructuredField: command class = 0x%02X\n", temp[2]));
   LOG (("WriteStructuredField: command type = 0x%02X\n", temp[3]));

   strm->QueryReply();
}

void Session::ReadScreenImmediate()
{
   // FIXME: Dependant on screen size.
   int row, col;
   unsigned char buffer[25][80];

   LOG (("ReadScreenImmediate: entered.\n"));

   for(row = 0; row < disp_buf.height (); row++) {
      for(col = 0; col < disp_buf.width (); col++) {
         buffer[row][col] = (disp_buf[row][col] & 0xff);
      }
   }

   strm->SendPacket(80*25, Record::DISPLAY, Record::H_NONE, Record::NO_OP,
   	(unsigned char *)&buffer[0]);
}

void Session::ReadInputFields()
{
   unsigned char CC1, CC2;

   LOG (("ReadInputFields: entered.\n"));

   CC1 = inpacket.GetByte();
   CC2 = inpacket.GetByte();

   LOG (("ReadInputFields: CC1 = 0x%02X; CC2 = 0x%02X\n", CC1, CC2));
   table.DumpMap();

   disp_buf.clear_indicator (DisplayBuffer::IND_X_SYSTEM
   	| DisplayBuffer::IND_X_CLOCK
	| DisplayBuffer::IND_INHIBIT);
   term->update_indicators (disp_buf);
}

void Session::ReadMDTFields()
{
   unsigned char CC1, CC2;

   LOG (("ReadMDTFields: entered.\n"));

   CC1 = inpacket.GetByte();
   CC2 = inpacket.GetByte();

   LOG (("ReadMDTFields: CC1 = 0x%02X; CC2 = 0x%02X\n", CC1, CC2));
   table.DumpMap();

   disp_buf.clear_indicator (DisplayBuffer::IND_X_SYSTEM
   	| DisplayBuffer::IND_X_CLOCK
	| DisplayBuffer::IND_INHIBIT);
   term->update_indicators (disp_buf);
}


void Session::Display_FFW(int field_number)
{

   LOG(("Session::Display_FFW FFW1 Bypass: %s, DupEnabled: %s, Field is Modified: %s\n",
	table.IsBypass(field_number) ? "Yes" : "No",
	table.IsDupEnable(field_number) ? "Yes" : "No",
	table.IsFieldModified(field_number) ? "Yes" : "No"));

   LOG(("Session::Display_FFW FFW1 Field Edit/Shift Spec: (%d) %s\n",
	table.FieldType(field_number), table.FieldDescription(field_number)));


   LOG(("Session::Display_FFW FFW2 Auto Enter: %s, Field Exit Req'd: %s, Upper Case: %s\n",
	table.IsAutoEnter(field_number) ? "Yes" : "No",
	table.IsFer(field_number) ? "Yes" : "No",
	table.IsMonocase(field_number) ? "Yes" : "No"));

   LOG(("Session::Display_FFW FFW2 Reserved: %s, Mandatory Entry: %s\n",
	table.IsReserved(field_number) ? "1" : "0",
	table.IsMandatory(field_number) ? "Yes" : "No"));

   LOG(("Session::Display_FFW FFW2 Right Adjust/Mandatory Fill (MF): (%d) %s\n",
	table.AdjustFill(field_number), table.AdjustDescription(field_number)));
}
