/* tn5250 -- an implentation of the 5250 telnet protocol.
 * Copyright (C) 1997 Michael Madore
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <assert.h>
#include <string.h>
#include "utility.h"
#include "displaybuf.h"

///////////////////////////////////////////////////////////////////////////////
// DisplayBuffer::Row class

void DisplayBuffer::Row::range_check (int pos) const
{
   ASSERT (pos >= 0);
   ASSERT (pos < w);
}

DisplayBuffer::Row::Row ()
{
   w = 0;
   data = NULL;
}

DisplayBuffer::Row::~Row ()
{
   if (data != NULL)
      delete[] data;
}

int DisplayBuffer::Row::width () const
{
   return w;
}

int DisplayBuffer::Row::width (int newwid)
{
   int oldwid = w;
   w = newwid;
   if (data)
      delete[] data;
   data = new unsigned char[w];
   return oldwid;
}

unsigned char DisplayBuffer::Row::at (int pos) const
{
   range_check (pos);
   return data[pos];
}

unsigned char& DisplayBuffer::Row::at (int pos)
{
   range_check (pos);
   return data[pos];
}

DisplayBuffer::Row& DisplayBuffer::Row::operator = (const DisplayBuffer::Row&r)
{
   ASSERT (w == r.w);
   memcpy (data, r.data, sizeof (unsigned char) * w);
   return *this;
}

///////////////////////////////////////////////////////////////////////////////
// DisplayBuffer class

void DisplayBuffer::range_check (int pos) const
{
   ASSERT (pos >= 0);
   ASSERT (pos < h);
}

DisplayBuffer::DisplayBuffer (int wd, int ht)
{
   w = wd;
   h = ht;
   cx = cy = 0;
   rows = new Row[h];
   for (int n = 0; n < h; n++)
   	rows[n].width (w);
   _rename_me_clear ();
   disp_indicators = 0;
}

DisplayBuffer::DisplayBuffer (const DisplayBuffer& rhs)
{
   w = rhs.w;
   h = rhs.h;
   cx = rhs.cx;
   cy = rhs.cy;
   tcx = rhs.tcx;
   tcy = rhs.tcy;
   disp_indicators = rhs.disp_indicators;

   rows = new Row[h];
   for (int n = 0; n < h; n++) {
      rows[n].width (w); 
      rows[n] = rhs.rows[n];
   }
}

DisplayBuffer::~DisplayBuffer ()
{
   delete[] rows;
}

int DisplayBuffer::width () const
{
   return w;
}

int DisplayBuffer::height () const
{
   return h;
}

int DisplayBuffer::cursor_x () const
{
   return cx;
}

int DisplayBuffer::cursor_y () const
{
   return cy;
}

int DisplayBuffer::cursor_x (int nx)
{
   int ox = cx;
   ASSERT (nx >= 0);
   ASSERT (nx < w);
   cx = nx;
   return ox;
}

int DisplayBuffer::cursor_y (int ny)
{
   int oy = cy;
   ASSERT (ny >= 0);
   ASSERT (ny < h);
   cy = ny;
   return oy;
}

void DisplayBuffer::cursor (int ny, int nx)
{
   cursor_x (nx);
   cursor_y (ny);
}

void DisplayBuffer::_rename_me_clear ()
{
   for (int r = 0; r < h; r++) {
      for (int c = 0; c < w; c++) {
	 rows[r].data[c] = 0;
      }
   }

   cx = cy = 0;
}

const DisplayBuffer::Row& DisplayBuffer::at (int pos) const
{
   range_check (pos);
   return rows[pos];
}

DisplayBuffer::Row& DisplayBuffer::at (int pos)
{
   range_check (pos);
   return rows[pos];
}

void DisplayBuffer::right (int n)
{
   cx += n;
   cy += (cx / w);
   cx = (cx % w);
   cy = (cy % h);
}

void DisplayBuffer::left ()
{
   cx--;
   if (cx == -1) {
      cx = w-1;
      cy--;
      if (cy == -1)
	 cy = h-1;
   }
}

void DisplayBuffer::up ()
{
   cy--;
   if (cy == -1)
      cy = h-1;
}

void DisplayBuffer::down ()
{
   cy++;
   if (cy == h)
      cy = 0;
}

void DisplayBuffer::gotoIC ()
{
   cy = tcy;
   cx = tcx;
}

void DisplayBuffer::add (unsigned char c)
{
   rows[cy][cx] = c;
   right ();
}

void DisplayBuffer::add (int y, int x, const unsigned char *str, int l)
{
   int ocx = cx, ocy = cy;
   cursor (y, x);
   for (; l; l--, str++)
      add (*str);
   cy = ocy;
   cx = ocx;
}

void DisplayBuffer::set_indicator (int inds)
{
   disp_indicators |= inds;
}

void DisplayBuffer::clear_indicator (int inds)
{
   disp_indicators &= ~inds;
}

int DisplayBuffer::indicators () const
{
   return disp_indicators;
}

int DisplayBuffer::indicators (int inds)
{
   int old = disp_indicators;
   disp_indicators = inds;
   return old;
}

void DisplayBuffer::del (int shiftcount)
{
   int x = cx, y = cy, fwdx, fwdy;

   for (int i = 0; i < shiftcount; i++) {
      fwdx = x+1;
      fwdy = y;
      if (fwdx == w) {
	 fwdx = 0;
	 fwdy++;
      }
      rows[y][x] = rows[fwdy][fwdx];
      x = fwdx;
      y = fwdy;
   }
}

void DisplayBuffer::ins (unsigned char c, int shiftcount)
{
   int x = cx, y = cy;
   unsigned char c2;
   
   for (int i = 0; i < shiftcount; i++) {
      c2 = rows[y][x];
      rows[y][x] = c;
      c = c2;
      if (++x == w) {
	 x = 0;
	 y++;
      }
   }
   right ();
}

void DisplayBuffer::set_new_ic ()
{
   cx = tcx;
   cy = tcy;
}

void DisplayBuffer::set_temp_ic (int y, int x)
{
   tcx = x;
   tcy = y;
}

void DisplayBuffer::roll (int top, int bot, int lines)
{
   int n;

   if (lines == 0)
      return;

   if (lines < 0) {
      /* Move text up */
      for (n = top; n <= bot; n++) {
	 if (n+lines >= top)
	    rows[n+lines] = rows[n];
      }
   } else {
      for (n = bot; n >= top; n--) {
	 if (n+lines <= bot)
	    rows[n+lines] = rows[n];
      }
   }
}

DisplayBuffer& DisplayBuffer::operator = (const DisplayBuffer& rhs)
{
   assert (rhs.width () > 0);

   w = rhs.w;
   h = rhs.h;
   cx = rhs.cx;
   cy = rhs.cy;
   tcx = rhs.tcx;
   tcy = rhs.tcy;
   disp_indicators = rhs.disp_indicators;

   if (rows != NULL)
      delete[] rows;
   rows = new Row[h];
   for (int n = 0; n < h; n++) {
      rows[n].width (w);
      rows[n] = rhs.rows[n];
   }

   return *this;
}
