/*
 * Copyright (C) 2000 WIDE Project.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 $Id: parser.c,v 1.3 2000/04/20 08:57:32 nishida Exp $
 */
#include <stdio.h>
#include <sys/param.h>
#include <sys/time.h>
#include <netinet/tcp.h>
#include "tcpillust.h"

#define UNSETVAL 99999

static void Usage __P((void));
static void generate_result __P((void));
static void generate_result_both __P((void));
static void generate_result_client __P((void));
static void generate_result_server __P((void));
static void regist_result __P((int, float, float, struct tcppkt, int));
static void determine_client __P((void));
static void calculate_timegap __P((void));
static void calculate_delay __P((void));
static void calculate_syndelay __P((void));
static void read_file __P((char *, int));
static int acksearch __P((long, long, struct tcppkt *, int, int));
static int idsearch __P((u_short, long, struct tcppkt *, int, int));
static int iltsearch __P((int, long, int));

int	pktcount[2];
struct tcppkt *pkt[2], *client = NULL, *server = NULL;

static int pkttype[2];
static int filecount;
static int clientcount;
static int servercount;
static int ilt_flag = 0;
static int ilt_stat;
static int csflag;
static int	file_type = 0;  /* default file type is binary */
static long clientaddr;
static long serveraddr;
static long last_cack, last_sack;
static long last_cseq, last_sseq;
static long cack_dupcnt, sack_dupcnt;
static long basegap;
static float timegap;
static float syndelay;
static float basetime;

void
parser(argc, argv)
	int		argc;
	char	**argv;
{
	int	opt;
	int xloc, yloc, width, height;
	extern char *optarg;
	extern int optind;

	timegap = delay = syndelay = UNSETVAL;

	if (argc == 1) return;
	while ((opt = getopt(argc, argv, "vVd:s:g:h")) != -1) { 
		switch(opt){
		case 'V':
			ilt_flag |= FL_VERBOSE;	
			break;

		case 'd':
			delay = atof(optarg);
			break;

		case 's':
			syndelay = atof(optarg);
			break;

		case 'g':
			XParseGeometry(optarg, &xloc, &yloc, &width, &height);
			appl_x = xloc;
			appl_y = yloc;
			appl_width = width;
			appl_height = height;
			break;

		case 'v':
			fprintf(stderr, "%s version %s\n", PROG_NAME, PROG_VERSION);
			exit(0);
			break;

#if 0 
		/* currently not implemented */
		case 'T':
			/* this is type option */
			if (optarg[0] == 'a'){
				file_type = 1;	
			} else { 
				if (optarg[0] == 'b'){
					file_type = 0;	
				} else {
					Usage();
				}
			}
			break;
#endif

		case 'h':
		default:
			Usage();
			break;
		}
	}

	argc -= optind;
	argv += optind;

	if (argc != 1 && argc != 2) Usage();

	load_file(argc, argv, 0);
}

void
load_file(argc, argv, clear)
	int argc;
	char **argv;
	int clear;
{
	if (clear){
		csflag = 0;
		timegap = delay = syndelay = UNSETVAL;
		pktcount[0] = pktcount[1] = 0;
		clientcount = servercount = 0;
		last_cack = last_sack = 0;
		last_cseq = last_sseq = 0;
		cack_dupcnt = sack_dupcnt = 0;
	}

	filecount = argc;
	read_file(argv[0], 0);
	if (csflag) pkttype[0] = csflag;

	if (filecount == 2){
		read_file(argv[1], 1);
		if (csflag) pkttype[1] = (csflag == 1?2:1);
	}

	/* which file is client ?? */
	if (!csflag){
		if (ilt_flag & FL_VERBOSE){
			fprintf(stderr, " determin client from SYN..\n");
		}
		determine_client();
		if (ilt_flag & FL_VERBOSE){
			fprintf(stderr, "   %s is ", argv[0]); 
			if (pkttype[0] == TYPE_CLIENT){
				fprintf(stderr, "client side.\n"); 
				if (filecount == 2) 
						fprintf(stderr, "   %s is server side.\n", argv[1]);
			} else {
				fprintf(stderr, "server side\n"); 
				if (filecount == 2) 
						fprintf(stderr, "   %s is client side.\n", argv[1]);
			}
		}
	}

	if (pkttype[0] == TYPE_CLIENT){
		client = pkt[0];
		clientcount = pktcount[0];
		clientaddr = client[0].src_addr;
		serveraddr = client[0].dst_addr;
	} else {
		server = pkt[0];
		servercount = pktcount[0];
		clientaddr = server[0].src_addr;
		serveraddr = server[0].dst_addr;
	}
	if (filecount == 2) {
		if (pkttype[1] == TYPE_CLIENT){
			client = pkt[1];
			clientcount = pktcount[1];
		} else {
			server = pkt[1];
			servercount = pktcount[1];
		}
	}

	/* set prepstat */
	if (filecount == 2) {
		ilt_stat = ST_BOTH;
	} else {
		if (client)
			ilt_stat = ST_CLIENT;
		else
			ilt_stat = ST_SERVER;
	}

	/* should I have to calculate time_gap ? */
	if (filecount == 2){
		if (timegap > UNSETVAL-1){
			if (ilt_flag & FL_VERBOSE){
				fprintf(stderr, " calculate time gap..\n");
			}
			calculate_timegap();
		}
	}

	/* should we have to calculate delay ? */
	if (delay > UNSETVAL-1) calculate_delay();
	if (syndelay > UNSETVAL-1) calculate_syndelay();

	if (delay > UNSETVAL-1 || delay < syndelay) {
		delay = syndelay;
		if (FL_VERBOSE) 
			fprintf(stderr, 
			" warning: failed to determine data delay. use syn delay as data delay\n");
	}

	if (ilt_flag & FL_VERBOSE){
		if (filecount == 2)
			fprintf(stderr, " time gap is %.06f sec\n", basegap + timegap);
		fprintf(stderr, " syn delay is %.06f sec\n", syndelay);
		fprintf(stderr, " data delay is %.06f sec\n", delay);
	}

	generate_result();
	loaded = 1;
}

void
read_file(filename, pktindex)
	char	*filename;
	int		pktindex;	
{
	if (ilt_flag & FL_VERBOSE) 
			fprintf(stderr, " reading from %s...", filename);
	if (!file_type) read_binfile(filename, pktindex);
	if (ilt_flag & FL_VERBOSE) fprintf(stderr, " done.\n");
}

void
Usage()
{
	char	name[] = PROG_NAME;

	fprintf(stderr, "Usage: %s [opts] file [file]\n", name);
	fprintf(stderr, "\t-V: be verbose\n");
	fprintf(stderr, "\t-d: specify data delay (sec)\n");
	fprintf(stderr, "\t-s: specify syn delay (sec)\n");
	fprintf(stderr, "\t-g <geometry>: specify window geometry\n");
	fprintf(stderr, "\t-v: show version number and quit\n");
	fprintf(stderr, "\t-h: display this help message\n");
#if 0
	fprintf(stderr, "\t-T [b|a]: Specify file type. bin or ascii\n");
#endif

	exit(0);
}

void
determine_client()
{
	float f11, f12, f21, f22;
	
	if ( !(pkt[0][0].flags & TH_SYN) ||
		 !(pkt[0][1].flags & TH_SYN) ||
	 	 !(pkt[0][1].flags & TH_ACK)) goto error;

	if ((pkt[0][0].src_addr != pkt[0][1].dst_addr) ||
		(pkt[0][1].src_addr != pkt[0][0].dst_addr)) goto error;
		 
	if (filecount == 2){
		if (!(pkt[1][0].flags & TH_SYN) ||
		 !(pkt[1][1].flags & TH_ACK) ||
		 !(pkt[1][1].flags & TH_SYN)) goto error;

		if ((pkt[1][0].src_addr != pkt[1][1].dst_addr) ||
			(pkt[1][1].src_addr != pkt[1][0].dst_addr)) goto error;
	}

	f11 = pkt[0][1].captime - pkt[0][0].captime;
	f12 = pkt[0][2].captime - pkt[0][1].captime;
	
	if (filecount == 2){
		f21 = pkt[1][1].captime - pkt[1][0].captime;
		f22 = pkt[1][2].captime - pkt[1][1].captime;

		if (f11 > f21){
			pkttype[0] = TYPE_CLIENT;
			pkttype[1] = TYPE_SERVER;
		} else {
			pkttype[1] = TYPE_CLIENT;
			pkttype[0] = TYPE_SERVER;
		}
		return;
	}

	if (f11 > f12)
		pkttype[0] = TYPE_CLIENT;
	else 
		pkttype[0] = TYPE_SERVER;

	return;

error:
	fprintf(stderr, 
		"fatal error. syn packet was lost!\n please use -c or -s option\n");
	exit(-1);
}

void
calculate_timegap()
{
	float c1, s1;

	if (syndelay > UNSETVAL -1){
		/* determin syndelay before calculate timegap */
		c1 = client[1].captime - client[0].captime;
		s1 = server[1].captime - server[0].captime;
		syndelay = (c1 - s1) / 2.0;
	}
	timegap = client[0].captime - server[0].captime + syndelay;

	return;
}

void
calculate_delay()
{
	float rtt, tmptime;
	int	i, count = 100, searchcount, found;

	if (ilt_stat == ST_BOTH || ilt_stat == ST_CLIENT){
		/*
		 * determin delay from client log
		 */
		searchcount = (100 > clientcount? clientcount: 100);
		for (i = 3; i < count; i ++){
			/*
			 * this packet is data packet from client side,
			 * search ack packet 
			 */
			if (client[i].datalen && client[i].src_addr == clientaddr){
				if ((found = acksearch(client[i].seq + client[i].datalen,
						client[i].dst_addr, client, i, searchcount)) > 0){
					/* use half of RTT as delay */	
					if (client[i].captime > client[found].captime) continue;
					rtt = client[found].captime - client[i].captime;
					tmptime = rtt / 2.0;
					if (delay > tmptime) delay = tmptime;
				}
			}
		}
	}

	if (ilt_stat == ST_BOTH || ilt_stat == ST_SERVER){
		/*
		 * determin delay from client log
		 */
		searchcount = (100 > servercount? servercount: 100);
		for (i = 3; i < count; i ++){
			if (server[i].datalen && server[i].src_addr == serveraddr) {
				/*
				 * this packet is data packet from server side,
				 * search ack packet
				 */
				if ((found = acksearch(server[i].seq + server[i].datalen,
					server[i].dst_addr, server, i, searchcount)) > 0){
					/* use half of RTT as delay */
					if (server[i].captime > server[found].captime) continue;
					rtt = server[found].captime - server[i].captime;
					tmptime = rtt / 2.0;
					if (delay > tmptime) delay = tmptime;
				}
			}
		}
	}
}

void
calculate_syndelay()
{
	float rtt;

	if (ilt_stat == ST_BOTH){
		syndelay = ((client[1].captime - client[0].captime) -
					(server[1].captime - server[0].captime)) / 2.0;
		return;
	}
	if (ilt_stat == ST_CLIENT){
		/* use half of RTT as syn delay */	
		rtt = client[1].captime - client[0].captime;
		syndelay = rtt / 2.0;
		return;
	}
	if (ilt_stat == ST_SERVER){
		/* use half of RTT as syn delay */	
		rtt = server[2].captime - server[1].captime;
		syndelay = rtt / 2.0;
		return;
	}
}

void
generate_result()
{
	switch(ilt_stat){
		case ST_SERVER:
			generate_result_server();
			break;
		case ST_CLIENT:
			generate_result_client();
			last_cack = last_sack = 0;
			break;
		case ST_BOTH:
			generate_result_both();
			break;
		default:
			fprintf(stderr, "fatal: ilt_stat is irregal\n");
			exit(-1);
		break;
	}
}

void
generate_result_server()
{
	int	i;
	float peer;

	basetime = server[0].captime - syndelay;

	for (i = 0; i < servercount; i ++){
		if (server[i].src_addr == clientaddr){
			/* this data was sent from client */
			if (server[i].datalen)
				peer = server[i].captime - delay;
			else
				peer = server[i].captime - syndelay;
			regist_result(TYPE_CLIENT, peer, server[i].captime, server[i], 1);
		} else {
			/* this data was sent from server */
			if (server[i].datalen)
				peer = server[i].captime + delay;
			else
				peer = server[i].captime + syndelay;
			regist_result(TYPE_SERVER, peer, server[i].captime, server[i], 1);
		}
	}
	free(server);
	server = NULL;
}

void
generate_result_client()
{
	int	i;
	float peer;

	basetime = client[0].captime;

	for (i = 0; i < clientcount; i ++){
		if (client[i].src_addr == clientaddr){
			/* this data was sent from client */
			if (client[i].datalen)
				peer = client[i].captime + delay;
			else
				peer = client[i].captime + syndelay;
			regist_result(TYPE_CLIENT, client[i].captime, peer, client[i], 1);
		} else {
			/* this data was sent from server */
			if (client[i].datalen)
				peer = client[i].captime - delay;
			else
				peer = client[i].captime - syndelay;
			regist_result(TYPE_SERVER, client[i].captime, peer, client[i], 1);
		}
	}
	free(client);
	client = NULL;
}

void
generate_result_both()
{
	float peer;
	int	ccnt, scnt, found, last_flag = 0;

	basetime = client[0].captime;
	ccnt = scnt = 0;

	while(ccnt != clientcount-1 || scnt != servercount-1){
		while(client[ccnt].src_addr != clientaddr && 
			ccnt < clientcount-1) ccnt ++;
		while(server[scnt].src_addr != serveraddr &&
			scnt < servercount-1) scnt ++;

last:
		if (client[ccnt].captime < server[scnt].captime + timegap){
			if (ccnt == clientcount) continue;
			found = idsearch(client[ccnt].id, clientaddr, 
					server, 0, servercount);
			if (found < 0) {
				/* we lost this packet XXX */
				peer = -1;
			} else
				peer = server[found].captime + timegap;
			regist_result(TYPE_CLIENT, client[ccnt].captime, 
					peer, client[ccnt], 0);
			if (ccnt < clientcount-1) ccnt ++;
		} else {
			if (scnt == servercount) continue;
			found = idsearch(server[scnt].id, serveraddr, 
					client, 0, clientcount);
			if (found < 0) {
				/* we lost this packet XXX */
				peer = -1;
			} else
				peer = client[found].captime;
			regist_result(TYPE_SERVER, peer, 
					server[scnt].captime + timegap, server[scnt], 0);
			if (scnt < servercount-1) scnt ++;
		}

		/* a bit tricky.. */
		if (ccnt == clientcount-1 && scnt == servercount-1 && !last_flag){
			last_flag = 1;
			goto last;
		}

	}
	free(client);
	free(server);
	client = server = NULL;
}

void
regist_result(flag, time1, time2, pkt, chkdupack)
	int	flag;
	float time1, time2;
	struct tcppkt pkt;
	int chkdupack;
{
	float rtime1, rtime2;
	long rseq, rack;
	float sendtime, recvtime; 
	static float cldelay = 0, srvdelay = 0;
	int state, lost;

	state = pkt.datalen? STAT_DATA: STAT_ACK;

	rtime1 = time1 - basetime;
	rtime2 = time2 - basetime;
	if (flag == TYPE_CLIENT){
		rseq = pkt.seq;
		rack = pkt.ack;
		if (rseq >= last_cseq) 
			last_cseq = rseq;
		else
			state = STAT_RETRANS;
		if (rack < last_sseq && rack == last_cack) 
			cack_dupcnt ++;
		else 
			cack_dupcnt = 0;
		last_cack = rack;
		if (chkdupack && cack_dupcnt == NUM_DUPACK -1){
			/* search lost packet */
			lost = iltsearch(TYPE_SERVER, rack, iltcnt);
			if (lost > 0) ilt[lost].state = STAT_LOST;
		}
		if (cack_dupcnt >= NUM_DUPACK -1) state = STAT_DUPACK;
		sendtime = rtime1;
		recvtime = rtime2;
		if (time2 < 0){
			state = STAT_REALLOST;
			recvtime = sendtime;
			recvtime = sendtime + cldelay;
		} else 
			cldelay = recvtime - sendtime;
	} else {
		rseq = pkt.seq;
		rack = pkt.ack;
		if (rseq >= last_sseq) 
			last_sseq = rseq;
		else
			state = STAT_RETRANS;
		if (rack < last_cseq && rack == last_sack) 
			sack_dupcnt ++;
		else
			sack_dupcnt = 0;
		last_sack = rack;
		if (chkdupack && sack_dupcnt == NUM_DUPACK -1){
			/* search lost packet */
			lost = iltsearch(TYPE_CLIENT, rack, iltcnt);
			if (lost > 0) ilt[lost].state = STAT_LOST;
		}
		if (sack_dupcnt >= NUM_DUPACK -1) state = STAT_DUPACK;
		sendtime = rtime2;
		recvtime = rtime1;
		if (time1 < 0){
			state = STAT_REALLOST;
			recvtime = sendtime + srvdelay;
		} else 
			srvdelay = recvtime - sendtime;
	}

	if (!iltcnt)
		ilt = (struct iltdata *)malloc(sizeof(struct iltdata));
	else
		ilt = (struct iltdata *)realloc(ilt, 
					sizeof(struct iltdata) * (iltcnt +1));

	memset(&ilt[iltcnt], 0, sizeof(struct iltdata));
	ilt[iltcnt].flag = flag;
	ilt[iltcnt].sendtime = sendtime;
	ilt[iltcnt].recvtime = recvtime;
	ilt[iltcnt].delay = recvtime - sendtime;
	ilt[iltcnt].state = state;
	ilt[iltcnt].seq = rseq;
	sprintf(ilt[iltcnt].text, " %d:%d(%d) ack %d win %d ", 
			rseq, rseq + pkt.datalen, pkt.datalen, rack, pkt.win);
	if (pkt.flags & (TH_SYN|TH_FIN|TH_RST|TH_PUSH)) {
		if (pkt.flags & TH_SYN) 
			strcat(ilt[iltcnt].text, "S");
		if (pkt.flags & TH_FIN)
			strcat(ilt[iltcnt].text, "F");
		if (pkt.flags & TH_RST)
			strcat(ilt[iltcnt].text, "R");
		if (pkt.flags & TH_PUSH)
			strcat(ilt[iltcnt].text, "P");
	} else
			strcat(ilt[iltcnt].text, ".");

	iltcnt ++;
}

static int
acksearch(ack, addr, pkt, start, num)
	long ack;
	long addr;
	struct tcppkt *pkt;
	int start;
	int num;
{
	int	i;
	for (i = start; i < num; i ++){
		if (pkt[i].src_addr != addr) continue;
		if (pkt[i].ack >= ack)
			return i;
	}
	return -1;
}


static int
seqsearch(seq, addr, pkt, start, num)
	long seq;
	long addr;
	struct tcppkt *pkt;
	int start;
	int num;
{
	int	i;
	for (i = start; i < num; i ++){
		if (pkt[i].src_addr != addr) continue;
		if (pkt[i].seq == seq)
			return i;
	}
	return -1;
}

static int
iltsearch(flag, seq, start)
	int flag;
	long seq;
	int start;
{
	int	i;
	for (i = start; i >= 0; i --){
		if (ilt[i].flag != flag) continue;
		if (ilt[i].seq == seq){
			return i;
		}
	}
	return -1;
}

static int
idsearch(id, addr, pkt, start, num)
	u_short id;
	long addr;
	struct tcppkt *pkt;
	int start;
	int num;
{
	int	i;
	for (i = start; i < num; i ++){
		if (pkt[i].src_addr != addr) continue;
		if (pkt[i].id == id)
			return i;
	}
	return -1;
}
