(* SwiftSurf
 * Sebastien Ailleret *)

(* type for describing a profile *)
type profile = {
    mutable auth_pass : string;
    mutable forbidden_urls : (string list * string) list;
    mutable allowed_urls : string list * string;
    mutable methods : string list;
    mutable canCONNECT : bool;
    mutable allCONNECT : bool;
    mutable req_1 : bool;
    mutable req_2 : bool;
    mutable req_in : bool;
    mutable req_out : bool;
    mutable ans_1 : bool;
    mutable ans_2 : bool;
    mutable ans_in : bool;
    mutable ans_out : bool
  }

(* state of a connexion *)
type state =
  | STARTING
  | DNS
  | DNSDONE of Unix.inet_addr
  | CONNECTING
  | ALIVE
  | FINISHING

(* what are we parsing now *)
type parse =
  | CMD_LINE
  | WAIT_DNS
  | HEADERS
  | HEADERSCONNECT
  | CONTENT
  | END

(* one connexion *)
type connexion = {
    client : Unix.file_descr;
    from : string;
    mutable timeout : float;
    mutable server : Unix.file_descr;
    mutable state : state;
    mutable auth : string;
    mutable prof : profile;
    mutable host : string * int;
    mutable need_proxy : (Unix.sockaddr * string option) option;
    mutable url : string;
    mutable proto_str : string;
    mutable meth : string;
    mutable read_req : Activebuffer.t;
    mutable write_req : Activebuffer.t;
    mutable size_req : int; (* avoid too much memory consumption *)
      (* give the number of byte we still accept to read *)
      (* this number must always remain < buf_size *)
    mutable state_req : parse;
    mutable len_post : int;
    mutable read_ans : Activebuffer.t;
    mutable write_ans : Activebuffer.t;
    mutable size_ans : int;
    mutable state_ans : parse
  }

(* value of the timeout *)
let timeout = ref 30.

(* maximum number of connexions *)
let max_conns = ref (-1)

(* what kind of output is required *)
let stats = ref 0
let debug = ref 0

(* fixed configuration *)
let buf_size = 65536
let dns_cache_size = 100

(* standard answers *)
let make_ans ans_code mime doc =
  Printf.sprintf
    "HTTP/1.0 %s\r\nServer: swiftsurf\r\nContent-Type: %s\r\nContent-Length: %d\r\n\r\n%s"
    ans_code mime (String.length doc) doc

let host_unreach =
  make_ans "404 Not found" "text/html"
    "<html>\n<head>\n<title>ERROR: Unable to locate Server</title>\n</head>\n<body>\n<h1>ERROR</h1>\n<h2>Unable to locate Server</h2></body></html>"

let invalid_req =
  make_ans "400 Proxy Error" "text/html"
    "<html>\n<head>\n<title>ERROR: Invalid HTTP Request</title>\n</head>\n<body>\n<h1>ERROR</h1>\n<h2>Invalid HTTP Request</h2></body></html>"

let unknown_host =
  make_ans "404 Not found" "text/html"
    "<html>\n<head>\n<title>ERROR: Unknown Host</title>\n</head>\n<body>\n<h1>ERROR</h1>\n<h2>cannot determine host ip address</h2></body></html>"

let secure_forbid =
  make_ans "403 Forbidden" "text/html"
    "<HTML><HEAD><TITLE>ERROR: The requested URL could not be retrieved</TITLE></HEAD><BODY>\n<h1>SECURITY ERROR</h1>\n<h2>The requested URL could not be retrieved (forbidden protocol)</h2></BODY></HTML>"

let need_auth = "HTTP/1.0 407 Proxy authentication required\r\nServer: swiftsurf\r\nProxy-Authenticate: Basic realm=\"Give a proxy authentication\"\r\nContent-length: 0\r\n\r\n"

let not_avail =
  make_ans "503 Service Unavailable" "text/html"
    "<html><head><title>ERROR: The requested URL could not be retrieved</title></head><body><h1>ERROR</h1><h2>The requested URL could not be retrieved</h2></body></html>"

(* where is the config file *)
let conf_file_dir = ref "."
