(* SwiftSurf
 * Sebastien Ailleret *)

open Unix
open Pervasives

open Types

(***************************)
(* the dns child processes *)
(***************************)

(* this process treat one call and die *)
let main_child host outch =
  let res =
    try
      let entry = gethostbyname host in
      Some entry.h_addr_list.(0)
    with _ ->
      None in
  output_value outch res;
  flush outch

let init_child host =
  let ans_in_fds, ans_out_fds = pipe () in
  let ans_out = out_channel_of_descr ans_out_fds in
  match fork () with
  | 0 -> (* Child process *)
      close ans_in_fds;
      main_child host ans_out;
      exit 0
  | pid -> (* main dns process *)
      close_out ans_out;
      ans_in_fds, pid


(************************************************)
(* part of the dns work inside the main process *)
(************************************************)

(* this includes everything except gethostbyname because it would block *)

exception Not_yet
exception Dns_queue

let wait_list = ref []

(* really read nb bytes on a fds *)
let really_read fds nb =
  let ans = String.create nb in
  let nb_read = ref 0 in
  while !nb_read < nb do
    nb_read := !nb_read + (Unix.read fds ans !nb_read (nb - !nb_read))
  done;
  ans

(* read an answer from a fds *)
let read_value fds =
  let header = really_read fds Marshal.header_size in
  let ds = Marshal.data_size header 0 in
  let data = really_read fds ds in
  Marshal.from_string (header^data) 0

(*************)
(* dns cache *)
(*************)

exception Not_in_cache
type cache_value =
  | Empty
  | Exist of string * Unix.inet_addr

let cache = Array.create Types.dns_cache_size Empty

(* calculate has code for a site name *)
let site_hashcode s =
  let res = ref 0 in
  for i=0 to (String.length s) - 1 do
    res := (!res * 23 + (Char.code s.[i])) mod Types.dns_cache_size;
  done;
  !res

(* add some value in the cache *)
let add_value host ip =
  let pos = site_hashcode host in
  cache.(pos) <- Exist (host, ip)

(* try to find the host in the cache *)
exception Is_in_cache of cache_value

let gethostbycache host =
  let pos = site_hashcode host in
  match cache.(pos) with
  | Empty ->
      raise Not_in_cache
  | Exist (s, ip) ->
      if s = host then ip else raise Not_in_cache

(* the gethostbyname is made in a separate process
 * because it can block everything *)
let my_gethostbyname conn host =
  try
    gethostbycache host
  with Not_in_cache ->
    if !Types.debug > 0 then
      (Printf.printf "dns call for %s (%d)\n" host (List.length !wait_list);
       flush stdout);
    let fds, pid = init_child host in
    wait_list := (conn, fds, pid, host)::!wait_list;
    raise Dns_queue


(**************************************)
(* Communication with the dns process *)
(**************************************)

(* forget this number *)
let forget conn =
  let rec new_list = function
    | [] -> []
    | (c, fds, pid, _)::l when c==conn ->
        Unix.close fds;
        Unix.kill pid 9;
        ignore (Unix.waitpid [] pid);
        l
    | a::l -> a::(new_list l) in
  wait_list := new_list !wait_list

(* read answers *)
let update active_read =
  let rec update_list = function
    | [] -> []
    | (c, fds, pid, host) as x::l ->
        if List.mem fds active_read then
          let ans = read_value fds in
          Unix.close fds;
          ignore (Unix.waitpid [] pid);
          (* add the answer in the cache and update the conn state *)
          (match ans with
          | Some ip ->
              add_value host ip;
              c.state <- DNSDONE ip
          | _ ->
              (* simplified version of Utils.finish *)
              c.write_ans <- Activebuffer.activebuffer_of_string unknown_host;
              c.state <- FINISHING);
          update_list l
        else
          x::(update_list l) in
  wait_list := update_list !wait_list

(* fds to use in select *)
let give_fds () =
  let rec aux res = function
    | [] -> res
    | (_, fds, _, _)::l ->
        aux (fds::res) l in
  aux [] !wait_list
