/*********************************************************************
 *
 * AUTHORIZATION TO USE AND DISTRIBUTE
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: 
 *
 * (1) source code distributions retain this paragraph in its entirety, 
 *  
 * (2) distributions including binary code include this paragraph in
 *     its entirety in the documentation or other materials provided 
 *     with the distribution, and 
 *
 * (3) all advertising materials mentioning features or use of this 
 *     software display the following acknowledgment:
 * 
 *      "This product includes software written and developed 
 *       by Brian Adamson and Joe Macker of the Naval Research 
 *       Laboratory (NRL)." 
 *         
 *  The name of NRL, the name(s) of NRL  employee(s), or any entity
 *  of the United States Government may not be used to endorse or
 *  promote  products derived from this software, nor does the 
 *  inclusion of the NRL written and developed software  directly or
 *  indirectly suggest NRL or United States  Government endorsement
 *  of this product.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 ********************************************************************/
 

#include "eventDispatcher.h"

#include <stdio.h>
#include <string.h>
#include <errno.h>

#ifdef UNIX
#include <unistd.h>
#include <sys/time.h>
#endif // UNIX

const EventDispatcher::Descriptor EventDispatcher::INVALID_DESCRIPTOR = INVALID_SOCKET;

/* Should we ever need a "normal" new operator
void* EventDispatcher::Item::operator new(size_t objSize)
{
    return (void*)::new int[(objSize/sizeof(int)];
}*/

void* EventDispatcher::Item::operator new(size_t objSize, EventDispatcher::Item* item)
{
    if (item)
        return ((void*)item);
    else
        return ((void*)new int[objSize/sizeof(int)]);
}  // end EventDispatcher::Item::operator new()
  
EventDispatcher::Item::Item()
 : flags(NONE), descriptor(INVALID_DESCRIPTOR), data(NULL), callback(NULL)
{
}


EventDispatcher::Item::Item(EventDispatcher::Descriptor theDescriptor, 
                            EventDispatcher::Callback func,
                            const void* userData)
 : type(GENERIC), descriptor(theDescriptor), data(userData), callback(func)
{
}

EventDispatcher::Item::Item(UdpSocket* theSocket)
 : type(SOCKET), descriptor(theSocket->Handle()), data(theSocket) 
{
}

EventDispatcher::Item::Item(ProtocolTimerMgr* timerMgr)
 : type(TIMER), data(timerMgr) 
{
   timer.Init(0.0, 0, (ProtocolTimerOwner*)this, 
              (ProtocolTimeoutFunc)&EventDispatcher::Item::OnTimeout);
}

EventDispatcher::Item::~Item()
{
    // (TBD) If open, dispatch "DETACH" event ???
}

void EventDispatcher::Item::OnInputEvent()
{
    switch (type)
    {
        case GENERIC:
            if (callback) callback(descriptor, EVENT_INPUT, data);
            break;
        case SOCKET:            
            ((UdpSocket*)data)->OnReceive();
            break;
        case TIMER:
            // Shouldn't happen
            break;
    }   
}  // end EventDispatcher::Item::OnInputEvent()

void EventDispatcher::Item::SetTimeout(double delay)     
{
    timer.SetInterval(delay);
    if (timer.IsActive()) timer.Reset();
}  // end EventDispatcher::Item::SetTimeout()

bool EventDispatcher::Item::OnTimeout()
{
    ((ProtocolTimerMgr*)data)->DoTimers();
    return true;
}  // end EventDispatcher::Item::OnTimeout()

#ifdef UNIX
// Function pointer for UNIX timer "installer" (select() timeout)
bool EventDispatcher::PrivateTimerInstaller(ProtocolTimerInstallCmd   cmd, 
                                     double                    theDelay,
                                     ProtocolTimerMgr*       /*timerMgr*/,
                                     const void*               installData)
{
    switch (cmd)
    {
        case PROTOCOL_TIMER_INSTALL:
        case PROTOCOL_TIMER_MODIFY:
            ((EventDispatcher*)installData)->SetTimeout(theDelay);
            break;
        case PROTOCOL_TIMER_REMOVE:
            ((EventDispatcher*)installData)->SetTimeout(-1.0);
            break;
    }
    return true;
}  // end EventDispatcher::PrivateTimerInstaller()
#endif // UNIX

bool EventDispatcher::SocketInstaller(UdpSocketCmd  cmd,
                                      UdpSocket*    theSocket,
                                      const void*   installData)
{
    switch (cmd)
    {
        case UDP_SOCKET_INSTALL:
        {             
            const Item* item = 
                ((EventDispatcher*)installData)->AddSocketInput(theSocket);
            if (!item) return false;
            break;
        }

        case UDP_SOCKET_REMOVE:
        {
            Item* item = 
                ((EventDispatcher*)installData)->FindItemByDescriptor(theSocket->Handle());
            if (item)
                ((EventDispatcher*)installData)->RemoveItem(item);
            else
                return false;
            break;
        }
    }
    return true;  
}  // end EventDispatcher::SocketInstaller()


bool EventDispatcher::TimerInstaller(
                        ProtocolTimerInstallCmd     cmd, 
                        double                      delay,
                        ProtocolTimerMgr*           timerMgr, 
                        const void*                 installData)
{
    EventDispatcher* dispatcher = (EventDispatcher*)installData;    
    switch(cmd)
    {
        case PROTOCOL_TIMER_INSTALL:
            if (!dispatcher->InstallTimerMgr(timerMgr, delay)) return false;
            break;

        case PROTOCOL_TIMER_MODIFY:
            if (!dispatcher->ModifyTimerMgr(timerMgr, delay)) return false;
            break;

        case PROTOCOL_TIMER_REMOVE:
            if (!dispatcher->RemoveTimerMgr(timerMgr)) return true;
            break;
    }  // end switch(cmd)
    return false;
}  // end EventDispatcher::TimerInstaller()           
            

bool EventDispatcher::InstallTimerMgr(ProtocolTimerMgr* timerMgr, double delay)
{
   Item* item = GetItemFromPool();
   item = new(item) Item(timerMgr); 
   if (item)
   {
       // Prepend item to beginning of item list
        item->prev = NULL;
        if ((item->next = top_item)) top_item->prev = item;
        top_item = item;
        item->SetTimeout(delay);
        InstallTimer(item->Timer());
        return true;
   }
   else
   {
       return false;
   }
}  // end bool EventDispatcher::InstallTimerMgr()  

bool EventDispatcher::ModifyTimerMgr(ProtocolTimerMgr* timerMgr, double delay)
{
    Item* item = top_item;
    while (item)
    {
        if ((Item::TIMER == item->type) && ((void*)timerMgr == item->data))
            break;
        item = item->next;   
    }
    if (item)
    {
        item->SetTimeout(delay);
        return true;
    }
    else
    {
        return false;   
    }
}  // end EventDispatcher::ModifyTimerMgr()

bool EventDispatcher::RemoveTimerMgr(ProtocolTimerMgr* timerMgr)
{
    Item* item = top_item;
    while (item)
    {
        if ((void*)timerMgr == item->data) break;
        item = item->next;   
    }
    if (item)
    {
        if (item->Timer()->IsActive())
            item->DeactivateTimer();
        RemoveItem(item);
        return true;
    }
    else
    {
        return false;   
    }
}  // end EventDispatcher::RemoveTimerMgr()



EventDispatcher::EventDispatcher()
	: delay(-1.0), top_item(NULL), item_pool(NULL), 
      max_descriptor(-1), run(false), exit_code(0)
#ifdef WIN32
	  ,msg_window(NULL), mm_timer_id(NULL), mm_timer_msg_pending(false)
#endif // WIN32
{
    timer_mgr.SetInstaller(EventDispatcher::PrivateTimerInstaller, this);
}

EventDispatcher::~EventDispatcher()
{
    Destroy();
}

void EventDispatcher::Destroy()
{
    Item* next;
    while ((next = top_item))
    {
        top_item = next->next;
        delete next;
    }
    while((next = item_pool))
    {
        item_pool = next->next;
        delete next;
    }
    delay = -1;
    max_descriptor = INVALID_DESCRIPTOR;
	// (TBD) WIN32 needs to destroy our window ???
    run = false;
}  // end EventDispatcher::Destroy()

const EventDispatcher::Item* EventDispatcher::AddGenericInput(EventDispatcher::Descriptor descriptor, 
                                                              Callback func,
                                                              const void* userData)
{
    Item* item = FindItemByDescriptor(descriptor);
	if (!item) 
	{
        item = GetItemFromPool();
		item = new(item) Item(descriptor, func, userData);   
		if (!item) return (Item*)NULL;
		// Prepend item to beginning of item list
		item->prev = NULL;
		if ((item->next = top_item)) top_item->prev = item;
		top_item = item;
		if ((INVALID_DESCRIPTOR == max_descriptor) || (descriptor > max_descriptor)) 
			max_descriptor = descriptor;
	}
    item->SetFlag(Item::INPUT);
    return item;
}  // end EventDispatcher::AddGenericInput()


const EventDispatcher::Item* EventDispatcher::AddSocketInput(UdpSocket* theSocket)
{
	Item* item = FindItemByDescriptor(theSocket->Handle());
	if (!item) 
	{
        item = GetItemFromPool();
		item = new(item) Item(theSocket);   
		if (!item) return (Item*)NULL;
		// Prepend item to beginning of item list
		item->prev = NULL;
		if ((item->next = top_item)) top_item->prev = item;
		top_item = item;
		if ((INVALID_DESCRIPTOR == max_descriptor) || (theSocket->Handle() > max_descriptor))
			max_descriptor = theSocket->Handle();
	}
	item->SetFlag(Item::INPUT);
#ifdef WIN32
	if (msg_window && !Win32Install(item))
	{
		item->UnsetFlag(Item::INPUT);
		if (item->IsActive() && !Win32Install(item)) RemoveItem(item);
		return NULL;
	}
#endif // WIN32  
	return item;
}  // end EventDispatcher::AddSocketInput()

void EventDispatcher::RemoveGenericInput(EventDispatcher::Descriptor descriptor)
{
    Item* item = FindItemByDescriptor(descriptor);
    if (item) RemoveInput(item);
}  // end EventDispatcher::RemoveGenericInput()

void EventDispatcher::RemoveSocketInput(UdpSocket* theSocket)
{
	Item* item = FindItemByDescriptor(theSocket->Handle());
	if (item) RemoveInput(item);
}  // end EventDispatcher::RemoveSocketInput()

void EventDispatcher::RemoveInput(Item *item)
{
	item->UnsetFlag(Item::INPUT);
	if (!item->IsActive())
	{
		RemoveItem(item);
	}
#ifdef WIN32
	else
	{
		if (!Win32Install(item)) RemoveItem(item);
	}
#endif 
}  // end EventDispatcher::RemoveInput()


EventDispatcher::Item* EventDispatcher::GetItemFromPool()
{   
    if (item_pool)
    {
        Item* item = item_pool;
        item_pool = item_pool->next;
        return item;   
    }
    else
    {
        return NULL;  // Pool empty
    }
}  // end EventDispatcher::GetItemFromPool()

void EventDispatcher::ReturnItemToPool(Item* item)
{   
    item->next = item_pool;
    item_pool = item;
}  // end EventDispatcher::GetItemFromPool()



void EventDispatcher::RemoveItem(EventDispatcher::Item* item)
{
    if (item->prev)
		item->prev->next = item->next;
	else
		top_item = item->next;	
	if (item->next)
		item->next->prev = item->prev;
	if (item->descriptor == max_descriptor) max_descriptor = FindMaxDescriptor();

#ifdef WIN32
	if (msg_window) WSAAsyncSelect(item->descriptor, 0, 0, 0);
#endif // WIN32
    delete item;	
}  // end void EventDispatcher::RemoveInput()


EventDispatcher::Descriptor EventDispatcher::FindMaxDescriptor()
{
	Item* next = top_item;
	if (next)
	{
		Descriptor max = next->descriptor;
		next = next->next;
		while (next)
		{
			if (next->descriptor > max) max = next->descriptor;
			next = next->next;
		}
		return max;
	}
	else
	{
		return INVALID_DESCRIPTOR;
	}
}  // end EventDispatcher::FindMaxDescriptor()

EventDispatcher::Item* EventDispatcher::FindItemByDescriptor(EventDispatcher::Descriptor descriptor)
{
	Item *next = top_item;
	while (next)
	{
		if (next->descriptor == descriptor) return next;
		next = next->next;
	}
	return NULL;
}  // end EventDispatcher::FindItemByDescriptor()

#ifdef UNIX
int EventDispatcher::Run()
{
	run = true;
	struct timeval timeout;
    struct timeval *timeout_ptr;			
	while (run)
	{		
		if (delay >= (double)0.0)
		{
			timeout.tv_sec = (unsigned long) delay;
	    	timeout.tv_usec = 
				(unsigned long)(1000000.0 * (delay - timeout.tv_sec));

			// If it's less than 10 msec, make it poll
        	if(!timeout.tv_sec && (timeout.tv_usec < 10000))
            	timeout.tv_usec = 0;

	    	timeout_ptr = &timeout;
    	}
		else
		{
			timeout_ptr = NULL;
		}
		
		fd_set inputs;
		FD_ZERO(&inputs);
		Item* next = top_item;
		while (next)
		{
			FD_SET(next->descriptor, &inputs);
			next = next->next;
		}		
		if (!timeout_ptr && (!top_item))
	    {
	        DMSG(0, "EventDispatcher::Run() stuck with infinite timeout & no inputs!\n");
	        run = false;
	        break;
	    }	
        
        int status = select(max_descriptor+1, (fd_set*)&inputs, 
			    			(fd_set*) NULL, (fd_set*) NULL, 
							timeout_ptr);
		switch (status)
		{
			case -1:
				if (EINTR != errno)
					TRACE("EventDispatcher::Run() select() error: %s\n", strerror(errno));
				break;
				
			case 0:
				// timeout
				break;
				
			default:
				next = top_item;
				while(next)
				{
					if (FD_ISSET(next->descriptor, &inputs)) next->OnInputEvent();
					next = next->next;
				}
				break;
		}
        timer_mgr.DoTimers();	
	}  // end while(run)
	return exit_code;
}  // end EventDispatcher::Run()
#endif // UNIX


#ifdef WIN32
bool EventDispatcher::Win32Init()
{
	HINSTANCE theInstance = GetModuleHandle(NULL);	
	// Register our msg_window class
	WNDCLASS cl;
    cl.style = CS_HREDRAW | CS_VREDRAW;
    cl.lpfnWndProc = MessageHandler;
    cl.cbClsExtra = 0;
    cl.cbWndExtra = 0;
    cl.hInstance = theInstance;
    cl.hIcon = NULL;
    cl.hCursor = NULL;
    cl.hbrBackground = NULL;
    cl.lpszMenuName = NULL;
    cl.lpszClassName = "ProteanEventDispatcher";
    if (!RegisterClass(&cl))
	{
		TRACE("EventDispatcher::Win32Init() Error registering message window class!\n");
		return false;
	}
	// Create msg_window to receive event messages
	msg_window = CreateWindow("ProteanEventDispatcher",  // registered class name
							  "ProteanEventDispatcher", // window name
							  WS_OVERLAPPED,        // window style
							  0,                // horizontal position of window
							  0,				// vertical position of window
							  0,              // window width
							  0,			 // window height
							  NULL,	 // handle to parent or owner window
							  NULL,          // menu handle or child identifier
							  theInstance,  // handle to application instance
							  this);        // window-creation data
	if (!msg_window)
	{
		TRACE("EventDispatcher::Win32Init() Error creating message window!\n");
		return false;
	}
	ShowWindow(msg_window, SW_HIDE);

	// Set multimedia timer resolution for 10 msec
	TIMECAPS tc;
	if (timeGetDevCaps(&tc, sizeof(TIMECAPS)) != TIMERR_NOERROR) 
	{
		TRACE("EventDispatcher::Run() timeGetDevCaps() error!\n");
	}
	mm_timer_resolution = min(max(tc.wPeriodMin, 10), tc.wPeriodMax);
	timeBeginPeriod(mm_timer_resolution);
	return true;
}  // end EventDispatcher::Win32Init()

int EventDispatcher::Run()
{

	ASSERT(msg_window);
	
	int exitCode = 0;  
	run = true;

	while (run)
	{
		HANDLE generic_handles[8];
		DWORD generic_count = 0;
		Item* next = top_item;
		while (next)
		{
			if (Item::GENERIC == next->type)
			{
				generic_handles[generic_count++] = (HANDLE)next->descriptor;
			}
			next = next->next;
		}	

		DWORD result = MsgWaitForMultipleObjectsEx(
								generic_count,		// number of handles in array
								generic_handles,	// object-handle array
								INFINITE,			// time-out interval
								QS_ALLEVENTS,		// input-event type
								MWMO_ALERTABLE | MWMO_INPUTAVAILABLE);
		if (result < (WAIT_OBJECT_0 + generic_count))
		{
			Item* item = FindItemByDescriptor((Descriptor)generic_handles[result - WAIT_OBJECT_0]);
			if (item && item->IsInput()) item->OnInputEvent();
		}
		else
		{
			MSG msg;
			if (GetMessage(&msg, NULL, 0, 0 ))
			{
				TranslateMessage(&msg);
				DispatchMessage(&msg);
			}
			else
			{
				exitCode = msg.wParam;
				break;  // we're done
			}
		}
	}  // end while(run)
	 
	timeEndPeriod(mm_timer_resolution);
	DestroyWindow(msg_window);
	msg_window = NULL;
	return exitCode;
}  // end EventDispatcher::Run()

LRESULT CALLBACK EventDispatcher::MessageHandler(HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message) 
	{
		case WM_CREATE:
		{
			CREATESTRUCT *info = (CREATESTRUCT *) lParam;
			EventDispatcher* dp = (EventDispatcher*)info->lpCreateParams;
			SetWindowLong(hwnd, GWL_USERDATA, (DWORD)dp);
			return 0;
		}	

		case WM_DESTROY:
			PostQuitMessage(0);
			return 0;
	
		case SOCKET_MSG:
		{
			EventDispatcher *dp = (EventDispatcher*)GetWindowLong(hwnd, GWL_USERDATA);
			if (dp)
			{
				Item* item = dp->FindItemByDescriptor(wParam);
				if (item)
				{
					long theEvent = WSAGETSELECTEVENT(lParam);
					switch (theEvent)
					{
						case FD_READ:
							item->OnInputEvent();
							break;
						default:
							break;
					}
				}
				return 0;
			}
			break;
		}

		case TIMER_MSG:
		{
			EventDispatcher *dp = (EventDispatcher*)GetWindowLong(hwnd, GWL_USERDATA);
			if (dp)
			{
				dp->mm_timer_msg_pending = false;
				dp->timer_mgr.DoTimers();
			}
			return 0;
		}

		default:
		    break;
	}
	return DefWindowProc(hwnd, message, wParam, lParam);
}  // end EventDispatcher::MessageHandler()


bool EventDispatcher::PrivateTimerInstaller(ProtocolTimerInstallCmd   cmd, 
											double                    delay,
											ProtocolTimerMgr*         timerMgr,
											const void*               installData)
{
	EventDispatcher* dp = (EventDispatcher*)installData;
	switch (cmd)
	{
		case PROTOCOL_TIMER_MODIFY:
		{
			if (dp->mm_timer_id)  // can't modify posted no-delay timeout
			{
				timeKillEvent(dp->mm_timer_id);
				int msec = (int) (delay * 1000.0);
				if (msec > 0)
				{
					if (!(dp->mm_timer_id = timeSetEvent(msec,				// delay (msec)
														 dp->mm_timer_resolution, // resolution
														 TimerCallback,	// callback function
														 (DWORD)dp,			// pass EventDispatcher instance 
														 TIME_ONESHOT)))
					{
						DMSG(0, "EventDispatcher::PrivateTimerInstaller(MODIFY) timeSetEvent(tout=%d msec) error!\n", msec);
						return false;
					}
				}
				else
				{
					dp->mm_timer_id = NULL;
					// For ZERO timeout, directly post message 
					if (!dp->mm_timer_msg_pending)
					{
						if (PostMessage(dp->msg_window, TIMER_MSG, (DWORD)timerMgr, (DWORD)dp))
						{
							dp->mm_timer_msg_pending = true;
						}
						else
						{
							DMSG(0, "EventDispatcher::PrivateTimerInstaller(MODIFY) PostMessage failed\n");
							return false;
						}
					}
				}
			}  // end if (timerId)
		}
		break;

		case PROTOCOL_TIMER_INSTALL:
		{
			if (dp->mm_timer_msg_pending) return true;
			int msec = (int) (delay * 1000.0);
			if (msec > 0)
			{
				if (!(dp->mm_timer_id = 
				timeSetEvent(msec,		      
							 dp->mm_timer_resolution,
						     TimerCallback,    
							 (DWORD)dp, 
							 TIME_ONESHOT)))
				{
					DMSG(0, "EventDispatcher::PrivateTimerInstaller(INSTALL) timeSetEvent(tout=%d msec) error!\n", msec);
					return false;
				}
			}
			else
			{
				// For ZERO timeout, directly post message 
				dp->mm_timer_id = NULL;
				if (PostMessage(dp->msg_window, TIMER_MSG, (DWORD)timerMgr, (DWORD)dp))
				{
					dp->mm_timer_msg_pending = true;
				}
				else
				{
					DMSG(0, "EventDispatcher::PrivateTimerInstaller(INSTALL) PostMessage error!\n");
					return false;
				}
			}
		}
		break;

		case PROTOCOL_TIMER_REMOVE:
		{
			if (dp->mm_timer_id) 
			{
				timeKillEvent(dp->mm_timer_id);
				dp->mm_timer_id = NULL;
			}
		}
		break;
	}  // end switch(cmd)
	return true;
}  // end EventDispatcher::PrivateTimerInstaller()

void CALLBACK EventDispatcher::TimerCallback(UINT uID, UINT uMsg, DWORD dwUser, 
										     DWORD dw1, DWORD dw2)
{
	EventDispatcher* dp = (EventDispatcher*)dwUser;
	if (uID == dp->mm_timer_id)
	{
		dp->mm_timer_id = NULL;
		if (dp->msg_window && !dp->mm_timer_msg_pending)
		{
			if (PostMessage(dp->msg_window, TIMER_MSG, (DWORD)&dp->timer_mgr, (DWORD)dwUser))
				dp->mm_timer_msg_pending = true;
			else
				DMSG(0, "EventDispatcher::TimerCallback() PostMessage failed!\n");
		}
	}
}  // end EventDispatcher::TimerCallback() 

bool EventDispatcher::Win32Install(EventDispatcher::Item* item)
{
	long events = 0;
	if (item->IsInput()) events |= (FD_READ);
	if (item->IsOutput()) events |= (FD_WRITE | FD_CONNECT);
	if (0 != events)
	{
		if (WSAAsyncSelect(item->descriptor, msg_window, SOCKET_MSG, events))
		{
			DMSG(0, "EventDispatcher::Win32Install() WSAAsyncSelect() error!\n");
			return false;
		}
		return true;
	}
	else
	{
		return false;
	}
}  // end EventDispatcher::Win32Install()

#endif // WIN32
		
