
#include "debug.h"

#include <stdio.h>   
#include <stdlib.h>  // for abort()
#include <stdarg.h>  // for variable args

#ifdef WIN32
#include <io.h>
#include <fcntl.h>
#include <Windows.h>
#endif // WIN32

#ifdef PROTO_DEBUG
              


unsigned int debug_level = 0;

FILE *log_file = stdout;  // log to stdout by default

void SetDebugLevel(unsigned int level)
{
#ifdef WIN32
	if (level)
	{
		if (!debug_level && (log_file == stdout)) OpenDebugWindow();
	}
	else
	{
		if (debug_level && (log_file == stdout)) CloseDebugWindow();
	}
#endif // WIN32


    debug_level = level;
}  // end SetDebugLevel()

unsigned int DebugLevel()
{
    return debug_level;
}


bool OpenDebugLog(const char *path)
{
	if (log_file != stdout) fclose(log_file);
    FILE* ptr = fopen(path, "w+");
    if (ptr)
    {
#ifdef WIN32
	if (debug_level && (log_file == stdout)) CloseDebugWindow();
#endif // WIN32
		log_file = ptr;
        return true;
    }
    else
    {
#ifdef WIN32
        if (debug_level && (log_file != stdout)) OpenDebugWindow();
#endif // WIN32
		log_file = stdout;
		TRACE("ivox: Error opening debug log file: %s\n", path);
        return false;
    }
}  // end OpenLogFile()

void CloseDebugLog()
{
    if (log_file != stdout)
    {
        fclose(log_file);
#ifdef WIN32
	if (debug_level) OpenDebugWindow();
#endif // WIN32
        log_file = stdout;
    }
}


//#if defined(UNIX) || (defined(WIN32) && defined(NDEBUG))
// Generic "printf()-like" function for printing debug messages
void TRACE(const char *format, ...)
{
    va_list args;
    va_start(args, format);
    vfprintf(log_file, format, args);
	fflush(log_file);
    va_end(args);
}  // end TRACE();
//#endif // UNIX || (WIN32 && NDEBUG)

// Display string if statement's debug level 
void DMSG(unsigned int level, const char *format, ...)
{
    if (level <= debug_level)
    {
		va_list args;
        va_start(args, format);
#if defined(WIN32) && !defined(NDEBUG)
		// This code puts DMSG printouts to Win32 Debugger
		// when running the code under the debugger
		if (stdout == log_file)
		{
			char tempText[1024];
			vsprintf(tempText, format, args);
			TRACE("%s", tempText);
			va_end(args);
			return;
		}
#endif  // WIN32 && !NDEBUG
        vfprintf(log_file, format, args);
        fflush(log_file);
        va_end(args);   
    }
}  // end DMSG();

void ABORT(const char *format, ...)
{
    va_list args;
    va_start(args, format);
    vfprintf(log_file, format, args);
    fflush(log_file);
    va_end(args); 
    abort();
}

#endif // if/else PROTO_DEBUG

#ifdef WIN32
// Open console for displaying debug messages
static unsigned int console_reference_count = 0;
void OpenDebugWindow()
{
	// Open a console window and redirect stdout to it
	if (0 == console_reference_count)
	{
		AllocConsole();
		int hCrt = _open_osfhandle((long) GetStdHandle(STD_OUTPUT_HANDLE), _O_TEXT);
		FILE* hf = _fdopen(hCrt, "w" );
		*stdout = *hf;
		int i = setvbuf(stdout, NULL, _IONBF, 0 );
		
		hCrt = _open_osfhandle((long) GetStdHandle(STD_INPUT_HANDLE), _O_TEXT);
		hf = _fdopen(hCrt, "r" );
		*stdin = *hf;

		hCrt = _open_osfhandle((long) GetStdHandle(STD_ERROR_HANDLE), _O_TEXT);
		hf = _fdopen(hCrt, "w" );
		*stderr = *hf;
		i = setvbuf(stderr, NULL, _IONBF, 0 );
	}
	console_reference_count++;
}  // end OpenDebugWindow() 

void CloseDebugWindow()
{
	if (console_reference_count > 0)
	{
		console_reference_count--;
		if (0 == console_reference_count) FreeConsole();
	}
}
#endif // WIN32
