/*
 * Copyright (C) 1983-1989  Masatoshi Kurihara <kurihara@sra.co.jp>
 * Copyright (C) 1999, 2000 and 2001
 * Jun-ichiro itojun Hagino <itojun@iijlab.net>
 * All rights reserved.
 *
 * Note well: this is not a normal 3-clause BSD copyright;
 * commercial use of the software is restricted.
 *
 * Redistribution and non-commercial use in source and binary
 * forms, with or without modification, are permitted provided that the
 * following conditions are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the authors nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * Personal Schedule Manager
 *
 *	% sch [-acp] [user | -f schedule]
 *		-acp      -- auto write, show calander, project code
 *		schedule  -- schedule file, default is ~/.schrc
 *		user      -- user name, file is ~user/.schrc
 *	% setenv SCHEDULE acp
 *		-acp      -- auto write, show calander, project code
 *
 *	% cc -s -O -o sch sch.c -lcurses -ltermlib
 *		-DBSD4_2  -- Berkelay 4.2 version
 *		-DSENDBUG -- send mail facility
 *		-DSENDUSR -- mail address, default is "srava!kurihara"
 *		-DMAXDATA -- max data line, default is 400
 *
 *	Author	Masatoshi Kurihara
 *	Date	Fri Apr  1 15:52:44 JST 1983
 *		Thu Dec  5 10:38:24 JST 1985
 *	Office	Software Tools and Technology Group
 *		Software Research Associates, Inc.
 *		1-1-1 Hirakawa-cho Chiyoda-ku Tokyo, Japan
 *		03 (234) 2611 ext 193
 *		03 (234) 2615 (night only)
 *	Home	Waseda House 3A 
 *		1-9-22 Nishiwaseda Shinjuku-ku Tokyo, Japan
 *		03 (208) 3875
 *
 *	Copyright (C) 1985 - 1989 by Software Research Associates, Inc.
 */

static	char	sccsid[] = "@(#)sch.c\t1.4 (SRA) 12/5/85";

#include <curses.h>
#include <common.h>
#ifdef MULTIBYTE
# include <wchar.h>
#endif
#include <err.h>

#include <def.h>
#include <entry.h>
#include <var.h>
#include <sub.h>
#include <sch.h>
#include <opts.h>

static void usage __P((void));
int main __P((int, char **));
static void initial __P((void));
static void headinit __P((void));
static void schinit __P((void));
static void screen_paintline __P((int));
static void screen_init __P((void));
static void setprjsw __P((int));

/*
 * Personal Schedule Manager
 *
 *	% sch [-acp] [-z homezone] [user | -f schedule]
 *		-acp      -- auto write, show calander, project code
 *		schedule  -- schedule file, default is ~/.schrc
 *		user      -- user name, file is ~user/.schrc
 *	% setenv SCHEDULE acp
 *		-acp      -- auto write, show calander, project code
 */
static void
usage()
{
	fprintf(stderr, "usage: sch [-acp] [-z homezone] [-f file | user]\n");
}

int
main(argc, argv)
	int argc;
	char **argv;
{
	char *opt;
	struct passwd *pwd;
	int c;
	char *fopt = NULL;
	extern int optind;
	extern char *optarg;

	setprjsw(FALSE);

	/*
	 * Option set
	 */
	if ((opt = getenv("SCHEDULE")) != NULL) {
		if (strchr(opt, 'a') || strchr(opt, 'A'))
			autowrite = TRUE;
		if (strchr(opt, 'c') || strchr(opt, 'C'))
			calsw = TRUE;
		if (strchr(opt, 'p') || strchr(opt, 'P'))
			setprjsw(TRUE);
	}
	while ((c = getopt(argc, argv, "acf:pz:")) != EOF) {
		switch (c) {
		case 'a':
			autowrite = TRUE;
			break;
		case 'c':
			calsw = TRUE;
			break;
		case 'f':
			fopt = optarg;
			break;
		case 'p':
			setprjsw(TRUE);
			break;
		case 'z':
			curzone = homezone = atoi(optarg);
			break;
		default:
			usage();
			exit(1);
		}
	}
	argc -= optind;
	argv += optind;

	/*
	 * Schedule file set
	 */
	if (argc == 1) {
		if (fopt) {
			usage();
			exit(1);
		}

		if (strcmp(*argv, getpwuid(geteuid())->pw_name))
			myfile = FALSE;
		if ((pwd = getpwnam(*argv)) != NULL) {
			strcpy(username, pwd->pw_name);
			strcpy(schfname, pwd->pw_dir);
			strcat(schfname, "/.schrc");
			strcpy(schsname, "~");
			strcat(schsname, pwd->pw_name);
			strcat(schsname, "/.schrc");
		} else {
			errx(1,"unknown user %s", *argv);
			/*NOTREACHED*/
		}
	} else if (argc == 0) {
		if (fopt) {
			strcpy(username, getpwuid(geteuid())->pw_name);
			strcpy(schfname, fopt);
			strcpy(schsname, fopt);
		} else {
			strcpy(username, getpwuid(geteuid())->pw_name);
			/* XXX $HOME, or pw_dir? */
			opt = getenv("HOME");
			if (opt == NULL) {
				errx(1, "no home directory");
				/*NOTREACHED*/
			}
			snprintf(schfname, sizeof(schfname),
				"%s/.schrc", opt);
			strcpy(schsname, "~/.schrc");
		}
	} else {
		usage();
		exit(1);
	}
	endpwent();

	/*
	 * Main processes
	 */
	initial();
	command();
	exit(0);
}

/*
 * Initial procedure
 */
static void
initial() 
{
	int cnt, lineno;
	char *pch;
	char line[LEN_BUFF];
	struct stat statbuf;
	int filever;
	FILE *fp;

	/*
	 * Time set
	 */
	time(&ltime);
	memcpy(&inttimebuf, localtime(&ltime), sizeof(inttimebuf));
	inttime = &inttimebuf;

	if (stat(schfname, &statbuf) == -1 || statbuf.st_size == NULL) {
		/*
		 * Schedule data set in new file
		 */
		if (myfile) {
			int fd;

			fprintf(stdout, "\"%s\" [New file] ", schsname);
			fflush(stdout);
			newsw = TRUE;
			sch[0] = sch_new();
			maxsch = 1;
			fd = open(schfname, O_CREAT|O_RDONLY, 0644);
			if (fd < 0)
				err(1, "open");
			if (flock(fd, LOCK_EX | LOCK_NB) == 0)
				writesw = TRUE;
			/* don't close it to preserve read lock */
		} else {
			errx(1, "%s not found", schsname);
			/*NOTREAHCED*/
		}
	} else {
		/*
		 * Schedule data get
		 */
		filever = 0;
		if ((fp = fopen(schfname, "r")) == NULL) {
			err(1, "%s", schsname);
			/*NOTREACHED*/
		}
		cnt = lineno = 0;
		while (1) {
			memset(line, 0, sizeof(line));
			if (fgets(line, sizeof(line), fp) == NULL)
				break;
			lineno++;

			if (cnt > MAXDATA) {
				errx(1, "too many lines");
				/*NOTREACHED*/
			}
			if ((pch = strchr(line, '\n')) != NULL)
				*pch = '\0';

			if (line[0] == '"')
				continue;
			if (strncmp(line, ":set", 4) == 0) {
				for (pch = line; *pch && !isspace(*pch); pch++)
					;
				for (/*nothing*/; *pch && isspace(*pch); pch++)
					;

				if (dosetit(pch, 0) < 0) {
					errx(1, "%s:%d: invalid directive",
						schsname, lineno);
					/*NOTREACHED*/
				}
				continue;
			}

			if (filever == 0) {
				sch[cnt] = sch_oparse(line);
				if (sch[cnt] == NULL)
					filever = 1;
			}
			if (filever == 1)
				sch[cnt] = sch_parse(line);
			if (sch[cnt] == NULL) {
				errx(1, "%s:%d: unparsable", schsname, lineno);
				/*NOTREACHED*/
			}
#if defined(HAVE_OFFTIME) && defined(HAVE_TIMEOFF)
			sch_tznorm(sch[cnt], curzone, homezone);
#else
			if (curzone != homezone) {
				errx(1, "timezone offset not supported");
				/*NOTREACHED*/
			}
#endif
			if (cnt == 0) {
				fprintf(stdout, "\"%s\"", schsname);
				fflush(stdout);
				fprintf(stdout, " date %d/%2.2d",
					 sch[cnt]->month, sch[cnt]->day);
				fflush(stdout);
			}

			cnt++;
		}
#ifdef MULTIBYTE
		if (!localeset) {
			char *opt;
			if (opt = getenv("LANG"))
				setrunelocale(opt);
			else
				setrunelocale("C");
		}
#endif

		if (access(schfname, 02) == 0) {
#ifdef HAVE_FLOCK
			if (flock(fileno(fp), LOCK_EX | LOCK_NB) == 0)
				writesw = TRUE;
#else
			writesw = TRUE;
#endif
		}
#ifndef HAVE_FLOCK
		fclose(fp);
#else
		/* don't close it to preserve read lock */
#endif
		maxsch = cnt;
		fprintf(stdout, " - %d/%2.2d, %d lines\n",
			 sch[cnt - 1]->month, sch[cnt - 1]->day, maxsch);
		fflush(stdout);
	}
	for (cnt = 0; cnt < 10; cnt++) {
		*yank[cnt].work = '\0';
		*yank[cnt].prj = '\0';
		*yank[cnt].msg = '\0';
		 mark[cnt] = -1;
	}

	/*
	 * Initialize screen
	 */
	if (prjsw)
		scrmsg  = SCR_MSG;
	else
		scrmsg  = SCR_WORK;
#ifdef HAVE_TERMIOS_H
	tcgetattr(STDIN_FILENO, &tio);
    {
	struct termios t;
	tcgetattr(STDIN_FILENO, &t);
	t.c_cc[VINTR] = _POSIX_VDISABLE;
	t.c_cc[VSUSP] = _POSIX_VDISABLE;
#ifdef VDSUSP
	t.c_cc[VDSUSP] = _POSIX_VDISABLE;
#endif
	tcsetattr(STDIN_FILENO, TCSAFLUSH, &t);
    }
#else
	gtty(0, &ttyb);
    {
	struct ltchars ltbuf;

	ioctl(STDIN_FILENO, TIOCGLTC, &ltbuf);
	ltbuf.t_suspc = TCHAR_DISABLE;
	ltbuf.t_dsuspc = TCHAR_DISABLE;
	ioctl(STDIN_FILENO, TIOCSLTC, &ltbuf);
    }
#endif
	initscr();

	/*
	 * Define signal handling procedure
	 */
#ifdef SIGTSTP
	/*signal(SIGTSTP, restart);*/
#endif
#ifdef SIGWINCH
	signal(SIGWINCH, resize);
#endif
	signal(SIGINT,  errbell);
	signal(SIGQUIT, errbell);

	screen_init();
}

/*
 * Headder set
 */
static void
headinit()
{
	int curx, cury;
	char *pch;
	char line[LEN_BUFF];

	/*
	 * Line 0
	 */
	mvaddstr(0, 0, " ------- ");
	strftime(line, sizeof(line), "%b %e, %Y", inttime);
	if (newsw)
		pch = "[New file]";
	else if (! writesw)
		pch = "[Read only]";
	else
		pch = NULL;
	if (pch) {
		strcat(line, "  ");
		strcat(line, pch);
	}
	standout();
	wprintw(stdscr, "  Personal Schedule  <%s>  %s  ",
		 username, line);
	standend();
	getyx(stdscr, cury, curx);
	strcpy(line, " ---------------------------------------");
	if (COLS - curx - 1 < strlen(line))
		line[COLS - curx - 1] = '\0';
	addstr(line);

	/*
	 * Line 1
	 */
	puthead();
}

void
puthead()
{
	int i;
	int x;

	move(1, 0);
	clrtoeol();
	move(1, SCR_BAR);
	for (i = daystart; i <= dayend; i++) {
		if (i == daystart || i == dayend || i % 3 == 0) {
			x = SCR_BAR + (i - daystart) * hourwidth;
			if (i == dayend && hourwidth < 2)
				x--;
			move(1, x);
			wprintw(stdscr, "%-d", i);
		}
	}
	if (prjsw) {
		move(1, SCR_WORK);
		addstr("work_time");
		move(1, SCR_PRJ);
		addstr("prj");
	}
	move(1, scrmsg);
	addstr("schedule");
}

/*
 * Final procedure
 */
void
fin()
{
	/*
	 * Mode reset and exit
	 */
	wmove(errwin, 0, 0);
	wclrtoeol(errwin);
	wrefresh(errwin);
	endwin();
	nocrmode();
	echo();
#ifdef HAVE_TERMIOS_H
	tcsetattr(STDIN_FILENO, TCSAFLUSH, &tio);
#endif
	exit(0);
	/*NOTREACHED*/
}

/*
 * data area clear
 */
void
clrline(line, size)
	char *line;
	int size;
{
	memset(line, 0, size);
}

/*
 * Write back schedule data
 */
void
writesch(name)
	const char *name;
{
	int cnt;
	FILE *fp;
	struct sch s;

	if ((fp = fopen(name, "w")) != NULL) {
		saveopts(fp, 0);
		for (cnt = 0; cnt < maxsch; cnt++) {
			memcpy(&s, sch[cnt], sizeof(struct sch));
#if defined(HAVE_OFFTIME) && defined(HAVE_TIMEOFF)
			sch_tznorm(&s, homezone, curzone);
#endif
			if (oldformat)
				sch_oprint(fp, &s);
			else
				sch_print(fp, &s);
		}
		fclose(fp);
	} else
		error(strerror(errno));
}

/*
 * Message get from window
 */
char *
wgets(win, line, col)
	WINDOW	*win;
	int line, col;
{
	int inchar, incnt, oldy, oldx;
	int kanjied = FALSE;
	static char getbuf[LEN_BUFF];

	/*
	 * Save cursor
	 */
	getyx(schwin, oldy, oldx);
	wmove(win, line, col);
	wrefresh(win);

	/*
	 * Command and string get
	 */
	incnt = 0;
	getbuf[incnt] = '\0';
	while (1) {
		inchar = getch();
		if (inchar == '\n' || inchar == ESCAPE)
			break;

		/*
		 * Convert erase character
		 */
#ifdef HAVE_TERMIOS_H
		if (inchar == tio.c_cc[VERASE])
			inchar = ERS_CHAR;
		else if (inchar == tio.c_cc[VKILL])
			inchar = ERS_LINE;
#else
		if (inchar == ttyb.sg_erase)
			inchar = ERS_CHAR;
		else if (inchar == ttyb.sg_kill)
			inchar = ERS_LINE;
#endif

		/* special rule */
		if (inchar == ERS_CHAR && incnt == 0)
			break;

		switch (inchar) {
		case ERS_CHAR:
			/*
			 * Erase character
			 */
			if (incnt) {
				getbuf[--incnt] = '\0';
#ifndef MULTIBYTE
				wmove(win, line, col + incnt);
#else
				wmove(win, line,
					col + strlen2scrwidth(getbuf, incnt));
#endif
				wclrtoeol(win);
				wrefresh(win);
			}
			break;

		case ERS_LINE:
			/*
			 * Erase line
			 */
			if (incnt) {
				wmove(win, line, col + (incnt = 0));
				wclrtoeol(win);
				wrefresh(win);
			}
			break;

		case ERS_WORD:
			/*
			 * Erase word
			 */
			if (incnt) {
				for (incnt--; incnt && isspace(getbuf[incnt]);
				     incnt--);
				for (       ; incnt && !isspace(getbuf[incnt]);
				     incnt--);
				if (incnt)
					incnt++;
#ifndef MULTIBYTE
				wmove(win, line, col + incnt);
#else
				wmove(win, line,
					col + strlen2scrwidth(getbuf, incnt));
#endif
				wclrtoeol(win);
				wrefresh(win);
			}
			break;

		case EOF:
			/*
			 * High value on "cbreak" mode,  should be ignore.
			 */
			break;

		default:
			/*
			 * Add new character
			 */
			if (!isprint(inchar) && !isspace(inchar)) {
				errbell(0);
				break;
			}
			waddch(win, (getbuf[incnt++] = inchar));
			if (!kanjied)
				wrefresh(win);
		}
	}
	if (inchar == ESCAPE)
		incnt = 0;
	getbuf[incnt] = '\0';

	/*
	 * Restore cursor
	 */
	wmove(schwin, oldy, oldx);
	wrefresh(schwin);
	return getbuf;
}

/*
 * Error message print
 */
void
error(msg)
	const char *msg;
{
	wmove(errwin, 0, 0);
	wstandout(errwin);
	waddstr(errwin, msg);
	wstandend(errwin);
	wclrtoeol(errwin);
	wrefresh(errwin);
	wrefresh(schwin);
	msgsw++;
}

/*
 * Message print
 */
void
mesg(col, msg)
	int col;
	const char *msg;
{
	if (col < 0) {
		if (msgsw) {
			wmove(errwin, 0, 0);
			wclrtoeol(errwin);
			wrefresh(errwin);
			wrefresh(schwin);
			msgsw = FALSE;
		}
	}
	else {
		wmove(errwin, 0, col);
		wclrtoeol(errwin);
		waddstr(errwin, msg);
		wrefresh(errwin);
		wrefresh(schwin);
		msgsw = TRUE;
	}
}

/*
 * CLear and redraw 
 */
void
refwin()
{
	if (calsw) {
		wclear(errwin);
		touchwin(calwin);
		touchwin(schwin);
		clearok(stdscr, TRUE);
		refresh();
		wrefresh(schwin);
		wrefresh(calwin);
		wrefresh(errwin);
		wrefresh(schwin);
	} else {
		wclear(errwin);
		touchwin(schwin);
		clearok(stdscr, TRUE);
		refresh();
		wrefresh(schwin);
		wrefresh(errwin);
		wrefresh(schwin);
	}
	msgsw = FALSE;
}

/*
 * Visible bell
 */
RETSIGTYPE
errbell(sig)
	int sig;
{
	signal(SIGINT,  errbell);
	signal(SIGQUIT, errbell);

#if 0
	if (VB && *VB)
		putpad(VB);
	else
#endif
	{
		fputc(BELL, stderr);
		fflush(stderr);
	}
}

#ifdef MULTIBYTE
/*
 * display multibyte string
 */
void
mbdsp(msg)
	const wchar_t *msg;
{
	const wchar_t *p;
	wchar_t wc[3];
	char mb[32];	/*XXX*/
	mbstate_t state;
	int i;

	for (p = msg; *p; p++) {
		wcrtomb(NULL, '\0', &state);	/*initialize state*/
		mb[0] = '\0';
#if 1
		wc[0] = *p;
		wc[1] = '\0';	/* quickhack! */
		i = wcsrtombs(mb, &wc[0], sizeof(mb), &state);
		if (0 < i)
			mb[i] = '\0';
#else
		i = wcrtomb(mb, *p, &state);
		if (0 < i)
			mb[i] = '\0';
#endif
		if (mb[0] != '\033' && !(mb[0] & 0x80))
			addstr("\033(B");
		addstr(mb);
	}
}

/*
 * display multibyte string
 */
void
wmbdsp(w, msg)
	WINDOW *w;
	const wchar_t *msg;
{
	const wchar_t *p;
	wchar_t wc[3];
	char mb[32];	/*XXX*/
	mbstate_t state;
	int i;

	for (p = msg; *p; p++) {
		wcrtomb(NULL, '\0', &state);	/*initialize state*/
		mb[0] = '\0';
#if 1
		wc[0] = *p;
		wc[1] = '\0';	/* quickhack! */
		i = wcsrtombs(mb, &wc[0], sizeof(mb), &state);
		if (0 < i)
			mb[i] = '\0';
#else
		i = wcrtomb(mb, *p, &state);
		if (0 < i)
			mb[i] = '\0';
#endif
		if (mb[0] != '\033' && !(mb[0] & 0x80))
			waddstr(w, "\033(B");
		waddstr(w, mb);
	}
}
#endif

/*
 * Convert Kanji to "[]"
 */
char *
jisdsp(msg)
	char *msg;
{
	return (msg);
}

/*
 * Compute the last day of a month.
 */
int
lastday(year, month)
	int year, month;
{
	struct tm tm, *tmp;
	time_t t;

	/* get time_t for 1st day, next month. */
	memset(&tm, 0, sizeof(tm));
	month++;
	tm.tm_year = year - 1900 + (month / 12);
	tm.tm_mon = (month % 12) - 1;
	tm.tm_mday = 1;
	tm.tm_hour = 12;	/* to avoid possible leap second at 0AM */
	t = mktime(&tm);

	/* get struct tm for one day before. */
	t -= 60 * 60 * 24;
	tmp = gmtime(&t);
	return tmp->tm_mday;
}

/*
 * Generate empty lines for the month
 */
void
genlines()
{
	int curx, cury;
	int n, i;
	char msgbuf[LEN_BUFF];

	if (cursch + 1 < maxsch
	 && sch[cursch + 1]->year == sch[cursch]->year
	 && sch[cursch + 1]->month == sch[cursch]->month) {
		n = sch[cursch + 1]->day - sch[cursch]->day - 1;
	} else {
		n = lastday(sch[cursch]->year, sch[cursch]->month) -
			sch[cursch]->day;
	}

	if (n < 1) {
		error("Enough entries already exist");
		return;
	}
	if (maxsch + n > MAXDATA) {
		error("Too many entries");
		return;
	}

	for (i = maxsch - 1; i > cursch; i--)
		sch[i + n] = sch[i];
	for (i = 0; i < n; i++) {
		sch[cursch + i + 1] = sch_new();
		sch[cursch + i + 1]->year = sch[cursch]->year;
		sch[cursch + i + 1]->month = sch[cursch]->month;
		sch[cursch + i + 1]->day = sch[cursch]->day + i + 1;
	}
	maxsch += n;

	sprintf(msgbuf, "Generated from %d/%d to %d/%d",
		 sch[cursch + 1]->month, sch[cursch + 1]->day,
		 sch[cursch + n]->month, sch[cursch + n]->day);
	mesg(0, msgbuf);

	getyx(schwin, cury, curx);
	if (cury + n < getmaxy(schwin))
		;
	else
		n = getmaxy(schwin) - (cury + 1);
	if (n > 0) {
		wmove(schwin, getmaxy(schwin) - n, 0);
		wclrtobot(schwin);
		wmove(schwin, cury + 1, curx);
		for (i = 0; i < n; i++)
			winsertln(schwin);
		for (i =0 ; i < n; i++) {
			wmove(schwin, cury + i + 1, 0);
			screen_paintline(cursch + i + 1);
		}
		wmove(schwin, cury, curx);
	}
	wrefresh(schwin);
	updated = TRUE;
}

/*
 * Print yanked list
 */
void
yanklist()
{
	int cnt, nodata;

#ifdef SIGTSTP
	signal(SIGTSTP, SIG_IGN);
#endif
	signal(SIGINT,  SIG_IGN);
	signal(SIGQUIT, SIG_IGN);

	/*
	 * Print yanked list 
	 */
	nodata = 1;
	for (cnt = 0; cnt < sizeof(mark) / sizeof(mark[0]); cnt++) {
		if (!validmark(mark[cnt]) &&
		    !isdisp(yank[cnt].work) && !isdisp(yank[cnt].prj) &&
		    !isdisp(yank[cnt].msg))
			continue;
		if (nodata) {
			wmove(errwin, 0, 0);
			wrefresh(errwin);
			fputc('\n', stdout);
			fflush(stdout);
			printf("%-2s  %-10s", "#", "date");
			if (prjsw)
				printf("  %-11s  %-3s", "work_time", "prj");
			printf("  sched\n");
			nodata = 0;
		}
		printf("%2d", cnt);
		if (validmark(mark[cnt])) {
			printf("  %02d/%2.2d/%04d", sch[mark[cnt]]->month,
				sch[mark[cnt]]->day, sch[mark[cnt]]->year);
		} else
			printf("  %10s", "");
		if (prjsw) {
			printf("  %11.11s", jisdsp(yank[cnt].work));
			printf("  %3.3s", jisdsp(yank[cnt].prj));
		}
		printf("  ");
#ifndef MULTIBYTE
		printf("%s\n", jisdsp(yank[cnt].msg));
#else
		mbdsp(yank[cnt].msg);
		addch('\n');
#endif
	}

	/*
	 * Redraw window
	 */
	if (nodata)
		error("No data in registers");
	else {
		standout();
		fputs("[Hit any to continue]", stdout);
		standend();
		fputc(SPACE, stdout);
		fflush(stdout);
		getch();
		refwin();
	}
#ifdef SIGTSTP
	signal(SIGTSTP, restart);
#endif
#ifdef SIGWINCH
	signal(SIGWINCH, resize);
#endif
	signal(SIGINT,  errbell);
	signal(SIGQUIT, errbell);
}

/*
 * SHELL command execute
 */
void
execom(shcom)
	const char *shcom;
{
	/*
	 * Mode set
	 */
	signal(SIGINT,  SIG_IGN);
	signal(SIGQUIT, SIG_IGN);
	echo();
	nocrmode();
	wmove(errwin, 0, 0);
	wrefresh(errwin);
	fputc('\n', stdout);
	fflush(stdout);

	/*
	 * Command execute
	 */
	exec_system(shcom);
#if 0
	standout();
#endif
	fputs("[Hit return to continue]", stdout);
#if 0
	standend();
#endif
	fputc(SPACE, stdout);
	fflush(stdout);

	/*
	 * Mode reset
	 */
	noecho();
	crmode();
	getch();
	refwin();
	signal(SIGINT,  errbell);
	signal(SIGQUIT, errbell);
}

/*
 * Calander data set
 */
void
calinit()
{
	FILE *fcal;
	char *pch;
	int cnt;
	char buf[LEN_BUFF], calbuf[1024];

	/*
	 * This month data get
	 */
	memset(calbuf, 0, sizeof(calbuf));
	sprintf(buf,  "%s %d %d", PATH_CAL,
		 inttime->tm_mon + 1, inttime->tm_year + 1900);
	if ((fcal = popen(buf, "r")) == NULL) {
		err(1, "%s", PATH_CAL);
		/*NOTREACHED*/
	}
	fread(calbuf, 1024, 1, fcal);
	pclose(fcal);
	wmove(calwin, cnt = 0, 3);
	for (pch = calbuf; *pch; pch++)
		if (*pch == '\n')
			wmove(calwin, ++cnt, 1);
		else
			waddch(calwin, *pch);

	/*
	 * Next month data get
	 */
	for (cnt = 0; cnt < 1024; cnt++)
		calbuf[cnt] = NULL;
	if (inttime->tm_mon == 11)
		sprintf(buf,  "%s 1 %d", PATH_CAL, inttime->tm_year + 1901);
	else
		sprintf(buf,  "%s %d %d", PATH_CAL,
			 inttime->tm_mon + 2, inttime->tm_year + 1900);
	if ((fcal = popen(buf, "r")) == NULL) {
		err(1, "%s", PATH_CAL);
		/*NOTREACHED*/
	}
	fread(calbuf, 1024, 1, fcal);
	pclose(fcal);
	wmove(calwin, cnt = 0, 27);
	for (pch = calbuf; *pch; pch++)
		if (*pch == '\n')
			wmove(calwin, ++cnt, 25);
		else
			waddch(calwin, *pch);

	/*
	 * Display help
	 */
	help(HLP_PHEAD);
}

/*
 * Display help
 */
void
help(page)
	int page;
{
	int cnt;
	static int helpnow = HLP_PHEAD;
	static char *helpmsg[][HLP_LINES] = {
{ /* table 0 */
  "  Help    [@ for next]",
  "cursor .. SP BS h j k l - CR",
  "          TAB w b H M L 0 $",
  "window .. ^F ^D ^B ^U",
  "search .. / ? n N # G",
  "update .. r R d D o O",
  "          i a p P y Y m",
  "other ... ZZ ^G W Q s g =",
},
{ /* table 1 */
  "  Help:   Cursor",
  "h, BS ... cursor left",
  "l, SP ... cursor right",
  "j, +, CR  next line",
  "k, - .... previous line",
  "W, TAB, B next/prev field",
  "w, b .... next/prev 3hour",
  "H, M, L . head, middle, last",
},
{ /* table 2 */
  "  Help:   Window, Serach",
  "^F, ^D .. forward [full, half]",
  "^B, ^U .. backward [full, half]",
  "/, ? .... search [for, back]",
  "n, N .... search next",
  "# ....... go to today",
  "' ....... go to mark",
  "",
},
{ /* table 3 */
  "  Help:   Update",
  "r, i, a . replace, edit mode",
  "R ....... clear all of data",
  "d, D .... delete line/field",
  "p, P .... put line/field",
  "o, O .... open below/above",
  "y, m .... yank, mark line",
  "",
},
{ /* table 4 */
  "  Help:   Other",
  "ZZ ...... write and exit",
  "QQ ...... quit",
  "^W ...... auto write",
  "^G, Y ... status, yank list",
  "^C, ^P .. change display mode",
  "s, g .... sort, generate line",
  "S, = .... send mail, hard copy",
} };

	if (page == HLP_PNEXT) {
		if (helpnow == HLP_POTHR)
			page = helpnow = HLP_PHEAD;
		else
			page = ++helpnow;
	} else
		helpnow = page;
	for (cnt = 0; cnt < HLP_LINES; cnt++) {
		wmove(calwin, cnt, 49);
		wclrtoeol(calwin);
		waddstr(calwin, helpmsg[page][cnt]);
	}
}

/*
 * Find today data
 */
static void
schinit()
{
	int year, mon, day;

	year = inttime->tm_year + 1900;
	mon = inttime->tm_mon + 1;
	day = inttime->tm_mday;
	for (cursch = 0; cursch < maxsch - 1; cursch++) {
		if (sch[cursch]->year >= year
		 && sch[cursch]->month >= mon && sch[cursch]->day >= day)
			break;
	}
	schput(cursch);
}

/*
 * Cut schedule message
 */
#ifndef MULTIBYTE
char *
#else
wchar_t *
#endif
schmsg(indx)
	int indx;
{
	int tail;
#ifndef MULTIBYTE
	static char msgbuf[LEN_BUFF];

	tail = COLS - scrmsg - 1;
	strcpy(msgbuf, jisdsp(sch[indx]->msg));
	msgbuf[tail] = '\0';
#else
	static wchar_t msgbuf[LEN_BUFF];
	int i;

#if 0
	memset(msgbuf, 0, sizeof(msgbuf));
#endif
	wcscpy(msgbuf, sch[indx]->msg);
	tail = COLS - scrmsg - 1;
	i = scrwidth2strlen(msgbuf, tail);
	if (wcslen(msgbuf) < i)
		msgbuf[i] = '\0';
#endif
	return (msgbuf);
}

/*
 * Create time bar image
 */
const char *
schbar(indx)
	int indx;
{
	int begin, end, i, j;
	static char barbuf[LEN_BUFF];
	char *p;

	p = barbuf;
	for (i = daystart; i < dayend; i++) {
		*p++ = (i % (hourwidth * 3) == 0) ? ':' : '.';
		for (j = 1; j < hourwidth; j++)
			*p++ = ' ';
	}
	*p++ = (dayend % (hourwidth * 3) == 0) ? ':' : '.';
	*p = '\0';

	if (sch[indx]->sthour == 0 && sch[indx]->enhour == 0)
		return barbuf;
	if (sch[indx]->sthour < 24 && sch[indx]->sthour >= 0) {
		begin = (sch[indx]->sthour - daystart) * hourwidth + sch[indx]->stmin / (60 / hourwidth);
		end   = (sch[indx]->enhour - daystart) * hourwidth + sch[indx]->enmin / (60 / hourwidth);
		if (begin <= (dayend - daystart) * hourwidth && begin >= 0)
			barbuf[begin] = '*';
		if (end   <= (dayend - daystart) * hourwidth && end   >= 0)
			barbuf[end  ] = '*';
		if (begin < 0)
			begin = -1;
		if (end > (dayend - daystart) * hourwidth)
			end   = (dayend - daystart) * hourwidth;
		while (++begin < end)
			barbuf[begin] = '=';
	}
	return barbuf;
}

/*
 * Clear and schedule data print
 */
void
schput(indx)
	int indx;
{
	int cnt, stcnt, maxcnt, lcnt;

	/*
	 * Set start and end index
	 */
	if (indx < 0)
		indx = 0;
	if (indx >= maxsch && maxsch)
		indx = maxsch - 1;
#if 1
	stcnt = indx;
#else
	stcnt = (indx >= 3) ? indx - 3 : 0;
#endif
	cnt = stcnt + getmaxy(schwin);
	maxcnt = (cnt > maxsch) ? maxsch : cnt;

	/*
	 * Clear and print
	 */
	wclear(schwin);
	for (lcnt = 0, cnt = stcnt; cnt < maxcnt; lcnt++, cnt++) {
		wmove(schwin, lcnt, SCR_DAY);
		screen_paintline(cnt);
	}
	wmove(schwin, indx - stcnt, SCR_EDAY);
	cursch = indx;
}

/*
 * Hard Copy Screen
 */
void
hardcopy(fname)
	const char *fname;
{
	FILE *fcpy;
	int endy, cnty, cntx;

	if ((fcpy = fopen(fname, "a")) != NULL) {
		wmove(errwin, 0, 0);
		wprintw(errwin, "\"%s\"", fname);
		wclrtoeol(errwin);
		wrefresh(errwin);
#ifdef SIGTSTP
		signal(SIGTSTP, SIG_IGN);
#endif
		signal(SIGINT,  SIG_IGN);
		signal(SIGQUIT, SIG_IGN);
		endy = getmaxy(schwin) + 2;
		for (cnty = 0; cnty < endy; cnty++) {
			for (cntx = 0; cntx < COLS - 1; cntx++) {
				wmove(curscr, cnty, cntx);
				fputc(winch(curscr), fcpy);
			}
			fputc('\n', fcpy);
		}
		fputc('\n', fcpy);
		fclose(fcpy);
#ifdef SIGTSTP
		signal(SIGTSTP, restart);
#endif
#ifdef SIGWINCH
		signal(SIGWINCH, resize);
#endif
		signal(SIGINT,  errbell);
		signal(SIGQUIT, errbell);
		wmove(curscr, 0, 0);	/* omajinai */
		wprintw(errwin, " %d lines", cnty);
		wrefresh(errwin);
		wrefresh(schwin);
		msgsw = TRUE;
	} else
		error("Permission denied");
}

/*
 * String search
 */
void
search(dir)
	int dir;
{
	int status, cnt, newy, cury, curx, column;
#ifdef HAVE_REGEX_H
	regmatch_t matched[1];
#endif
	char buf[LEN_BUFF];

	for (cnt = cursch + dir; cnt != cursch; cnt += dir) {
		/*
		 * Wrap around set
		 */
		if (cnt == maxsch) {
			cnt = -1;
			continue;
		}
		if (cnt == -1) {
			cnt = maxsch;
			continue;
		}

		status = 0;

		/*
		 * Regular expression 
		 */
		sprintf(buf, "%02d/%02d", sch[cnt]->month, sch[cnt]->day);
#ifdef HAVE_REGEX_H
		if (regexec(psearch, buf, 1, &matched[0], 0) == 0)
			status += 1000;
#else
		status += 1000 * regexec(psearch, buf);
#endif

		sprintf(buf, "%d/%d", sch[cnt]->month, sch[cnt]->day);
#ifdef HAVE_REGEX_H
		if (regexec(psearch, buf, 1, &matched[0], 0) == 0)
			status += 1000;
#else
		status += 1000 * regexec(psearch, buf);
#endif

		if (prjsw) {
#ifdef HAVE_REGEX_H
			if (regexec(psearch, sch[cnt]->work, 1, &matched[0], 0) == 0)
				status += 100;
			if (regexec(psearch, sch[cnt]->prj, 1, &matched[0], 0) == 0)
				status += 10;
			if (regexec(psearch, sch[cnt]->msg, 1, &matched[0], 0) == 0)
				status += 1;	/*XXX*/
#else
			status += 100 * regexec(psearch, sch[cnt]->work);
			status += 10 * regexec(psearch, sch[cnt]->prj);
			status += 1 * regexec(psearch, sch[cnt]->msg);	/*XXX*/
#endif
		} else {
#ifdef HAVE_REGEX_H
			if (regexec(psearch, sch[cnt]->msg, 1, &matched[0], 0) == 0)	/*XXX*/
				status += 1;
#else
			status += 1 * regexec(psearch, sch[cnt]->msg);	/*XXX*/
#endif
		}
		if (status) {
			if (dir == SRCH_FOR) {
				if (status >= 1000)
					column = SCR_DAY;
				else if (status >= 100)
					column = SCR_WORK;
				else if (status >= 10)
					column = SCR_PRJ;
				else 
					column = scrmsg;
			} else {
				if (status < 10)
					column = scrmsg;
				else if (status < 100)
					column = SCR_PRJ;
				else if (status < 1000)
					column = SCR_WORK;
				else
					column = SCR_DAY;
			}
			getyx(schwin, cury, curx);
			newy = cury + cnt - cursch;
			if (newy >= 0 && newy < getmaxy(schwin)) {
				wmove(schwin, newy, column);
				cursch = cnt;
			} else {
				schput(cnt);
				getyx(schwin, cury, curx);
				wmove(schwin, cury, column);
			}
			wrefresh(schwin);
			return;
		}
	}
	error("Pattern not found");
}

/*
 * Scroll lines
 */
void
roll(offset)
	int offset;
{
	int curx, cury, goal;
	int topoff;
	int i;

	if (offset == 0)
		return;

	getyx(schwin, cury, curx);
	topoff = cursch - cury;
	goal = cursch + offset;
	if (goal < 0)
		goal = 0;
	else if (goal >= maxsch)
		goal = maxsch - 1;

	/* make sure screen shift does not do out-of-range movement */
	if (offset > 0) {
		if (topoff + offset + getmaxy(schwin) >= maxsch)
			offset = maxsch - getmaxy(schwin) - topoff;
		/* avoid unnecessary scroll */
		if (offset < 0)
			offset = 0;
	} else if (offset < 0) {
		if (topoff + offset < 0)
			offset = -1 * topoff;
	}

	if (0 < offset && offset < getmaxy(schwin)) {
		wmove(schwin, 0, 0);
		for (i = 0; i < offset; i++) {
			wdeleteln(schwin);
			topoff++;
		}
		wmove(schwin, getmaxy(schwin) - offset, 0);
		wclrtobot(schwin);
		for (i = getmaxy(schwin) - offset; i < getmaxy(schwin); i++) {
			if (topoff + i >= maxsch)
				break;
			wmove(schwin, i, 0);
			screen_paintline(topoff + i);
		}
		offset = 0;
	} else if (offset < 0 && offset * -1 < getmaxy(schwin)) {
		wmove(schwin, getmaxy(schwin) + offset, 0);
		wclrtobot(schwin);
		wmove(schwin, 0, 0);
		for (i = 0; i < offset * -1; i++) {
			winsertln(schwin);
			topoff--;
		}
		wmove(schwin, 0, 0);
		for (i = 0; i < offset * -1; i++) {
			wmove(schwin, i, 0);
			screen_paintline(topoff + i);
		}
		offset = 0;
	}
	if (offset > 0) {
		topoff += offset;
		if (topoff + getmaxy(schwin) >= maxsch)
			topoff = maxsch - getmaxy(schwin);
		schput(topoff);
	} else if (offset < 0) {
		topoff += offset;
		if (topoff < 0)
			topoff = 0;
		schput(topoff);
	}

	if (goal < topoff || goal >= topoff + getmaxy(schwin))
		abort();
	wmove(schwin, goal - topoff, SCR_EDAY);
	cursch = goal;
}

/*
 * Goto line
 */
void
goline(lineno)
	int lineno;
{
	int newy, cury, curx;

	if (lineno != cursch) {
		getyx(schwin, cury, curx);
		newy = cury + lineno - cursch;
		if (newy >= 0 && newy < getmaxy(schwin)) {
			wmove(schwin, newy, SCR_EDAY);
			cursch = lineno;
		} else if (lineno + getmaxy(schwin) < maxsch)
			schput(lineno);
		else {
			schput(maxsch - getmaxy(schwin));
			wmove(schwin, lineno - (maxsch - getmaxy(schwin)),
				SCR_EDAY);
			cursch = lineno;
		}

		wrefresh(schwin);
	}
}

/*
 * Search date
 */
void
godate(sdate)
	const char *sdate;
{
	const char *p;
	int cnt, newy, start, stop, cury, curx;
	int month, day;

	/*
	 * Month/day or day only?
	 */
	p = sdate;
	while (p && *p && isdigit(*p))
		p++;
	while (p && *p && !isdigit(*p))
		p++;
	if (*p) {
		month = atoi(sdate);
		day = atoi(p);
	} else {
		month = sch[cursch]->month;
		day = atoi(sdate);
	}

	/*
	 * Today Check
	 */
	if (month == sch[cursch]->month && day == sch[cursch]->day)
		return;

	/*
	 * Start and end index set
	 */
	if ((month > sch[cursch]->month) ||
	    (month == sch[cursch]->month && day >= sch[cursch]->day)) {
		start = cursch + 1;
		stop = maxsch;
	} else {
		start = 0;
		stop = cursch;
	}

	/*
	 * Search then go there
	 */
	for (cnt = start; cnt < stop; cnt++)
		if (month < sch[cnt]->month ||
		    (month == sch[cnt]->month && day <= sch[cnt]->day))
			break;
	if (cnt != cursch) {
		getyx(schwin, cury, curx);
		newy = cury + cnt - cursch;
		if (newy >= 0 && newy < getmaxy(schwin)) {
			wmove(schwin, newy, SCR_EDAY);
			cursch = cnt;
		} else
			schput(cnt);
		wrefresh(schwin);
	}
}

/*
 * Field check
 */
int
ismsg()
{
	int curx, cury;

	getyx(schwin, cury, curx);
	if (curx >= SCR_BAR && curx < SCR_WORK - 1)
		return (0);
	if (curx < SCR_BAR - 1)
		return (FLD_DAY);
	else if (prjsw && curx < SCR_PRJ - 1)
		return (FLD_WORK);
	else if (prjsw && curx < scrmsg - 1)
		return (FLD_PRJ);
	else
		return (FLD_MSG);
}

/*
 * Time bar set
 */
void
getbar()
{
	int startx, curx, cury, inchar;
	const char *pbar;
	int i;

	/*
	 * Save cursor
	 */
	pbar = schbar(cursch);
	getyx(schwin, cury, startx);
	waddch(schwin, '*');
	wmove(schwin, cury, curx = startx);
	wrefresh(schwin);

	/*
	 * Get command and display bar
	 */
	while ((inchar = getch()) != ESCAPE && inchar != '\n' &&
	       curx < SCR_WORK - 2) {
#ifdef HAVE_TERMIOS_H
		if (inchar == tio.c_cc[VERASE])
			inchar = ERS_CHAR;
		else if (inchar == tio.c_cc[VKILL])
			inchar = ERS_LINE;
#else
		if (inchar == ttyb.sg_erase)
			inchar = ERS_CHAR;
		else if (inchar == ttyb.sg_kill)
			inchar = ERS_LINE;
#endif

		switch (inchar) {
		case SPACE:
			/*
			 * Next character
			 */
			if (curx > startx)
				waddch(schwin, '=');
			else
				waddch(schwin, '*');
			wrefresh(schwin);
			curx++;
			break;

		case TAB:
			/*
			 * Advance for (tabhour) hours
			 */
			for (i = 0; i < tabhour * hourwidth; i++) {
				if (curx >= SCR_WORK - 2)
					break;
				if (curx > startx)
					waddch(schwin, '=');
				else
					waddch(schwin, '*');
				wrefresh(schwin);
				curx++;
			}
			break;

		case ERS_CHAR:
			/*
			 * Previous character
			 */
			if (curx > startx) {
				wmove(schwin, cury, --curx);
				waddch(schwin, pbar[curx - SCR_BAR]);
				wmove(schwin, cury, curx);
				wrefresh(schwin);
			}
			break;

		case ERS_WORD:
		case ERS_LINE:
			/*
			 * Back to first column
			 */
			if (curx > startx) {
				wmove(schwin, cury, SCR_BAR);
				waddstr(schwin, pbar);
				wmove(schwin, cury, curx = startx);
				waddch(schwin, '*');
				wmove(schwin, cury, curx);
				wrefresh(schwin);
			}
			break;

		case EOF:
			/*
			 * High value on "cbreak" mode,  should be ignore.
			 */
			break;

		  default:
			errbell(0);
		}
	}

	/*
	 * Set time and refreshwindow
	 */
	sch[cursch]->sthour = (startx - SCR_BAR) / hourwidth + daystart;
	sch[cursch]->stmin  = (startx - SCR_BAR) % hourwidth * (60 / hourwidth);
	sch[cursch]->enhour = (curx - SCR_BAR) / hourwidth + daystart;
	sch[cursch]->enmin  = (curx - SCR_BAR) % hourwidth * (60 / hourwidth);
	wmove(schwin, cury, 0);
	screen_paintline(cursch);
	wmove(schwin, cury, curx);
	wrefresh(schwin);
	updated = TRUE;

	/*
	 * Move cursor next line
	 */
	if (inchar == '\n') {
		if (cursch < maxsch - 1) {
			if (cury < getmaxy(schwin) - 1) {
				wmove(schwin, cury + 1, SCR_EDAY);
				cursch++;
			} else
				schput(cursch + 1);
		} else
			wmove(schwin, cury, SCR_EDAY);
	}
	wrefresh(schwin);
}

/*
 * Open line
 */
void
openline(newy, template)
	int newy;
	int template;
{
	int cnt, starty, curx;

	/*
	 * Save cursor
	 */
	getyx(schwin, starty, curx);
	if (starty == getmaxy(schwin) - 1) {
		schput(cursch);
		getyx(schwin, starty, curx);
	}

	/*
	 * Element shift
	 */
	for (cnt = maxsch; cnt > newy; cnt--)
		sch[cnt] = sch[cnt - 1];
	sch[newy] = sch_new();
	maxsch++;

	sch[newy]->year = sch[newy + template]->year;
	sch[newy]->month = sch[newy + template]->month;
	sch[newy]->day = sch[newy + template]->day;

	/*
	 * Refresh window
	 */
	wmove(schwin, getmaxy(schwin) - 1, 0);
	wclrtobot(schwin);
	wmove(schwin, newy - (cursch - starty), 0);
	winsertln(schwin);
	screen_paintline(newy);

	/*
	 * Restore cursor
	 */
	if (cursch == newy)
		wmove(schwin, starty, SCR_EDAY);
	else {	/* XXX assumes newy == cursch + 1 */
		wmove(schwin, starty + 1, SCR_EDAY);
		cursch = newy;
	}
	wrefresh(schwin);
	updated = TRUE;
}

/*
 * Field restore
 */
void
fldrest(regno, field)
	int regno, field;
{
	int curx, cury;

	getyx(schwin, cury, curx);
	if (prjsw) {
		switch (field) {
		case FLD_WORK:
			strcpy(sch[cursch]->work, yank[regno].work);
			wmove(schwin, cury, SCR_WORK);
			waddstr(schwin, "           ");
			wmove(schwin, cury, SCR_WORK);
			waddstr(schwin, jisdsp(sch[cursch]->work));
			wmove(schwin, cury, SCR_WORK);
			break;
		case FLD_PRJ:
			strcpy(sch[cursch]->prj, yank[regno].prj);
			wmove(schwin, cury, SCR_PRJ);
			waddstr(schwin, "   ");
			wmove(schwin, cury, SCR_PRJ);
			waddstr(schwin, jisdsp(sch[cursch]->prj));
			wmove(schwin, cury, SCR_PRJ);
			break;
		case FLD_MSG:
#ifndef MULTIBYTE
			strcpy(sch[cursch]->msg, yank[regno].msg);
#else
			wcscpy(sch[cursch]->msg, yank[regno].msg);
#endif
			wmove(schwin, cury, scrmsg);
			wclrtoeol(schwin);
#ifndef MULTIBYTE
			waddstr(schwin, schmsg(cursch));
#else
			wmbdsp(schwin, schmsg(cursch));
#endif
			wmove(schwin, cury, scrmsg);
		}
	} else {
#ifndef MULTIBYTE
		strcpy(sch[cursch]->msg, yank[regno].msg);
#else
		wcscpy(sch[cursch]->msg, yank[regno].msg);
#endif
		wmove(schwin, cury, scrmsg);
		wclrtoeol(schwin);
#ifndef MULTIBYTE
		waddstr(schwin, schmsg(cursch));
#else
		wmbdsp(schwin, schmsg(cursch));
#endif
		wmove(schwin, cury, scrmsg);
	}
	wrefresh(schwin);
	updated = TRUE;
}

/*
 * Element restore
 */
void
elmrest(regno)
	int regno;
{
	int curx, cury;

	strcpy(sch[cursch]->work, yank[regno].work);
	strcpy(sch[cursch]->prj , yank[regno].prj);
#ifndef MULTIBYTE
	strcpy(sch[cursch]->msg , yank[regno].msg);
#else
	wcscpy(sch[cursch]->msg, yank[regno].msg);
#endif

	getyx(schwin, cury, curx);
	if (prjsw) {
		wmove(schwin, cury, SCR_WORK);
		wclrtoeol(schwin);
		waddstr(schwin, jisdsp(sch[cursch]->work));
		wmove(schwin, cury, SCR_PRJ);
		waddstr(schwin, jisdsp(sch[cursch]->prj));
		wmove(schwin, cury, SCR_MSG);
#ifndef MULTIBYTE
		waddstr(schwin, schmsg(cursch));
#else
		wmbdsp(schwin, schmsg(cursch));
#endif
	} else {
		wmove(schwin, cury, scrmsg);
		wclrtoeol(schwin);
#ifndef MULTIBYTE
		waddstr(schwin, schmsg(cursch));
#else
		wmbdsp(schwin, schmsg(cursch));
#endif
	}
	wmove(schwin, cury, SCR_EDAY);
	wrefresh(schwin);
	updated = TRUE;
}

/*
 * Element save
 */
void
elmsave(regno)
	int regno;
{
	strcpy(yank[regno].work, sch[cursch]->work);
	strcpy(yank[regno].prj,  sch[cursch]->prj );
#ifndef MULTIBYTE
	strcpy(yank[regno].msg,  sch[cursch]->msg );
#else
	wcscpy(yank[regno].msg, sch[cursch]->msg);
#endif
}

/*
 * Erase filed
 */
void
delfld(field)
	int field;
{
	int curx, cury;

	getyx(schwin, cury, curx);
	if (prjsw) {
		switch (field) {
		case FLD_WORK:
			strcpy(yank[0].work, sch[cursch]->work);
			*sch[cursch]->work = '\0';
			wmove(schwin, cury, SCR_WORK);
			waddstr(schwin, "           ");
			wmove(schwin, cury, SCR_WORK);
			break;
		case FLD_PRJ:
			strcpy(yank[0].prj, sch[cursch]->prj);
			*sch[cursch]->prj = '\0';
			wmove(schwin, cury, SCR_PRJ);
			waddstr(schwin, "   ");
			wmove(schwin, cury, SCR_PRJ);
			break;
		case FLD_MSG:
#ifndef MULTIBYTE
			strcpy(yank[0].msg, sch[cursch]->msg);
#else
			wcscpy(yank[0].msg, sch[cursch]->msg);
#endif
			*sch[cursch]->msg = '\0';
			wmove(schwin, cury, scrmsg);
			wclrtoeol(schwin);
		}
	} else {
		*sch[cursch]->msg = '\0';
		wmove(schwin, cury, scrmsg);
		wclrtoeol(schwin);
	}
	wrefresh(schwin);
	updated = TRUE;
}

/*
 * Delete line
 */
void
delline(yankbuf)
	int yankbuf;
{
	int cnt, curx, cury, starty;

	/*
	 * Save cursor
	 */
	getyx(schwin, starty, curx);
	elmsave(yankbuf);

	/*
	 * Shift
	 */
	sch_free(sch[cursch]);
	for (cnt = cursch + 1; cnt < maxsch; cnt++)
		sch[cnt - 1] = sch[cnt];
	maxsch--;

	/*
	 * Refresh window
	 */
	wdeleteln(schwin);
	cnt = cursch + (getmaxy(schwin) - 1 - starty);
	if (cursch == maxsch)
		wclrtobot(schwin);
	else if (cursch < maxsch && cnt < maxsch) {
		cury = getmaxy(schwin) - 1;
		wmove(schwin, cury, SCR_DAY);
		screen_paintline(cnt);
	}

	/*
	 * Restore cursor
	 */
	if (cursch == maxsch) {
		cursch--;
		if (starty)
			wmove(schwin, starty - 1, SCR_EDAY);
		else
			schput(cursch);
	} else
		wmove(schwin, starty, SCR_EDAY);
	wrefresh(schwin);
	updated = TRUE;
}

/*
 * Reset data all in line 
 */
void
chnline()
{
	int curx, cury;

	getyx(schwin, cury, curx);
	sch[cursch]->sthour = 0;
	sch[cursch]->stmin = 0;
	sch[cursch]->enhour = 0;
	sch[cursch]->enmin = 0;
	*sch[cursch]->work = '\0';
	*sch[cursch]->prj = '\0';
	*sch[cursch]->msg = '\0';
	mvwaddstr(schwin, cury, SCR_BAR, schbar(cursch));
	wclrtoeol(schwin);
	wmove(schwin, cury, SCR_EDAY);
	wrefresh(schwin);
	updated = TRUE;
}

/*
 * Change data
 */
void
chndate(sdate)
	const char *sdate;
{
	const char *p;
	int newm, newd, newy, cury, curx;

	for (p = sdate; p && *p && (isdigit(*p) || *p == '/'); p++)
		;
	if (*p) {
		error("Incorrect date");
		return;
	}

	switch (sscanf(sdate, "%d/%d/%d", &newm, &newd, &newy)) {
	case 1:
		newd = newm;
		newy = sch[cursch]->year;
		newm = sch[cursch]->month;
		break;
	case 2:
		newy = sch[cursch]->year;
		break;
	case 3:
		break;
	default:
		error("Incorrect date");
		return;
	}

	if (newm < 1 || 12 < newm || newd < 1 || 31 < newd) {
		error("Incorrect date");
		return;
	}

	sch[cursch]->year = newy;
	sch[cursch]->month = newm;
	sch[cursch]->day = newd;

	/*
	 * Refresh window
	 */
	getyx(schwin, cury, curx);
	wmove(schwin, cury, 0);
	screen_paintline(cursch);
	wmove(schwin, cury, curx);
	wrefresh(schwin);
	updated = TRUE;
}

/*
 * Column cut
 */
char *
colcut(line, tail)
	char *line;
	int tail;
{
	static char msgbuf[LEN_BUFF];

	strcpy(msgbuf, line);
	msgbuf[tail] = '\0';
	return msgbuf;
}

/*
 * Date compare
 */
int
schcmp(lelmp, relmp)
	const void *lelmp;
	const void *relmp;
{
	struct sch *lelm = *(struct sch **)lelmp;
	struct sch *relm = *(struct sch **)relmp;
#define CMP(elm) \
do {								\
	if (lelm->elm != relm->elm)				\
		return lelm->elm < relm->elm ? -1 : 1;		\
} while (0)

	CMP(year);
	CMP(month);
	CMP(day);
	if (lelm->sthour * 60 + lelm->stmin != relm->sthour * 60 + relm->stmin) {
		if (lelm->sthour < relm->sthour)
			return -1;
		else if (lelm->sthour > relm->sthour)
			return 1;
		else if (lelm->stmin < relm->stmin)
			return -1;
		else if (lelm->stmin > relm->stmin)
			return 1;
	}
	if (lelm->enhour * 60 + lelm->enmin != relm->enhour * 60 + relm->enmin) {
		if (lelm->enhour < relm->enhour)
			return -1;
		else if (lelm->enhour > relm->enhour)
			return 1;
		else if (lelm->enmin < relm->enmin)
			return -1;
		else if (lelm->enmin > relm->enmin)
			return 1;
	}
#ifndef MULTIBYTE
	return strcmp(lelm->msg, relm->msg);
#else
	return wcscmp(lelm->msg, relm->msg);
#endif
#undef CMP
}

/*
 * Check non space
 */
int
isdisp(line)
	const char *line;
{
	const char *p;

	p = line;
	while (*p && isspace(*p))
		p++;

	if (*p && !isspace(*p))
		return TRUE;
	else
		return FALSE;
}

/*
 * Send Bug to Author
 */
void
sendbug()
{
#ifdef SENDBUG
	char *keyin;
	char subject[LEN_BUFF];

	mesg(0, "Send Bug Really ? ");
	keyin = wgets(errwin, 0, 18);
	if (*keyin == 'Y' || *keyin == 'y') {
		signal(SIGINT,  SIG_IGN);
		signal(SIGQUIT, SIG_IGN);
#ifdef SIGTSTP
		signal(SIGTSTP, SIG_IGN);
#endif
		mvcur(0, COLS - 1, LINES - 1, 0);
		endwin();
		nocrmode();
		echo();
		sprintf(subject, "`sch' [%s] BUG Report", VERSION);
		printf("\n%% mail %s\n", SENDUSR);
		fflush(stdout);
		signal(SIGINT,  SIG_DFL);
		signal(SIGQUIT, SIG_DFL);
#ifdef SIGTSTP
		signal(SIGTSTP, SIG_DFL);
#endif
#ifdef PATH_BIGMAIL
		execl(PATH_BIGMAIL, "mail", "-s", subject, SENDUSR, NULL);
#endif
#ifdef PATH_MAIL
		execl(PATH_MAIL, "mail", SENDUSR, NULL);
#endif
		errx(1, "can't exec external mail program");
		/*NOTREACHED*/
	}
	mesg(-1, "");
#else
	error("Send bug facilty not supported");
#endif
}

/*
 * String put without delay
 */
void
putpad(str)
	char *str;
{
	int ch;

	while (isdigit(ch = *str) || ch == '*')
		str++;
	fputs(str, stdout);
	fflush(stdout);
}

#ifdef SIGTSTP
/*
 * Handle stop and start signals
 */
RETSIGTYPE
restart(sig)
	int sig;
{
#ifdef BSD4_2
	int omask;
#endif
#ifdef HAVE_TERMIOS_H
	struct termios tiob;
#else
	struct sgttyb tty;
#endif


	if (writesw && autowrite && updated) {
		wmove(errwin, 0, 0);
		if (newsw) {
			wprintw(errwin, "\"%s\" [New file]", schsname);
			newsw = FALSE;
		} else
			wprintw(errwin, "\"%s\"", schsname);
		wclrtoeol(errwin);
		wrefresh(errwin);
		writesch(schfname);
		wprintw(errwin, " %d lines ", maxsch);
		wrefresh(errwin);
		updated = FALSE;
		msgsw = TRUE;
	}
#ifdef HAVE_TERMIOS_H
	tcgetattr(STDIN_FILENO, &tiob);
#else
	tty = _tty;
#endif
	mvcur(0, COLS - 1, LINES - 1, 0);
	endwin();
	fflush(stdout);
	signal(SIGTSTP, SIG_DFL);
#ifdef BSD4_2
#define	mask(s)	(1 << (s - 1))
	omask = sigsetmask(sigblock(0) &~ mask(SIGTSTP));
#endif
#ifdef HAVE_TERMIOS_H
	tcsetattr(STDIN_FILENO, TCSAFLUSH, &tio);
#endif
	kill(getpid(), SIGTSTP);

#ifdef BSD4_2
	sigblock(mask(SIGTSTP));
#endif
	signal(SIGTSTP, restart);
#ifdef HAVE_TERMIOS_H
	tcsetattr(STDIN_FILENO, TCSAFLUSH, &tiob);
#else
	_tty = tty;
	stty(_tty_ch, &_tty);
#endif
	resize(0);
	wrefresh(curscr);
	wrefresh(schwin);
}

/*
 * Execute command
 */
void
exec_system(scom)
	const char *scom;
{
	RETSIGTYPE (*sstat) __P((int)), (*istat) __P((int)), (*qstat) __P((int));
	int status, pid, w;
#ifdef HAVE_TERMIOS_H
	struct termios tiob;

	tcgetattr(STDIN_FILENO, &tiob);
#else
	struct sgttyb tty;

	tty = _tty;
#endif
	endwin();
	if ((pid = vfork()) == 0) {
		execl("/bin/sh", "sh", "-c", scom, 0);
		_exit(127);
	}
	sstat = signal(SIGTSTP, SIG_DFL);
	istat = signal(SIGINT,  SIG_IGN);
	qstat = signal(SIGQUIT, SIG_IGN);
	while ((w = wait(&status)) != pid && w != -1)
		;
#ifdef HAVE_TERMIOS_H
	tcsetattr(STDIN_FILENO, TCSAFLUSH, &tiob);
#else
	_tty = tty;
	stty(_tty_ch, &_tty);
#endif
	signal(SIGTSTP, sstat);
	signal(SIGINT, istat);
	signal(SIGQUIT, qstat);
}
#endif

#ifdef SIGWINCH
RETSIGTYPE
resize(sig)
	int sig;
{
	endwin();
	initscr();

	screen_init();
}
#endif

/*
 * It is caller's responsibility to update cursch.
 */
void
screen_scroll(offset)
	int offset;
{
	int topoff, curx, cury;

	getyx(schwin, cury, curx);
	topoff = cursch - cury;
	if (offset > 0) {
		while (offset > 0 && topoff < maxsch) {
			wmove(schwin, 0, 0);
			wdeleteln(schwin);
			topoff++;
			if (topoff + getmaxy(schwin) - 1 < maxsch) {
				wmove(schwin, getmaxy(schwin) - 1, 0);
				screen_paintline(topoff + getmaxy(schwin) - 1);
			}
			offset--;
		}
	} else if (offset < 0) {
		while (offset < 0 && topoff > 0) {
			wmove(schwin, getmaxy(schwin) - 1, 0);
			wclrtobot(schwin);
			wmove(schwin, 0, 0);
			winsertln(schwin);
			topoff--;
			wmove(schwin, 0, 0);
			screen_paintline(topoff);
			offset++;
		}
	}
}

static void
screen_paintline(line)
	int line;
{
	int curx, cury;
	int daymatch;
	int wday;
	char wch;
	int revday;

	if (inttime->tm_mon + 1 == sch[line]->month
	 && inttime->tm_mday == sch[line]->day) {
		daymatch = 1;
	} else
		daymatch = 0;
	wday = sch_wday(sch[line], curzone);

	wch = '\0';
	revday = 0;
	if (daymatch)
		revday = 1;
	if (inttime->tm_year + 1900 > sch[line]->year)
		wch = '-';
	else if (inttime->tm_year + 1900 < sch[line]->year)
		wch = '+';
	else {
		switch (wday) {
		case 0:	/* sunday */
			wch = 'S';
			break;
		case 6:	/* saturday */
			wch = 's';
			break;
		}
	}

	getyx(schwin, cury, curx);
	wmove(schwin, cury, 0);
	wclrtoeol(schwin);
	if (revday)
		wstandout(schwin);
	wprintw(schwin, "%2d/%2.2d", sch[line]->month, sch[line]->day);
	if (revday)
		wstandend(schwin);
	if (wch)
		wprintw(schwin, "%c", wch);
	mvwaddstr(schwin, cury, SCR_BAR, schbar(line));
	if (prjsw) {
		mvwaddstr(schwin, cury, SCR_WORK,
			   jisdsp(sch[line]->work));
		mvwaddstr(schwin, cury, SCR_PRJ ,
			   jisdsp(sch[line]->prj));
	}
#ifndef MULTIBYTE
	mvwaddstr(schwin, cury, scrmsg, schmsg(line));
#else
	wmove(schwin, cury, scrmsg);
	wmbdsp(schwin, schmsg(line));
#endif
}

static void
screen_init()
{

	/*
	 * Initialize subwindows
	 */
	crmode();
	noecho();
	if (COLS < 80 || LEN_FLWSCH <= 0 || LEN_HFWSCH <= 0) {
		errx(1, "screen too small");
		/*NOTREACHED*/
	}
	schfull = newwin(LEN_FLWSCH, 0, 2, 0);
	schhalf = newwin(LEN_HFWSCH, 0, 2, 0);
	calwin = newwin(8, 0, LINES - 9, 0);
	errwin = newwin(1, 0, LINES - 1, 0);
	if (calsw)
		schwin = schhalf;
	else
		schwin = schfull;

	/*
	 * Screen setup
	 */
	headinit();
	schinit();
	calinit();
	refwin();
}

static void
setprjsw(x)
	int x;
{
	prjsw = x;
	scrmsg = SCR_WORK;
}
