# SPDX-FileCopyrightText: 2013 Hynek Schlawack <hs@ox.cx>
#
# SPDX-License-Identifier: MIT

CERT_PEMS = [
    b"""-----BEGIN CERTIFICATE-----
MIIBfDCCATagAwIBAgIJAK94OSlzVBsWMA0GCSqGSIb3DQEBBQUAMBYxFDASBgNV
BAMTC3BlbS5pbnZhbGlkMB4XDTEzMDcxNzE0NDAyMFoXDTIzMDcxNTE0NDAyMFow
FjEUMBIGA1UEAxMLcGVtLmludmFsaWQwTDANBgkqhkiG9w0BAQEFAAM7ADA4AjEA
vtIM2QADJDHcqxZugx7MULbenrNUFrmoMDfEaedYveWY3wBxOw642L4nFWxN/fwL
AgMBAAGjdzB1MB0GA1UdDgQWBBQ4O0ZSUfTA6C+Y+QZ3MpeMhysxYjBGBgNVHSME
PzA9gBQ4O0ZSUfTA6C+Y+QZ3MpeMhysxYqEapBgwFjEUMBIGA1UEAxMLcGVtLmlu
dmFsaWSCCQCveDkpc1QbFjAMBgNVHRMEBTADAQH/MA0GCSqGSIb3DQEBBQUAAzEA
XwKIF+Kf4OhcqbdQp253HG2KBt/WZwvNLo/bBlkrGYwfacbGuWT8nKJG70ujdKKf
-----END CERTIFICATE-----
""",
    b"""-----BEGIN CERTIFICATE-----
MIIBfDCCATagAwIBAgIJAK9X9aUr9pYtMA0GCSqGSIb3DQEBBQUAMBYxFDASBgNV
BAMTC3BlbS5pbnZhbGlkMB4XDTEzMDcxNzE0NDAyMFoXDTIzMDcxNTE0NDAyMFow
FjEUMBIGA1UEAxMLcGVtLmludmFsaWQwTDANBgkqhkiG9w0BAQEFAAM7ADA4AjEA
v401YT8GeCt6oG076W/n7hxUsFO7sd74/4+2+4OcwMiLEp8BSRdWTk3g/tdF1YHT
AgMBAAGjdzB1MB0GA1UdDgQWBBT/dStoZFKGlnfedA7gtJV1K8JYKDBGBgNVHSME
PzA9gBT/dStoZFKGlnfedA7gtJV1K8JYKKEapBgwFjEUMBIGA1UEAxMLcGVtLmlu
dmFsaWSCCQCvV/WlK/aWLTAMBgNVHRMEBTADAQH/MA0GCSqGSIb3DQEBBQUAAzEA
uuXLfr1DgOMNt9JGNY5mBjabj3P7ALQYglygEe+QB7d0b/mFngn/aG35TuF5aud9
-----END CERTIFICATE-----
""",
    b"""-----BEGIN CERTIFICATE-----
MIIBfDCCATagAwIBAgIJAK4oWdJCuqj2MA0GCSqGSIb3DQEBBQUAMBYxFDASBgNV
BAMTC3BlbS5pbnZhbGlkMB4XDTEzMDcxNzE0NDAyMFoXDTIzMDcxNTE0NDAyMFow
FjEUMBIGA1UEAxMLcGVtLmludmFsaWQwTDANBgkqhkiG9w0BAQEFAAM7ADA4AjEA
wfq/eNemUKJ287E0ydVkzBxX44E6WhmnAN3oq7M881CxXLseNXHe/CRqYGpmziN5
AgMBAAGjdzB1MB0GA1UdDgQWBBQNtv8Fx7AEj4VCmX1I08mk4/viVzBGBgNVHSME
PzA9gBQNtv8Fx7AEj4VCmX1I08mk4/viV6EapBgwFjEUMBIGA1UEAxMLcGVtLmlu
dmFsaWSCCQCuKFnSQrqo9jAMBgNVHRMEBTADAQH/MA0GCSqGSIb3DQEBBQUAAzEA
i3qou3qkVXNKuiAFe9dBvz0nhcpAZpXrpwc9R4Qk+rirEqkdCZI1feQKBz4J3ikm
-----END CERTIFICATE-----
""",
]

# This is the first certificate from CERT_PEMS on which the
# trusted certificate extesion was applied using:
# openssl x509 -in cert.pem -trustout -addtrust emailProtection
CERT_PEM_OPENSSL_TRUSTED = b"""\
-----BEGIN TRUSTED CERTIFICATE-----
MIIBfDCCATagAwIBAgIJAK94OSlzVBsWMA0GCSqGSIb3DQEBBQUAMBYxFDASBgNV
BAMTC3BlbS5pbnZhbGlkMB4XDTEzMDcxNzE0NDAyMFoXDTIzMDcxNTE0NDAyMFow
FjEUMBIGA1UEAxMLcGVtLmludmFsaWQwTDANBgkqhkiG9w0BAQEFAAM7ADA4AjEA
vtIM2QADJDHcqxZugx7MULbenrNUFrmoMDfEaedYveWY3wBxOw642L4nFWxN/fwL
AgMBAAGjdzB1MB0GA1UdDgQWBBQ4O0ZSUfTA6C+Y+QZ3MpeMhysxYjBGBgNVHSME
PzA9gBQ4O0ZSUfTA6C+Y+QZ3MpeMhysxYqEapBgwFjEUMBIGA1UEAxMLcGVtLmlu
dmFsaWSCCQCveDkpc1QbFjAMBgNVHRMEBTADAQH/MA0GCSqGSIb3DQEBBQUAAzEA
XwKIF+Kf4OhcqbdQp253HG2KBt/WZwvNLo/bBlkrGYwfacbGuWT8nKJG70ujdKKf
MAwwCgYIKwYBBQUHAwQ=
-----END TRUSTED CERTIFICATE-----
"""

KEY_PEM = b"""\
-----BEGIN RSA PRIVATE KEY-----
MIHyAgEAAjEAvtIM2QADJDHcqxZugx7MULbenrNUFrmoMDfEaedYveWY3wBxOw64
2L4nFWxN/fwLAgMBAAECMCwqsCCV+SQqilnrQj8FJONVwGdZOJBd/iHi6ZXI2zbD
Q9Rv3iOsmqoCb5mqiDra0QIZAPbJRoliNA+2w7/dfttmWcQzcq8xL8qnEwIZAMXx
3hQNtUjuvgohXhZeBkyjP+7G0tceKQIZAPD9sFHsgiZuNU2hgIXDtxkvnGiUQbVF
3QIYMSmKQ6bH8K5DCtcQvDNsExq0pURCV2VJAhgnscmQDJ+DZblOG4zzn4pK7POX
OzCeivo=
-----END RSA PRIVATE KEY-----
"""


KEY_PEM2 = b"""\
-----BEGIN RSA PRIVATE KEY-----
MIH0AgEAAjEAv401YT8GeCt6oG076W/n7hxUsFO7sd74/4+2+4OcwMiLEp8BSRdW
Tk3g/tdF1YHTAgMBAAECMCus59Hvi+sUhtZTccitMmXRYeH+hZpt61RidFRLWzwe
nxAWvPxLtU9HC0Pc+zYBWQIZAP8ks93ruPqtoczsmiK+YSoyU+I4bKxM/wIZAMAx
2S5sDr/R+mizU6c8KnSRyQ60jY8HLQIZALZ8b9F4ObPB4IoLaCsVc7WUjX6t0Lxj
zQIZAKm8nHjiF9iSwlsrXMrKWRhgFDf3fzl89QIZALgkMvFA5CmRO+DMECBMsxIb
kjBF/mzooA==
-----END RSA PRIVATE KEY-----
"""

# KEY_PEM_PKCS8_* and KEY_PEM_PKCS5_* contain the same private key, but in
# different formats.

# PKCS#5 RSA unencrypted.
# Generated with:
# openssl genrsa -out private.pem 512
KEY_PEM_PKCS5_UNENCRYPTED = b"""\
-----BEGIN RSA PRIVATE KEY-----
MIIBOwIBAAJBAKX6cRhPHvdyoftEHGiRje3tTLRDnddg01AvgsJJcCFoIjwdgfa9
aKFdzCcgD/htjvfRZl24M7E89sMUBMNHk8ECAwEAAQJABcBi8OO1AAAh6tIWZe09
TNRfRxPcwVzilbG/xznCP/YMf72E8hsZazu+HGMKITg9dFeJOyjXZ4e8sD/pL/I6
0QIhANzULu4JjJxpoTK8NnF/CemF7neLROA18NDB/mao5ZZtAiEAwGnYobinxuHS
UQh8cT3w7aLsVlarZmCtoapxjW+ObiUCIQCcAltVV/G63vU/PrDH5hQ+opwiYIW8
UN9c3HC6XkA00QIhAJ8YpfwKgAfNfyZrmuHTspv7Q+mb3jtXoxnyodOtsxpVAiBC
a4FDqkr+bDwV4SwaGdG/AC40fR3P8hhOADAhtFDwlw==
-----END RSA PRIVATE KEY-----
"""

KEY_PEM_PKCS5_UNENCRYPTED_PAYLOAD = (
    b"MIIBOwIBAAJBAKX6cRhPHvdyoftEHGiRje3tTLRDnddg01AvgsJJcCFoIjwdgfa9"
    b"aKFdzCcgD/htjvfRZl24M7E89sMUBMNHk8ECAwEAAQJABcBi8OO1AAAh6tIWZe09"
    b"TNRfRxPcwVzilbG/xznCP/YMf72E8hsZazu+HGMKITg9dFeJOyjXZ4e8sD/pL/I6"
    b"0QIhANzULu4JjJxpoTK8NnF/CemF7neLROA18NDB/mao5ZZtAiEAwGnYobinxuHS"
    b"UQh8cT3w7aLsVlarZmCtoapxjW+ObiUCIQCcAltVV/G63vU/PrDH5hQ+opwiYIW8"
    b"UN9c3HC6XkA00QIhAJ8YpfwKgAfNfyZrmuHTspv7Q+mb3jtXoxnyodOtsxpVAiBC"
    b"a4FDqkr+bDwV4SwaGdG/AC40fR3P8hhOADAhtFDwlw=="
)

# PKCS#5 RSA encrypted with `test` as password.
# Generated with:
# openssl genrsa -des3 -out private.pem 512
KEY_PEM_PKCS5_ENCRYPTED = b"""\
-----BEGIN RSA PRIVATE KEY-----
Proc-Type: 4,ENCRYPTED
DEK-Info: DES-EDE3-CBC,8A72BD2DC1C9092F

6LgvCNeXdcuTayEOKhQo2N4IveCP0S3t8xJCeihW9yizLeQFzSjqSfKtmRyImjfg
fMl8IMDFozR+xVE9uWaIo98wKWpjyu6cytYyjL/8SP3jswBoSP5P9OekUSLifPWM
ghUEu6tGissqSs/8i2wzLIdho3DdUnUMPZIprENmK6HrYmdRtJT3qMgkFTCtCS9Q
r9oPm7xKPsfKBhaUHK51JcsPkPjrny8Dl56W0IYf/dfvRPwSr5yFQFLk6Nbgnx0N
32aT3ZMRCEvOTxhX1cO3f5JqYLxFAGKBFwvsulTisJ6rGYOEDSMBDwZc3sqLvt5g
h0vKRPqSkylQ0W5shNg0bwbxySiRxJPBL8kWDAbJVfauArabLPuNkUNwmYhIjy7j
lY0oYw2xeJ9hTUly/Zg3+DI8oYYY3z7WaxPHXEoicCE=
-----END RSA PRIVATE KEY-----
"""

# PKCS#8 RSA encrypted with `test` as password.
# Generated with pkc5 as intermediate file:
# openssl genrsa -des3 -out private.pem 512
# openssl pkcs8 -topk8 -in private.pem
KEY_PEM_PKCS8_ENCRYPTED = b"""\
-----BEGIN ENCRYPTED PRIVATE KEY-----
MIIBvTBXBgkqhkiG9w0BBQ0wSjApBgkqhkiG9w0BBQwwHAQIyqwWErm7rlcCAggA
MAwGCCqGSIb3DQIJBQAwHQYJYIZIAWUDBAEqBBAkVu+KRbmcfWIGKzgnjjBMBIIB
YI3aRS0ebuzb1Tq26/HAq8pplPu+96dM1SnRNXwH0ijmP3fLBjEDH4hB/X9H8arT
xWSfKQ80+FKI07DsLQKmO+cuB12MAWPSoCNBRtLwGUiwYvlMcBp6XR4NQQ+YG/Nw
OgZ1InH2w7uSnDPdxV9dZculYWzJE82IohnFVZokO2nYSEfIqr1xVQZht6lfzpx2
aRje42fpYfgkEm13w4oJKIlekzA9M4CeYku7Q4l9GDSHRmoeypMSHPI8RFV9pxub
ME3AMXGcRioJ0Ic/cpmwqFaJbTVRPsqFVEsMCz1T/CQ4oLjPTWg+zkxfsPIyGj7L
K3yLZmTA6IxSu+wuO/bsbqiM3x718AW6U0FHXd4zk+llu3mUfhTiMYPvN/cedv/M
wsT85CHM6reIBopGMqeZD965tNEcWPGMEvXXnG71dxxgrfHFv7l/o8+moVRNIQCh
EArlaXgT3MlI1jb9HoNvVNg=
-----END ENCRYPTED PRIVATE KEY-----
"""

# RSA unencrypted
# Generated with pkc5 as intermediate file:
# openssl genrsa -des3 -out private.pem 512
# openssl pkcs8 -topk8 -in private.pem -nocrypt
KEY_PEM_PKCS8_UNENCRYPTED = b"""\
-----BEGIN PRIVATE KEY-----
MIIBVQIBADANBgkqhkiG9w0BAQEFAASCAT8wggE7AgEAAkEApfpxGE8e93Kh+0Qc
aJGN7e1MtEOd12DTUC+CwklwIWgiPB2B9r1ooV3MJyAP+G2O99FmXbgzsTz2wxQE
w0eTwQIDAQABAkAFwGLw47UAACHq0hZl7T1M1F9HE9zBXOKVsb/HOcI/9gx/vYTy
GxlrO74cYwohOD10V4k7KNdnh7ywP+kv8jrRAiEA3NQu7gmMnGmhMrw2cX8J6YXu
d4tE4DXw0MH+Zqjllm0CIQDAadihuKfG4dJRCHxxPfDtouxWVqtmYK2hqnGNb45u
JQIhAJwCW1VX8bre9T8+sMfmFD6inCJghbxQ31zccLpeQDTRAiEAnxil/AqAB81/
Jmua4dOym/tD6ZveO1ejGfKh062zGlUCIEJrgUOqSv5sPBXhLBoZ0b8ALjR9Hc/y
GE4AMCG0UPCX
-----END PRIVATE KEY-----
"""


DH_PEM = b"""\
-----BEGIN DH PARAMETERS-----
MIICCAKCAgEAj9/hwPNNKlQEANXqFBXViNy9nVpYlqIIHaLhoKdwAFzgYM+9hNSz
FM/k+K5FS5dXrM63Zh9NgTI1M+ZRHJAxM2hhsG8AA333PN+c3exTRGwjQhU16XJg
Jw/r/jYfsAyKRXqZnMuXRg+3ALEHEkvNt/vMm7Zdo6SvNIN1B2I/qEGlqFRYSgx8
dS7CRxfVf9lHE+q6HB2l/bmwZQzxvt900Dclp5DZaUnS0zS8OGsx3/QcDCZ4/4nB
Gs53uDcFzb6BnsoT3oowvzCGTihHXGmS5dJaMEWwdoL1/X9ZrN3+Sex9XZufZLoO
nGWjEI4yEQJbPql1iutLBR723ZDLbz9bbo86vhJJqYPJyDvxLHv6moh27PCL95JH
Cg1mp4DCrjPYyd65fDvd/gP2Fa7Y+yH74CsV3LLzkxIEv9WobpxRamIljehVAdUA
ZHXSLSROnc5iGNT3gJ7jcFucD7pi5MbSa1nKVc0C4Vp4f1enrVvNB/je3no0KTtN
0aY+RqJTqypmsiv6b6zdLfaZHBcqy7/qmeJu/HTIiAYhk6LW6RDFcBLJCFEujV5Y
5qjyk41M4luGVr0m41LkiXPgCPko8kiqnzDi7inVjKjrmKfspCzROw9edIdvAtfH
vX0dAb0xNIpm5IRffJTVVpeZwNXuE7cNvBSZ0GY4MPkiO2dcPHeV28MCAQI=
-----END DH PARAMETERS-----
"""

CERT_NO_NEW_LINE = b"""\
-----BEGIN CERTIFICATE-----
MIIBfDCCATagAwIBAgIJAK4oWdJCuqj2MA0GCSqGSIb3DQEBBQUAMBYxFDASBgNV
BAMTC3BlbS5pbnZhbGlkMB4XDTEzMDcxNzE0NDAyMFoXDTIzMDcxNTE0NDAyMFow
FjEUMBIGA1UEAxMLcGVtLmludmFsaWQwTDANBgkqhkiG9w0BAQEFAAM7ADA4AjEA
wfq/eNemUKJ287E0ydVkzBxX44E6WhmnAN3oq7M881CxXLseNXHe/CRqYGpmziN5
AgMBAAGjdzB1MB0GA1UdDgQWBBQNtv8Fx7AEj4VCmX1I08mk4/viVzBGBgNVHSME
PzA9gBQNtv8Fx7AEj4VCmX1I08mk4/viV6EapBgwFjEUMBIGA1UEAxMLcGVtLmlu
dmFsaWSCCQCuKFnSQrqo9jAMBgNVHRMEBTADAQH/MA0GCSqGSIb3DQEBBQUAAzEA
i3qou3qkVXNKuiAFe9dBvz0nhcpAZpXrpwc9R4Qk+rirEqkdCZI1feQKBz4J3ikm
-----END CERTIFICATE-----"""

CERT_PEMS_NO_NEW_LINE = [
    b"""-----BEGIN CERTIFICATE-----
MIIBfDCCATagAwIBAgIJAK94OSlzVBsWMA0GCSqGSIb3DQEBBQUAMBYxFDASBgNV
BAMTC3BlbS5pbnZhbGlkMB4XDTEzMDcxNzE0NDAyMFoXDTIzMDcxNTE0NDAyMFow
FjEUMBIGA1UEAxMLcGVtLmludmFsaWQwTDANBgkqhkiG9w0BAQEFAAM7ADA4AjEA
vtIM2QADJDHcqxZugx7MULbenrNUFrmoMDfEaedYveWY3wBxOw642L4nFWxN/fwL
AgMBAAGjdzB1MB0GA1UdDgQWBBQ4O0ZSUfTA6C+Y+QZ3MpeMhysxYjBGBgNVHSME
PzA9gBQ4O0ZSUfTA6C+Y+QZ3MpeMhysxYqEapBgwFjEUMBIGA1UEAxMLcGVtLmlu
dmFsaWSCCQCveDkpc1QbFjAMBgNVHRMEBTADAQH/MA0GCSqGSIb3DQEBBQUAAzEA
XwKIF+Kf4OhcqbdQp253HG2KBt/WZwvNLo/bBlkrGYwfacbGuWT8nKJG70ujdKKf
-----END CERTIFICATE-----""",
    b"""-----BEGIN CERTIFICATE-----
MIIBfDCCATagAwIBAgIJAK9X9aUr9pYtMA0GCSqGSIb3DQEBBQUAMBYxFDASBgNV
BAMTC3BlbS5pbnZhbGlkMB4XDTEzMDcxNzE0NDAyMFoXDTIzMDcxNTE0NDAyMFow
FjEUMBIGA1UEAxMLcGVtLmludmFsaWQwTDANBgkqhkiG9w0BAQEFAAM7ADA4AjEA
v401YT8GeCt6oG076W/n7hxUsFO7sd74/4+2+4OcwMiLEp8BSRdWTk3g/tdF1YHT
AgMBAAGjdzB1MB0GA1UdDgQWBBT/dStoZFKGlnfedA7gtJV1K8JYKDBGBgNVHSME
PzA9gBT/dStoZFKGlnfedA7gtJV1K8JYKKEapBgwFjEUMBIGA1UEAxMLcGVtLmlu
dmFsaWSCCQCvV/WlK/aWLTAMBgNVHRMEBTADAQH/MA0GCSqGSIb3DQEBBQUAAzEA
uuXLfr1DgOMNt9JGNY5mBjabj3P7ALQYglygEe+QB7d0b/mFngn/aG35TuF5aud9
-----END CERTIFICATE-----""",
    b"""-----BEGIN CERTIFICATE-----
MIIBfDCCATagAwIBAgIJAK4oWdJCuqj2MA0GCSqGSIb3DQEBBQUAMBYxFDASBgNV
BAMTC3BlbS5pbnZhbGlkMB4XDTEzMDcxNzE0NDAyMFoXDTIzMDcxNTE0NDAyMFow
FjEUMBIGA1UEAxMLcGVtLmludmFsaWQwTDANBgkqhkiG9w0BAQEFAAM7ADA4AjEA
wfq/eNemUKJ287E0ydVkzBxX44E6WhmnAN3oq7M881CxXLseNXHe/CRqYGpmziN5
AgMBAAGjdzB1MB0GA1UdDgQWBBQNtv8Fx7AEj4VCmX1I08mk4/viVzBGBgNVHSME
PzA9gBQNtv8Fx7AEj4VCmX1I08mk4/viV6EapBgwFjEUMBIGA1UEAxMLcGVtLmlu
dmFsaWSCCQCuKFnSQrqo9jAMBgNVHRMEBTADAQH/MA0GCSqGSIb3DQEBBQUAAzEA
i3qou3qkVXNKuiAFe9dBvz0nhcpAZpXrpwc9R4Qk+rirEqkdCZI1feQKBz4J3ikm
-----END CERTIFICATE-----""",
]

CRL_PEMS = [
    b"""-----BEGIN X509 CRL-----
MIIBHTCBhwIBATANBgkqhkiG9w0BAQsFADBAMRcwFQYDVQQKDA50cnVzdG1lIHYw
LjQuMDElMCMGA1UECwwcVGVzdGluZyBDQSAjOW5QSzlVdHdiUXBsWVlKRxcNMDAw
MTAxMDAwMDAwWhgPMzAwMDAxMDEwMDAwMDBaoBEwDzANBgNVHRQEBgIEW1he8TAN
BgkqhkiG9w0BAQsFAAOBgQCFvaiM6sky2LaFU5gcI3KPdmLKiAnZRSbIkMvflsAm
bFqBGT72XWeLZe9Yuf5LrXrT8V+iZzzxA3UmQn36BVSbY4ZQzOryQVh1w0mELYmO
tRx8uHnC2sodKxpwelfkVkoTyUvMHLDmsfBhpNnRO5Uh7aIq/ZWt2tzxiCJHZgLD
gQ==
-----END X509 CRL-----
""",
    b"""-----BEGIN X509 CRL-----
MIIBRjCBsAIBATANBgkqhkiG9w0BAQsFADBAMRcwFQYDVQQKDA50cnVzdG1lIHYw
LjQuMDElMCMGA1UECwwcVGVzdGluZyBDQSAjNnEtVlBkeUpVcGdaOXN6TxcNMDAw
MTAxMDAwMDAwWhgPMzAwMDAxMDEwMDAwMDBaMCcwJQIUc5dlqlXTL5nQqyF72UKz
8y8sjWMXDTAwMDEwMTAwMDAwMFqgETAPMA0GA1UdFAQGAgRbWGEaMA0GCSqGSIb3
DQEBCwUAA4GBACt6+sxvplrTiaAl3pAu1/Xgj45PN6YWGpNl/h0A6///2EcfnioF
tL640vzHDe6wVO7pmxRkYORsplEVWog6P+AFh3n5MYc+MFBVn94hq8OGQlE6kKJ3
Sb6rAvgbb2gPFFZr5fHTBwSpWZhuVJ32eqNdvkLBjdNgYZhFVonkVIw1
-----END X509 CRL-----
""",
]
KEY_PEM_RSA_PUBLIC = b"""\
-----BEGIN RSA PUBLIC KEY-----
MIIBCgKCAQEAq4a0j5Za0zUrkJjWWCEXBH44L3+wWQf2VRwNA3ICUHfjwbb0aGwp
4PaqY9CS88Tzmrwn9yvfkuaxa3dTIaCaoW62C6CwdCu7O2QMFZi4H2oO9NBM2ni1
rzaKulkvZV4iPGyQSiqJqMy//DAEPVx/kyOkH9oNA62srv95gs8j83inTShhUfF6
wVtZzTKDkCiNtq9ZEXl4bJnEo4CmPhpI6AKCNxztocyKDU2rV6igIfo3UjV3U8nU
DPzjzHrBPoXbzXEgY6RkmgJwzCTIkv2BYkMwafY9ogDo4e1fY6JiwWzZlSEgScZb
VfLmAh4rZNe5PiQtDZiwZkvfSK7+Sxaa4QIDAQAB
-----END RSA PUBLIC KEY-----"""
KEY_PEM_PUBLIC = b"""\
-----BEGIN PUBLIC KEY-----
MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEA61BjmfXGEvWmegnBGSuS
+rU9soUg2FnODva32D1AqhwdziwHINFaD1MVlcrYG6XRKfkcxnaXGfFDWHLEvNBS
EVCgJjtHAGZIm5GL/KA86KDp/CwDFMSwluowcXwDwoyinmeOY9eKyh6aY72xJh7n
oLBBq1N0bWi1e2i+83txOCg4yV2oVXhBo8pYEJ8LT3el6Smxol3C1oFMVdwPgc0v
Tl25XucMcG/ALE/KNY6pqC2AQ6R2ERlVgPiUWOPatVkt7+Bs3h5Ramxh7XjBOXeu
lmCpGSynXNcpZ/06+vofGi/2MlpQZNhHAo8eayMp6FcvNucIpUndo1X8dKMv3Y26
ZQIDAQAB
-----END PUBLIC KEY-----"""

# generated with:
# openssl ecparam -name secp256k1 -genkey -noout -out key.pem
# Documented at https://tools.ietf.org/html/rfc5915
KEY_PEM_EC_PRIVATE = b"""\
-----BEGIN EC PRIVATE KEY-----
MHQCAQEEIGTpm0NjJRU5dYDrRPh+C9agdudJvCGSBd1hah5jnMYPoAcGBSuBBAAK
oUQDQgAEyEVc22cgFbf0Ey4W7tNawHEW4o+4xNVznhP4et9g53Puw1KSVD/GOfl1
95dwvaIyx2XZG8tn9DRsVyjy8fjgFA==
-----END EC PRIVATE KEY-----"""

# generated with:
# ssh-keygen -t ecdsa
KEY_PEM_OPENSSH = b"""\
-----BEGIN OPENSSH PRIVATE KEY-----
b3BlbnNzaC1rZXktdjEAAAAABG5vbmUAAAAEbm9uZQAAAAAAAAABAAAAaAAAABNlY2RzYS
1zaGEyLW5pc3RwMjU2AAAACG5pc3RwMjU2AAAAQQTw5dr6HzmoqoF1tJkUiCrFvbPbe18u
iJ2gmS+0mmdwledpUh8g9VWeCGUpa50MYWH83Y+cT0TY9jjRUhkNCchzAAAAuB5SwggeUs
IIAAAAE2VjZHNhLXNoYTItbmlzdHAyNTYAAAAIbmlzdHAyNTYAAABBBPDl2vofOaiqgXW0
mRSIKsW9s9t7Xy6InaCZL7SaZ3CV52lSHyD1VZ4IZSlrnQxhYfzdj5xPRNj2ONFSGQ0JyH
MAAAAhAMP/HkDnx5kbDXrh2EMYhj5FFAB2jbwXRVvJqeM6jD09AAAAHmJ1ZGR5QEJ1ZGR5
cy1NYWNCb29rLVByby5sb2NhbAE=
-----END OPENSSH PRIVATE KEY-----"""

# OpenSSH legacy PEM private key format:
# ssh-keygen -t dsa -m PEM
# OpenSSL DSA key.
# openssl dsaparam -out key.pem -genkey 1024:
KEY_PEM_DSA_PRIVATE = b"""\
-----BEGIN DSA PRIVATE KEY-----
MIIBugIBAAKBgQCMaLmHH3HlQwVVp2mJq2Peblj+rjeLfN20fFHNm5LecTP0XIO9
48chkzea4Ma2lv/hGTkYLlA0dQwYoAFO9rwrZo9HU+CXpx3A9BYVk3IWif7rSXZP
3HCmQ0vN7nZhzW32Us6LM8MW5ZW1tJYQLPsDpGKgNVdKLZt+nhzJ3MxF3QIVAJKB
6g66zmFYO+SN6zFYkco0wrNHAoGABZ0tQnYYj6uQoTK0mE90jsnUA3WpXenZDKBt
TcryHe0ijwr4hzqGRJNmxBKgCX7mhYP2j5Kyd91BQDDNc9K41xeH3ikTal6O2b4J
ckkxAyjhZccxwkvBKJVXC/g9I5ePbWGDheq3TO76sJNOcHHt0/KTGKb0Zy5rtgOn
CSJ5eD0CgYBX0u5FjDreCn+4vqIwQyPgIvIIsq4S2WNWMEp1JvxA5OB+2BZh83Ua
Qrb0riZLOXc966m9uXkBJE+Eimh+Jed/qfbwNuTZbxVz9rmsnbGHj8kvJT4c3J27
NRrjPxY+c3X65vSaThscOQ0SHm5bRhX2YNRhgnZPznUnMXfE8yRLdgIUUS6kFIid
HhSy7IHLTHWGoNdmwLo=
-----END DSA PRIVATE KEY-----"""

# Taken from https://datatracker.ietf.org/doc/html/rfc4716#section-3.6.
KEY_PEM_RFC4716_PUBLIC = rb"""---- BEGIN SSH2 PUBLIC KEY ----
Subject: me
Comment: 1024-bit rsa, created by me@example.com Mon Jan 15 08:31:24 2001
AAAAB3NzaC1yc2EAAAABJQAAAIEAiPWx6WM4lhHNedGfBpPJNPpZ7yKu+dnn1SJejgt4
596k6YjzGGphH2TUxwKzxcKDKKezwkpfnxPkSMkuEspGRt/aZZ9wa++Oi7Qkr8prgHc4
soW6NUlfDzpvZK2H5E7eQaSeP3SAwGmQKUFHCddNaP0L+hM7zhFNzjFvpaMgJw0=
---- END SSH2 PUBLIC KEY ----
"""

# SSH.COM / Tectia private (encrypted or plain) key.
# The non-encrypted key has the same armor.
# puttygen -t rsa -O private-sshcom
KEY_PEM_SSHCOM_PRIVATE = b"""\
---- BEGIN SSH2 ENCRYPTED PRIVATE KEY ----
Comment: "rsa-key-20210120"
P2/56wAAAUwAAAA3aWYtbW9kbntzaWdue3JzYS1wa2NzMS1zaGExfSxlbmNyeXB0e3JzYS
1wa2NzMXYyLW9hZXB9fQAAAARub25lAAAA/QAAAPkAAAAGJQAAAf0VIAK0MvdpUXEG6obL
3F5n0UimJWvwhJIb5AGyd++EdYvimCOg9iM2E75dDj89Ap7S5l4IS40fZO/5UjzYQxitAA
ACAMNoGQLXcI4xVX/5Xt22aUBP4ADaJnDKR4H9D7LVZ4lBDUP8RBTmowCv9p3Hz7KvVw3R
TX8BNF72gEuSOvruUAUAAAD9Hs7Zn1KbFR29ujFEv+d50/7rjMU7Ox4tzDeTSE6PBhsAAA
EA3m/0JWkf61807iZ7AV8umYJMmNQ35HadG53n9nitpFEAAAEA4OQI1Rrh8e1EZ5qJBV8o
gGyxzt4OdoXzuOtxkbHUB3U=
---- END SSH2 ENCRYPTED PRIVATE KEY ----
"""

# https://www.intel.com/content/www/us/en/security-center/pgp-public-key.html
KEY_PEM_OPENPGP_PUBLIC = b"""\
-----BEGIN PGP PUBLIC KEY BLOCK-----
Version: Encryption Desktop 10.4.2 (Build 289)

mQINBGLoNzcBEACXineYjpC9Am/qiSe1DOiCkf1hWeiGsoqMFToy3/qKmjccLUkc
90Ap5BCkrdl2zEWsYUfegZ6F+XQmlKK5nttVbP9Ezoz+Mn2SwPZgw7u/ZYpGPWv2
fYJbO/yjYTfNjgfCdWnRsYAmE6bcu7mT6MoKQ/dM//plyFEwVlzjlnceVzwekeAl
SpntCnB5nEcKnG/2c9FfD8Z0vgGcHTYkP/HffSPTcpDOADlFSnjENL1v3wi2xDo/
SRyfsTLrndkytNDyRTpVEfCduzOcAsK4/DIygucqj2Lx4kSQGpYO2lD/MV6LWH+H
NVsiOeiCREdczu6JAdnCadYqtslPL7pJ0Hg2Ji+Wo7jq8Km2p7gmFnCXHVY90nfU
z4j3nwP5Z0j3jocYd5sbks29eI0a/i+sq+JwFbfQed4PaB/onpSignyQuofEMdaF
HJeN/ld+MdeE1+xh7arzgCYY2ZVvkoEgJgE6K6N0/+teHSLKZFzTntLoeEFd0Ffa
0oiBUYNdo5di7T8XN8Q5mDa45vFBXe941Q19AaVDEsxFv7g71sZ4YEC511oT0lcQ
D/VVKtYg7kYFFmxlw9TvpKRYc0+j1sGGwxdm8s5eXGtY/eEDIfk8QI5aYnthjFbb
ZL2edcMCBHYElsR92MTb+s2c4/BKUs/q/IOoe3IkYlNUqI1H9cYoHuaLDwARAQAB
tEBJbnRlbCBQcm9kdWN0IFNlY3VyaXR5IEluY2lkZW50IFJlc3BvbnNlIFRlYW0g
PHNlY3VyZUBpbnRlbC5jb20+iQJ4BBABAgBiBQJi6Dc3BQkB3pCAMBSAAAAAACAA
B3ByZWZlcnJlZC1lbWFpbC1lbmNvZGluZ0BwZ3AuY29tcGdwbWltZQgLCQgHAwIB
CgIZAQUbAwAAAAUWAAMCAQUeAQAAAAYVCAkKAwIACgkQwOey70F1YHgnQQ/+L7n2
7cCqUtLMqwH6W1wrQxqRIhqflmSifX8qRiYwBi/EcN1Vf958W9zcqEvX6qMyX16m
SGCKC8KjpPw22XiTKuFocSEzs9pzNRc8p6AGHdISKIMUlpr5JzruvMnraSTlJZ9h
IAuXu+6/U7Y1YG2i9VAg0dSvlpa6YLQWfZ8vB7YS0txSESx4krzF683oWW+BC5jD
01pQgmjuX1fb9KAOoIQ2GLUBZ9gAorz3Dvvas30/JG7mNYI5K6cwOILO3UVu0Hx9
LsF/ut4iFkHNf6uQpcdmaOdcM7HME1K8khddGaop7NozXW7/LKMTwSGOCIRbqBCE
FiPF0NeE+a3tc6W1+ciOU2KFQJIIqplAN5B+kuUSYDH1JobivArtOIrV1tmfhF9H
Bbm30hw5XF0fLtCokpaUZtvGIjVcT9epaSGaEYdzkQJRI8jHsNQlms8E1G8QDDnl
29775JV+EMqQrOn9DdfgLeuRpMNeMpu6XvA6gZVr4OBO5m03ckjfx9uC11e84cNy
oN+5bacZRTQAUw7pOs8mUEYgtzDEomUnKpah/EY8H2QzheogZYwrj+hxGG9+JBm+
L2KRzZ7N6DevcMoGwYFPUOAslzvPlLAr0atm3yNNu2HR9vFXQNkQXDJhusfLMDWQ
q5L5rjiWVDY7CFT4Sl+ULVtO6+Dwt6QERV0rt1C0QUludGVsIFByb2R1Y3QgU2Vj
dXJpdHkgQW5ub3VuY2VtZW50IDxzZWN1cml0eS1hbm5vdW5jZUBpbnRlbC5jb20+
iQJ1BBABAgBfBQJi6DecBQkB3pCAMBSAAAAAACAAB3ByZWZlcnJlZC1lbWFpbC1l
bmNvZGluZ0BwZ3AuY29tcGdwbWltZQgLCQgHAwIBCgUbAwAAAAUWAAMCAQUeAQAA
AAYVCAkKAwIACgkQwOey70F1YHitKQ/8CbtlRKAV0d/2tMK1h1xzH6NAYF//9XsM
jW45Il05c5cIwRSybgl6YQAV+WFG24ucEwxnLpyaYPQDNHGtyM4DNZW1BA+1DE5D
A6WF8juVAw+406Uz1jmS+bg2m+lL57AMGPB1TSNR7QY/dv3McUYxSxltF61SqYvB
M/TcV5v1EirHVAjhHoAJDRCXJW19zn0lxM3ef0w2IXVbSL+1wr0mHouc4aCMHypt
NJF1Lm6qtVCWi5pQU83cqqCSwa41axMONVUKE6CI0a5DuvSHA0fOmARWBu/Wdd5A
MoE3SpiCB9LK5iaZI9rm1bL6bbj+CSnPIwEG5wS8iZNUIDqjA7l8IP34psyHWyc/
OwbHzIkZ0Bbqiyw0+g/LB6sxOjY78PqPoHYr59DTrNflMqMgmMFSt8Ug5AbkirYn
sqnIGJzoNeWfRtN2VAa3KYxdofpnunY8BklONRwMgfgjkWtdQju6x/HJSIqQoX/4
0F9E6fAQt4FrSGNQgNeR7X6rbCXs2z+bBS1gTOV4tlSsKDbycZjAjufSyrmCWD8J
r40Gq8FLY1r85kiwwksW2iZlhble9MlvKXIyvkuGxBrokhMLGGrSbsjPcL8t53TW
NtMo49Vd0o1AM8bLjcLzL0LqN8FKEmM9LnPaysu0XBt04g4piKxw7cgPVugUMP6g
xSQ5vvdcmpq5Ag0EYug3OAEQANP1PkBrX2wi4zrAaZ+1gdZjLwLQ08h+bCGgqxSr
aSShdl7al1sKl8enJwkHzr3Ry7nkJoVkAC0bR9BjnQZuvaZQTWA8QM/sCl/CQT6x
l7oBh3sXax3G+lLlT1XAh65dQYOOj0qYeTYqCcseNeykqv9vtHtxAdb/Cfwua9Hu
6mVWLYxW8pAP6kmG7WUCq6J6jaRby/JFXbkLoKBhRKSrJ6Vud3gq5gpEif3G81db
ApDCaPe/oguf2Sr24QFV4c+Q9JMoE6RN4DjFm5/bYcY3LJDRUsriIAOvL12cDNs6
Tqw5/Dy7ZYtYXN6tTF3QNfpXFUEGMMb25viOsDELV7bzm4m2fFp8tC3b0TbTDGuz
lRlZanbFT7UB7SsGZWzM9limdeuPKZ5Wh25cAfKjCP4BEUM+nga3KU3LKLGd1PXN
S56KrzTQnzVvr92mdTWBy5qb7L+iX6/g67nWLrICvnLwV/9wY37EH1dQ+9Y44PcD
50X7b0CHR3Ug0HNNapnPswZOWgyEJOtPGpBU1v1W/3+uyXE8GJpmroaUTs+ECmPM
DVEszYMALf40r/ZBm1YQ+ruiHetz511Dj0wNUAnBhLfZvU/DKp38eSzkzVNetRQt
I5FauRdAlZV4T3VDeRIz+0Xg2cD9mv/gBeyiI/ybotIcfWDos0GwBvYNMnCcYB78
GNWjABEBAAGJBEcEGAECAjEFAmLoNzkFCQHekIAFGwwAAADBXSAEGQEIAAYFAmLo
NzgACgkQdYwkt+4BKuOIwhAAn/qmTDQTE6nEEYp/u/WTpbSvKyX1G0yrLGZ15mZO
nr2tCaXlFqTMF9nc2ygXFtjyEXrfYsEaA3f6JjXCBnJttxVbiHHKAROGAn4Sod6n
ISptqmeWsMEL1NAxzN9NJJrdh/lFGUCIvr/xmOb2PBaFU0gHAMAnsS70WRfm5ud2
13nv9FEOSDF3IcqFbiHNyk26aTCO4jKErVJkdAYuq5Je3jXi5cU/yso6v+TzCn8Z
YyDP13qBXvNoJEnmItLSc/LGIaSwXhGtI/hTWEV0LgeTRlYVzfG3i5xkS15GWaAP
QE9iVQHSI9uZF/g1sKBUU5bJRZ7I41xFpfeqNB63DlsBst0QX0sR7jgIqI/Vy0SB
yctmdQ6SmG8l8I6PgwmKCV/PqHICA3ZQj04MxIoh65l0bOI5bW3YvAWheMVBf5NB
DrmxBGZd/ineeTCH981m3QbmM+9Z17NksBZMOspXsBxXlvpGioQ9Z2PPTm14PXZN
52KEEJTPKaaUbiyo7SwMyeyAg1JoNGIAn8SmIOnm4i2vLEq5dDNas+L3MG5TbDAh
209BXbYTt4QOD5Fb1UDHYZFUnnDMxv78huVcQrOsHkXSGgosuK91ebTb4M8lw9Te
kyGVJ/vwG/wx4rZSV56TrzynAep+9d2BjwXhSLilmf2kwFYac/IM6Tj6KRmQOXxX
jlUACgkQwOey70F1YHiaMg/8CkX+f/Y0SzD/Tvbj06rM5W9JLIX175PItcoSJqlq
XZ94tCs8rcporX1nmj6moxob9MyYqxQuQ1y6sBnerLw7KmxIe+6D+jsRV+lvl7Qq
DNvoeynMIipVUIzdFlxo1e9fQjgu01wV80U20HjpMc6WTU3QtixHd0fQv1PinbvY
d802JlEYlNOdJRDvu1zt25PM8+4aGbJbd5d4E0qyZcs+V8tKwbtmGchXOaPhMeq7
XpYjwYhlI54vU+HM8mcZTSqmezUUGqk0ySwgHB06I5QuJ6QktALZYbWofAnUpFW7
ZDMBIibCJb8uYzLubWZE/1BPlZxg7jgy5IEKfh9RxYdSkg6tnf5Hae1Eg6V7H8EC
Uaa4E6FUEVDQrg2oujEp0aY3LRX9VaCT62Hw+IBFSF02w28qyl/pohWWf/oGsB/J
tesUKGwv0Mpi0ZqQXGSVDqXLF7UUSbHoQOotGFcZ+J9qLnDnGYO6DAPge7xR9ZKJ
3vFqyaIoi+CdU8UBSTx6GGveuUAM44G8oqQclv6UszVznL9z6CWFY2lgG8VbaS9e
S+KBZ/Z/HZwO/j9+6fdA0di5KVd1EXtBjmmFtnYFzfm0pVF53sMfR2hIu1Jauvbc
8+OnsR0t7fjotTjCAjiBcHIMCpFJjv+AsnXZEib9z206L4I6rieXTQoaazEgJcNm
kss=
=hCPU
-----END PGP PUBLIC KEY BLOCK-----
"""

# From https://www.ietf.org/archive/id/draft-bre-openpgp-samples-01.html
KEY_PEM_OPENPGP_PRIVATE = b"""\
-----BEGIN PGP PRIVATE KEY BLOCK-----
Comment: Alice's OpenPGP Transferable Secret Key
Comment: https://www.ietf.org/id/draft-bre-openpgp-samples-01.html

lFgEXEcE6RYJKwYBBAHaRw8BAQdArjWwk3FAqyiFbFBKT4TzXcVBqPTB3gmzlC/U
b7O1u10AAP9XBeW6lzGOLx7zHH9AsUDUTb2pggYGMzd0P3ulJ2AfvQ4RtCZBbGlj
ZSBMb3ZlbGFjZSA8YWxpY2VAb3BlbnBncC5leGFtcGxlPoiQBBMWCAA4AhsDBQsJ
CAcCBhUKCQgLAgQWAgMBAh4BAheAFiEE64W7X6M6deFelE5j8jFVDE9H444FAl2l
nzoACgkQ8jFVDE9H447pKwD6A5xwUqIDprBzrHfahrImaYEZzncqb25vkLV2arYf
a78A/R3AwtLQvjxwLDuzk4dUtUwvUYibL2sAHwj2kGaHnfICnF0EXEcE6RIKKwYB
BAGXVQEFAQEHQEL/BiGtq0k84Km1wqQw2DIikVYrQrMttN8d7BPfnr4iAwEIBwAA
/3/xFPG6U17rhTuq+07gmEvaFYKfxRB6sgAYiW6TMTpQEK6IeAQYFggAIBYhBOuF
u1+jOnXhXpROY/IxVQxPR+OOBQJcRwTpAhsMAAoJEPIxVQxPR+OOWdABAMUdSzpM
hzGs1O0RkWNQWbUzQ8nUOeD9wNbjE3zR+yfRAQDbYqvtWQKN4AQLTxVJN5X5AWyb
Pnn+We1aTBhaGa86AQ==
=n8OM
-----END PGP PRIVATE KEY BLOCK-----
"""

# From https://datatracker.ietf.org/doc/html/rfc7468#section-5.2
CERT_PEM_CERTIFICATE_EXPLANATORY = b"""\
Subject: CN=Atlantis
Issuer: CN=Atlantis
Validity: from 7/9/2012 3:10:38 AM UTC to 7/9/2013 3:10:37 AM UTC
-----BEGIN CERTIFICATE-----
MIIBmTCCAUegAwIBAgIBKjAJBgUrDgMCHQUAMBMxETAPBgNVBAMTCEF0bGFudGlz
MB4XDTEyMDcwOTAzMTAzOFoXDTEzMDcwOTAzMTAzN1owEzERMA8GA1UEAxMIQXRs
YW50aXMwXDANBgkqhkiG9w0BAQEFAANLADBIAkEAu+BXo+miabDIHHx+yquqzqNh
Ryn/XtkJIIHVcYtHvIX+S1x5ErgMoHehycpoxbErZmVR4GCq1S2diNmRFZCRtQID
AQABo4GJMIGGMAwGA1UdEwEB/wQCMAAwIAYDVR0EAQH/BBYwFDAOMAwGCisGAQQB
gjcCARUDAgeAMB0GA1UdJQQWMBQGCCsGAQUFBwMCBggrBgEFBQcDAzA1BgNVHQEE
LjAsgBA0jOnSSuIHYmnVryHAdywMoRUwEzERMA8GA1UEAxMIQXRsYW50aXOCASow
CQYFKw4DAh0FAANBAKi6HRBaNEL5R0n56nvfclQNaXiDT174uf+lojzA4lhVInc0
ILwpnZ1izL4MlI9eCSHhVQBHEp2uQdXJB+d5Byg=
-----END CERTIFICATE-----
"""
