/* $Id: match.c,v 1.4 2000/10/10 13:02:57 keiji Exp $ */
/*
** signature.c : signature part of pakemon.
**
** Pakemon is an abbreviation of "Packet Monster", a simple packet
** monitoring misuse detector.
**
** Copyright (C) 1999, 2000 Keiji Takeda <keiji@sfc.keio.ac.jp>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pakemon.h"
#include "match.h"
#include "util.h"

/***** main function for local testing 

int main(int args, char **argv)
{

  u_char *delta1, *delta2;

  int i;

  u_char payload[] = "hogehogekeiji";

  u_char signature[] = "keiji"; 

#ifdef DEBUG
  printf("payload=%s(%d)\n", payload, sizeof(payload)-1 );

  printf("signature=%s(%d)\n", signature, sizeof(signature)-1 );

  printf("qd_match = %d\n", qd_match(payload, sizeof(payload)-1, signature, sizeof(signature)-1 ) );
#endif

  delta1 = bm_init_delta1( signature, sizeof(signature)-1 );

  delta2 = bm_init_delta2( signature, sizeof(signature)-1 );

#ifdef DEBUG
  for ( i = 0; i < sizeof( signature )-1; i++)
    {
      printf("delta2[%d](%c) = %d\n", i, signature[i], delta2[i]);
    }

  printf("bm_match = %d\n", bm_match(payload, sizeof(payload)-1, signature, sizeof(signature)-1 , delta1, delta2 ));
#endif

  free_check(delta1);

  free_check(delta2);

}

******/

/************************************************
  Function: qd_match
  Purpose: Quadratic Algorithm Pattern Matching 
  Argument:
   u_char * data
   int data_length
   u_char * pattern
   int pattern_length
  Returns:
   TRUE :found
   FALSE : not found
 ***********************************************/
int qd_match ( u_char * data, int data_length, u_char * pattern, int pattern_length )
{

  int i; /* loop counter */

  u_char *i_data = data; /* index pointer to check */

  for ( i = data_length; i >= pattern_length; i-- )
    {

      if( !memcmp( i_data, pattern, pattern_length ) )
	{
	  return TRUE;
	}

      i_data++;

    }/* for */

  return FALSE;

}/* pattern_match */

/************************************************
  Function: bm_match
  Purpose: Boyer-Moore Algorithm Pattern Matching 
  Argument:
   u_char * data
   int data_length
   u_char * pattern
   int pattern_length
  Returns:
   TRUE :found
   FALSE : not found
 ***********************************************/
int bm_match ( u_char * data, int data_length, u_char * pattern, int pattern_length, u_char *delta1, u_char *delta2 )
{

  int i; /* data index */

  int j; /* pattern index */

  int d1, d2; /* values from delta table of BM */
#ifdef DEBUG
  int k;
#endif
  i = pattern_length -1; /* from last char */
    
  while( i >= 0 && i < data_length ){

    j = pattern_length -1;

#ifdef DEBUG
    printf("i=%d\n", i);
    printf("[pattern]=");
    for( k=0; k <= i - pattern_length; k++)
      putchar('.');
    printf("%s\n", pattern );
    printf("[d a t a]=");
    for( k=0; k <= data_length; k++)
      {
	if( isprint(data[k]) )
	  putchar(data[k]);
	else
	  putchar('.');
      }
#endif

    while( data[i] == pattern[j] )
      {
	if( j == 0)

	  return TRUE;

	/* if you want return position of matched data

	  return i + 1; 

	*/

	i--; j--;
      }

    d1 = delta1[ data[i] ];

    d2 = delta2[j];

#ifdef DEBUG
    printf("i=%d, j=%d, d1=%d, d2=%d\n", i, j, d1, d2 );
#endif
    if (d1 > d2)
      i += d1;
    else
      i += d2;

    continue;
  }

  return FALSE;
}

/************************************************
  Function: bm_init_delta1
  Purpose: Genearate Delta1 Table for 
           Boyer-Moore Algorithm Pattern Matching 
  Argument:
   u_char * pattern
   int pattern_length
  Returns:
   int * : Delta1 Table for BM
 ***********************************************/
u_char *bm_init_delta1( u_char *pattern, u_char pattern_length )
{
  int i;

  u_char *delta1;

  if( pattern_length <= 0 )
      return (u_char *)NULL;

  if( pattern_length > MAX_PATTERN_SIZE)
    {
      fprintf(stderr, "Too large patttern for delta1.\n");
      final( 0 ); /* failure */ 
    }

  delta1 = (u_char *)malloc_check( 256 * sizeof(u_char), "bm_init_delta1:delta1" );

  if ( delta1 == (u_char *)NULL )
    {
      fprintf(stderr, "Memory allocation error for delta1\n");
      final( 0 );
    }
  
  for( i = 0; i < 256; i++)
    delta1[i] = pattern_length;

  for( i = 0; i < pattern_length; i++ )
    {
#ifdef DEBUG
      printf("delta1[%c]=%d\n", pattern[i], pattern_length -i -1);
#endif
      delta1[ pattern[i] ] = pattern_length - i -1;
    }

  return delta1;

}

/************************************************
  Function: bm_init_delta2
  Purpose: Genearate Delta2 Table for 
           Boyer-Moore Algorithm Pattern Matching 
  Argument:
   u_char * pattern
   int pattern_length
  Returns:
   u_char * : Delta2 Table for BM
 ***********************************************/
u_char *bm_init_delta2( u_char *pattern, u_char pattern_length )
{

    int i_delta, j_pattern, k_pattern, reader1, reader2;

    u_char last_char;

    u_char *delta2;

    if( pattern_length == 0 )
      return (u_char *)NULL;

    delta2 = (u_char *)malloc_check( pattern_length * sizeof(u_char) + 1, "bm_init_delta2" );

    if ( delta2 == (u_char *)NULL )
      {
	fputs( "Memory allocation error for delta2.",stderr );
	final( 0 );
      }

    i_delta = j_pattern = pattern_length -1;
    last_char = pattern[pattern_length - 1]; /* the last letter */

    last_char = pattern[pattern_length - 1]; /* the last letter */

    delta2[i_delta] = 0; /* for the last char */

    while (i_delta-- > 0)
      {

	k_pattern = pattern_length - 3; 

        do
        {
	  while ( k_pattern >= 0 && pattern[k_pattern--] != last_char);

	  reader1 = pattern_length - 2;
	  reader2 = k_pattern;

	  while ( reader2 >= 0 && pattern[reader2--] == pattern[reader1--] 
		 && reader1 >= j_pattern);

        }
        while ( reader2 >= 0 && reader1 >= j_pattern);

	delta2[i_delta] = pattern_length - i_delta + reader1 - reader2 -1;

#ifdef DEBUG

	printf("delta2[%d](%c)=%d\n", i_delta, pattern[i_delta], delta2[i_delta]);
#endif

        j_pattern--;
    }
    return delta2;
}     







