#pragma once

// Author: Torarin Hals Bakke (2012)

// Implementation of the scope guard idiom
// (http://www.drdobbs.com/cpp/generic-change-the-way-you-write-excepti/184403758),
// including a SCOPE_EXIT macro a la the D programming language.

// Boost Software License - Version 1.0 - August 17th, 2003

// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:

// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#include <cstdlib>
#include <exception>
#include <utility>
#include <type_traits>
#include <typeinfo>
#include <boost/config.hpp>

#include "utils/log.h"

template<typename T>
class Scope_guard {
  T fun_;

public:
  Scope_guard& operator=(const Scope_guard&) = delete;

  // This copy constructor cannot be deleted due to the SCOPE_EXIT macro,
  // but the compiler should elide it
  Scope_guard(const Scope_guard&);

  explicit Scope_guard(const T& fun): fun_(fun) {}
  explicit Scope_guard(T&& fun): fun_(std::move(fun)) {}

  ~Scope_guard() BOOST_NOEXCEPT {
    try {
      fun_();
    } catch (std::exception& e) {
      CLog::Log(LOGSEVERE, "Scope_guard exit function threw %s (%s)",
                typeid(e).name(), e.what());
#ifndef NDEBUG
      printf("Scope_guard exit function threw %s (%s)\n",
             typeid(e).name(), e.what());
      abort();
#endif
    } catch (...) {
      CLog::Log(LOGSEVERE, "Scope_guard exit function threw an object unrelated to std::exception");
#ifndef NDEBUG
      printf("Scope_guard exit function threw an object unrelated to std::exception\n");
      abort();
#endif
    }
  }
};

struct SCOPE_EXIT_tag {};

template <typename T>
Scope_guard<typename std::decay<T>::type> operator+(SCOPE_EXIT_tag, T&& fun) {
  return Scope_guard<typename std::decay<T>::type>(std::forward<T>(fun));
}

// Usage:
//   SCOPE_EXIT {
//     ...
//   };

#define SCOPE_EXIT \
  auto BOOST_JOIN(scope_guard_, __LINE__) = SCOPE_EXIT_tag{} + [&]
