/* -*- Mode: C; c-file-style: "gnu" -*-
   random.c -- native methods for java/io/RandomAccessFile.
   Created: Chris Toshok <toshok@hungry.com>, 6-Feb-1998.
 */
/*
  This file is part of Japhar, the GNU Virtual Machine for Java Bytecodes.
  Japhar is a project of The Hungry Programmers, GNU, and OryxSoft.

  Copyright (C) 1998, 1999 The Hungry Programmers

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include "jni.h"
#include "interp.h"
#include "exceptions.h" /* For throw_Exception() */
#include "common.h"
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <assert.h>

#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif
#ifdef HAVE_IO_H
#  include <io.h>
#endif

JNIEXPORT void JNICALL
Java_java_io_RandomAccessFile_open(JNIEnv *env,
				   jobject obj,
				   jobject name,
				   jboolean writeable)
{
  int flags = O_RDONLY;
  if (writeable)
    flags = O_RDWR | O_CREAT
#ifdef O_LARGEFILE
      | O_LARGEFILE
#endif
      ;
  file_open(env, obj, name, flags);
}

JNIEXPORT jint JNICALL
Java_java_io_RandomAccessFile_read(JNIEnv *env,
				   jobject obj)
{
  return file_read(env, obj);
}

JNIEXPORT jint JNICALL
Java_java_io_RandomAccessFile_readBytes(JNIEnv *env,
					jobject obj,
					jbyteArray array,
					jint offset,
					jint length)
{
  return file_readBytes(env, obj, array, offset, length);
}

JNIEXPORT void JNICALL
Java_java_io_RandomAccessFile_write(JNIEnv *env,
				    jobject obj,
				    jint byte)
{
  file_write(env, obj, byte);
}

JNIEXPORT void JNICALL
Java_java_io_RandomAccessFile_writeBytes(JNIEnv *env,
					 jobject obj,
					 jbyteArray array,
					 jint offset,
					 jint length)
{
  file_writeBytes(env, obj, array, offset, length);
}

JNIEXPORT jlong JNICALL
Java_java_io_RandomAccessFile_getFilePointer(JNIEnv *env,
					     jobject obj)
{
  int fd = get_file_descriptor(env, obj);
  off_t ret_val;

  if (fd == -1)
    {
      throw_Exception(env, "java/io/IOException", "getFilePointer() called on unopened file.");
      return - 1;
    }

  ret_val = lseek(fd, 0, SEEK_CUR);

  assert( sizeof(off_t) <= sizeof(jlong) );
  
  if (-1 == ret_val) {
    switch (errno) {
    case EBADF:
    case EINVAL:
    case ESPIPE:
    default: {
      throw_Exception(env, "java/io/IOException", "lseek() failed");
      break;
    }
    }

  }
  return (jlong)ret_val;
}

JNIEXPORT void JNICALL
Java_java_io_RandomAccessFile_seek(JNIEnv *env,
				   jobject obj,
				   jlong offset)
{
  int fd = get_file_descriptor(env, obj);

  if (fd == -1)
    {
      throw_Exception(env, "java/io/IOException", "seek() called on unopened file.");
      return;
    }

  if (-1 == lseek(fd, (off_t)offset, SEEK_SET) ) {
    switch (errno) {
    case EBADF:
    case EINVAL:
    case ESPIPE:
    default: {
      throw_Exception(env, "java/io/IOException", "lseek() failed");
      break;
    }
    }
  }
}

JNIEXPORT jlong JNICALL
Java_java_io_RandomAccessFile_length(JNIEnv *env,
				     jobject obj)
{
  int fd = get_file_descriptor(env, obj);
  struct stat stat_buf;
  int res = -1;

  if (fd == -1)
    {
      throw_Exception(env, "java/io/IOException", "length() called on unopened file.");
      return -1;
    }

  res = fstat(fd, &stat_buf);
  if (-1 == res)
    {
      throw_Exception(env, "java/io/IOException", "fstat() failed");
      return -1;
    }

  assert( sizeof(stat_buf.st_size) <= sizeof(jlong) );

  return (jlong)stat_buf.st_size;
}

JNIEXPORT void JNICALL
Java_java_io_RandomAccessFile_close(JNIEnv *env,
				    jobject obj)
{
  file_close(env, obj);
}

/* JDK1.2 */
JNIEXPORT void JNICALL
Java_java_io_RandomAccessFile_setLength(JNIEnv *env,
					jobject obj,
					jlong new_length)
{
  (*env)->FatalError(env, "Java_java_io_RandomAccessFile_setLength not implemented");
}

JNIEXPORT void JNICALL
Java_java_io_RandomAccessFile_initIDs(JNIEnv *env,
				      jclass cls)
{
  (*env)->FatalError(env, "Java_java_io_RandomAccessFile_initIDs not implemented");
}
