---
id: cache
title: Cache
---

Cache middleware for [Fiber](https://github.com/gofiber/fiber) designed to intercept responses and cache them. This middleware will cache the `Body`, `Content-Type` and `StatusCode` using the `c.Path()` as unique identifier. Special thanks to [@codemicro](https://github.com/codemicro/fiber-cache) for creating this middleware for Fiber core!

Request Directives<br />
`Cache-Control: no-cache` will return the up-to-date response but still caches it. You will always get a `miss` cache status.<br />
`Cache-Control: no-store` will refrain from caching. You will always get the up-to-date response.

## Signatures

```go
func New(config ...Config) fiber.Handler
```

## Examples

Import the middleware package that is part of the Fiber web framework

```go
import (
    "github.com/gofiber/fiber/v2"
    "github.com/gofiber/fiber/v2/middleware/cache"
)
```

After you initiate your Fiber app, you can use the following possibilities:

```go
// Initialize default config
app.Use(cache.New())

// Or extend your config for customization
app.Use(cache.New(cache.Config{
    Next: func(c *fiber.Ctx) bool {
        return c.Query("refresh") == "true"
    },
    Expiration: 30 * time.Minute,
    CacheControl: true,
}))
```

Or you can custom key and expire time like this:

```go
app.Use(cache.New(cache.Config{
    ExpirationGenerator: func(c *fiber.Ctx, cfg *cache.Config) time.Duration {
        newCacheTime, _ := strconv.Atoi(c.GetRespHeader("Cache-Time", "600"))
        return time.Second * time.Duration(newCacheTime)
    },
    KeyGenerator: func(c *fiber.Ctx) string {
		return utils.CopyString(c.Path())
    },
}))

app.Get("/", func(c *fiber.Ctx) error {
    c.Response().Header.Add("Cache-Time", "6000")
    return c.SendString("hi")
})
```

## Config

```go
// Config defines the config for middleware.
type Config struct {
    // Next defines a function to skip this middleware when returned true.
    //
    // Optional. Default: nil
    Next func(c *fiber.Ctx) bool

    // Expiration is the time that an cached response will live
    //
    // Optional. Default: 1 * time.Minute
    Expiration time.Duration

	// CacheHeader header on response header, indicate cache status, with the following possible return value
	//
	// hit, miss, unreachable
	//
	// Optional. Default: X-Cache
	CacheHeader string

    // CacheControl enables client side caching if set to true
    //
    // Optional. Default: false
    CacheControl bool

    // Key allows you to generate custom keys, by default c.Path() is used
    //
    // Default: func(c *fiber.Ctx) string {
    //   return utils.CopyString(c.Path())
    // }
    KeyGenerator func(*fiber.Ctx) string

    // allows you to generate custom Expiration Key By Key, default is Expiration (Optional)
    //
    // Default: nil
    ExpirationGenerator func(*fiber.Ctx, *Config) time.Duration

    // Store is used to store the state of the middleware
    //
    // Default: an in memory store for this process only
    Storage fiber.Storage

    // allows you to store additional headers generated by next middlewares & handler
    //
    // Default: false
    StoreResponseHeaders bool

    // Max number of bytes of response bodies simultaneously stored in cache. When limit is reached,
    // entries with the nearest expiration are deleted to make room for new.
    // 0 means no limit
    //
    // Default: 0
    MaxBytes uint

    // You can specify HTTP methods to cache.
    // The middleware just caches the routes of its methods in this slice.
    //
    // Default: []string{fiber.MethodGet, fiber.MethodHead}
    Methods []string
}
```

## Default Config

```go
var ConfigDefault = Config{
    Next:         nil,
    Expiration:   1 * time.Minute,
	CacheHeader:  "X-Cache",
    CacheControl: false,
    KeyGenerator: func(c *fiber.Ctx) string {
        return utils.CopyString(c.Path())
    },
    ExpirationGenerator:  nil,
    StoreResponseHeaders: false,
    Storage:              nil,
    MaxBytes:             0,
    Methods: []string{fiber.MethodGet, fiber.MethodHead},
}
```
