package util

import (
	"fmt"
	"strconv"
	"strings"
	"time"

	"github.com/jaswdr/faker"
)

// To avoid polluting init(), we initialize this the first time Fake() is
// called.
var fakeCache FakeOptions
var fakeCacheReady bool = false

// GetDefaultFakeOptions returns a set of reasonable default options to use
// with the faker API.
//
// * The maximum time is 256 years after 1970-01-01T00:00:00+00:00.
// * The minimum time is 256 years before 1970-01-01T00:00:00+00:00.
// * The maximum length for binary strings is 32.
// * Directory lengths are chosen to be random values between 0 and 7.
//
// These date values are intentionally chosen to be earlier than 1900 (should
// catch date errors with 2-digit base 10 years and old Excel versions, among
// other incorrectly written software) and after the Y2038 bug (should catch
// date errors with old *NIX/libc) but also be relatively close to the present
// day, that make sense to reason about. Production software should be able to
// handle any dates in this range with ease.
func GetDefaultFakeOptions() FakeOptions {

	// Just to convince ourselves: https://play.golang.com/p/aXfIurffjuh
	//
	//	package main
	//
	//	import (
	//		"fmt"
	//		"time"
	//	)
	//
	//	func main() {
	//		t, err := time.Parse(time.RFC3339, "1970-01-01T00:00:00+00:00")
	//		fmt.Printf("t=+%v err=%s sec=%d nsec=%d\n", t, err, t.Unix(), t.UnixNano())
	//
	//		// exactly 256 years
	//		d, err := time.ParseDuration("2242560h0m0s")
	//		fmt.Printf("d=+%v err=%s s=%s\n", d, err, d.String())
	//
	//		tmax := t.Add(d)
	//		fmt.Printf("tmax=+%v err=%s sec=%d nsec=%d\n", tmax, err, tmax.Unix(), tmax.UnixNano())
	//
	//		tmin := t.Add(-1 * d)
	//		fmt.Printf("tmin=+%v err=%s sec=%d nsec=%d\n", tmin, err, tmin.Unix(), tmin.UnixNano())
	//	}
	//
	//
	// t=+1970-01-01 00:00:00 +0000 UTC err=%!s(<nil>) sec=0 nsec=0
	// d=+2242560h0m0s err=%!s(<nil>) s=2242560h0m0s
	// tmax=+2225-10-31 00:00:00 +0000 UTC err=%!s(<nil>) sec=8073216000 nsec=8073216000000000000
	// tmin=+1714-03-04 00:00:00 +0000 UTC err=%!s(<nil>) sec=-8073216000 nsec=-8073216000000000000

	// This construction of the time values means we avoid the possibility
	// of an error from Parse().
	return FakeOptions{
		Faker:                 faker.New(),
		MinTime:               time.Unix(8073216000, 8073216000000000000),
		MaxTime:               time.Unix(-8073216000, -8073216000000000000),
		BinaryStringMaxLength: 32,
		PathLength:            -7,
		LoremParagraphLength:  3,
	}
}

// Fake provides a simpler calling convention for fake data generation with
// reasonable defaults are assumed for faker functions that require arguments,
// as described on the GetDefaultFakeOptions() function.
//
// Note that this function's seed cannot be changed. If you need a custom seed,
// you should instantiate a FakeOptions object and use that.
func Fake(kind string) (interface{}, error) {
	if fakeCacheReady == false {
		fakeCache = GetDefaultFakeOptions()
		fakeCacheReady = true

	}

	return fakeCache.Fake(kind)
}

// FakeOptions reprints the entire domain for all faker functions that are
// wrapped by this API.
type FakeOptions struct {
	// Faker is the faker.Faker instance to use for calls to .Fake().
	Faker faker.Faker

	// MinTime is the time to use for faker functions that accept a minimum time.
	MinTime time.Time

	// Maxtime is the time to use for faker functions that accept a maximum time.
	MaxTime time.Time

	// BinaryStringMaxLength is The maximum length of binary strings to generate.
	BinaryStringMaxLength int

	// PathLength is length to use for directory or file paths. If the
	// value is negative, then the directory length is instead randomized
	// each call to be between `0` and `-1 * PathLength`, inclusive.
	PathLength int

	// LoremParagraphLength is used as the value for nbParagraph in calls
	// to the Lorem API.
	LoremParagraphLength int
}

// Fake is a wrapper around github.com/jaswdr/faker. It's in the util package
// so that it can be exposed in rq's string templating functions, as well as in
// it's builtin repertoire.
//
// The available `kind` strings correspond closely to faker functions,
// typically by converting to lower snake case, and eliminating spaces and
// `()`. For example, `Faker.Address.City()` corresponds to the kind
// `address.city`. Some of the kinds have aliases, such as `city` being
// equivalent to `address.city`.
//
// The suggested way to determine the available `kind` values is to read the
// source code of this function.
func (o FakeOptions) Fake(kind string) (interface{}, error) {
	pathLength := o.PathLength
	if pathLength < 0 {
		pathLength = o.Faker.IntBetween(0, -1*pathLength)
	}

	kind = strings.ToLower(kind)

	// Recursive calls are explicitly avoided for performance. I'm hoping
	// that with newer Go versions, the compiler will optimize this to a
	// jump table.
	switch kind {
	case "address":
		return strings.TrimPrefix(o.Faker.Address().Address(), "%"), nil
	case "address.building_number":
		return strings.TrimPrefix(o.Faker.Address().BuildingNumber(), "%"), nil
	case "city":
		return o.Faker.Address().City(), nil
	case "address.city":
		return o.Faker.Address().City(), nil
	case "address.city_prefix":
		return o.Faker.Address().CityPrefix(), nil
	case "address.city_suffix":
		return o.Faker.Address().CitySuffix(), nil
	case "address.country":
		return o.Faker.Address().Country(), nil
	case "address.country_abbr":
		return o.Faker.Address().CountryAbbr(), nil
	case "address.country_code":
		return o.Faker.Address().CountryCode(), nil
	case "address.latitude":
		return o.Faker.Address().Latitude(), nil
	case "address.longitude":
		return o.Faker.Address().Longitude(), nil
	case "address.post_code":
		return o.Faker.Address().PostCode(), nil
	case "address.secondary_address":
		return o.Faker.Address().SecondaryAddress(), nil
	case "address.state":
		return o.Faker.Address().State(), nil
	case "address.state_abbr":
		return o.Faker.Address().StateAbbr(), nil
	case "address.street_address":
		return strings.TrimPrefix(o.Faker.Address().StreetAddress(), "%"), nil
	case "address.street_name":
		return o.Faker.Address().StreetName(), nil
	case "address.street_suffix":
		return o.Faker.Address().StreetSuffix(), nil

	case "app":
		return o.Faker.App().Name(), nil
	case "app.name":
		return o.Faker.App().Name(), nil
	case "app.version":
		return o.Faker.App().Version(), nil

	case "beer":
		return o.Faker.Beer().Name(), nil
	case "beer.alcohol":
		return o.Faker.Beer().Alcohol(), nil
	case "beer.blg":
		return o.Faker.Beer().Blg(), nil
	case "beer.hop":
		return o.Faker.Beer().Hop(), nil
	case "beer.ibu":
		return o.Faker.Beer().Ibu(), nil
	case "beer.malt":
		return o.Faker.Beer().Malt(), nil
	case "beer.name":
		return o.Faker.Beer().Name(), nil
	case "beer.style":
		return o.Faker.Beer().Style(), nil

	case "binary_string":
		return o.Faker.BinaryString().BinaryString(o.BinaryStringMaxLength), nil
	case "binary_string.binary_string":
		return o.Faker.BinaryString().BinaryString(o.BinaryStringMaxLength), nil

	case "blood":
		return o.Faker.Blood().Name(), nil
	case "blood.name":
		return o.Faker.Blood().Name(), nil

	case "bool":
		return o.Faker.Boolean().Bool(), nil
	case "boolean":
		return o.Faker.Boolean().Bool(), nil
	case "boolean.bool":
		return o.Faker.Boolean().Bool(), nil
	case "boolean.bool_int":
		return o.Faker.Boolean().BoolInt(), nil

	case "car":
		return o.Faker.Car().Model(), nil
	case "car.category":
		return o.Faker.Car().Category(), nil
	case "car.fuel_type":
		return o.Faker.Car().FuelType(), nil
	case "car.maker":
		return o.Faker.Car().Maker(), nil
	case "car.model":
		return o.Faker.Car().Model(), nil
	case "car.plate":
		return o.Faker.Car().Plate(), nil
	case "car.transmission_gear":
		return o.Faker.Car().TransmissionGear(), nil

	case "color":
		return o.Faker.Color().ColorName(), nil
	case "color.css":
		return o.Faker.Color().CSS(), nil
	case "color.color_name":
		return o.Faker.Color().ColorName(), nil
	case "color.hex":
		return o.Faker.Color().Hex(), nil
	case "color.rgb":
		return o.Faker.Color().RGB(), nil
	case "color.rgb_as_array":
		rgb := o.Faker.Color().RGBAsArray()
		return "[" + rgb[0] + "," + rgb[1] + "," + rgb[2] + "]", nil
	case "color.safe_color_name":
		return o.Faker.Color().SafeColorName(), nil

	case "company":
		return o.Faker.Company().Name(), nil
	case "company.bs":
		return o.Faker.Company().BS(), nil
	case "company.catch_phrase":
		return o.Faker.Company().CatchPhrase(), nil
	case "company.ein":
		return o.Faker.Company().EIN(), nil
	case "company.job_title":
		return o.Faker.Company().JobTitle(), nil
	case "company.name":
		return o.Faker.Company().Name(), nil
	case "company.suffix":
		return o.Faker.Company().Suffix(), nil

	case "crypto.bech32_address":
		return o.Faker.Crypto().Bech32Address(), nil
	case "crypto.bitcoin_address":
		return o.Faker.Crypto().BitcoinAddress(), nil
	case "crypto.etherium_address":
		return o.Faker.Crypto().EtheriumAddress(), nil
	case "crypto.p2pkh_address":
		return o.Faker.Crypto().P2PKHAddress(), nil
	case "crypto.p2sh_address":
		return o.Faker.Crypto().P2SHAddress(), nil

	case "money":
		return strconv.Itoa(o.Faker.Currency().Number()) + " " + o.Faker.Currency().Code(), nil
	case "currency":
		curr, code := o.Faker.Currency().CurrencyAndCode()
		return curr + " " + code, nil
	case "currency.code":
		return o.Faker.Currency().Code(), nil
	case "currency.country":
		return o.Faker.Currency().Country(), nil
	case "currency.currency":
		return o.Faker.Currency().Currency(), nil
	case "currency.currency_and_code":
		curr, code := o.Faker.Currency().CurrencyAndCode()
		return curr + " " + code, nil
	case "currency.number":
		return o.Faker.Currency().Number(), nil

	case "dir":
		return o.Faker.Directory().Directory(pathLength), nil
	case "directory":
		return o.Faker.Directory().Directory(pathLength), nil
	case "directory.directory":
		return o.Faker.Directory().Directory(pathLength), nil
	case "directory.drive_letter":
		return o.Faker.Directory().DriveLetter(), nil
	case "directory.unix_directory":
		return o.Faker.Directory().UnixDirectory(pathLength), nil
	case "directory.windows_directory":
		return o.Faker.Directory().WindowsDirectory(pathLength), nil

	case "emoji":
		return o.Faker.Emoji().Emoji(), nil
	case "emoji.emoji":
		return o.Faker.Emoji().Emoji(), nil
	case "emoji.emoji_code":
		return o.Faker.Emoji().EmojiCode(), nil

	case "file":
		return o.Faker.File().AbsoluteFilePath(pathLength), nil
	case "file.absolute_file_path":
		return o.Faker.File().AbsoluteFilePath(pathLength), nil
	case "abspath":
		return o.Faker.File().AbsoluteFilePath(pathLength), nil
	case "file.absolute_file_path_unix":
		return o.Faker.File().AbsoluteFilePathForUnix(pathLength), nil
	case "file.absolute_file_path_for_unix":
		return o.Faker.File().AbsoluteFilePathForUnix(pathLength), nil
	case "file.absolute_file_path_for_windows":
		return o.Faker.File().AbsoluteFilePathForWindows(pathLength), nil
	case "file.extension":
		return o.Faker.File().Extension(), nil
	case "file.filename_with_extension":
		return o.Faker.File().FilenameWithExtension(), nil

	case "food.fruit":
		return o.Faker.Food().Fruit(), nil
	case "food.vegetable":
		return o.Faker.Food().Vegetable(), nil

	case "gamer.tag":
		return o.Faker.Gamer().Tag(), nil

	case "gender.abbr":
		return o.Faker.Gender().Abbr(), nil
	case "gender.name":
		return o.Faker.Gender().Name(), nil

	case "genre.name":
		return o.Faker.Genre().Name(), nil

	case "hash.md5":
		return o.Faker.Hash().MD5(), nil
	case "hash.sha256":
		return o.Faker.Hash().SHA256(), nil
	case "hash.sha512":
		return o.Faker.Hash().SHA512(), nil

	case "internet.company_email":
		return o.Faker.Internet().CompanyEmail(), nil
	case "internet.domain":
		return o.Faker.Internet().Domain(), nil
	case "email":
		return o.Faker.Internet().SafeEmail(), nil
	case "internet.email":
		return o.Faker.Internet().Email(), nil
	case "internet.free_email":
		return o.Faker.Internet().FreeEmail(), nil
	case "internet.free_email_domain":
		return o.Faker.Internet().FreeEmailDomain(), nil
	case "internet.http_method":
		return o.Faker.Internet().HTTPMethod(), nil
	case "ipv4":
		return o.Faker.Internet().Ipv4(), nil
	case "internet.ipv4":
		return o.Faker.Internet().Ipv4(), nil
	case "ipv6":
		return o.Faker.Internet().Ipv6(), nil
	case "internet.ipv6":
		return o.Faker.Internet().Ipv6(), nil
	case "internet.local_ipv4":
		return o.Faker.Internet().LocalIpv4(), nil
	case "mac":
		return o.Faker.Internet().MacAddress(), nil
	case "internet.mac_address":
		return o.Faker.Internet().MacAddress(), nil
	case "password":
		return o.Faker.Internet().Password(), nil
	case "internet.password":
		return o.Faker.Internet().Password(), nil
	case "internet.query":
		return o.Faker.Internet().Query(), nil
	case "internet.safe_email":
		return o.Faker.Internet().SafeEmail(), nil
	case "internet.safe_email_domain":
		return o.Faker.Internet().SafeEmailDomain(), nil
	case "internet.slug":
		return o.Faker.Internet().Slug(), nil
	case "status_code":
		return o.Faker.Internet().StatusCode(), nil
	case "internet.status_code":
		return o.Faker.Internet().StatusCode(), nil
	case "internet.status_code_message":
		return o.Faker.Internet().StatusCodeMessage(), nil
	case "internet.status_code_with_message":
		return o.Faker.Internet().StatusCodeWithMessage(), nil
	case "internet.tld":
		return o.Faker.Internet().TLD(), nil
	case "url":
		return o.Faker.Internet().URL(), nil
	case "internet.url":
		return o.Faker.Internet().URL(), nil
	case "user":
		return o.Faker.Internet().User(), nil
	case "internet.user":
		return o.Faker.Internet().User(), nil

	case "language":
		return o.Faker.Language().Language(), nil
	case "language.language":
		return o.Faker.Language().Language(), nil
	case "language.language_abbr":
		return o.Faker.Language().LanguageAbbr(), nil
	case "language.programming_language":
		return o.Faker.Language().ProgrammingLanguage(), nil

	case "lorem":
		return o.Faker.Lorem().Paragraph(o.LoremParagraphLength), nil
	case "lorem.paragraph":
		return o.Faker.Lorem().Paragraph(o.LoremParagraphLength), nil

	case "mime_type":
		return o.Faker.MimeType().MimeType(), nil
	case "mime_type.mime_type":
		return o.Faker.MimeType().MimeType(), nil

	case "music.author":
		return o.Faker.Music().Author(), nil
	case "music.genre":
		return o.Faker.Music().Genre(), nil
	case "music.length":
		return o.Faker.Music().Length(), nil
	case "music.name":
		return o.Faker.Music().Name(), nil

	case "payment.credit_card_expiration":
		return o.Faker.Payment().CreditCardExpirationDateString(), nil
	case "payment.credit_card_expiration_date_string":
		return o.Faker.Payment().CreditCardExpirationDateString(), nil
	case "payment.credit_card_number":
		return o.Faker.Payment().CreditCardNumber(), nil
	case "cc":
		return o.Faker.Payment().CreditCardNumber(), nil
	case "payment.credit_card_type":
		return o.Faker.Payment().CreditCardType(), nil

	case "name":
		return o.Faker.Person().Name(), nil
	case "person.first_name":
		return o.Faker.Person().FirstName(), nil
	case "person.first_name_female":
		return o.Faker.Person().FirstNameFemale(), nil
	case "person.first_name_male":
		return o.Faker.Person().FirstNameMale(), nil
	case "person.gender":
		return o.Faker.Person().Gender(), nil
	case "person.gender_female":
		return o.Faker.Person().GenderFemale(), nil
	case "person.gender_male":
		return o.Faker.Person().GenderMale(), nil
	case "person.last_name":
		return o.Faker.Person().LastName(), nil
	case "person.name":
		return o.Faker.Person().Name(), nil
	case "person.name_female":
		return o.Faker.Person().NameFemale(), nil
	case "person.name_male":
		return o.Faker.Person().NameMale(), nil
	case "person.ssn":
		return o.Faker.Person().SSN(), nil
	case "person.suffix":
		return o.Faker.Person().Suffix(), nil
	case "person.title":
		return o.Faker.Person().Title(), nil
	case "person.title_female":
		return o.Faker.Person().TitleFemale(), nil
	case "person.title_male":
		return o.Faker.Person().TitleMale(), nil

	case "pet.cat":
		return o.Faker.Pet().Cat(), nil
	case "pet.dog":
		return o.Faker.Pet().Dog(), nil
	case "pet.name":
		return o.Faker.Pet().Name(), nil

	case "phone.area_code":
		return o.Faker.Phone().AreaCode(), nil
	case "phone.e164_number":
		return o.Faker.Phone().E164Number(), nil
	case "phone.exchange_code":
		return o.Faker.Phone().ExchangeCode(), nil
	case "phone":
		return o.Faker.Phone().Number(), nil
	case "phone.number":
		return o.Faker.Phone().Number(), nil
	case "phone.toll_free_area_code":
		return o.Faker.Phone().TollFreeAreaCode(), nil
	case "phone.tool_free_number":
		return o.Faker.Phone().ToolFreeNumber(), nil

	case "time":
		return o.Faker.Time().RFC3339(o.MaxTime), nil
	case "time.ansic":
		return o.Faker.Time().ANSIC(o.MaxTime), nil
	case "time.ampm":
		return o.Faker.Time().AmPm(), nil
	case "time.century":
		return o.Faker.Time().Century(), nil
	case "time.day_of_month":
		return o.Faker.Time().DayOfMonth(), nil
	case "time.day_of_week":
		return o.Faker.Time().DayOfWeek(), nil
	case "time.iso8601":
		return o.Faker.Time().ISO8601(o.MaxTime), nil
	case "time.kitchen":
		return o.Faker.Time().Kitchen(o.MaxTime), nil
	case "time.month":
		return o.Faker.Time().Month(), nil
	case "time.month_name":
		return o.Faker.Time().MonthName(), nil
	case "rfc3339":
		return o.Faker.Time().RFC3339(o.MaxTime), nil
	case "time.rfc3339":
		return o.Faker.Time().RFC3339(o.MaxTime), nil
	case "rfc3339_nano":
		return o.Faker.Time().RFC3339Nano(o.MaxTime), nil
	case "time.rfc3339_nano":
		return o.Faker.Time().RFC3339Nano(o.MaxTime), nil
	case "rfc822":
		return o.Faker.Time().RFC822(o.MaxTime), nil
	case "time.rfc822":
		return o.Faker.Time().RFC822(o.MaxTime), nil
	case "rfc850":
		return o.Faker.Time().RFC850(o.MaxTime), nil
	case "time.rfc850":
		return o.Faker.Time().RFC850(o.MaxTime), nil
	case "time.ruby_date":
		return o.Faker.Time().RubyDate(o.MaxTime), nil
	case "time.time":
		return o.Faker.Time().Time(o.MaxTime), nil
	case "time.time_between":
		return o.Faker.Time().TimeBetween(o.MinTime, o.MaxTime), nil
	case "time.timezone":
		return o.Faker.Time().Timezone(), nil
	case "time.unix":
		return o.Faker.Time().Unix(o.MaxTime), nil
	case "time.unix_date":
		return o.Faker.Time().UnixDate(o.MaxTime), nil
	case "time.year":
		return o.Faker.Time().Year(), nil

	case "uuid":
		return o.Faker.UUID().V4(), nil
	case "uuid.v4":
		return o.Faker.UUID().V4(), nil

	case "user_agent":
		return o.Faker.UserAgent().UserAgent(), nil
	case "user_agent.chrome":
		return o.Faker.UserAgent().Chrome(), nil
	case "user_agent.firefox":
		return o.Faker.UserAgent().Firefox(), nil
	case "user_agent.ie":
		return o.Faker.UserAgent().InternetExplorer(), nil
	case "user_agent.internet_explorer":
		return o.Faker.UserAgent().InternetExplorer(), nil
	case "user_agent.opera":
		return o.Faker.UserAgent().Opera(), nil
	case "user_agent.safari":
		return o.Faker.UserAgent().Safari(), nil
	case "user_agent.user_agent":
		return o.Faker.UserAgent().UserAgent(), nil

	case "youtube.embeded_url":
		return o.Faker.YouTube().GenerateEmbededURL(), nil
	case "youtube.full_url":
		return o.Faker.YouTube().GenerateFullURL(), nil
	case "youtube.generate_share_url":
		return o.Faker.YouTube().GenerateShareURL(), nil
	case "youtube.video_id":
		return o.Faker.YouTube().GenerateVideoID(), nil

	}

	return nil, fmt.Errorf("unknown kind of fake data: %s", kind)
}
